/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <string.h>
#include "batch_importer.h"

// callback functions for DownloadImporter ---
static char* on_get (DownloadImporter* dim, gpointer data)
{
	BatchImporter* bim = data;

	if (bim->min_value > bim->max_value)
		return NULL;
	if (bim->char_value) {
		g_string_printf (bim->gstr, "%.*s%c%.*s",
		                 bim->len_pre, bim->pattern_pre,
		                 bim->min_value,
		                 bim->len_post, bim->pattern_post);
	} else {
		g_string_printf (bim->gstr, "%.*s%.*d%.*s",
		                 bim->len_pre, bim->pattern_pre,
		                 bim->value_width, bim->min_value,
		                 bim->len_post, bim->pattern_post);
	}

	bim->min_value++;
	bim->n_current++;
	download_importer_set_progress (dim, bim->n_current, bim->n_total);

	return bim->gstr->str;
}

// signal handler
static void on_batch_response (GtkDialog *dlg, gint response, gpointer data)
{
	BatchImporter* bim = data;
	gint  mode;

	if (response == GTK_RESPONSE_OK) {
		gtk_widget_hide (bim->batch_dialog->self);
		mode = batch_dialog_get_mode (bim->batch_dialog);
		bim->pattern_pre = g_strdup (batch_dialog_get_pattern (bim->batch_dialog));
		bim->pattern_post = strchr (bim->pattern_pre, '*') + 1;
		bim->len_pre  = bim->pattern_post - bim->pattern_pre - 1;
		bim->len_post = strlen (bim->pattern_pre) - bim->len_pre;
		if (bim->len_pre && bim->pattern_pre[bim->len_pre-1] == '(' &&
		    bim->pattern_post[0] == ')')
		{
			bim->len_pre--;
			bim->len_post--;
			bim->pattern_post++;
		}
		bim->char_value = (batch_dialog_get_mode (bim->batch_dialog) == BATCH_MODE_CHAR);
		if (bim->char_value) {
			bim->min_value = batch_dialog_get_char_min (bim->batch_dialog);
			bim->max_value = batch_dialog_get_char_max (bim->batch_dialog);
		} else {
			bim->min_value = batch_dialog_get_int_min (bim->batch_dialog);
			bim->max_value = batch_dialog_get_int_max (bim->batch_dialog);
			bim->value_width = batch_dialog_get_int_width (bim->batch_dialog);
		}
		bim->n_total = bim->max_value - bim->min_value + 1;
//		g_message ("%.*s", bim->len_pre, bim->pattern_pre);
//		g_message ("%.*s", bim->len_post, bim->pattern_post);
		download_importer_activate (DOWNLOAD_IMPORTER (bim));
		return;
	}

	batch_importer_destroy (bim);
}

BatchImporter* batch_importer_new (GtkWindow* parent, MainApp* app)
{
	BatchImporter* bim = g_malloc0 (sizeof (BatchImporter));
	CategoryNode* cnode = NULL;

	if (app->window.queue_ctrl)
		cnode = app->window.queue_ctrl->cnode;

	// initialize base class
	download_importer_init (DOWNLOAD_IMPORTER (bim), parent, app);
	download_dialog_set_category (DOWNLOAD_DIALOG (bim), app->rnode, cnode);
	bim->finalize  = (DownloadDialogFunc) batch_importer_finalize;
	bim->get       = on_get;
	bim->user_data = bim;

	bim->batch_dialog = batch_dialog_new (parent);
	bim->gstr = g_string_sized_new (128);

	// signals
	g_signal_connect (bim->batch_dialog->self, "response",
	                  G_CALLBACK (on_batch_response), bim);
	return bim;
}

void  batch_importer_activate (BatchImporter* bim)
{
	gtk_widget_show (bim->batch_dialog->self);
}

// protected function
void batch_importer_finalize (BatchImporter* bim)
{
	batch_dialog_destroy (bim->batch_dialog);
	g_string_free (bim->gstr, TRUE);
	g_free (bim->pattern_pre);

	download_importer_finalize (DOWNLOAD_IMPORTER (bim));
}

