/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

/*
 *  Code base on Gtk+ 2.0 Tree View tutorial 
 *      - a tutorial covering GtkTreeView by Tim-Philipp Mu"ller
 */

#include "category_tree_model.h"

/* boring declarations of local functions */

static void         category_tree_model_init            (CategoryTreeModel      *pkg_tree);

static void         category_tree_model_class_init      (CategoryTreeModelClass *klass);

static void         category_tree_model_tree_model_init (GtkTreeModelIface *iface);

static void         category_tree_model_finalize        (GObject           *object);

static GtkTreeModelFlags category_tree_model_get_flags  (GtkTreeModel      *tree_model);

static gint         category_tree_model_get_n_columns   (GtkTreeModel      *tree_model);

static GType        category_tree_model_get_column_type (GtkTreeModel      *tree_model,
                                                     gint               index);

static gboolean     category_tree_model_get_iter        (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     GtkTreePath       *path);

static GtkTreePath *category_tree_model_get_path        (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter);

static void         category_tree_model_get_value       (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     gint               column,
                                                     GValue            *value);

static gboolean     category_tree_model_iter_next       (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter);

static gboolean     category_tree_model_iter_children   (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     GtkTreeIter       *parent);

static gboolean     category_tree_model_iter_has_child  (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter);

static gint         category_tree_model_iter_n_children (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter);

static gboolean     category_tree_model_iter_nth_child  (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     GtkTreeIter       *parent,
                                                     gint               n);

static gboolean     category_tree_model_iter_parent     (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     GtkTreeIter       *child);

static GObjectClass *parent_class = NULL;  /* GObject stuff - nothing to worry about */


// Change linking of node for display
// From :
//
// RootNode
// |
// `--- CategoryNode
//      |
//      +--- QueueNode (waiting queue)
//      |
//      +--- QueueNode (completed queue)
//      |
//      `--- QueueNode (recycled queue)
//
// To :
//
// RootNode
// |
// `--- QueueNode (waiting queue)
//      |
//      +--- QueueNode (completed queue)
//      |
//      `--- QueueNode (recycled queue)
//



/*****************************************************************************
 *
 *                        here we register our new type and its interfaces
 *                        with the type system. If you want to implement
 *                        additional interfaces like GtkTreeSortable, you
 *                        will need to do it here.
 *
 *****************************************************************************/

GType category_tree_model_get_type (void)
{
	static GType category_tree_model_type = 0;

	if (category_tree_model_type)
		return category_tree_model_type;

	/* Some boilerplate type registration stuff */
	if (1) {
		static const GTypeInfo category_tree_model_info = {
			sizeof (CategoryTreeModelClass),
			NULL,                                         /* base_init */
			NULL,                                         /* base_finalize */
			(GClassInitFunc) category_tree_model_class_init,
			NULL,                                         /* class finalize */
			NULL,                                         /* class_data */
			sizeof (CategoryTreeModel),
			0,                                            /* n_preallocs */
			(GInstanceInitFunc) category_tree_model_init
		};

		category_tree_model_type = g_type_register_static (G_TYPE_OBJECT,
		                                               "CategoryTreeModel",
		                                               &category_tree_model_info,
		                                               (GTypeFlags)0);
	}

	if (1) {
		static const GInterfaceInfo tree_model_info = {
			(GInterfaceInitFunc) category_tree_model_tree_model_init,
			NULL,
			NULL
		};

		g_type_add_interface_static (category_tree_model_type,
		                             GTK_TYPE_TREE_MODEL,
		                             &tree_model_info);
	}

	return category_tree_model_type;
}


/*****************************************************************************
 *
 *                          more boilerplate GObject/GType stuff.
 *                          Init callback for the type system,
 *                          called once when our new class is created.
 *
 *****************************************************************************/

static void category_tree_model_class_init (CategoryTreeModelClass *klass)
{
	GObjectClass* object_class;

	// parent_class is a global value...
	parent_class = (GObjectClass*) g_type_class_peek_parent (klass);
	object_class = (GObjectClass*) klass;

	object_class->finalize = category_tree_model_finalize;
}

/*****************************************************************************
 *
 *                               init callback for the interface registration
 *                               in custom_list_get_type. Here we override
 *                               the GtkTreeModel interface functions that
 *                               we implement.
 *
 *****************************************************************************/

static void
category_tree_model_tree_model_init (GtkTreeModelIface *iface)
{
	iface->get_flags       = category_tree_model_get_flags;
	iface->get_n_columns   = category_tree_model_get_n_columns;
	iface->get_column_type = category_tree_model_get_column_type;
	iface->get_iter        = category_tree_model_get_iter;
	iface->get_path        = category_tree_model_get_path;
	iface->get_value       = category_tree_model_get_value;
	iface->iter_next       = category_tree_model_iter_next;
	iface->iter_children   = category_tree_model_iter_children;
	iface->iter_has_child  = category_tree_model_iter_has_child;
	iface->iter_n_children = category_tree_model_iter_n_children;
	iface->iter_nth_child  = category_tree_model_iter_nth_child;
	iface->iter_parent     = category_tree_model_iter_parent;
}


/*****************************************************************************
 *
 *  category_tree_model_init: this is called everytime a new custom list object
 *                    instance is created (we do that in category_tree_model_new).
 *                    Initialise the list structure's fields here.
 *
 *****************************************************************************/

static void category_tree_model_init (CategoryTreeModel *category_tree_model)
{
	/* Random int to check whether an iter belongs to our model */
	category_tree_model->stamp = g_random_int();
}


/*****************************************************************************
 *
 *  category_tree_model_finalize: this is called just before a custom list is
 *                        destroyed. Free dynamically allocated memory here.
 *
 *****************************************************************************/

static void category_tree_model_finalize (GObject *object)
{
	CategoryTreeModel* model = CATEGORY_TREE_MODEL (object);

	base_node_inserted_disconnect (model->node, model->inserted_signal);
	base_node_removed_disconnect (model->node, model->removed_signal);
	base_node_unref (model->node);
	(* parent_class->finalize) (object);
}


/*****************************************************************************
 *
 *  category_tree_model_get_flags: tells the rest of the world whether our tree model
 *                         has any special characteristics. In our case,
 *                         we have a list model (instead of a tree), and each
 *                         tree iter is valid as long as the row in question
 *                         exists, as it only contains a pointer to our struct.
 *
 *****************************************************************************/

static GtkTreeModelFlags
category_tree_model_get_flags (GtkTreeModel *tree_model)
{
	g_return_val_if_fail (IS_CATEGORY_TREE_MODEL(tree_model), (GtkTreeModelFlags)0);

	return GTK_TREE_MODEL_ITERS_PERSIST;
}


/*****************************************************************************
 *
 *  category_tree_model_get_n_columns: tells the rest of the world how many data
 *                             columns we export via the tree model interface
 *
 *****************************************************************************/

static gint category_tree_model_get_n_columns (GtkTreeModel *tree_model)
{
	return 1;
}


/*****************************************************************************
 *
 *  category_tree_model_get_column_type: tells the rest of the world which type of
 *                               data an exported model column contains
 *
 *****************************************************************************/

static GType
category_tree_model_get_column_type (GtkTreeModel *tree_model,
                                     gint          index)
{
	g_return_val_if_fail (IS_CATEGORY_TREE_MODEL(tree_model), G_TYPE_INVALID);
	g_return_val_if_fail (index < 1 && index >= 0, G_TYPE_INVALID);
	return G_TYPE_POINTER;
}


/*****************************************************************************
 *
 *  category_tree_model_get_iter: converts a tree path (physical position) into a
 *                        tree iter structure (the content of the iter
 *                        fields will only be used internally by our model).
 *                        We simply store a pointer to our CustomRecord
 *                        structure that represents that row in the tree iter.
 *
 *****************************************************************************/

static gboolean
category_tree_model_get_iter (GtkTreeModel *tree_model,
                              GtkTreeIter  *iter,
                              GtkTreePath  *path)
{
	CategoryTreeModel  *category_tree_model;
	gint               *indices, n, depth;
	CategoryNode*  cnode;
	BaseNode*      node;

	g_assert(IS_CATEGORY_TREE_MODEL(tree_model));
	g_assert(path!=NULL);

	category_tree_model = CATEGORY_TREE_MODEL(tree_model);

	indices = gtk_tree_path_get_indices(path);
	depth   = gtk_tree_path_get_depth(path);

	// category_tree_model only use 2 level
	g_assert(depth < 3);

	// skip first child because it display as parent
	n = (depth == 2) ? indices[1] + 1 : 0;

	cnode = (CategoryNode*)base_node_nth_child (category_tree_model->node, indices[0]);
	if (cnode) {
		node = base_node_nth_child (BASE_NODE (cnode), n);
		if (node) {
			iter->stamp      = category_tree_model->stamp;
			iter->user_data  = node;
			iter->user_data2 = NULL;   /* unused */
			iter->user_data3 = NULL;   /* unused */
			return TRUE;
		}
	}

	return FALSE;
}


/*****************************************************************************
 *
 *  category_tree_model_get_path: converts a tree iter into a tree path (ie. the
 *                        physical position of that row in the list).
 *
 *****************************************************************************/

static GtkTreePath *
category_tree_model_get_path (GtkTreeModel *tree_model,
                              GtkTreeIter  *iter)
{
	GtkTreePath  *path;
	CategoryTreeModel  *category_tree_model;
	CategoryNode* cnode;
	QueueNode*    qnode;
	int           index;

	g_return_val_if_fail (IS_CATEGORY_TREE_MODEL(tree_model), NULL);
	g_return_val_if_fail (iter != NULL,               NULL);
	g_return_val_if_fail (iter->user_data != NULL,    NULL);

	category_tree_model = CATEGORY_TREE_MODEL(tree_model);

	qnode = (QueueNode*) iter->user_data;
	cnode = queue_node_get_category (qnode);

	path = gtk_tree_path_new();
	index = base_node_child_position (category_tree_model->node, cnode);
	gtk_tree_path_append_index (path, index);

	index = base_node_child_position (cnode, qnode);
	if (index > 0)
		gtk_tree_path_append_index(path, index - 1);

	return path;
}


/*****************************************************************************
 *
 *  category_tree_model_get_value: Returns a row's exported data columns
 *                         (_get_value is what gtk_tree_model_get uses)
 *
 *****************************************************************************/

static void
category_tree_model_get_value (GtkTreeModel *tree_model,
                               GtkTreeIter  *iter,
                               gint          column,
                               GValue       *value)
{
	BaseNode*  node;
	CategoryTreeModel    *category_tree_model;

	g_return_if_fail (IS_CATEGORY_TREE_MODEL (tree_model));
	g_return_if_fail (iter != NULL);

	g_value_init (value, G_TYPE_POINTER);

	category_tree_model = CATEGORY_TREE_MODEL(tree_model);

	node = (BaseNode*) iter->user_data;

	g_value_set_pointer(value, node);
}


/*****************************************************************************
 *
 *  category_tree_model_iter_next: Takes an iter structure and sets it to point
 *                         to the next row.
 *
 *****************************************************************************/

static gboolean
category_tree_model_iter_next (GtkTreeModel  *tree_model,
                               GtkTreeIter   *iter)
{
	CategoryTreeModel    *category_tree_model;
	CategoryNode*  cnode;
	QueueNode*     qnode;
	BaseNode*      node;
	int            index;

	g_return_val_if_fail (IS_CATEGORY_TREE_MODEL (tree_model), FALSE);

	if (iter == NULL || iter->user_data == NULL)
		return FALSE;

	category_tree_model = CATEGORY_TREE_MODEL(tree_model);

	node  = NULL;
	qnode = iter->user_data;
	cnode = queue_node_get_category (qnode);
	index = base_node_child_position (BASE_NODE (cnode), BASE_NODE (qnode));

	if (index > 0)
		node = base_node_next (BASE_NODE (qnode));
	else {
		cnode = (CategoryNode*) base_node_next (BASE_NODE (cnode));
		node = (cnode) ? base_node_first_child (BASE_NODE (cnode)) : NULL;
	}

	if (node) {
		iter->stamp     = category_tree_model->stamp;
		iter->user_data = node;
		return TRUE;
	}
	return FALSE;
}


/*****************************************************************************
 *
 *  category_tree_model_iter_children: Returns TRUE or FALSE depending on whether
 *                             the row specified by 'parent' has any children.
 *                             If it has children, then 'iter' is set to
 *                             point to the first child. Special case: if
 *                             'parent' is NULL, then the first top-level
 *                             row should be returned if it exists.
 *
 *****************************************************************************/

static gboolean
category_tree_model_iter_children (GtkTreeModel *tree_model,
                                   GtkTreeIter  *iter,
                                   GtkTreeIter  *parent)
{
	CategoryTreeModel  *category_tree_model;
	CategoryNode*  cnode;
	QueueNode*     qnode;
	int            index;

	g_return_val_if_fail (parent == NULL || parent->user_data != NULL, FALSE);
	g_return_val_if_fail (IS_CATEGORY_TREE_MODEL (tree_model), FALSE);

	category_tree_model = CATEGORY_TREE_MODEL (tree_model);

	/* parent == NULL is a special case; we need to return the first top-level row */
	if (parent) {
		qnode = parent->user_data;
		cnode = (qnode) ? queue_node_get_category (qnode) : NULL;
	} else {
		cnode = (CategoryNode*) base_node_first_child (category_tree_model->node);
		qnode = (cnode) ? category_node_waiting_queue (cnode) : NULL;
	}

	if (qnode == NULL)
		return FALSE;

	index = base_node_child_position (BASE_NODE (cnode), BASE_NODE (qnode));

	if (index==0) {
		iter->stamp     = category_tree_model->stamp;
		iter->user_data = base_node_next (BASE_NODE (qnode));
		return TRUE;
	}
	return FALSE;
}


/*****************************************************************************
 *
 *  category_tree_model_iter_has_child: Returns TRUE or FALSE depending on whether
 *                              the row specified by 'iter' has any children.
 *                              We only have a list and thus no children.
 *
 *****************************************************************************/

static gboolean
category_tree_model_iter_has_child (GtkTreeModel *tree_model,
                                    GtkTreeIter  *iter)
{
	CategoryNode*  cnode;
	QueueNode*     qnode;

	g_return_val_if_fail (iter == NULL || iter->user_data != NULL, FALSE);
	g_return_val_if_fail (IS_CATEGORY_TREE_MODEL (tree_model), FALSE);

	qnode = iter->user_data;
	cnode = queue_node_get_category (qnode);

	if (base_node_child_position (BASE_NODE (cnode), BASE_NODE (qnode)) == 0) {
		if ( base_node_next (BASE_NODE (qnode)) )
			return TRUE;
	}

	return FALSE;
}


/*****************************************************************************
 *
 *  category_tree_model_iter_n_children: Returns the number of children the row
 *                               specified by 'iter' has. This is usually 0,
 *                               as we only have a list and thus do not have
 *                               any children to any rows. A special case is
 *                               when 'iter' is NULL, in which case we need
 *                               to return the number of top-level nodes,
 *                               ie. the number of rows in our list.
 *
 *****************************************************************************/

static gint
category_tree_model_iter_n_children (GtkTreeModel *tree_model,
                                     GtkTreeIter  *iter)
{
	CategoryTreeModel    *category_tree_model;
	CategoryNode*  cnode;
	QueueNode*     qnode;

	g_return_val_if_fail (IS_CATEGORY_TREE_MODEL (tree_model), -1);
	g_return_val_if_fail (iter == NULL || iter->user_data != NULL, FALSE);

	category_tree_model = CATEGORY_TREE_MODEL(tree_model);
	/* special case: if iter == NULL, return number of top-level rows */
	if (iter == NULL)
		return base_node_n_children (category_tree_model->node);

	qnode = iter->user_data;
	cnode = queue_node_get_category (qnode);

	if (base_node_child_position (BASE_NODE (cnode), BASE_NODE (qnode)) == 0)
		return base_node_n_children (BASE_NODE (cnode)) - 1;

	return 0;
}


/*****************************************************************************
 *
 *  category_tree_model_iter_nth_child: If the row specified by 'parent' has any
 *                              children, set 'iter' to the n-th child and
 *                              return TRUE if it exists, otherwise FALSE.
 *                              A special case is when 'parent' is NULL, in
 *                              which case we need to set 'iter' to the n-th
 *                              row if it exists.
 *
 *****************************************************************************/

static gboolean
category_tree_model_iter_nth_child (GtkTreeModel *tree_model,
                                    GtkTreeIter  *iter,
                                    GtkTreeIter  *parent,
                                    gint          n)
{
	CategoryTreeModel    *category_tree_model;
	CategoryNode*  cnode;
	QueueNode*     qnode;
	int            index;

	g_return_val_if_fail (IS_CATEGORY_TREE_MODEL (tree_model), FALSE);

	category_tree_model = CATEGORY_TREE_MODEL(tree_model);

	qnode = NULL;
	/* special case: if parent == NULL, set iter to n-th top-level row */
	if (parent==NULL) {
		cnode = (CategoryNode*) base_node_nth_child (category_tree_model->node, n);
		qnode = (cnode) ? category_node_waiting_queue (cnode) : NULL;
	} else {
		qnode = iter->user_data;
		cnode = (qnode) ? queue_node_get_category (qnode) : NULL;
	}

	if (qnode == NULL)
		return FALSE;

	index = base_node_child_position (BASE_NODE (cnode), BASE_NODE (qnode));
	if (index > 0)
		return FALSE;

	qnode = (QueueNode*) base_node_nth_child (BASE_NODE (cnode), n + 1);
	if (qnode) {
		iter->stamp = category_tree_model->stamp;
		iter->user_data = qnode;
		return TRUE;
	}

	return FALSE;
}


/*****************************************************************************
 *
 *  category_tree_model_iter_parent: Point 'iter' to the parent node of 'child'. As
 *                           we have a list and thus no children and no
 *                           parents of children, we can just return FALSE.
 *
 *****************************************************************************/

static gboolean
category_tree_model_iter_parent (GtkTreeModel *tree_model,
                                 GtkTreeIter  *iter,
                                 GtkTreeIter  *child)
{
	CategoryTreeModel    *category_tree_model;
	CategoryNode*  cnode;
	QueueNode*     qnode;
	int            index;

	g_return_val_if_fail (IS_CATEGORY_TREE_MODEL (tree_model), FALSE);

	category_tree_model = CATEGORY_TREE_MODEL(tree_model);
	qnode = child->user_data;
	cnode = queue_node_get_category (qnode);
	index = base_node_child_position (BASE_NODE (cnode), BASE_NODE (qnode));

	if (index > 0) {
		iter->stamp = category_tree_model->stamp;
		iter->user_data = category_node_waiting_queue (cnode);
		return TRUE;
	}

	return FALSE;
}


/*****************************************************************************
 *
 *  category_tree_model_new:  This is what you use in your own code to create a
 *                    new custom list tree model for you to use.
 *
 *****************************************************************************/

static void on_node_insert (BaseNode* parent, BaseNode* child, int nth, gpointer data)
{
	GtkTreeModel* model = data;
	GtkTreeIter   iter;
	GtkTreePath*  path;

	// get waiting queue
	child = base_node_first_child (child);

	if (child == NULL)
		return;
	iter.stamp     = CATEGORY_TREE_MODEL (data)->stamp;
	iter.user_data = child;
	path = gtk_tree_path_new ();
	gtk_tree_path_append_index (path, nth);
	gtk_tree_model_row_inserted (model, path, &iter);

	// completed queue
	child = base_node_next (child);
	if (child) {
		iter.user_data = child;
		gtk_tree_path_append_index (path, 0);
		gtk_tree_model_row_inserted (model, path, &iter);

		// recycled queue
		child = base_node_next (child);
		if (child) {
			iter.user_data = child;
			gtk_tree_path_next (path);
			gtk_tree_model_row_inserted (model, path, &iter);
		}
	}

	gtk_tree_path_free (path);
}

static void on_node_remove (BaseNode* parent, BaseNode* child, int nth, gpointer data)
{
	GtkTreeModel* model = data;
	GtkTreePath*  path;

	path = gtk_tree_path_new ();
	gtk_tree_path_append_index (path, nth);
	gtk_tree_model_row_deleted (model, path);
	gtk_tree_path_free (path);
}

CategoryTreeModel* category_tree_model_new (RootNode* parent)
{
	BaseNode* node = BASE_NODE (parent);
	CategoryTreeModel *newlist;

	newlist = (CategoryTreeModel*) g_object_new (TYPE_CATEGORY_TREE_MODEL, NULL);
	g_assert( newlist != NULL );

	base_node_ref (node);
	newlist->node = node;
	newlist->inserted_signal = base_node_inserted_connect (node, on_node_insert, newlist);
	newlist->removed_signal = base_node_removed_connect (node, on_node_remove, newlist);

	return newlist;
}

