/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include "download_applying.h"

static gboolean load_node (DownloadApplying* dapl, gboolean next)
{
	GList* list = dapl->list_cur;

	if (list == NULL)
		return FALSE;

	if (next) {
		if (list->next)
			list = list->next;
		else
			return FALSE;
		dapl->list_cur = list;
	}

	dapl->n_current++;
	download_info_view_set (&dapl->dinfo_view, list->data);
	dapl->dinfo_view.auto_complete = TRUE;
	download_importer_set_progress (DOWNLOAD_IMPORTER (dapl), dapl->n_current, dapl->n_total);
	return TRUE;
}

static void on_response (GtkDialog *dlg, gint response, gpointer data)
{
	DownloadApplying* dapl = data;
	DownloadImporter* dim  = data;
	DownloadDialog*   dd   = data;
	DownloadNode* dnode;
	DownloadNode* dnode_temp;
	GList*   list;

	list = dapl->list_cur;

	switch (response) {
	case GTK_RESPONSE_OK:
		if (download_dialog_handle_error (dd))
			return;

		dnode = list->data;
		download_info_view_get (&dd->dinfo_view, dnode);

		// apply all jobs
		if (download_importer_get_mode (dim) == 0) {
			dnode_temp = download_node_new ();
			download_info_view_get (&dd->dinfo_view, dnode_temp);
			for (list=list->next; list; list=list->next) {
				dnode = list->data;
				download_node_apply (dnode, dnode_temp, DOWNLOAD_APPLY_ALL);
			}
			download_node_unref (dnode_temp);
			dapl->list_cur = NULL;
		}
		// Make sure main_app_on_timer () will start.
		main_app_job_changed (dim->main_app);
		main_window_message_area_refresh (&dim->main_app->window);
		// don't break here
	case GTK_RESPONSE_REJECT:
		g_message ("Get next or skip this");
		if (load_node (dapl, TRUE))
			return;
		// don't break here
	case GTK_RESPONSE_DELETE_EVENT:
	case GTK_RESPONSE_CANCEL:
		g_message ("Call download_applying_destroy ()");
		download_applying_destroy (dapl);
		break;
	}
}


DownloadApplying* download_applying_new (GtkWindow* parent, MainApp* app)
{
	DownloadApplying* dapl = g_malloc0 (sizeof (DownloadApplying));
	CategoryNode* cnode = NULL;

	if (app->window.queue_ctrl)
		cnode = app->window.queue_ctrl->cnode;

	// initialize base class
	download_importer_init (DOWNLOAD_IMPORTER (dapl), parent, app);
	download_dialog_set_category (DOWNLOAD_DIALOG (dapl), app->rnode, cnode);
	dapl->finalize  = (DownloadDialogFunc) download_applying_finalize;
	dapl->user_data = dapl;
	gtk_widget_set_sensitive (GTK_WIDGET (dapl->treeview), FALSE);
	gtk_widget_set_sensitive (GTK_WIDGET (dapl->title_category), FALSE);

	return dapl;
}

void download_applying_set (DownloadApplying* dapl, GList* download_list)
{
	g_list_foreach (download_list, (GFunc) base_node_ref, NULL);
	dapl->download_list = download_list;
	dapl->list_cur = download_list;
	dapl->n_total = g_list_length (download_list);
}

void download_applying_activate (DownloadApplying* dapl)
{
	if (load_node (dapl, FALSE)==FALSE) {
		download_applying_destroy (dapl);
		return;
	}

	// signals
	g_signal_connect (dapl->self, "response", G_CALLBACK (on_response), dapl);
	gtk_widget_show (GTK_WIDGET (dapl->self));
}

// protected function
void download_applying_finalize (DownloadApplying* dapl)
{
	g_list_foreach (dapl->download_list, (GFunc) base_node_unref, NULL);
	g_list_free (dapl->download_list);

	download_importer_finalize (DOWNLOAD_IMPORTER (dapl));
}

