/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include "download_importer.h"
#include <urlglib/ug_i18n.h>

static void on_response (GtkDialog *dlg, gint response, gpointer data)
{
	DownloadImporter* dim = data;
	DownloadDialog*   dd  = data;
	CategoryNode* cnode;
	QueueNode*    qnode;
	DownloadNode* dnode;
	DownloadNode* dnode_other;
	char* url;

	switch (response) {
	case GTK_RESPONSE_OK:
		if (download_dialog_handle_error (dd))
			return;

		g_message ("Url check pass");
		dnode = download_node_new ();
		cnode = download_dialog_get_category (dd);
		qnode = category_node_waiting_queue (cnode);
		download_info_view_get (&dd->dinfo_view, dnode);
		queue_node_append (qnode, dnode);

		// apply all jobs
		if (download_importer_get_mode (dim) == 0) {
			url = dim->get (dim, dim->user_data);
			while (url) {
				dnode_other = download_node_new ();
				download_node_apply (dnode_other, dnode, DOWNLOAD_APPLY_ALL);
				download_node_set_url (dnode_other, url);
				queue_node_append (qnode, dnode_other);
				url = dim->get (dim, dim->user_data);
			}
		}
		// Make sure main_app_on_timer () will start.
		main_app_job_changed (dim->main_app);
		// don't break here
	case GTK_RESPONSE_REJECT:
		g_message ("Get next or skip this");
		url = dim->get (dim, dim->user_data);
		if (url) {
			download_info_view_set_url (&dd->dinfo_view, url);
			break;
		}
		// don't break here
	case GTK_RESPONSE_DELETE_EVENT:
	case GTK_RESPONSE_CANCEL:
		g_message ("Call download_importer_destroy ()");
		download_importer_destroy (dim);
		break;
	}
}

static char* download_importer_get_none (DownloadImporter* dim, gpointer data)
{
	return NULL;
}

// DownloadImporter function

DownloadImporter* download_importer_new (GtkWindow* parent, MainApp* app)
{
	DownloadImporter* dim = g_malloc0 (sizeof (DownloadImporter));

	download_importer_init (dim, parent, app);
	return dim;
}

void  download_importer_activate (DownloadImporter* dim)
{
	char* url;

	// signals
	g_signal_connect (dim->self, "response", G_CALLBACK (on_response), dim);

	if (dim->get) {
		url = dim->get (dim, dim->user_data);
		if (url) {
			download_info_view_set_url (&dim->dinfo_view, url);
			gtk_widget_show (GTK_WIDGET (dim->self));
			return;
		}
	}
	download_importer_destroy (dim);
}

void  download_importer_activate_1 (DownloadImporter* dim, const gchar* url)
{
	// signals
	g_signal_connect (dim->self, "response", G_CALLBACK (on_response), dim);

	download_info_view_set_url (&dim->dinfo_view, url ? url : "");
	gtk_widget_show (GTK_WIDGET (dim->self));
	return;
}

void  download_importer_set_mode (DownloadImporter* dim, int mode)
{
	if (mode == 1)
		gtk_toggle_button_set_active (dim->toggle_one_by_one, TRUE);
	else
		gtk_toggle_button_set_active (dim->toggle_apply_all, TRUE);
}

int   download_importer_get_mode (DownloadImporter* dim)
{
	if (gtk_toggle_button_get_active (dim->toggle_apply_all))
		return 0;
	return 1;
}

void  download_importer_set_progress (DownloadImporter* dim, int n_current, int n_total)
{
	char* title_str;
	char* progress_str;

	if (n_current > n_total)
		n_total = n_current;

	if (dim->title_base)
		title_str = dim->title_base;
	else
		title_str = _("Download property");

	if (n_total > 1) {
		progress_str = g_strdup_printf ("%s ( %d / %d )", title_str, n_current, n_total);
		gtk_window_set_title (GTK_WINDOW (dim->self), progress_str);
		g_free (progress_str);

		gtk_widget_set_sensitive (dim->button_skip, TRUE);
		gtk_widget_show (dim->button_skip);
		gtk_widget_show_all (dim->frame_import_mode);
	} else {
		gtk_widget_set_sensitive (dim->button_skip, FALSE);
		gtk_widget_hide (dim->button_skip);
		gtk_widget_hide (dim->frame_import_mode);
	}
}

void  download_importer_set_title (DownloadImporter* dim, gchar* title)
{
	g_free (dim->title_base);
	dim->title_base = g_strdup (title);
}

// protected functions
void download_importer_init (DownloadImporter* dim, GtkWindow* parent, MainApp* app)
{
	GtkWidget* button;
	GtkBox*    box;

	download_dialog_init (DOWNLOAD_DIALOG (dim), parent);
	dim->finalize   = (DownloadDialogFunc) download_importer_finalize;
	dim->get        = download_importer_get_none;
	dim->main_app   = app;
	dim->title_base = NULL; //g_strdup (_("Download property"));

	dim->frame_import_mode = gtk_frame_new (_("Apply mode:"));
	box = (GtkBox*)gtk_vbox_new (TRUE, 2);
	gtk_container_add (GTK_CONTAINER (dim->frame_import_mode), (GtkWidget*)box);
	button = gtk_radio_button_new_with_mnemonic (NULL, _("Apply all jobs"));
	gtk_box_pack_start (box, button, FALSE, FALSE, 0);
	dim->toggle_apply_all = GTK_TOGGLE_BUTTON (button);

	button = gtk_radio_button_new_with_mnemonic_from_widget ((GtkRadioButton*)button, _("One by one"));
	gtk_box_pack_start (box, button, FALSE, FALSE, 0);
	dim->toggle_one_by_one = GTK_TOGGLE_BUTTON (button);

	gtk_container_set_border_width (GTK_CONTAINER (box), 3);
	gtk_toggle_button_set_active (dim->toggle_apply_all, TRUE);
//	gtk_widget_show_all (dim->frame_import_mode);

	gtk_table_attach (dim->table, dim->frame_import_mode, 0, 1, 2, 3,
	                  GTK_FILL, GTK_SHRINK, 3, 3);
}

void download_importer_finalize (DownloadImporter* dim)
{
	if (dim->destroy)
		dim->destroy (dim, dim->user_data);

	g_free (dim->title_base);
	download_dialog_finalize (DOWNLOAD_DIALOG (dim));
}

