/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include "filter_importer.h"

// callback functions for DownloadImporter ---
static char* on_get (DownloadImporter* dim, gpointer data)
{
	FilterImporter* fim = data;
	UrlItem* uitem;
	GSList*  slist_next;

	if (fim->marked_list) {
		uitem = fim->marked_list->data;
		slist_next = fim->marked_list->next;
		g_slist_free_1 (fim->marked_list);
		fim->marked_list = slist_next;
		fim->n_current++;
		download_importer_set_progress (dim, fim->n_current, fim->n_total);
		return uitem->string;
	}
	return NULL;
}

// signal handler
static void on_filter_response (GtkDialog *dlg, gint response, gpointer data)
{
	FilterImporter* fim = data;
	UrlFilter* filter;
	GSList* slist;
	GSList* slist_filter;
	int     n_marked;

	switch (response) {
	case GTK_RESPONSE_OK:
		gtk_widget_hide (GTK_WIDGET (dlg));
		for (slist_filter=fim->filter_list; slist_filter; slist_filter=slist_filter->next) {
			filter = slist_filter->data;
			slist  = url_filter_get_marked_item (filter, &n_marked);
			fim->marked_list = g_slist_concat (fim->marked_list, slist);
			fim->n_total += n_marked;
		}
		download_importer_activate (DOWNLOAD_IMPORTER (fim));
		break;
	case GTK_RESPONSE_DELETE_EVENT:
	case GTK_RESPONSE_CANCEL:
	default:
		filter_importer_destroy (fim);
		break;
	}
}

FilterImporter* filter_importer_new (GtkWindow* parent, MainApp* app)
{
	FilterImporter* fim = g_malloc0 (sizeof (FilterImporter));
	CategoryNode* cnode = NULL;

	if (app->window.queue_ctrl)
		cnode = app->window.queue_ctrl->cnode;

	// initialize base class
	download_importer_init (DOWNLOAD_IMPORTER (fim), parent, app);
	download_dialog_set_category (DOWNLOAD_DIALOG (fim), app->rnode, cnode);
	fim->finalize  = (DownloadDialogFunc) filter_importer_finalize;
	fim->get       = on_get;
	fim->user_data = fim;

	fim->filter_dialog = url_filter_dialog_new (parent);

	// signals
	g_signal_connect (fim->filter_dialog->self, "response",
	                  G_CALLBACK (on_filter_response), fim);

	return fim;
}

void  filter_importer_add (FilterImporter* fim, UrlFilter* filter, gchar* label_name)
{
	fim->filter_list = g_slist_append (fim->filter_list, filter);
	url_filter_dialog_add_filter (fim->filter_dialog, filter, label_name);
}

void filter_importer_set_title (FilterImporter* fim, gchar* title)
{
	gtk_window_set_title (GTK_WINDOW (fim->filter_dialog->self), title);
	download_importer_set_title (DOWNLOAD_IMPORTER (fim), title);
}

void  filter_importer_activate (FilterImporter* fim)
{
	UrlItemList* item_list;
	int  filter_len;

	if (fim->filter_list == NULL)
		filter_importer_destroy (fim);

	if (g_slist_length (fim->filter_list) == 1) {
		item_list = fim->filter_list->data;  // first UrlFilter
		filter_len = url_item_list_length (item_list);
		if (filter_len == 1) {
			// only one item - skip filter_dialog
			url_filter_set_item_all (item_list, TRUE, 0);
			on_filter_response (fim->filter_dialog->self, GTK_RESPONSE_OK, fim);
			return;
		} else if (filter_len == 0) {
			// No item - destory directly
			filter_importer_destroy (fim);
			return;
		}
	}

	gtk_widget_show (GTK_WIDGET (fim->filter_dialog->self));
}

// protected function
void filter_importer_finalize (FilterImporter* fim)
{
	url_filter_dialog_destroy (fim->filter_dialog);
	g_slist_foreach (fim->filter_list, (GFunc)url_item_list_unref, NULL);
	g_slist_free (fim->marked_list);
	g_slist_free (fim->filter_list);

	download_importer_finalize (DOWNLOAD_IMPORTER (fim));
}

