/*
 *
 *   Copyright (C) 2002-2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <urlglib/ug_i18n.h>
#include <gdk/gdkkeysyms.h>
#include "main_window.h"
#include "urlgfe_util.h"
#include "category_tree_view.h"


// utility functions ------------------------------------------------

// utility for create button
static GtkWidget* tool_button_new (const gchar* stock_id,
                                   const gchar* tip_str,
                                   GtkTooltips* tips)
{
	GtkWidget* button;
	GtkWidget* image;

	button = gtk_button_new();
	image = gtk_image_new_from_stock (stock_id, GTK_ICON_SIZE_SMALL_TOOLBAR);

	gtk_container_add (GTK_CONTAINER(button), image);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);

	gtk_tooltips_set_tip (tips, button, tip_str, NULL);

	return button;
}

// utility for GtkStatusbar
static void statusbar_set_n_selected (GtkStatusbar* sb, guint n_selected)
{
	static guint last_id=0;
	gchar* string;

	if (last_id)
		gtk_statusbar_pop (sb, last_id);

	string = g_strdup_printf (_("Selected %d files"), n_selected);
	last_id = gtk_statusbar_get_context_id (sb, string);
	gtk_statusbar_push (sb, last_id, string);
	g_free (string);
}

// utility for GtkTextBuffer
static void text_buffer_add_text (GtkTextBuffer* buffer,
                                  GtkTextTag* text_tag,
                                  gchar *name, gchar *str,
                                  gboolean next_line)
{
	GtkTextIter    text_iter;
	GtkTextIter    text_iter2;
	GtkTextMark*   mark;

	gtk_text_buffer_get_end_iter (buffer, &text_iter2);
	mark = gtk_text_buffer_create_mark (buffer, NULL, &text_iter2, TRUE);
	gtk_text_buffer_insert (buffer, &text_iter2, name, -1);
	// apply tag
	gtk_text_buffer_get_end_iter (buffer, &text_iter2);
	gtk_text_buffer_get_iter_at_mark (buffer, &text_iter, mark);
	gtk_text_buffer_apply_tag (buffer, text_tag,
	                           &text_iter, &text_iter2);
	gtk_text_buffer_delete_mark (buffer, mark);

	gtk_text_buffer_insert (buffer, &text_iter2, (str) ? str : "", -1);
	// to next line
	if (next_line) {
		gtk_text_buffer_get_end_iter (buffer, &text_iter2);
		gtk_text_buffer_insert (buffer, &text_iter2, "\n", -1);
	}
}

// End of utility functions =========================================

// Signal handler
static void on_category_selection_changed (GtkTreeSelection* selection, MainWindow* mw)
{
	if (gtk_tree_selection_count_selected_rows (selection))
		main_window_set_category_sensitive (mw, TRUE);
	else
		main_window_set_category_sensitive (mw, FALSE);
}

static void on_category_row_collapsed (GtkTreeView* tview, GtkTreeIter* iter,
                                       GtkTreePath* path, gpointer user_data)
{
	GtkTreeSelection* selection;

	// select category when row collapsed
	selection = gtk_tree_view_get_selection (tview);
	if (gtk_tree_selection_count_selected_rows (selection)==0)
		gtk_tree_view_set_cursor (tview, path, NULL, FALSE);
}

static void on_download_cursor_changed (GtkTreeView* view, MainWindow* mw)
{
	main_window_message_area_refresh (mw);
}

static void on_download_selection_changed (GtkTreeSelection* selection, MainWindow* mw)
{
	// Don't handle "selection_changed" when moving download.
	if (mw->download_moving)
		return;

	if (gtk_tree_selection_count_selected_rows (selection))
		main_window_set_download_sensitive (mw, TRUE);
	else
		main_window_set_download_sensitive (mw, FALSE);

	main_window_statusbar_refresh (mw);
}

static void on_view_toggled (GtkCheckMenuItem* cmi, gpointer data)
{
	MainWindow* mw = data;

	if (mw->view_setting_loading == FALSE)
		main_window_view_get (mw);

	main_window_view_set (mw);
}


// download view and store signal handler ---------------------------

static gboolean on_download_button_event (GtkWidget* widget,
                                          GdkEventButton* event,
                                          MainWindow* mw)
{
	if (event->type==GDK_BUTTON_PRESS && event->button==3) {
		// press right button
		gtk_menu_popup (mw->download_menu.self,
		                NULL,
		                NULL,
		                NULL,
		                NULL,
		                0,
		                gtk_get_current_event_time());
		return TRUE;
	}
//	main_window_decide_download_sensitive (mw);
	return FALSE;
}

// category view signal handler -------------------------------------
static gboolean on_category_button_event (GtkWidget* widget,
                                          GdkEventButton* event,
                                          MainWindow* mw)
{
	if (event->type==GDK_BUTTON_RELEASE && event->button==3) {
		// press right button
		gtk_menu_popup (mw->category_menu.self,
		                NULL,
		                NULL,
		                NULL,
		                NULL,
		                0,
		                gtk_get_current_event_time());
		return TRUE;
	}
	return FALSE;
}


// Menu

void file_menu_init (FileMenu* fm, MainWindow* mw)
{
	GtkWidget* image;

	fm->self = (GtkMenu*)gtk_menu_new ();

	fm->import_html = gtk_image_menu_item_new_with_mnemonic (_("Import _HTML file..."));
	image = gtk_image_new_from_stock (GTK_STOCK_CONVERT, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)fm->import_html, image);

	fm->import_url = gtk_image_menu_item_new_with_mnemonic (_("Import _URLs..."));
	image = gtk_image_new_from_stock (GTK_STOCK_CONVERT, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)fm->import_url, image);

	fm->export_url = gtk_image_menu_item_new_with_mnemonic (_("_Export URLs..."));
	image = gtk_image_new_from_stock (GTK_STOCK_CONVERT, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)fm->export_url, image);

	fm->save_all = gtk_image_menu_item_new_with_mnemonic (_("_Save All"));
	image = gtk_image_new_from_stock (GTK_STOCK_SAVE, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)fm->save_all, image);

	fm->quit = gtk_image_menu_item_new_from_stock (GTK_STOCK_QUIT, mw->accelgroup);

//	menu.append( gtk_tearoff_menu_item_new() );
	gtk_menu_append (fm->self, fm->import_html);
	gtk_menu_append (fm->self, fm->import_url);
	gtk_menu_append (fm->self, fm->export_url);
	gtk_menu_append (fm->self, fm->save_all);
	gtk_menu_append (fm->self, fm->quit);
}

void category_menu_init (CategoryMenu* cm)
{
	GtkWidget* image;

	cm->self = (GtkMenu*)gtk_menu_new ();

	cm->create = gtk_image_menu_item_new_with_mnemonic (_("_New..."));
	image = gtk_image_new_from_stock (GTK_STOCK_NEW, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)cm->create, image);

	cm->erase = gtk_image_menu_item_new_with_mnemonic (_("_Erase"));
	image = gtk_image_new_from_stock (GTK_STOCK_CLOSE, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)cm->erase, image);

	cm->properties = gtk_image_menu_item_new_with_mnemonic (_("_Properties..."));
	image = gtk_image_new_from_stock (GTK_STOCK_PROPERTIES, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)cm->properties ,image);

	cm->set_default = gtk_menu_item_new_with_mnemonic (_("_Default of new Category..."));

//	menu.append( gtk_tearoff_menu_item_new() );
	gtk_menu_append (cm->self, cm->create);
	gtk_menu_append (cm->self, cm->erase);
	gtk_menu_append (cm->self, cm->properties);
	gtk_menu_append (cm->self, cm->set_default);
//	gtk_menu_append (cm->self, gtk_separator_menu_item_new() );
}

void download_menu_init (DownloadMenu* dm)
{
	GtkWidget* image;

	dm->self = (GtkMenu*)gtk_menu_new ();

	dm->create = gtk_image_menu_item_new_with_mnemonic (_("_New Download..."));
	image = gtk_image_new_from_stock (GTK_STOCK_NEW, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)dm->create, image);

	dm->add_batch = gtk_image_menu_item_new_with_mnemonic (_("_Add Batch Download..."));
	image = gtk_image_new_from_stock (GTK_STOCK_SORT_ASCENDING, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)dm->add_batch, image);

	dm->erase = gtk_image_menu_item_new_with_mnemonic(_("_Erase"));
	image = gtk_image_new_from_stock(GTK_STOCK_DELETE, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)dm->erase, image);

	dm->move_to = gtk_image_menu_item_new_with_mnemonic(_("_Move To..."));
	image = gtk_image_new_from_stock(GTK_STOCK_DND_MULTIPLE, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)dm->move_to, image);

	dm->move_up = gtk_image_menu_item_new_with_mnemonic(_("Move _Up"));
	image = gtk_image_new_from_stock(GTK_STOCK_GO_UP, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)dm->move_up, image);

	dm->move_down = gtk_image_menu_item_new_with_mnemonic(_("Move _Down"));
	image = gtk_image_new_from_stock(GTK_STOCK_GO_DOWN, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)dm->move_down, image);

	dm->move_top = gtk_image_menu_item_new_with_mnemonic(_("Move _Top"));
	image = gtk_image_new_from_stock(GTK_STOCK_GOTO_TOP, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)dm->move_top, image);

	dm->move_bottom = gtk_image_menu_item_new_with_mnemonic(_("Move _Bottom"));
	image = gtk_image_new_from_stock(GTK_STOCK_GOTO_BOTTOM, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)dm->move_bottom, image);

	dm->start = gtk_image_menu_item_new_with_mnemonic(_("_Start"));
	image = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)dm->start, image);

	dm->stop  = gtk_image_menu_item_new_with_mnemonic(_("St_op"));
	image = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PAUSE, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)dm->stop ,image);

	dm->properties = gtk_image_menu_item_new_with_mnemonic(_("_Properties..."));
	image = gtk_image_new_from_stock(GTK_STOCK_PROPERTIES, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)dm->properties, image);

//	menu.append( gtk_tearoff_menu_item_new() );
	gtk_menu_append (dm->self, dm->create);
	gtk_menu_append (dm->self, dm->add_batch);
	gtk_menu_append (dm->self, dm->erase);
	gtk_menu_append (dm->self, gtk_separator_menu_item_new() );
	gtk_menu_append (dm->self, dm->move_to);
	gtk_menu_append (dm->self, dm->move_up);
	gtk_menu_append (dm->self, dm->move_down);
	gtk_menu_append (dm->self, dm->move_top);
	gtk_menu_append (dm->self, dm->move_bottom);
	gtk_menu_append (dm->self, gtk_separator_menu_item_new() );
	gtk_menu_append (dm->self, dm->start);
	gtk_menu_append (dm->self, dm->stop);
	gtk_menu_append (dm->self, dm->properties);
}

void view_menu_init (ViewMenu* vm)
{
	vm->self = (GtkMenu*)gtk_menu_new ();

	vm->toolbar = gtk_check_menu_item_new_with_mnemonic (_("_Toolbar"));
	vm->statusbar = gtk_check_menu_item_new_with_mnemonic (_("_Status Bar"));

//	vm->category_header = gtk_check_menu_item_new_with_mnemonic (_("_Category Header"));
	vm->msg_area = gtk_check_menu_item_new_with_mnemonic (_("_Message Area"));
	vm->msg_area_items = gtk_menu_item_new_with_mnemonic (_("M_essage Items..."));

	vm->download_rules_hint = gtk_check_menu_item_new_with_mnemonic (_("Download _Rules Hint"));
	vm->columns = gtk_menu_item_new_with_mnemonic (_("Download C_olumns..."));

	gtk_menu_append (vm->self, vm->toolbar);
	gtk_menu_append (vm->self, vm->statusbar);
	gtk_menu_append (vm->self, gtk_separator_menu_item_new() );
//	gtk_menu_append (vm->self, vm->category_header);
	gtk_menu_append (vm->self, vm->msg_area);
	gtk_menu_append (vm->self, vm->msg_area_items);
	gtk_menu_append (vm->self, gtk_separator_menu_item_new() );
	gtk_menu_append (vm->self, vm->download_rules_hint);
	gtk_menu_append (vm->self, vm->columns);
}

void global_menu_init (GlobalMenu* gm)
{
	GtkWidget* image;

	gm->self = (GtkMenu*)gtk_menu_new ();

	gm->start_all = gtk_image_menu_item_new_with_mnemonic (_("_Start All"));
	image = gtk_image_new_from_stock (GTK_STOCK_EXECUTE, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)gm->start_all, image);
	gm->pause_all = gtk_image_menu_item_new_with_mnemonic (_("_Pause All"));
	image = gtk_image_new_from_stock (GTK_STOCK_STOP, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)gm->pause_all, image);

//	gm->start_queuing = gtk_image_menu_item_new_with_mnemonic (_("S_tart Queuing"));
//	image = gtk_image_new_from_stock (GTK_STOCK_YES, GTK_ICON_SIZE_MENU);
//	gtk_image_menu_item_set_image ((GtkImageMenuItem*)gm->start_queuing, image);
//	gm->pause_queuing = gtk_image_menu_item_new_with_mnemonic (_("P_ause Queuing"));
//	image = gtk_image_new_from_stock (GTK_STOCK_NO, GTK_ICON_SIZE_MENU);
//	gtk_image_menu_item_set_image ((GtkImageMenuItem*)gm->pause_queuing, image);

	gm->setting = gtk_menu_item_new_with_mnemonic (_("_Global Setting..."));

	gtk_menu_append (gm->self, gm->start_all);
	gtk_menu_append (gm->self, gm->pause_all);
//	gtk_menu_append (gm->self, gtk_separator_menu_item_new() );
//	gtk_menu_append (gm->self, gm->start_queuing);
//	gtk_menu_append (gm->self, gm->pause_queuing);
	gtk_menu_append (gm->self, gtk_separator_menu_item_new() );
	gtk_menu_append (gm->self, gm->setting);
}

void help_menu_init (HelpMenu* hm)
{
	GtkWidget* image;

	hm->self = (GtkMenu*)gtk_menu_new ();
	hm->about_urlgfe = gtk_image_menu_item_new_with_mnemonic (_("_About Urlgfe..."));
	image = gtk_image_new_from_stock (GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image ((GtkImageMenuItem*)hm->about_urlgfe, image);

	gtk_menu_append (hm->self, hm->about_urlgfe);
}

// MainWindow
void main_window_init_toolbar (MainWindow* wnd, GtkWidget* hbox)
{
	GtkTooltips* tips;

	tips = gtk_tooltips_new ();

	// button create Category
	wnd->button_category_new = tool_button_new (GTK_STOCK_DND_MULTIPLE,
	                                            _("Create a Category"),
	                                            tips);
	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_category_new,
	                    FALSE, FALSE, 1);

	// button erase Category
	wnd->button_category_erase = tool_button_new (GTK_STOCK_CLOSE,
	                                              _("Delete a Category"),
	                                              tips);
	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_category_erase,
	                    FALSE, FALSE, 1);

	// button import Category
//	wnd->button_import =  tool_button_new (GTK_STOCK_CONVERT,
//	                                       _("Import ..."),
//	                                       tips);
//	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_import,
//	                    FALSE, FALSE, 1);

	// button import Category
//	wnd->button_export =  tool_button_new (GTK_STOCK_CONVERT,
//	                                       _("Export ..."),
//	                                       tips);
//	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_export,
//	                    FALSE, FALSE, 1);

	// add separator
	gtk_box_pack_start (GTK_BOX(hbox), gtk_vseparator_new (), FALSE, FALSE, 1);

	// button Start Download
	wnd->button_start = tool_button_new (GTK_STOCK_MEDIA_PLAY, _("Start Automatically"), tips);
	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_start, FALSE, FALSE, 1);

	// button Stop Download
	wnd->button_stop = tool_button_new (GTK_STOCK_MEDIA_PAUSE, _("Stop"), tips);
	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_stop, FALSE, FALSE, 1);

	// button New Download
	wnd->button_new = tool_button_new (GTK_STOCK_NEW,
	                                   _("Create new download"),
	                                   tips);
	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_new,
	                    FALSE, FALSE, 1);

	// button Add batch
	wnd->button_batch = tool_button_new (GTK_STOCK_SORT_ASCENDING,
	                                     _("Add Batch Download"),
	                                     tips);
	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_batch,
	                    FALSE, FALSE, 1);

	// button Delete
	wnd->button_erase = tool_button_new (GTK_STOCK_DELETE,
	                                     _("Delete selected jobs"),
	                                     tips);
	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_erase,
	                    FALSE, FALSE, 1);

	// button Property
	wnd->button_property = tool_button_new (GTK_STOCK_PROPERTIES,
	                                        _("Properties"),
	                                        tips);
	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_property,
	                    FALSE, FALSE, 1);

	// button Move up
	wnd->button_move_up = tool_button_new (GTK_STOCK_GO_UP,
	                                       _("Move Up"),
	                                       tips);
	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_move_up,
	                    FALSE, FALSE, 1);

	// button Move down
	wnd->button_move_down = tool_button_new (GTK_STOCK_GO_DOWN,
	                                         _("Move Down"),
	                                         tips);
	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_move_down,
	                    FALSE, FALSE, 1);

	// button Move top
	wnd->button_move_top = tool_button_new (GTK_STOCK_GOTO_TOP,
	                                        _("Move Top"),
	                                        tips);
	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_move_top,
	                    FALSE, FALSE, 1);

	// button Move bottom
	wnd->button_move_bottom = tool_button_new (GTK_STOCK_GOTO_BOTTOM,
	                                           _("Move Bottom"),
	                                           tips);
	gtk_box_pack_start (GTK_BOX(hbox), wnd->button_move_bottom,
	                    FALSE, FALSE, 1);

	gtk_widget_show_all (hbox);
}

void main_window_init (MainWindow* mw)
{
	GtkWidget*  scrolled;
	GtkWidget*  menu_item;
	GtkTextBuffer*    buffer;
	GtkTreeSelection* selection;
#ifndef _WIN32
	GdkPixbuf*  icon_pixbuf;
	gchar*      icon_path;
#endif

	mw->vbox = gtk_vbox_new (FALSE, 3);
	mw->self = (GtkWindow*)gtk_window_new (GTK_WINDOW_TOPLEVEL);

	// accelerators
	mw->accelgroup = gtk_accel_group_new ();
	gtk_window_add_accel_group (mw->self, mw->accelgroup);
	gtk_accel_group_connect (mw->accelgroup, GDK_q, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE, g_cclosure_new (G_CALLBACK(gtk_main_quit), NULL, NULL));
	//gtk_accel_group_lock (mw->accelgroup);
	//gtk_accel_groups_activate (G_OBJECT(mw->self), GDK_q, GDK_CONTROL_MASK);

	gtk_window_set_title (mw->self, URLGFE_TITLE);
	gtk_window_set_default_size (mw->self, 515, 350);
	view_setting_init (&mw->view_setting);
	mw->queue_ctrl = NULL;
	mw->id_cursor_changed = 0;
	mw->id_selection = 0;
	mw->id_button_event = 0;
	mw->view_setting_loading = FALSE;
	mw->download_moving = FALSE;

#ifndef _WIN32
	icon_path   = g_build_filename (DATADIR, "pixmaps", "urlgfe-icon.png", NULL);
	icon_pixbuf = gdk_pixbuf_new_from_file (icon_path, NULL);
	g_free (icon_path);
	if (icon_pixbuf)
		gtk_window_set_icon (mw->self, icon_pixbuf);
#endif

	// menu bar --------------------------------------------------
	mw->menu_bar = gtk_menu_bar_new ();

	// file menu
	file_menu_init (&mw->file_menu, mw);
	menu_item = gtk_menu_item_new_with_mnemonic (_("_File"));
	gtk_menu_item_set_submenu ((GtkMenuItem*)menu_item,
	                           (GtkWidget*)mw->file_menu.self);
	gtk_menu_shell_append ((GtkMenuShell *)mw->menu_bar, menu_item);

	// category menu
	category_menu_init (&mw->category_menu);
	menu_item = gtk_menu_item_new_with_mnemonic (_("_Category"));
	gtk_menu_item_set_submenu ((GtkMenuItem*)menu_item,
	                           (GtkWidget*)mw->category_menu.self);
	gtk_menu_shell_append ((GtkMenuShell *)mw->menu_bar, menu_item);

	// download menu
	download_menu_init (&mw->download_menu);
	menu_item = gtk_menu_item_new_with_mnemonic(_("_Download"));
	gtk_menu_item_set_submenu ((GtkMenuItem*)menu_item,
	                           (GtkWidget*)mw->download_menu.self);
	gtk_menu_shell_append ((GtkMenuShell *)mw->menu_bar, menu_item);

	// global menu
	global_menu_init (&mw->global_menu);
	menu_item = gtk_menu_item_new_with_mnemonic (_("_Global"));
	gtk_menu_shell_append ((GtkMenuShell *)mw->menu_bar, menu_item);
	gtk_menu_item_set_submenu ((GtkMenuItem*)menu_item,
	                           (GtkWidget*)mw->global_menu.self);

	// view menu
	view_menu_init (&mw->view_menu);
	menu_item = gtk_menu_item_new_with_mnemonic (_("_View"));
	gtk_menu_shell_append ((GtkMenuShell *)mw->menu_bar, menu_item);
	gtk_menu_item_set_submenu ((GtkMenuItem*)menu_item,
	                           (GtkWidget*)mw->view_menu.self);

	// about
	help_menu_init (&mw->help_menu);
	menu_item = gtk_menu_item_new_with_mnemonic(_("_Help"));
	gtk_menu_item_set_submenu ((GtkMenuItem*)menu_item,
	                           (GtkWidget*)mw->help_menu.self);
	gtk_menu_shell_append ((GtkMenuShell *)mw->menu_bar, menu_item);
	// End of menu bar ==============================================

	// toolbar
	mw->toolbar = gtk_hbox_new (FALSE, 0);
	main_window_init_toolbar (mw, mw->toolbar);

	// category tree view
	mw->category_view = category_tree_view_new ();
	gtk_widget_set_size_request ((GtkWidget*)mw->category_view, 165, 100);
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled),
	                                     GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
	                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(scrolled), GTK_WIDGET(mw->category_view));

	// create and setup TextView
	mw->text_view = (GtkTextView*)gtk_text_view_new ();
	gtk_text_view_set_left_margin (mw->text_view, 2);
	gtk_text_view_set_right_margin (mw->text_view, 2);
	gtk_text_view_set_editable (mw->text_view, FALSE);
	gtk_text_view_set_cursor_visible (mw->text_view, FALSE);
	gtk_widget_set_size_request ((GtkWidget*)mw->text_view, 35, 35);
	buffer = gtk_text_view_get_buffer (mw->text_view);
	mw->text_tag = gtk_text_buffer_create_tag (buffer, "blue_foreground",
	                                           "foreground", "blue", NULL);
	mw->message_area = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (mw->message_area),
	                                     GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (mw->message_area),
	                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_set_size_request (mw->message_area, 50, 60);
	gtk_container_add (GTK_CONTAINER(mw->message_area), GTK_WIDGET(mw->text_view));

	// pack paned
	mw->vpaned = (GtkPaned*)gtk_vpaned_new ();
	mw->hpaned = (GtkPaned*)gtk_hpaned_new ();
	gtk_paned_pack1 (mw->hpaned, (GtkWidget*)scrolled, FALSE, TRUE);
	gtk_paned_add2  (mw->hpaned, (GtkWidget*)mw->vpaned);
	gtk_paned_pack2 (mw->vpaned, (GtkWidget*)mw->message_area, FALSE, TRUE);

	// status bar + status image
	mw->status_hbox = gtk_hbox_new (FALSE, 0);
	mw->statusbar = (GtkStatusbar*) gtk_statusbar_new ();
	statusbar_set_n_selected (mw->statusbar, 0);
	mw->status_image = (GtkImage*) gtk_image_new ();
	gtk_box_pack_start (GTK_BOX (mw->status_hbox), GTK_WIDGET (mw->status_image),
	                    FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (mw->status_hbox), GTK_WIDGET (mw->statusbar),
	                    TRUE, TRUE, 0);


	// pack to vbox
	gtk_box_pack_start (GTK_BOX (mw->vbox), mw->menu_bar,
	                    FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (mw->vbox), GTK_WIDGET (mw->toolbar),
	                    FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (mw->vbox), GTK_WIDGET (mw->hpaned),
	                    TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (mw->vbox), GTK_WIDGET (mw->status_hbox),
	                    FALSE, FALSE, 0);

	gtk_container_add (GTK_CONTAINER (mw->self), GTK_WIDGET (mw->vbox));

	// signal connect
	selection = gtk_tree_view_get_selection (mw->category_view);
	g_signal_connect (selection, "changed",
	                  G_CALLBACK (on_category_selection_changed), mw);
	g_signal_connect (mw->category_view, "button-release-event",
	                  G_CALLBACK(on_category_button_event), (gpointer)mw);
	g_signal_connect (mw->category_view, "row-collapsed",
	                  G_CALLBACK(on_category_row_collapsed), (gpointer)mw);

	g_signal_connect (mw->view_menu.toolbar, "toggled", G_CALLBACK (on_view_toggled), mw);
	g_signal_connect (mw->view_menu.statusbar, "toggled", G_CALLBACK (on_view_toggled), mw);
//	g_signal_connect (mw->view_menu.category_header, "toggled", G_CALLBACK (on_view_toggled), mw);
	g_signal_connect (mw->view_menu.msg_area, "toggled", G_CALLBACK (on_view_toggled), mw);
	g_signal_connect (mw->view_menu.download_rules_hint, "toggled", G_CALLBACK (on_view_toggled), mw);

/*
	g_signal_connect_swapped (mw->self, "destroy",
	                          G_CALLBACK (main_window_finalize), mw);

	selection = gtk_tree_view_get_selection (mw->category_view);
	g_signal_connect_swapped (selection, "changed",
	                          G_CALLBACK (main_window_on_selection_changed), mw);
	signal_proxy_connect_swapped (&mw->browser.dselection_signal_proxy, "changed",
	                              G_CALLBACK (main_window_on_browser_changed), mw);
*/

//	main_window_on_selection_changed (mw);
	main_window_set_category_sensitive (mw, FALSE);
	main_window_set_download_sensitive (mw, FALSE);
	gtk_widget_show_all ((GtkWidget*)mw->vbox);
}

void main_window_set_queue_ctrl (MainWindow* mw, QueueCtrl* ctrl)
{
	GtkTreeView*      treeview;
	GtkTreeSelection* selection;

	// avoid crash if mw->queue_ctrl == ctrl
	if (ctrl)
		queue_ctrl_ref (ctrl);

	if (mw->queue_ctrl) {
		treeview = mw->queue_ctrl->view;
		selection = gtk_tree_view_get_selection (treeview);
		g_signal_handler_disconnect (treeview, mw->id_cursor_changed);
		g_signal_handler_disconnect (selection, mw->id_selection);
		g_signal_handler_disconnect (treeview, mw->id_button_event);
		g_object_ref (mw->queue_ctrl->top_widget);
		gtk_container_remove (GTK_CONTAINER (mw->vpaned),
		                      mw->queue_ctrl->top_widget);
		queue_ctrl_unref (mw->queue_ctrl);
	}

	if (ctrl) {
		queue_ctrl_apply_setting (ctrl, &mw->view_setting);
		treeview = ctrl->view;
		selection = gtk_tree_view_get_selection (treeview);
		gtk_paned_add1 (mw->vpaned, ctrl->top_widget);
		mw->id_cursor_changed = g_signal_connect (treeview, "cursor-changed",
		                        G_CALLBACK(on_download_cursor_changed), mw);
		mw->id_selection = g_signal_connect (selection, "changed",
		                        G_CALLBACK(on_download_selection_changed), mw);
		mw->id_button_event = g_signal_connect (treeview, "button-press-event",
		                        G_CALLBACK(on_download_button_event), mw);
		on_download_selection_changed (selection, mw);
		gtk_tree_view_set_rules_hint (ctrl->view, mw->view_setting.download.rules_hint);
	}

	mw->queue_ctrl = ctrl;
	main_window_message_area_refresh (mw);
	main_window_statusbar_refresh (mw);
}

void main_window_set_queue_state (MainWindow* mw, gboolean queuing)
{
	if (queuing) {
//		gtk_widget_set_sensitive (mw->global_menu.start_all, FALSE);
		gtk_widget_set_sensitive (mw->global_menu.pause_all, TRUE);
		gtk_image_set_from_stock (mw->status_image, GTK_STOCK_YES, GTK_ICON_SIZE_MENU);
	} else {
		gtk_widget_set_sensitive (mw->global_menu.start_all, TRUE);
		gtk_widget_set_sensitive (mw->global_menu.pause_all, FALSE);
		gtk_image_set_from_stock (mw->status_image, GTK_STOCK_NO, GTK_ICON_SIZE_MENU);
	}
}

void main_window_finalize (MainWindow* mw)
{
	if (mw->queue_ctrl) {
		queue_ctrl_unref (mw->queue_ctrl);
	}
}

void main_window_set_download_sensitive (MainWindow* mw, gboolean sens)
{
	// tool button
	gtk_widget_set_sensitive (mw->button_start, sens);
	gtk_widget_set_sensitive (mw->button_stop, sens);
	gtk_widget_set_sensitive (mw->button_erase, sens);
	gtk_widget_set_sensitive (mw->button_property, sens);
	gtk_widget_set_sensitive (mw->button_move_up, sens);
	gtk_widget_set_sensitive (mw->button_move_down, sens);
	gtk_widget_set_sensitive (mw->button_move_top, sens);
	gtk_widget_set_sensitive (mw->button_move_bottom, sens);
	// menu item
	gtk_widget_set_sensitive (mw->download_menu.start, sens);
	gtk_widget_set_sensitive (mw->download_menu.stop, sens);
	gtk_widget_set_sensitive (mw->download_menu.erase, sens);
	gtk_widget_set_sensitive (mw->download_menu.move_to, sens);
	gtk_widget_set_sensitive (mw->download_menu.properties, sens);
	gtk_widget_set_sensitive (mw->download_menu.move_up, sens);
	gtk_widget_set_sensitive (mw->download_menu.move_down, sens);
	gtk_widget_set_sensitive (mw->download_menu.move_top, sens);
	gtk_widget_set_sensitive (mw->download_menu.move_bottom, sens);
}

void main_window_set_category_sensitive (MainWindow* mw, gboolean sens)
{
	// tool button for category
	gtk_widget_set_sensitive (mw->button_category_erase, sens);
//	gtk_widget_set_sensitive (mw->button_export, sens);
	// tool button
	gtk_widget_set_sensitive (mw->button_new, sens);
	gtk_widget_set_sensitive (mw->button_batch, sens);
	// download menu item
	gtk_widget_set_sensitive (mw->download_menu.create, sens);
	gtk_widget_set_sensitive (mw->download_menu.add_batch, sens);
	// category menu item
	gtk_widget_set_sensitive (mw->category_menu.erase, sens);
	gtk_widget_set_sensitive (mw->category_menu.properties, sens);
	// file menu item
	gtk_widget_set_sensitive (mw->file_menu.export_url, sens);
}

void main_window_view_set (MainWindow* mw)
{
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mw->view_menu.toolbar),
	                                mw->view_setting.toolbar_visible);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mw->view_menu.statusbar),
	                                mw->view_setting.statusbar_visible);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mw->view_menu.msg_area),
	                                mw->view_setting.message.visible);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mw->view_menu.download_rules_hint),
	                                mw->view_setting.download.rules_hint);

	if (mw->view_setting.message.visible)
		gtk_widget_show (GTK_WIDGET (mw->message_area));
	else
		gtk_widget_hide (GTK_WIDGET (mw->message_area));

	if (mw->view_setting.toolbar_visible)
		gtk_widget_show (GTK_WIDGET (mw->toolbar));
	else
		gtk_widget_hide (GTK_WIDGET (mw->toolbar));

	if (mw->view_setting.statusbar_visible) {
		gtk_widget_show (GTK_WIDGET (mw->status_image));
		gtk_widget_show (GTK_WIDGET (mw->statusbar));
	} else {
		gtk_widget_hide (GTK_WIDGET (mw->status_image));
		gtk_widget_hide (GTK_WIDGET (mw->statusbar));
	}

	if (mw->queue_ctrl) {
		gtk_tree_view_set_rules_hint (mw->queue_ctrl->view, mw->view_setting.download.rules_hint);
	}
}

void main_window_view_get (MainWindow* mw)
{
	gboolean visible;

	visible = gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (mw->view_menu.toolbar));
	mw->view_setting.toolbar_visible = visible;

	visible = gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (mw->view_menu.statusbar));
	mw->view_setting.statusbar_visible = visible;

	visible = gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (mw->view_menu.msg_area));
	mw->view_setting.message.visible = visible;

	visible = gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (mw->view_menu.download_rules_hint));
	mw->view_setting.download.rules_hint = visible;
}

void main_window_view_apply (MainWindow* mw)
{
	mw->view_setting_loading = TRUE;

	main_window_view_set (mw);

	mw->view_setting_loading = FALSE;
}

// Message area
void main_window_message_area_refresh (MainWindow* mw)
{
	DownloadNode* dnode;
	GtkTreeView*  treeview;
	GtkTreeIter   iter;
	GtkTreePath*  path;
	GtkTreeModel* model;

	if (mw->queue_ctrl == NULL) {
		main_window_message_area_set (mw, NULL);
		return;
	}

	treeview = mw->queue_ctrl->view;
	gtk_tree_view_get_cursor (treeview, &path, NULL);
	if (path==NULL) {
		main_window_message_area_set (mw, NULL);
	} else {
		model = gtk_tree_view_get_model (treeview);
		gtk_tree_model_get_iter (model, &iter, path);
		gtk_tree_path_free (path);
		gtk_tree_model_get (model, &iter, 0, &dnode, -1);
		main_window_message_area_set (mw, dnode);
	}
}

void main_window_message_area_set (MainWindow* mw, DownloadNode* dnode)
{
	GtkTextBuffer*   textbuf;

	textbuf = gtk_text_view_get_buffer (mw->text_view);
	gtk_text_buffer_set_text (textbuf, "", 0);

	if (dnode==NULL)
		return;

	if (mw->view_setting.message.filename.visible) {
		text_buffer_add_text (textbuf, mw->text_tag, _("Filename:"),
		                      dnode->filename, TRUE);
	}
	if (mw->view_setting.message.directory.visible) {
		text_buffer_add_text (textbuf, mw->text_tag, _("Directory:"),
		                      dnode->directory, TRUE);
	}
	if (mw->view_setting.message.url.visible) {
		text_buffer_add_text (textbuf, mw->text_tag, _("URL:"),
		                      dnode->url, TRUE);
	}
	if (mw->view_setting.message.message.visible) {
		text_buffer_add_text (textbuf, mw->text_tag, _("Last Message:"),
		                      (dnode->message) ? dnode->message : _("None"),
		                      FALSE);
	}
}

// Status bar

void main_window_statusbar_refresh (MainWindow* mw)
{
	int  n_selected;
	GtkTreeView*  treeview;
	GtkTreeSelection* selection;

	if (mw->queue_ctrl == NULL) {
		statusbar_set_n_selected (mw->statusbar, 0);
		return;
	}

	treeview = mw->queue_ctrl->view;
	selection = gtk_tree_view_get_selection (treeview);
	n_selected = gtk_tree_selection_count_selected_rows (selection);
	statusbar_set_n_selected (mw->statusbar, n_selected);
}

