/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

/*
 *  Code base on Gtk+ 2.0 Tree View tutorial 
 *      - a tutorial covering GtkTreeView by Tim-Philipp Mu"ller
 */


#include "node_list_model.h"


/* boring declarations of local functions */

static void         node_list_model_init            (NodeListModel      *pkg_tree);

static void         node_list_model_class_init      (NodeListModelClass *klass);

static void         node_list_model_tree_model_init (GtkTreeModelIface *iface);

static void         node_list_model_finalize        (GObject           *object);

static GtkTreeModelFlags node_list_model_get_flags  (GtkTreeModel      *tree_model);

static gint         node_list_model_get_n_columns   (GtkTreeModel      *tree_model);

static GType        node_list_model_get_column_type (GtkTreeModel      *tree_model,
                                                     gint               index);

static gboolean     node_list_model_get_iter        (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     GtkTreePath       *path);

static GtkTreePath *node_list_model_get_path        (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter);

static void         node_list_model_get_value       (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     gint               column,
                                                     GValue            *value);

static gboolean     node_list_model_iter_next       (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter);

static gboolean     node_list_model_iter_children   (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     GtkTreeIter       *parent);

static gboolean     node_list_model_iter_has_child  (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter);

static gint         node_list_model_iter_n_children (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter);

static gboolean     node_list_model_iter_nth_child  (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     GtkTreeIter       *parent,
                                                     gint               n);

static gboolean     node_list_model_iter_parent     (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     GtkTreeIter       *child);

static GObjectClass *parent_class = NULL;  /* GObject stuff - nothing to worry about */


/*****************************************************************************
 *
 *                        here we register our new type and its interfaces
 *                        with the type system. If you want to implement
 *                        additional interfaces like GtkTreeSortable, you
 *                        will need to do it here.
 *
 *****************************************************************************/

GType node_list_model_get_type (void)
{
	static GType node_list_model_type = 0;

	if (node_list_model_type)
		return node_list_model_type;

	/* Some boilerplate type registration stuff */
	if (1) {
		static const GTypeInfo node_list_model_info = {
			sizeof (NodeListModelClass),
			NULL,                                         /* base_init */
			NULL,                                         /* base_finalize */
			(GClassInitFunc) node_list_model_class_init,
			NULL,                                         /* class finalize */
			NULL,                                         /* class_data */
			sizeof (NodeListModel),
			0,                                            /* n_preallocs */
			(GInstanceInitFunc) node_list_model_init
		};

		node_list_model_type = g_type_register_static (G_TYPE_OBJECT,
		                                               "NodeListModel",
		                                               &node_list_model_info,
		                                               (GTypeFlags)0);
	}

	if (1) {
		static const GInterfaceInfo tree_model_info = {
			(GInterfaceInitFunc) node_list_model_tree_model_init,
			NULL,
			NULL
		};

		g_type_add_interface_static (node_list_model_type,
		                             GTK_TYPE_TREE_MODEL,
		                             &tree_model_info);
	}

	return node_list_model_type;
}


/*****************************************************************************
 *
 *                          more boilerplate GObject/GType stuff.
 *                          Init callback for the type system,
 *                          called once when our new class is created.
 *
 *****************************************************************************/

static void node_list_model_class_init (NodeListModelClass *klass)
{
	GObjectClass* object_class;

	// parent_class is a global value...
	parent_class = (GObjectClass*) g_type_class_peek_parent (klass);
	object_class = (GObjectClass*) klass;

	object_class->finalize = node_list_model_finalize;
}

/*****************************************************************************
 *
 *                               init callback for the interface registration
 *                               in custom_list_get_type. Here we override
 *                               the GtkTreeModel interface functions that
 *                               we implement.
 *
 *****************************************************************************/

static void
node_list_model_tree_model_init (GtkTreeModelIface *iface)
{
	iface->get_flags       = node_list_model_get_flags;
	iface->get_n_columns   = node_list_model_get_n_columns;
	iface->get_column_type = node_list_model_get_column_type;
	iface->get_iter        = node_list_model_get_iter;
	iface->get_path        = node_list_model_get_path;
	iface->get_value       = node_list_model_get_value;
	iface->iter_next       = node_list_model_iter_next;
	iface->iter_children   = node_list_model_iter_children;
	iface->iter_has_child  = node_list_model_iter_has_child;
	iface->iter_n_children = node_list_model_iter_n_children;
	iface->iter_nth_child  = node_list_model_iter_nth_child;
	iface->iter_parent     = node_list_model_iter_parent;
}


/*****************************************************************************
 *
 *  node_list_model_init: this is called everytime a new custom list object
 *                    instance is created (we do that in node_list_model_new).
 *                    Initialise the list structure's fields here.
 *
 *****************************************************************************/

static void node_list_model_init (NodeListModel *node_list_model)
{
	/* Random int to check whether an iter belongs to our model */
	node_list_model->stamp = g_random_int();
}


/*****************************************************************************
 *
 *  node_list_model_finalize: this is called just before a custom list is
 *                        destroyed. Free dynamically allocated memory here.
 *
 *****************************************************************************/

static void node_list_model_finalize (GObject *object)
{
	NodeListModel* model = NODE_LIST_MODEL (object);

	base_node_inserted_disconnect (model->node, model->inserted_signal);
	base_node_removed_disconnect (model->node, model->removed_signal);
	base_node_unref (model->node);
	(* parent_class->finalize) (object);
}


/*****************************************************************************
 *
 *  node_list_model_get_flags: tells the rest of the world whether our tree model
 *                         has any special characteristics. In our case,
 *                         we have a list model (instead of a tree), and each
 *                         tree iter is valid as long as the row in question
 *                         exists, as it only contains a pointer to our struct.
 *
 *****************************************************************************/

static GtkTreeModelFlags
node_list_model_get_flags (GtkTreeModel *tree_model)
{
	g_return_val_if_fail (IS_NODE_LIST_MODEL(tree_model), (GtkTreeModelFlags)0);

	return (GTK_TREE_MODEL_LIST_ONLY | GTK_TREE_MODEL_ITERS_PERSIST);
}


/*****************************************************************************
 *
 *  node_list_model_get_n_columns: tells the rest of the world how many data
 *                             columns we export via the tree model interface
 *
 *****************************************************************************/

static gint node_list_model_get_n_columns (GtkTreeModel *tree_model)
{
	return 1;
}


/*****************************************************************************
 *
 *  node_list_model_get_column_type: tells the rest of the world which type of
 *                               data an exported model column contains
 *
 *****************************************************************************/

static GType
node_list_model_get_column_type (GtkTreeModel *tree_model,
                                 gint          index)
{
	g_return_val_if_fail (IS_NODE_LIST_MODEL(tree_model), G_TYPE_INVALID);
	g_return_val_if_fail (index < 1 && index >= 0, G_TYPE_INVALID);
	return G_TYPE_POINTER;
}


/*****************************************************************************
 *
 *  node_list_model_get_iter: converts a tree path (physical position) into a
 *                        tree iter structure (the content of the iter
 *                        fields will only be used internally by our model).
 *                        We simply store a pointer to our CustomRecord
 *                        structure that represents that row in the tree iter.
 *
 *****************************************************************************/

static gboolean
node_list_model_get_iter (GtkTreeModel *tree_model,
                          GtkTreeIter  *iter,
                          GtkTreePath  *path)
{
	NodeListModel  *node_list_model;
	gint           *indices, n, depth;

	g_assert(IS_NODE_LIST_MODEL(tree_model));
	g_assert(path!=NULL);

	node_list_model = NODE_LIST_MODEL(tree_model);

	indices = gtk_tree_path_get_indices(path);
	depth   = gtk_tree_path_get_depth(path);

	/* we do not allow children */
	g_assert(depth == 1); /* depth 1 = top level; a list only has top level nodes and no children */

	n = indices[0]; /* the n-th top level row */

	iter->stamp      = node_list_model->stamp;
	iter->user_data  = base_node_nth_child (node_list_model->node, n);
	iter->user_data2 = NULL;   /* unused */
	iter->user_data3 = NULL;   /* unused */

	return (iter->user_data) ? TRUE : FALSE;
}


/*****************************************************************************
 *
 *  node_list_model_get_path: converts a tree iter into a tree path (ie. the
 *                        physical position of that row in the list).
 *
 *****************************************************************************/

static GtkTreePath *
node_list_model_get_path (GtkTreeModel *tree_model,
                          GtkTreeIter  *iter)
{
	BaseNode  *node;
	GtkTreePath  *path;
	NodeListModel  *node_list_model;

	g_return_val_if_fail (IS_NODE_LIST_MODEL(tree_model), NULL);
	g_return_val_if_fail (iter != NULL,               NULL);
	g_return_val_if_fail (iter->user_data != NULL,    NULL);

	node_list_model = NODE_LIST_MODEL(tree_model);

	node = (BaseNode*) iter->user_data;

	path = gtk_tree_path_new();
	gtk_tree_path_append_index(path, base_node_child_position (node_list_model->node, node));

	return path;
}


/*****************************************************************************
 *
 *  node_list_model_get_value: Returns a row's exported data columns
 *                         (_get_value is what gtk_tree_model_get uses)
 *
 *****************************************************************************/

static void
node_list_model_get_value (GtkTreeModel *tree_model,
                           GtkTreeIter  *iter,
                           gint          column,
                           GValue       *value)
{
	BaseNode*  node;
	NodeListModel    *node_list_model;

	g_return_if_fail (IS_NODE_LIST_MODEL (tree_model));
	g_return_if_fail (iter != NULL);

	g_value_init (value, G_TYPE_POINTER);

	node_list_model = NODE_LIST_MODEL(tree_model);

	node = (BaseNode*) iter->user_data;

	g_value_set_pointer(value, node);
}


/*****************************************************************************
 *
 *  node_list_model_iter_next: Takes an iter structure and sets it to point
 *                         to the next row.
 *
 *****************************************************************************/

static gboolean
node_list_model_iter_next (GtkTreeModel  *tree_model,
                           GtkTreeIter   *iter)
{
	BaseNode*  node;
	NodeListModel    *node_list_model;

	g_return_val_if_fail (IS_NODE_LIST_MODEL (tree_model), FALSE);

	if (iter == NULL || iter->user_data == NULL)
		return FALSE;

	node_list_model = NODE_LIST_MODEL(tree_model);

	node = iter->user_data;
	node = base_node_next (node);

	if (node == NULL)
		return FALSE;

	iter->stamp     = node_list_model->stamp;
	iter->user_data = node;

	return TRUE;
}


/*****************************************************************************
 *
 *  node_list_model_iter_children: Returns TRUE or FALSE depending on whether
 *                             the row specified by 'parent' has any children.
 *                             If it has children, then 'iter' is set to
 *                             point to the first child. Special case: if
 *                             'parent' is NULL, then the first top-level
 *                             row should be returned if it exists.
 *
 *****************************************************************************/

static gboolean
node_list_model_iter_children (GtkTreeModel *tree_model,
                               GtkTreeIter  *iter,
                               GtkTreeIter  *parent)
{
	BaseNode*  node;
	NodeListModel  *node_list_model;

	g_return_val_if_fail (parent == NULL || parent->user_data != NULL, FALSE);

	/* this is a list, nodes have no children */
	if (parent)
		return FALSE;

	/* parent == NULL is a special case; we need to return the first top-level row */

	g_return_val_if_fail (IS_NODE_LIST_MODEL (tree_model), FALSE);

	node_list_model = NODE_LIST_MODEL(tree_model);

	node = base_node_first_child (node_list_model->node);
	if (node == NULL)
		return FALSE;

	/* Set iter to first item in list */
	iter->stamp     = node_list_model->stamp;
	iter->user_data = node;

	return TRUE;
}


/*****************************************************************************
 *
 *  node_list_model_iter_has_child: Returns TRUE or FALSE depending on whether
 *                              the row specified by 'iter' has any children.
 *                              We only have a list and thus no children.
 *
 *****************************************************************************/

static gboolean
node_list_model_iter_has_child (GtkTreeModel *tree_model,
                                GtkTreeIter  *iter)
{
	return FALSE;
}


/*****************************************************************************
 *
 *  node_list_model_iter_n_children: Returns the number of children the row
 *                               specified by 'iter' has. This is usually 0,
 *                               as we only have a list and thus do not have
 *                               any children to any rows. A special case is
 *                               when 'iter' is NULL, in which case we need
 *                               to return the number of top-level nodes,
 *                               ie. the number of rows in our list.
 *
 *****************************************************************************/

static gint
node_list_model_iter_n_children (GtkTreeModel *tree_model,
                                 GtkTreeIter  *iter)
{
	NodeListModel  *node_list_model;

	g_return_val_if_fail (IS_NODE_LIST_MODEL (tree_model), -1);
	g_return_val_if_fail (iter == NULL || iter->user_data != NULL, FALSE);

	node_list_model = NODE_LIST_MODEL(tree_model);

	/* special case: if iter == NULL, return number of top-level rows */
	if (!iter)
		return base_node_n_children (node_list_model->node);

	return 0; /* otherwise, this is easy again for a list */
}


/*****************************************************************************
 *
 *  node_list_model_iter_nth_child: If the row specified by 'parent' has any
 *                              children, set 'iter' to the n-th child and
 *                              return TRUE if it exists, otherwise FALSE.
 *                              A special case is when 'parent' is NULL, in
 *                              which case we need to set 'iter' to the n-th
 *                              row if it exists.
 *
 *****************************************************************************/

static gboolean
node_list_model_iter_nth_child (GtkTreeModel *tree_model,
                                GtkTreeIter  *iter,
                                GtkTreeIter  *parent,
                                gint          n)
{
	BaseNode  *node;
	NodeListModel    *node_list_model;

	g_return_val_if_fail (IS_NODE_LIST_MODEL (tree_model), FALSE);

	node_list_model = NODE_LIST_MODEL(tree_model);

	/* a list has only top-level rows */
	if(parent)
		return FALSE;

	/* special case: if parent == NULL, set iter to n-th top-level row */

	node = base_node_nth_child (node_list_model->node, n);
	if( node == NULL )
		return FALSE;

	iter->stamp = node_list_model->stamp;
	iter->user_data = node;

	return TRUE;
}


/*****************************************************************************
 *
 *  node_list_model_iter_parent: Point 'iter' to the parent node of 'child'. As
 *                           we have a list and thus no children and no
 *                           parents of children, we can just return FALSE.
 *
 *****************************************************************************/

static gboolean
node_list_model_iter_parent (GtkTreeModel *tree_model,
                             GtkTreeIter  *iter,
                             GtkTreeIter  *child)
{
	return FALSE;
}


/*****************************************************************************
 *
 *  node_list_model_new:  This is what you use in your own code to create a
 *                    new custom list tree model for you to use.
 *
 *****************************************************************************/

static void on_node_insert (BaseNode* parent, BaseNode* child, int nth, gpointer data)
{
	GtkTreeModel* model = data;
	GtkTreeIter   iter;
	GtkTreePath*  path;

	iter.stamp     = NODE_LIST_MODEL (data)->stamp;
	iter.user_data = child;
	path = gtk_tree_path_new ();
	gtk_tree_path_append_index (path, nth);
	gtk_tree_model_row_inserted (model, path, &iter);
	gtk_tree_path_free (path);
}

static void on_node_remove (BaseNode* parent, BaseNode* child, int nth, gpointer data)
{
	GtkTreeModel* model = data;
	GtkTreePath*  path;

	path = gtk_tree_path_new ();
	gtk_tree_path_append_index (path, nth);
	gtk_tree_model_row_deleted (model, path);
	gtk_tree_path_free (path);
}

NodeListModel* node_list_model_new (BaseNode* parent)
{
	NodeListModel *newlist;

	newlist = (NodeListModel*) g_object_new (TYPE_NODE_LIST_MODEL, NULL);
	g_assert( newlist != NULL );

	base_node_ref (parent);
	newlist->node = parent;
	newlist->inserted_signal = base_node_inserted_connect (parent, on_node_insert, newlist);
	newlist->removed_signal = base_node_removed_connect (parent, on_node_remove, newlist);

	return newlist;
}

