/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <urlglib/ug_i18n.h>
#include "url_filter_dialog.h"
#include "item_list_view.h"
#include "item_list_model.h"

// signal handler ----------------------------------------------------
static void on_choose(GtkWidget* w, UrlFilterDialog* ufd)
{
	gint   n_page;
	UrlFilterPage* page;

	n_page = gtk_notebook_get_current_page (GTK_NOTEBOOK(ufd->notebook));
	if (n_page < 0)
		return;

	page   = g_slist_nth (ufd->page_list, n_page)->data;
	gtk_tree_view_set_model (ufd->address_view, page->model_addr);
	gtk_tree_view_set_model (ufd->extension_view, page->model_ext);
	url_item_list_set_item_all (URL_ITEM_LIST (page->class_addr), TRUE, 0);
	url_item_list_set_item_all (URL_ITEM_LIST (page->class_ext), FALSE, 0);

	if (gtk_dialog_run (ufd->chooser_dialog)==GTK_RESPONSE_OK) {
		url_item_list_set_item_all (URL_ITEM_LIST (page->url_filter), FALSE, 0);
		url_filter_pick_by_marked_class (page->url_filter);
		url_filter_mark_item (page->url_filter, 2, 2);
	}
	gtk_widget_hide (GTK_WIDGET (ufd->chooser_dialog));
	gtk_widget_queue_draw (GTK_WIDGET (page->view));
}

static void on_select_all(GtkWidget* w, UrlFilterDialog* ufd)
{
	gint   n_page;
	UrlFilterPage* page;

	n_page = gtk_notebook_get_current_page (GTK_NOTEBOOK(ufd->notebook));
	if (n_page < 0)
		return;

	page   = g_slist_nth (ufd->page_list, n_page)->data;
	url_filter_set_item_all (page->url_filter, TRUE, 0);
	gtk_widget_queue_draw (GTK_WIDGET (page->view));
}

static void on_select_none (GtkWidget* w, UrlFilterDialog* ufd)
{
	gint   n_page;
	UrlFilterPage* page;

	n_page = gtk_notebook_get_current_page (GTK_NOTEBOOK(ufd->notebook));
	if (n_page < 0)
		return;

	page   = g_slist_nth (ufd->page_list, n_page)->data;
	url_filter_set_item_all (page->url_filter, FALSE, 0);
	gtk_widget_queue_draw (GTK_WIDGET (page->view));
}

static void on_class_select_all (GtkWidget* w, UrlFilterDialog* ufd)
{
	gint   n_page;
	UrlFilterPage* page;

	n_page = gtk_notebook_get_current_page (GTK_NOTEBOOK(ufd->notebook));
	page   = g_slist_nth (ufd->page_list, n_page)->data;

	if (w == ufd->button_all_address) {
		url_filter_class_set_item_all (page->class_addr, TRUE, 0);
		gtk_widget_queue_draw (GTK_WIDGET (ufd->address_view));
	}
	else {
		url_filter_class_set_item_all (page->class_ext, TRUE, 0);
		gtk_widget_queue_draw (GTK_WIDGET (ufd->extension_view));
	}
}

static void on_class_select_none (GtkWidget* w, UrlFilterDialog* ufd)
{
	gint   n_page;
	UrlFilterPage* page;

	n_page = gtk_notebook_get_current_page (GTK_NOTEBOOK(ufd->notebook));
	page   = g_slist_nth (ufd->page_list, n_page)->data;

	if (w == ufd->button_none_address) {
		url_filter_class_set_item_all (page->class_addr, FALSE, 0);
		gtk_widget_queue_draw (GTK_WIDGET (ufd->address_view));
	}
	else {
		url_filter_class_set_item_all (page->class_ext, FALSE, 0);
		gtk_widget_queue_draw (GTK_WIDGET (ufd->extension_view));
	}
}

// member functions --------------------------------------------------
UrlFilterDialog* url_filter_dialog_new (GtkWindow* parent)
{
	UrlFilterDialog* ufd = g_malloc0 (sizeof (UrlFilterDialog));
	GtkWidget* vbox;
	GtkWidget* hbox;
	GtkWidget* action_area;
	GtkWidget* button;
	GtkTable*  table;
	GtkScrolledWindow* scrolled;

	ufd->page_list = NULL;
	ufd->self = (GtkDialog*)gtk_dialog_new ();
	gtk_window_set_destroy_with_parent (GTK_WINDOW (ufd->self), TRUE);
	g_object_set (ufd->self, "has-separator", FALSE, NULL);
	if (parent)
		gtk_window_set_transient_for (GTK_WINDOW (ufd->self), parent);

	vbox = GTK_DIALOG (ufd->self)->vbox;
	action_area = GTK_DIALOG (ufd->self)->action_area;

	ufd->notebook = gtk_notebook_new ();
	ufd->separator = gtk_hseparator_new ();

	// setup dialog & vbox
	gtk_window_set_default_size (GTK_WINDOW(ufd->self), 450, 350);
//	gtk_container_set_border_width (GTK_CONTAINER(ufd->self), 5);
	gtk_box_set_spacing (GTK_BOX(vbox), 5);

	// action_area ( GtkHButtonBox )
	button = gtk_button_new_with_label (_("Select none"));
	gtk_box_pack_start (GTK_BOX(action_area), button, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT(button), "clicked",
	                  G_CALLBACK(on_select_none), (gpointer)ufd);

	button = gtk_button_new_with_label (_("Select all"));
	gtk_box_pack_start (GTK_BOX(action_area), button, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT(button), "clicked",
	                  G_CALLBACK(on_select_all), (gpointer)ufd);

	button = gtk_button_new_with_label (_("Choose"));
	gtk_box_pack_start (GTK_BOX(action_area), button, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT(button), "clicked", 
	                  G_CALLBACK(on_choose), (gpointer)ufd);

	// response buttons
	gtk_dialog_add_button (GTK_DIALOG(ufd->self),
	                       GTK_STOCK_CANCEL,
	                       GTK_RESPONSE_CANCEL);
	gtk_dialog_add_button (GTK_DIALOG(ufd->self),
	                       GTK_STOCK_OK,
	                       GTK_RESPONSE_OK);

	// vbox
	hbox = gtk_hbox_new (FALSE, 5);
	ufd->entry_base_href = gtk_entry_new ();
	ufd->label_base_href = gtk_label_new_with_mnemonic (_("_Base href"));
	gtk_label_set_mnemonic_widget (GTK_LABEL(ufd->label_base_href), ufd->entry_base_href);
	gtk_box_pack_start (GTK_BOX(hbox), ufd->label_base_href, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox), ufd->entry_base_href, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), ufd->separator, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), ufd->notebook, TRUE, TRUE, 0);
	gtk_widget_show_all (vbox);

	// class chooser dialog
	ufd->chooser_dialog = (GtkDialog*)gtk_dialog_new_with_buttons (
	                                                      _("Chose by condition"),
	                                                      GTK_WINDOW(ufd->self),
	                                                      GTK_DIALOG_MODAL,
	                                                      GTK_STOCK_CANCEL,
	                                                      GTK_RESPONSE_CANCEL,
	                                                      GTK_STOCK_OK,
	                                                      GTK_RESPONSE_OK,
	                                                      NULL);
	vbox = GTK_DIALOG (ufd->chooser_dialog)->vbox;
	table = (GtkTable*)gtk_table_new (3, 5, FALSE);
	gtk_table_attach (table, gtk_label_new (_("URLs selected by host name and file extension")), 0, 5, 0, 1,
	                  GTK_EXPAND | GTK_FILL, GTK_SHRINK, 1, 2);

	ufd->button_all_address = gtk_button_new_with_mnemonic (_("Select _All"));
	ufd->button_none_address = gtk_button_new_with_mnemonic (_("Select _None"));
	ufd->button_all_extension = gtk_button_new_with_mnemonic (_("Select A_ll"));
	ufd->button_none_extension = gtk_button_new_with_mnemonic (_("Select Non_e"));
	gtk_table_attach (table, ufd->button_all_address, 0, 1, 2, 3,
	                  GTK_SHRINK, GTK_SHRINK, 1, 2);
	gtk_table_attach (table, ufd->button_none_address, 1, 2, 2, 3,
	                  GTK_SHRINK, GTK_SHRINK, 1, 2);
	gtk_table_attach (table, gtk_label_new (NULL), 2, 3, 2, 3,
	                  GTK_EXPAND, GTK_SHRINK, 1, 2);
	gtk_table_attach (table, ufd->button_all_extension, 3, 4, 2, 3,
	                  GTK_SHRINK, GTK_SHRINK, 1, 2);
	gtk_table_attach (table, ufd->button_none_extension, 4, 5, 2, 3,
	                  GTK_SHRINK, GTK_SHRINK, 1, 2);

	// create address_view & extension_view
	ufd->address_view   = item_list_view_new (_("Host Name"));
	scrolled = (GtkScrolledWindow*) gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_set_size_request (GTK_WIDGET (scrolled), 120, 150);
	gtk_scrolled_window_set_shadow_type (scrolled, GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (scrolled, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scrolled), GTK_WIDGET (ufd->address_view));
	gtk_table_attach (table, GTK_WIDGET (scrolled), 0, 3, 1, 2,
	                  GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 1, 2);

	ufd->extension_view = item_list_view_new (_("File Extension"));
	scrolled = (GtkScrolledWindow*) gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_set_size_request (GTK_WIDGET (scrolled), 120, 150);
	gtk_scrolled_window_set_shadow_type (scrolled, GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (scrolled, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scrolled), GTK_WIDGET (ufd->extension_view));
	gtk_table_attach (table, GTK_WIDGET (scrolled), 3, 5, 1, 2,
	                  GTK_FILL, GTK_EXPAND | GTK_FILL, 1, 2);

	// setup dialog & vbox
	gtk_box_set_spacing (GTK_BOX(vbox), 5);
	gtk_window_set_default_size (GTK_WINDOW(ufd->chooser_dialog), 420, 320);
//	gtk_container_set_border_width (GTK_CONTAINER(ufd->chooser_dialog), 5);

	gtk_box_pack_start (GTK_BOX(vbox), GTK_WIDGET (table), TRUE, TRUE, 0);
	gtk_widget_show_all (vbox);

	// signal connect
	g_signal_connect (G_OBJECT(ufd->button_all_address), "clicked",
	                  G_CALLBACK(on_class_select_all), (gpointer)ufd);
	g_signal_connect (G_OBJECT(ufd->button_none_address), "clicked",
	                  G_CALLBACK(on_class_select_none), (gpointer)ufd);
	g_signal_connect (G_OBJECT(ufd->button_all_extension), "clicked",
	                  G_CALLBACK(on_class_select_all), (gpointer)ufd);
	g_signal_connect (G_OBJECT(ufd->button_none_extension), "clicked",
	                  G_CALLBACK(on_class_select_none), (gpointer)ufd);

	return ufd;
}

void url_filter_dialog_destroy (UrlFilterDialog* ufd)
{
	GSList* node;
	UrlFilterPage* page;

	gtk_widget_destroy ((GtkWidget*)ufd->chooser_dialog);
	gtk_widget_destroy ((GtkWidget*)ufd->self);

	for (node=ufd->page_list; node; node=node->next) {
		page = node->data;
		g_object_unref (page->model_url);
		g_object_unref (page->model_ext);
		g_object_unref (page->model_addr);
		url_filter_unref (page->url_filter);
		g_free (page);
	}
	g_slist_free (ufd->page_list);
	g_free (ufd);
}

void url_filter_dialog_add_filter (UrlFilterDialog* ufd, UrlFilter* uf, gchar* label_string)
{
	UrlFilterPage* page;
	GtkWidget*     label;

	page = g_malloc (sizeof(UrlFilterPage));
	ufd->page_list = g_slist_append (ufd->page_list, page);
	url_filter_ref (uf);
	url_item_list_set_item_all (URL_ITEM_LIST (uf), TRUE, 0);
	page->url_filter = uf;

	page->class_addr = url_filter_get_class (uf, url_filter_class_address);
	page->model_addr = (GtkTreeModel*)item_list_model_new (URL_ITEM_LIST (page->class_addr));

	page->class_ext  = url_filter_get_class (uf, url_filter_class_extension);
	page->model_ext  = (GtkTreeModel*)item_list_model_new (URL_ITEM_LIST (page->class_ext));

	page->model_url  = (GtkTreeModel*)item_list_model_new (URL_ITEM_LIST (uf));
	page->view = item_list_view_new (_("URL"));
	gtk_tree_view_set_model (page->view, page->model_url);
	gtk_container_set_border_width (GTK_CONTAINER (page->view), 5);

	page->scrolled = (GtkScrolledWindow*) gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_set_size_request (GTK_WIDGET (page->scrolled), 120, 150);
	gtk_scrolled_window_set_shadow_type (page->scrolled, GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (page->scrolled, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (page->scrolled), GTK_WIDGET (page->view));
	gtk_widget_show_all (GTK_WIDGET (page->scrolled));

	label = gtk_label_new_with_mnemonic (label_string);
	gtk_notebook_append_page (GTK_NOTEBOOK(ufd->notebook), GTK_WIDGET (page->scrolled), label);
}

