/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <glib.h>
#include "urlgfe_util.h"

static const gchar* _urlgfe_prog_dir   = NULL;
static const gchar* _urlgfe_conf_dir   = NULL;
static const gchar* _urlgfe_image_dir  = NULL;
static const gchar* _urlgfe_locale_dir = NULL;
static const gchar* _urlgfe_data_file  = NULL;
static const gchar* _urlgfe_conf_file  = NULL;

#ifdef _WIN32
#include <windows.h>

void  urlgfe_path_init (int argc, char *argv[])
{
	const gchar* home_dir = g_get_home_dir ();
	const gchar* prog_path = NULL;
	HMODULE hmod;

	hmod = GetModuleHandle(NULL);
	if( hmod ) {
		prog_path = g_malloc (256);
		GetModuleFileName( hmod, (LPTSTR)prog_path, 256 );
		_urlgfe_prog_dir = g_path_get_dirname (prog_path);
		g_free ((gpointer)prog_path);
	} else {
		_urlgfe_prog_dir = g_path_get_dirname (argv[0]);
	}

	_urlgfe_conf_dir  = g_strdup_printf ("%s" G_DIR_SEPARATOR_S "Application Data"
	                                        G_DIR_SEPARATOR_S ".urlgfe", home_dir);
	_urlgfe_conf_file = g_strdup_printf ("%s" G_DIR_SEPARATOR_S "setting.xml", _urlgfe_conf_dir);
	_urlgfe_data_file = g_strdup_printf ("%s" G_DIR_SEPARATOR_S "category.xml", _urlgfe_conf_dir);

	_urlgfe_image_dir  = g_strdup_printf ("%s" G_DIR_SEPARATOR_S "pixmaps", _urlgfe_prog_dir);
	_urlgfe_locale_dir = g_strdup_printf ("%s" G_DIR_SEPARATOR_S "locale", _urlgfe_prog_dir);
}
#else
void  urlgfe_path_init (int argc, char *argv[])
{
	const gchar* home_dir = g_get_home_dir ();

	_urlgfe_prog_dir  = g_path_get_dirname (argv[0]);
	_urlgfe_conf_dir  = g_strdup_printf ("%s" G_DIR_SEPARATOR_S ".urlgfe", home_dir);
	_urlgfe_conf_file = g_strdup_printf ("%s" G_DIR_SEPARATOR_S "setting.xml", _urlgfe_conf_dir);
	_urlgfe_data_file = g_strdup_printf ("%s" G_DIR_SEPARATOR_S "category.xml", _urlgfe_conf_dir);

# ifdef DATADIR
	_urlgfe_image_dir  = DATADIR "/pixmaps";
	_urlgfe_locale_dir = DATADIR "/locale";
# endif
}
#endif

const gchar* urlgfe_path_prog_dir ()
{
	return _urlgfe_prog_dir;
}

const gchar* urlgfe_path_conf_dir ()
{
	return _urlgfe_conf_dir;
}

const gchar* urlgfe_path_image_dir ()
{
	return _urlgfe_image_dir;
}

const gchar* urlgfe_path_locale_dir ()
{
	return _urlgfe_locale_dir;
}

const gchar* urlgfe_path_data_file ()
{
	return _urlgfe_data_file;
}

const gchar* urlgfe_path_conf_file ()
{
	return _urlgfe_conf_file;
}

// Directory

static GList* _direcotry_list = NULL;

void   directory_history_set (const gchar* directory)
{
	GList* cur_lnode;
	gchar* dir_str;
	int    length = 0;

	if (directory==NULL || *directory==0)
		return;

	for (cur_lnode=_direcotry_list; cur_lnode; cur_lnode=cur_lnode->next) {
		dir_str = cur_lnode->data;
		if (strcmp (dir_str, directory)==0) {
			_direcotry_list = g_list_delete_link (_direcotry_list, cur_lnode);
			_direcotry_list = g_list_prepend (_direcotry_list, dir_str);
			return;
		}

		length++;
		if (length==8) {
			g_free (cur_lnode->data);
			_direcotry_list = g_list_delete_link (_direcotry_list, cur_lnode);
			break;
		}
	}

	// add new directory
	dir_str = g_strdup (directory);
	_direcotry_list = g_list_prepend (_direcotry_list, dir_str);
}

GList* directory_history_get ()
{
	return _direcotry_list;
}

// log handler
static void dummy_print( const gchar* string )
{
	return;
}

static void dummy_log_handler (const gchar    *domain,
                               GLogLevelFlags  flags,
                               const gchar    *msg,
                               gpointer        user_data)
{
	return;
}

// disable log and console output in win32
void urlgfe_disable_console_output ()
{
	g_log_set_handler (NULL,
	                   G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
	                   dummy_log_handler, NULL);	
	g_log_set_handler ("Gdk",
	                   G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
	                   dummy_log_handler, NULL);	
	g_log_set_handler ("Gtk",
	                   G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
	                   dummy_log_handler, NULL);	
	g_log_set_handler ("GLib",
	                   G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
	                   dummy_log_handler, NULL);	
	g_log_set_handler ("GModule",
	                   G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
	                   dummy_log_handler, NULL);	
	g_log_set_handler ("GLib-GObject",
	                   G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
	                   dummy_log_handler, NULL);	
	g_log_set_handler ("GThread",
	                   G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
	                   dummy_log_handler, NULL);	

	g_set_print_handler( dummy_print );
}
