/*
 *
 *   Copyright (C) 2003-2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <stdarg.h>
#include <string.h>
#include <glib.h>
#include <urlglib/conf_writer.h>
#include <urlglib/urlglib_util.h>

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

void conf_writer_init (ConfWriter* cw, int formating)
{
	cw->element_status = 1;
	cw->formating = formating;
	cw->level = 0;
	cw->file = NULL;
}

void conf_writer_finalize (ConfWriter* cw)
{
	if (cw->file)
		fclose (cw->file);
}

int  conf_writer_decide_ended (ConfWriter* cw, int is_ending)
{
    char* string;

	if (cw->element_status == 0) {
//		cw->element_status = 1;
		string = (is_ending) ? "/>" : ">";
		fputs (string, cw->file);
		return TRUE;
	}

	return FALSE;
}

int  conf_writer_open_file (ConfWriter* cw, const char* filename)
{
	cw->file = ug_fopen (filename, "w");
	if (cw->file) {
		fputs ("<?xml version=\"1.0\"?>\n", cw->file);
		return TRUE;
	}
	return FALSE;
}

void conf_writer_start_element (ConfWriter* cw, const char* format_str, ...)
{
	FILE* file = cw->file;
	char* esc_text;
	va_list  arg_list;

	conf_writer_decide_ended (cw, FALSE);

	if (cw->formating) {
	    if (cw->level > 0)
			fprintf (file, "\n%*c", cw->level, ' ');
		cw->level ++;
	}
	fputc ('<', file);

	va_start (arg_list, format_str);
	esc_text = g_markup_vprintf_escaped (format_str, arg_list);
	va_end (arg_list);

	fputs (esc_text, file);
	g_free (esc_text);

	cw->element_status = 0;
}

void conf_writer_end_element (ConfWriter* cw, const char* element_name)
{
	FILE* file = cw->file;
	int   is_ended;

	is_ended = conf_writer_decide_ended (cw, TRUE);

	if (cw->formating) {
		cw->level --;
		if (cw->element_status != 2 && is_ended == FALSE) {
			fputc ('\n', file);
			if (cw->level > 0)
				fprintf (file, "%*c", cw->level, ' ');
		}
	}

	if (is_ended == FALSE)
		fprintf (file, "</%s>", element_name);

	cw->element_status = 1;
}

void conf_writer_text (ConfWriter* cw, const char* text)
{
	conf_writer_text_len (cw, text, -1);
}

void conf_writer_text_len (ConfWriter* cw, const char* text, int len)
{
	char* esc_text;

	if (text) {
		if (len == -1)
			len = strlen (text);
		if (len == 0)
			return;
		esc_text = g_markup_escape_text (text, len);
		conf_writer_decide_ended (cw, FALSE);
		fputs (esc_text, cw->file);
		cw->element_status = 2;
		g_free (esc_text);
	}
}

