/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#ifndef __DOWNLOAD_TASK_H
#define __DOWNLOAD_TASK_H

#include <glib.h>
#include <urlglib/download_node.h>

// Hierarchy :
//
// DownloadTask
// |
// +--- DownloadTaskCurl
// |
// +--- DownloadTaskWget ?
// |
// +--- DownloadTaskRtsp ?
// |
// `--- DownloadTaskMms  ?
//

#ifdef __cplusplus
extern "C" {
#endif

#define DOWNLOAD_TASK(obj)             ((DownloadTask*)(obj))

#define  DOWNLOAD_TASK_FIELDS            \
	DownloadTaskFinalizeFunc finalize;   \
	DownloadTaskRunFunc      run;        \
	int            reference_count;      \
	GThread*       thread;               \
	DownloadNode*  download;             \
	ProxyNode*     proxy;                \
	char*          fullpath;             \
	char*          url;                  \
	char*          referer;              \
	gpointer       user_data

typedef struct _DownloadTask         DownloadTask;

typedef DownloadTask* (*DownloadTaskNewFunc) ();
typedef void   (*DownloadTaskFinalizeFunc) (DownloadTask* dtask);
typedef void   (*DownloadTaskResetFunc)    (DownloadTask* dtask);
typedef int    (*DownloadTaskRunFunc)      (DownloadTask* dtask);

struct _DownloadTask {
    DOWNLOAD_TASK_FIELDS;
};

typedef enum {
	DOWNLOAD_TASK_OK,
	DOWNLOAD_TASK_ABORT,
	DOWNLOAD_TASK_CAN_RETRY,
	DOWNLOAD_TASK_ERROR,
	DOWNLOAD_TASK_ERROR_PROXY,
} DownloadTaskCode;

DownloadTask* download_task_new_from_protocol (const char* protocol, int protocol_len);
void download_task_instance_init (DownloadTask* dtask);
void download_task_instance_finalize (DownloadTask* dtask);

void download_task_ref (DownloadTask* dtask);
void download_task_unref (DownloadTask* dtask);

void download_task_retry_delay (DownloadTask* dtask);
void download_task_activate (DownloadTask* dtask);
void download_task_set (DownloadTask* dtask, DownloadNode* dnode, gpointer data);

#define download_task_is_active(dtask)    ( ((dtask)->thread) ? TRUE : FALSE )

// copy current proxy from DownloadNode, store it in dtask->proxy
// @return copied porxy
ProxyNode* download_task_load_proxy (DownloadTask* dtask);

// copy & encode current url & referer from DownloadNode, store it in dtask->url & dtask->referer
// if referer==NULL, generate it from url.
// @return copied url
char*      download_task_load_url (DownloadTask* dtask, gboolean encoding);

// copy & check current filename from DownloadNode, store it in dtask->filename
// @return copied filename
char*      download_task_load_fullpath (DownloadTask* dtask);

// Global functions --------------------------------------------------
void download_task_add_protocol  (const char*  protocol,  DownloadTaskNewFunc fn);
void download_task_add_protocols (const char** protocols, DownloadTaskNewFunc fn);

#ifdef __cplusplus
}
#endif

#endif  // End of __DOWNLOAD_TASK_H
