/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <string.h>
#include <stdlib.h>
#include <urlglib/proxy_node.h>
#include <urlglib/download_node.h>

static void proxy_node_finalize (ProxyNode* node);

ProxyNode* proxy_node_new ()
{
	ProxyNode*  node = g_malloc0 (sizeof (ProxyNode));

	// initialize base class
	base_node_instance_init (BASE_NODE (node));
	node->finalize = (BaseNodeFinalizeFunc) proxy_node_finalize;

//	node->type = 0;
//	node->authentication = 0;
//	node->port = 0;
//	node->address  = NULL;
//	node->username = NULL;
//	node->password = NULL;

	return node;
}

static void proxy_node_finalize (ProxyNode* node)
{
	g_free (node->host);
	g_free (node->username);
	g_free (node->password);

	// finalize base class
	base_node_instance_finalize (BASE_NODE (node));
}

void proxy_node_unref (ProxyNode* pnode)
{
	if (pnode->reference_count == 1) {
		proxy_node_unlink (pnode);
		base_node_unref (BASE_NODE (pnode));
	}
}

void proxy_node_set_host (ProxyNode* pnode, const char* host)
{
	DownloadNode* dnode = proxy_node_get_download (pnode);

	if (dnode)
		download_node_lock (dnode);

	str_replace (&pnode->host, host, -1);

	if (dnode)
		download_node_unlock (dnode);
}

void proxy_node_set_username (ProxyNode* pnode, const char* username)
{
	DownloadNode* dnode = proxy_node_get_download (pnode);

	if (dnode)
		download_node_lock (dnode);

	str_replace (&pnode->username, username, -1);

	if (dnode)
		download_node_unlock (dnode);
}

void proxy_node_set_password (ProxyNode* pnode, const char* password)
{
	DownloadNode* dnode = proxy_node_get_download (pnode);

	if (dnode)
		download_node_lock (dnode);

	str_replace (&pnode->password, password, -1);

	if (dnode)
		download_node_unlock (dnode);
}

void proxy_node_assign (ProxyNode* dest, ProxyNode* src)
{
	dest->type           = src->type;
	dest->authentication = src->authentication;
	dest->port           = src->port;

	proxy_node_set_host (dest, src->host);
	proxy_node_set_username (dest, src->username);
	proxy_node_set_password (dest, src->password);
}

void proxy_node_unlink (ProxyNode* pnode)
{
	DownloadNode* dnode = proxy_node_get_download (pnode);

	if (dnode)
		download_node_lock (dnode);

	base_node_unlink (BASE_NODE (pnode));

	if (dnode)
		download_node_unlock (dnode);
}

// config writer & parser --------------------------------------------

void proxy_node_write_conf (ProxyNode* node, ConfWriter* cw)
{
	conf_writer_start_element (cw, PROXY_NODE_TAG
	         " type=\"%d\""
	         " authentication=\"%d\""
	         " port=\"%d\""
	         " host=\"%s\""
	         " username=\"%s\""
	         " password=\"%s\"",
	         node->type,
	         node->authentication,
	         node->port,
	         (node->host) ? node->host : "",
	         (node->username) ? node->username : "",
	         (node->password) ? node->password : ""
	         );
	conf_writer_end_element (cw, PROXY_NODE_TAG);
}

// parser

void proxy_node_parser_attr (const gchar**  attr_names,
                             const gchar**  attr_values,
                             ProxyNode*     pnode)
{
	int ii;

	for (ii=0; attr_names[ii]; ii++) {
		if (strcmp (attr_names[ii], "type")==0)
			pnode->type = atoi (attr_values[ii]);
		else if (strcmp (attr_names[ii], "authentication")==0)
			pnode->authentication = atoi (attr_values[ii]);
		else if (strcmp (attr_names[ii], "port")==0)
			pnode->port = atoi (attr_values[ii]);
		else if (strcmp (attr_names[ii], "host")==0)
			str_replace_no_crlf (&pnode->host, attr_values[ii], -1);
		else if (strcmp (attr_names[ii], "username")==0)
			str_replace_no_crlf (&pnode->username, attr_values[ii], -1);
		else if (strcmp (attr_names[ii], "password")==0)
			str_replace_no_crlf (&pnode->password, attr_values[ii], -1);
	}
}

