/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <stdlib.h>
#include <string.h>
#include <urlglib/queue_node.h>
#include <urlglib/category_node.h>

static void queue_node_finalize (QueueNode* qnode)
{
	urlglib_signal_list_free (&qnode->changed_signal);
	base_node_instance_finalize (BASE_NODE (qnode));
}

QueueNode* queue_node_new ()
{
	QueueNode* qnode = g_malloc0 (sizeof (QueueNode));

	base_node_instance_init (BASE_NODE (qnode));
	qnode->finalize = (BaseNodeFinalizeFunc) queue_node_finalize;
	qnode->add = (QueueNodeAddFunc) base_node_append;
	return qnode;
}

QueueNode* queue_node_new_reverse ()
{
	QueueNode* qnode = queue_node_new ();

	qnode->add = (QueueNodeAddFunc) base_node_prepend;
	return qnode;
}

DownloadNode* queue_node_find (QueueNode* qnode, DownloadNodeState state,
                               DownloadNode* after_sibling)
{
	DownloadNode* dnode;

	if (after_sibling)
		dnode = download_node_next (after_sibling);
	else
		dnode = queue_node_first_download (qnode);

	while (dnode) {
		if (dnode->state == state)
			return dnode;
		dnode = download_node_next (dnode);
	}

	return NULL;
}

void queue_node_remove_after_nth (QueueNode* qnode, int n)
{
	DownloadNode* dnode;
	DownloadNode* dnode_next;

	dnode = queue_node_nth_download (qnode, n);
	while (dnode) {
		dnode_next = download_node_next (dnode);
		download_node_unlink (dnode);
		dnode = dnode_next;
	}
}

void queue_node_add (QueueNode* qnode, DownloadNode* dnode)
{
	qnode->add (qnode, dnode);
}

gboolean queue_node_is_waiting (QueueNode* qnode)
{
	CategoryNode* cnode;

	cnode = queue_node_get_category (qnode);
	if (cnode && category_node_waiting_queue (cnode) == qnode)
		return TRUE;

	return FALSE;
}

gboolean queue_node_is_completed (QueueNode* qnode)
{
	CategoryNode* cnode;

	cnode = queue_node_get_category (qnode);
	if (cnode && category_node_completed_queue (cnode) == qnode)
		return TRUE;

	return FALSE;
}

gboolean queue_node_is_recycled (QueueNode* qnode)
{
	CategoryNode* cnode;

	cnode = queue_node_get_category (qnode);
	if (cnode && category_node_recycled_queue (cnode) == qnode)
		return TRUE;

	return FALSE;
}

// config writer & parser --------------------------------------------
void  queue_node_write_conf (QueueNode* qnode, ConfWriter* cw)
{
	DownloadNode* dnode;
	CategoryNode* cnode;
	int  position;

	cnode = queue_node_get_category (qnode);
	position = base_node_child_position (BASE_NODE (cnode), BASE_NODE (qnode));
	conf_writer_start_element (cw, QUEUE_NODE_TAG " type=\"%d\"", position);

	dnode = queue_node_last_download (qnode);
	while (dnode) {
		download_node_write_conf (dnode, cw);
		dnode = download_node_prev (dnode);
	}

	conf_writer_end_element (cw, QUEUE_NODE_TAG);
}

// parser ---

int  queue_node_parser_attr (const gchar**     attr_names,
                             const gchar**     attr_values)
{
	int ii;

	for (ii=0; attr_names[ii]; ii++) {
		if (strcmp (attr_names[ii], "type")==0)
			return atoi (attr_values[ii]);
	}
	return 0;
}

void queue_node_parser_start_element (GMarkupParseContext* gmpc,
                                      const gchar*    element_name,
                                      const gchar**   attr_names,
                                      const gchar**   attr_values,
                                      gpointer        cparser,
                                      GError**        error)
{
	QueueNode*    qnode = CONF_PARSER_DATA (cparser);
	DownloadNode* dnode;

	if (strcmp (element_name, DOWNLOAD_NODE_TAG)==0) {
		dnode = download_node_new ();
		queue_node_prepend (qnode, dnode);
		conf_parser_push (cparser, &download_node_parser , dnode);
		download_node_parser_attr (attr_names, attr_values, dnode);
	}
}

void queue_node_parser_end_element (GMarkupParseContext*  gmpc,
                                    const gchar*      element_name,
                                    gpointer          cparser,
                                    GError**          error)
{
	if (strcmp (element_name, QUEUE_NODE_TAG)==0)
		conf_parser_pop (cparser);
}

const GMarkupParser queue_node_parser = {
	queue_node_parser_start_element,
	queue_node_parser_end_element,
	NULL,
	NULL,
	NULL
};

