/*
 *
 *   Copyright (C) 2005 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <urlglib/root_node.h>
#include <urlglib/download_task_curl.h>
#include <urlglib/url_filter.h>
#include <urlglib/urlglib_util.h>

#include <pcre.h>

void test_pcre ()
{
	pcre*  p_regexp;
	const char*  error_string;
	int    error_len;

	p_regexp = pcre_compile ("rar|r[0-9][0-9]", PCRE_CASELESS | PCRE_UTF8,
	                         &error_string, &error_len, NULL);

	g_message ("%d", pcre_exec (p_regexp, NULL, ".rar", 4, 0,
	                            PCRE_NO_UTF8_CHECK, NULL, 0));
	pcre_free (p_regexp);
}

#ifdef _WIN32
#include <windows.h>
void sleep_1 ()
{
	Sleep (1000);
}
#else
#include <unistd.h>
void sleep_1 ()
{
	usleep (1000000);
}
#endif

void queue_node_changed_notify (QueueNode* qnode, DownloadNode* dnode,
                                int nth, gpointer data)
{
	g_message ("changed");
}

void queue_node_removed_notify (QueueNode* qnode, DownloadNode* dnode,
                                int nth, gpointer data)
{
	g_message ("removed");
}

void queue_node_inserted_notify (QueueNode* qnode, DownloadNode* dnode,
                                 int nth, gpointer data)
{
	g_message ("inserted");
}

void test_node ()
{
	RootNode*     rnode  = root_node_new ();
	CategoryNode* cnode  = category_node_new ();
	DownloadNode* dnode1 = download_node_new ();
	DownloadNode* dnode2 = download_node_new ();
	QueueNode*    qnode  = category_node_waiting_queue (cnode);
	ProxyNode*    pnode1 = proxy_node_new ();
	ProxyNode*    pnode2 = proxy_node_new ();

	root_node_append (rnode, cnode);
	queue_node_changed_connect (qnode, queue_node_changed_notify, NULL);
	queue_node_removed_connect (qnode, queue_node_removed_notify, NULL);
	queue_node_inserted_connect (qnode, queue_node_inserted_notify, NULL);
	queue_node_append (qnode, dnode1);
	queue_node_append (qnode, dnode2);

	download_node_set_url (dnode1, "http://homepage.ntu.edu.tw/~b91612003/GSD20000.pdf");
	download_node_set_directory (dnode1, "c:\\");
	download_node_set_url (dnode2, "http://homepage.ntu.edu.tw/~b91612019/GSD20001.pdf");
	download_node_set_directory (dnode2, "c:\\");
	download_node_append_proxy (dnode1, pnode1);
	download_node_append_proxy (dnode1, pnode2);
	g_message ("pnode1 %p", pnode1);
	g_message ("pnode2 %p", pnode2);
	g_message ("dnode1->reference_count %d", dnode1->reference_count);
	g_message ("pnode1->reference_count %d", pnode1->reference_count);
	g_message ("pnode2->reference_count %d", pnode2->reference_count);
	g_message ("dnode1 child %p", base_node_child (BASE_NODE (dnode1)));

	g_message ("---");
	proxy_node_unref (pnode1);
	g_message ("dnode1 child %p", base_node_child (BASE_NODE (dnode1)));


//	while (queue_node_first_download (qnode)) {
//		root_node_start (rnode);
//		sleep_1 (2);
//	}

	root_node_unref (rnode);
	g_message ("---");
}

// test message server/client
#include <urlglib/msg_server.h>
#include <urlglib/msg_client.h>

void test_msg_server_client ()
{
	MsgServer* msgsvr;

	msgsvr = msg_server_new ();
	printf ("%p\n", msgsvr);
	if (msgsvr) {
		msg_server_activate (msgsvr);

		printf ("client check : %d\n", msg_client_ack ());
		msg_client_add_url ("http://test.org/test.url");

		sleep_1 ();
		msg_server_lock (msgsvr);
		if (msgsvr->url_list)
			printf ("server url_list : %s\n", msgsvr->url_list->data);
		msg_server_unlock (msgsvr);

		msg_server_stop (msgsvr);
		msg_server_unref (msgsvr);
	}
}



// UrlFilter -------

void on_url_added (UrlItemList *uilist, UrlItem* uitem, gpointer data)
{
	g_message ("URL added : %s", uitem->string);
}

void on_class_protocol_added (UrlItemList *uilist, UrlItem* uitem, gpointer data)
{
	g_message ("class of protocol added : %s", uitem->string);
}

void on_class_ext_added (UrlItemList *uilist, UrlItem* uitem, gpointer data)
{
	g_message ("class of ext added : %s", uitem->string);
}

void test_url_filter ()
{
	UrlFilter* filter = url_filter_new ();
	UrlFilterClass* fclass_protocol;
	UrlFilterClass* fclass_ext;

	fclass_protocol = url_filter_class_new (url_filter_class_protocol);
	fclass_ext = url_filter_class_new (url_filter_class_extension);

	url_item_list_added_connect (URL_ITEM_LIST (filter), on_url_added, NULL);
	url_item_list_added_connect (URL_ITEM_LIST (fclass_protocol), on_class_protocol_added, NULL);
	url_item_list_added_connect (URL_ITEM_LIST (fclass_ext), on_class_ext_added, NULL);

	url_filter_add_class (filter, fclass_protocol);
	url_filter_add_class (filter, fclass_ext);
	url_filter_add_item (filter, "http://aaa.www/asd/file.ext", NULL);
	url_filter_add_item (filter, "ftp://aaa.www/asd/file.zip", NULL);

	url_filter_class_set_item_all (fclass_ext, TRUE, 0);
	url_filter_class_pick_marked_item (fclass_ext);
	url_filter_mark_item (filter, 1, 1); 

	url_filter_unref (filter);
}

// XML config -----------

void test_write_conf ()
{
	ConfWriter cw;
	RootNode*     rnode  = root_node_new ();
	CategoryNode* cnode  = category_node_new ();
	DownloadNode* dnode1 = download_node_new ();
	QueueNode*    qnode  = category_node_waiting_queue (cnode);
	ProxyNode*    pnode1 = proxy_node_new ();
	ProxyNode*    pnode2 = proxy_node_new ();

	conf_writer_init (&cw , 1);
	conf_writer_open_file (&cw, "c:\\test_w.xml");

	root_node_append (rnode, cnode);

	queue_node_changed_connect (qnode, queue_node_changed_notify, NULL);
	queue_node_removed_connect (qnode, queue_node_removed_notify, NULL);
	queue_node_inserted_connect (qnode, queue_node_inserted_notify, NULL);
	queue_node_append (qnode, dnode1);


	download_node_set_url (dnode1, "http://www.geocities.jp/psobbtw/bityou.zip");
	download_node_set_directory (dnode1, "c:\\");
	download_node_append_proxy (dnode1, pnode1);
	download_node_append_proxy (dnode1, pnode2);
	proxy_node_set_host (pnode1, "127.0.0.1");
	g_message ("pnode1 %p", pnode1);
	g_message ("pnode2 %p", pnode2);
	g_message ("dnode1->reference_count %d", dnode1->reference_count);
	g_message ("pnode1->reference_count %d", pnode1->reference_count);
	g_message ("pnode2->reference_count %d", pnode2->reference_count);
	g_message ("dnode1 child %p", base_node_child (BASE_NODE (dnode1)));

	root_node_write_conf (rnode, &cw);

	g_message ("---");
	proxy_node_unref (pnode1);
	g_message ("dnode1 child %p", base_node_child (BASE_NODE (dnode1)));

	g_message ("download_node finalize---");
	category_node_unref (cnode);
	g_message ("---");

	conf_writer_finalize (&cw);

}

void test_parse_conf ()
{
	ConfParser cp;
	ConfWriter cw;
	RootNode* rnode = root_node_new ();

	conf_parser_init (&cp);

	conf_parser_push (&cp, &root_node_parser, rnode);
	conf_parser_read_file (&cp, "c:\\test_w.xml", NULL);

	conf_parser_finalize (&cp);

	// ---
	conf_writer_init (&cw , 1);
	conf_writer_open_file (&cw, "c:\\test_w1.xml");
	root_node_write_conf (rnode, &cw);
	conf_writer_finalize (&cw);
}

void test_parse_str ()
{
	char* str_ = "  \r\n  http://wxf/gfg/xc  \n\r";
	char* url;
	int   url_len;
	char* ret_val;

	ret_val = (char*) text_get_line_beg (str_, &url, &url_len);
	printf ("url %.*s", url_len, url);
	printf ("ret_val %s", ret_val);
}

int  main (void)
{
	download_task_curl_global_init ();

	if( g_thread_supported()==FALSE )
		g_thread_init(NULL);

//	test_node ();
	test_msg_server_client ();
//	test_url_filter ();
//	test_write_conf ();
//	test_parse_conf ();
//	test_pcre ();
//	test_parse_str ();

#ifdef _WIN32
	system ("pause");
#endif

	download_task_curl_global_finalize ();
	return 0;
}

