/***************************************************************************
                          dcchat.cpp  -  description
                             -------------------
    begin                : Don Mär 28 2002
    copyright            : (C) 2002 by Mathias Küster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "dcchat.h"

#include <qdatetime.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qclipboard.h>
#include <qapplication.h>
#include <qlabel.h>
#include <qmessagebox.h>
#include <qregexp.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qlayout.h>
#include <qprocess.h>
#include <qglobal.h>
#include <qstatusbar.h>
#include <qtooltip.h>
#include <qfile.h>
#include <qfiledialog.h>
#include <qsplitter.h>
#include <qtabwidget.h>
#include <qworkspace.h>
#include <qcstring.h>
#include <qurl.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qcombobox.h>

#include "dcevent.h"
#include "dcclient.h"
#include "dcmenuhandler.h"
#include "dcconfig.h"
#include "dctranslator.h"
#include "cdialogpicturemap.h"
#include "dcfiletool.h"
#include "dchublistmanager.h"
#include "dcconnectionmanager.h"
#include "dcuserslist.h"
#include "dcqtextedit.h"
#include "dciconloader.h"
#include "dctransferview.h"
#include "dchubsearch.h"
#include "dcshellcommandrunner.h"
#include "dcguiutils.h"
#include "dcqtextedit.h"

#include "ui/DCDialogMagnet.h"

#include <dclib/cfilemanager.h>
#include <dclib/dcos.h>
#include <dclib/cutils.h>
#include <dclib/core/cbase64.h>
#include <dclib/core/cdir.h>
#include <dclib/dclib.h>
#include <dclib/clistenmanager.h>
#include <dclib/core/cbytearray.h>

/* to enabled the /pretend command */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/** the link list */
const QStringList DCChat::g_LinkList = QStringList::split(',',"HTTP://,HTTPS://,NEWS://,FILE://,WWW.,FTP://,FTP.,HKP://,LDAP://,DCFILE://,DCHUB://,DCHUBS://,DCCMD://,MAGNET:");

#define DCFILE_LINK_INDEX  9
#define DCHUB_LINK_INDEX  10
#define DCHUBS_LINK_INDEX 11
#define DCCMD_LINK_INDEX  12
#define MAGNET_LINK_INDEX 13

#define MAX_HISTORY_COUNT 25

/** */
DCChat::DCChat( QWidget * parent, const char *name, int wflags, DCClient * client, bool bprivate ) : DCDialogChat(parent, name, wflags)
{
	ASSERT(client);

	// set default icon
	setIcon( g_pIconLoader->GetPixmap(eiGLOBE) );

	m_pParent      = parent;
	m_bPrivateChat = bprivate;
	m_pClient      = client;
	m_bSendAway    = true;
	m_eSecureState = esecsNONE;
	m_nCurrentHistoryIndex = -1;

	m_sTimeStamp  = QDateTime::currentDateTime().toString("yyyyMMdd-hhmmss");

	m_nTabStart   = -1;
	m_nTabEnd     = -1;
	m_nTabPressed = 0;
	m_sTabSaved   = "";
	m_sTabNick    = "";
	
	m_pShellRunners = new QPtrList<DCShellCommandRunner>();
	
	InitDocument();
}

/** */
DCChat::~DCChat()
{
	/*
	 * Call delete on all running shell commands.
	 * The DCShellCommand destructor will wait
	 * for the thread to finish.
	 */
	
	for ( unsigned int i = 0; i < m_pShellRunners->count(); i++ )
	{
		delete m_pShellRunners->at(i);
	}
	
	m_pShellRunners->clear();
	delete m_pShellRunners;
}

/** */
void DCChat::QT4TranslationSync()
{
	QString qt41 = tr("Process still running after 2 minutes, killing process...");
}

/** */
void DCChat::InitDocument()
{
	m_pStatusBar = new QStatusBar(this);
	m_pStatusBar->setSizeGripEnabled(false);
	DCDialogChatLayout->addWidget( m_pStatusBar, 1, 0 );
	m_pStatusCrypt = new QLabel(m_pStatusBar);
	m_pStatusBar->addWidget(m_pStatusCrypt);

	SetCrypt(esecsNONE);

	if ( !m_bPrivateChat )
	{
		m_pStatusBar->hide();
	}
	else
	{
		m_pStatusBar->show();
	}

/*	if ( g_pConfig->ShowChatStatusBar() == false )
	{
		m_pStatusBar->hide();
	}
*/
	if ( g_pConfig->GetShowChatSendButton() )
	{
		connect( PushButton_SEND, SIGNAL(clicked()), this, SLOT(slotSendChat()) );
		slotTextChangedChatInput();
	}
	else
	{
		// remove send button and recreate layout
		delete PushButton_SEND;
		PushButton_SEND = NULL;
		delete Frame_SEND->layout();
		QGridLayout * gl = new QGridLayout( Frame_SEND, 1, 1, 1, 1, "GridLayout_CHATINPUT");
		gl->addWidget( TextEdit_CHATINPUT, 0, 0 );
	}

	m_pTextEdit_CHATOUTPUT = new DCQTextEdit( Frame_OUTPUT, "TextEdit_CHATOUTPUT" );
	
	// QSizePolicy::Preferred is the default anyway...
	m_pTextEdit_CHATOUTPUT->setSizePolicy( QSizePolicy( QSizePolicy::Preferred, QSizePolicy::Preferred, 0, 0, m_pTextEdit_CHATOUTPUT->sizePolicy().hasHeightForWidth() ) );
	m_pTextEdit_CHATOUTPUT->setMinimumSize( QSize( 0, 60 ) );
	m_pTextEdit_CHATOUTPUT->setTextFormat( QTextEdit::RichText );
	m_pTextEdit_CHATOUTPUT->setWordWrap( QTextEdit::WidgetWidth );
	m_pTextEdit_CHATOUTPUT->setWrapPolicy( QTextEdit::AtWordOrDocumentBoundary );
    	m_pTextEdit_CHATOUTPUT->setReadOnly( true );
	Frame_OUTPUTLayout->addWidget(m_pTextEdit_CHATOUTPUT,0,0);

	connect( m_pTextEdit_CHATOUTPUT, SIGNAL(rightButtonClicked( const QPoint& )), this, SLOT(slotRightButtonClickedChatOutput( const QPoint& )) );
#if QT_VERSION >= 0x030100
	connect( m_pTextEdit_CHATOUTPUT, SIGNAL(clicked( int, int )), this, SLOT(slotClickedChatOutput( int, int )) );
	connect( m_pTextEdit_CHATOUTPUT, SIGNAL(doubleClicked( int, int )), this, SLOT(slotDoubleClickedChatOutput( int, int )) );
#endif
	if ( PushButton_SEND )
	{
		connect( TextEdit_CHATINPUT, SIGNAL(textChanged()), this, SLOT(slotTextChangedChatInput()) );
	}

	connect( g_pConfig, SIGNAL(chatBackgroundColorChanged()), this, SLOT(slotBGColorChanged()) );
	if ( g_pConfig->GetChatBackgroundColorEnabled() )
	{
		QBrush paper = m_pTextEdit_CHATOUTPUT->paper();
		paper.setColor(g_pConfig->GetChatBackgroundColor());
		m_pTextEdit_CHATOUTPUT->setPaper(paper);
		
		paper = TextEdit_CHATINPUT->paper();
		paper.setColor(g_pConfig->GetChatBackgroundColor());
		TextEdit_CHATINPUT->setPaper(paper);
	}

	m_pTextEdit_CHATOUTPUT->installEventFilter(this);
	TextEdit_CHATINPUT->installEventFilter(this);
	TextEdit_CHATINPUT->setFocus();
}

/** event filter */
bool DCChat::eventFilter( QObject * object, QEvent * event )
{
	if ((event->type() == QEvent::KeyPress)&&((QTextEdit*)object==TextEdit_CHATINPUT))
	{
		QKeyEvent * e = (QKeyEvent*)event;

		bool send = false;
		if ( g_pConfig->GetSendChat() == 1 )
		{
			if ( (e->state() != ControlButton) && ((e->key() == Key_Enter) || (e->key() == Key_Return)) )
			{
				send = true;
			}
		}
		else if ( g_pConfig->GetSendChat() == 0 )
		{
			if ( (e->state() == ControlButton) && ((e->key() == Key_Enter) || (e->key() == Key_Return)) )
			{
				send = true;
			}
		}
		else if ( g_pConfig->GetSendChat() == 2 )
		{
			if ( (e->state() == AltButton) && ((e->key() == Key_Enter) || (e->key() == Key_Return)) )
			{
				send = true;
			}
		}
		else if ( g_pConfig->GetSendChat() == 3 )
		{
			if ( (e->state() == AltButton) && (e->key() == Key_S) )
			{
				send = true;
			}
		}

		if ( send )
		{
			SendChat();
			e->accept();
			return true;
		}
		else if ( e->state() == ControlButton )
		{
			int histindex = -1;

			// check for history
			if ( (e->key() == Key_Down) && (m_nCurrentHistoryIndex != -1) )
			{
				histindex = m_nCurrentHistoryIndex - 1;

			}
			else if ( e->key() == Key_Up )
			{
				if ( m_nCurrentHistoryIndex == -1 )
				{
					histindex = 0;
					
					// save current text
					m_sHistoryTempString = TextEdit_CHATINPUT->text();
					// save current cursor position
					TextEdit_CHATINPUT->getCursorPosition( &m_nHistoryTempPara, &m_nHistoryTempIndex );
				}
				else
				{
					histindex = m_nCurrentHistoryIndex + 1;
					if ( histindex >= (int) m_lHistory.size() )
					{
						histindex = -1;
					}
				}
			}

			if ( (histindex >= 0) && (histindex < (int) m_lHistory.size()) )
			{
				m_nCurrentHistoryIndex = histindex;
				TextEdit_CHATINPUT->setText(m_lHistory[histindex]);
			}
			else if (e->key() == Key_Down)
			{
				m_nCurrentHistoryIndex = histindex;
				TextEdit_CHATINPUT->setText(m_sHistoryTempString);
				TextEdit_CHATINPUT->setCursorPosition( m_nHistoryTempPara, m_nHistoryTempIndex );
			}
		}

		if ( e->key() == Key_Tab )
		{
			NickCompletion();
			e->accept();
			return true;
		}
		else if ( m_nTabPressed != 0 )
		{
			// reset tab settings
			m_nTabStart   = -1;
			m_nTabEnd     = -1;
			m_nTabPressed = 0;
			m_sTabSaved   = "";
			//m_sTabNick    = "";
		}
	}
	else if ( (event->type() == QEvent::Resize) && (object == m_pTextEdit_CHATOUTPUT) )
	{
		bool bscroll, b;

		if ( m_pTextEdit_CHATOUTPUT->verticalScrollBar()->maxValue() == m_pTextEdit_CHATOUTPUT->verticalScrollBar()->value() )
		{
			bscroll = true;
		}
		else
		{
			bscroll = false;
		}

		b = QWidget::eventFilter( object, event );

		if ( bscroll )
		{
			m_pTextEdit_CHATOUTPUT->scrollToBottom();
			m_pTextEdit_CHATOUTPUT->moveCursor( QTextEdit::MoveEnd, false );
		}
		
		return b;
	}

	return QWidget::eventFilter( object, event );    // standard event processing
}

/** handle translation and shell command finished events */
void DCChat::customEvent( QCustomEvent * event )
{
	if ( event->type() == EVENT_TRANSLATION )
	{
		DC_TranslationEvent *e = (DC_TranslationEvent*)event;

		if ( e->m_bTranslate )
		{
			AddStatus( tr("Translation: ") + "'" + e->m_sOriginal + "' -> '" + e->m_sTranslation + "'" );
		}
		else
		{
			AddStatus( tr("Translation failed: ") + "'" + e->m_sOriginal + "'" + " ( " +tr("Error was: ") + "'" + e->m_sTranslation + "' )" );
		}
	}
	else if ( event->type() == EVENT_SHELL_COMMAND_FINISHED )
	{
		DC_ShellCommandFinishedEvent * e = (DC_ShellCommandFinishedEvent*)event;
		{
			int pos = m_pShellRunners->find(e->m_pShellCommandRunner);
			if ( pos == -1 )
			{
				AddStatus( tr("Shell command event not found in list") );
				AddStatus( e->m_pShellCommandRunner->getOutput() );
			}
			else
			{
				if ( e->m_pShellCommandRunner->getSucceeded() )
				{
					SendChat( e->m_pShellCommandRunner->getOutput() );
				}
				else
				{
					AddStatus( e->m_pShellCommandRunner->getOutput() );
				}
				
				m_pShellRunners->remove(pos);
				delete e->m_pShellCommandRunner;
			}
		}
	}
	else
	{
		QWidget::customEvent( event );
	}

}

/** */
void DCChat::AddHistory( QString message )
{
	if ( m_lHistory.size() == MAX_HISTORY_COUNT )
	{
		m_lHistory.pop_back();
	}

	m_nCurrentHistoryIndex = -1;
	m_lHistory.prepend(message);
}

/** */
void DCChat::SetCrypt( eSecureState e )
{
	if ( e == esecsENCRYPTED )
	{
		QToolTip::remove(m_pStatusCrypt);
		QToolTip::add(m_pStatusCrypt, tr("Line is encrypted."));
		m_pStatusCrypt->setPixmap( g_pIconLoader->GetPixmap(eiSSL_YES) );
	}
	else
	{
		QToolTip::remove(m_pStatusCrypt);
		QToolTip::add(m_pStatusCrypt, tr("Line is not encrypted."));
		m_pStatusCrypt->setPixmap( g_pIconLoader->GetPixmap(eiSSL_NO) );
	}

	m_eSecureState = e;
}

/** */
void DCChat::slotTextChangedChatInput()
{
	if ( PushButton_SEND )
	{
		if ( TextEdit_CHATINPUT->text().isEmpty() )
		{
			PushButton_SEND->setEnabled(false);
		}
		else
		{
			PushButton_SEND->setEnabled(true);
		}
	}
}

/** */
void DCChat::slotClickedChatOutput( int, int )
{
	int x,y;

	QPoint p = m_pTextEdit_CHATOUTPUT->mapFromGlobal(QCursor::pos());
	m_pTextEdit_CHATOUTPUT->viewportToContents(p.x(),p.y(),x,y);
	p.setX(x);
	p.setY(y);
	QString pressedLink = m_pTextEdit_CHATOUTPUT->anchorAt(p);

//	printf("%s\n",pressedLink.ascii());

	for ( unsigned int i = 0; i < g_LinkList.size(); ++i )
	{
		if ( pressedLink.startsWith(g_LinkList[i],false) )
		{
			if ( i == DCFILE_LINK_INDEX )
			{
				// DC-File-Link
				CString hubhost,hubname,nick,file,tth;
				ulonglong size;

				if ( CUtils::ConvertDCLink( pressedLink.ascii(), hubhost, hubname, nick, size, file, tth ) == false )
				{
					return;
				}

				QPopupMenu * m = new QPopupMenu(this);

				DCMenuHandler::InsertMenu( m, emiDOWNLOAD );
				DCMenuHandler::InsertMenu( m, emiDOWNLOAD_TO );
				DCMenuHandler::InsertMenu( m, emiDOWNLOAD_AS );
				DCMenuHandler::InsertMenu( m, emiDOWNLOAD_IN );

				int id = m->exec(QCursor::pos());

				delete m;

				if ( (id == emiDOWNLOAD) || (id == emiDOWNLOAD_AS) || (id == emiDOWNLOAD_TO) )
				{
					QString localrootpath;
					QString localname;

					// select downloadfolder for all selected files
					if ( id == emiDOWNLOAD_TO )
					{
						localrootpath = QFileDialog::getExistingDirectory( QString(), this, "bdf", tr("Select download folder"), true );

						if ( localrootpath.isEmpty() )
							return;
					}

					CDir d;
					d.SetPath(file);
					QFileInfo fi(d.DirName().Data());
					localname = fi.fileName();

					if ( id == emiDOWNLOAD_AS )
					{
						localrootpath = QFileDialog::getSaveFileName( localname, QString(), this, "bdf", tr("Select file for")+" "+localname );

						if ( localrootpath.isEmpty() )
							return;

						QFileInfo fi(localrootpath);
						localrootpath = fi.dirPath();
						localname     = fi.fileName();

						if ( (localrootpath.isEmpty()) || (localname.isEmpty()) )
							return;
					}

					// add transfer to the waitlist
					DCFileTool::CheckFile( this, nick, hubname, hubhost,
							file, localname.ascii(), CString(), localrootpath.ascii(), eltFILE,
							size, tth );
				}
				else if ( id == emiDOWNLOAD_IN )
				{
					QString localrootpath;
					QString localname;
					QString localpath;

					if ( DCFileTool::SelectFileSource( this, size, QString(tth.Data()), localname, localrootpath, localpath ) == false )
					{
						return;
					}

					// add transfer to the waitlist
					DCFileTool::CheckFile( this, nick, hubname, hubhost,
							  file, localname.ascii(), localpath.ascii(), localrootpath.ascii(), eltFILE,
					  		size, tth, true);
				}
			}
			else if ( i == DCHUB_LINK_INDEX )
			{
				pressedLink = pressedLink.right( pressedLink.length()-8 );
				// remove all '/' chars
				pressedLink.remove('/');

				g_pConnectionManager->Connect( pressedLink.ascii(), pressedLink.ascii() );
			}
			else if ( i == DCHUBS_LINK_INDEX )
			{
				pressedLink = pressedLink.right( pressedLink.length()-9 );
				// remove all '/' chars
				pressedLink.remove('/');

				g_pConnectionManager->Connect( pressedLink.ascii(), pressedLink.ascii(), true );
			}
			/* else if ( i == DCCMD_LINK_INDEX )
			{
				pressedLink = pressedLink.right( pressedLink.length()-8 );
			} */
			else if ( i == MAGNET_LINK_INDEX )
			{
				//magnet:?xt=urn:tree:tiger:EOSA5AGTL5SD3VWCF3R2OH2WMGXV3S3R7SYN4YA&xl=708780032&dn=FC-6-i386-disc1.iso
				
				QString key = QString::fromAscii("xt=urn:tree:tiger:");
				int i = pressedLink.find(key);
				if (i == -1)
				{
					key = QString::fromAscii("xt.1=urn:tree:tiger:");
					i = pressedLink.find(key);
				}
				
				int end = pressedLink.find("&",i+1);
				
				QString tth;
				if ( i != -1 )
				{
					i += key.length();
					if ( end == -1 )
					{
						tth = pressedLink.mid( i );
					}
					else
					{
						tth = pressedLink.mid( i, end - i );
					}
				}

				switch ( g_pConfig->GetMagnetAction(eChatMagnet) )
				{
					case eMagnetPrompt:
					{
						DCDialogMagnet * dialog = new DCDialogMagnet(this);
						
						dialog->LineEdit_LINK->setText(pressedLink);
						dialog->LineEdit_TTH->setText(tth);
						
						key = QString::fromAscii("dn=");
						i = pressedLink.find(key);
						if ( i != -1 )
						{
							end = pressedLink.find("&",i+1);
							i += key.length();
							QString dispname;
							if ( end == -1 )
							{
								dispname = pressedLink.mid( i );
							}
							else
							{
								dispname = pressedLink.mid( i, end - i );
							}
							
							dispname.replace("+","%20");
							QUrl::decode( dispname );
							
							dialog->LineEdit_NAME->setText(dispname);
						}
						
						key = QString::fromAscii("xl=");
						i = pressedLink.find(key);
						if ( i != -1 )
						{
							end = pressedLink.find("&",i+1);
							i += key.length();
							ulonglong size;
							if ( end == -1 )
							{
								size = pressedLink.mid( i ).toULongLong();
							}
							else
							{
								size = pressedLink.mid( i , end - i ).toULongLong();
							}
							
							dialog->LineEdit_SIZE->setText( DCGuiUtils::GetSizeString(size) );
							dialog->LineEdit_EXACT_SIZE->setText( QString::number(size) );
						}
						
						if ( dialog->exec() == QDialog::Accepted )
						{
							if ( dialog->CheckBox_SET_DEFAULT->isChecked() )
							{
								if ( dialog->RadioButton_SEARCH->isChecked() )
								{
									g_pConfig->SetMagnetAction( eMagnetSearch, eChatMagnet );
								}
								else if ( dialog->RadioButton_DO_NOTHING->isChecked() )
								{
									g_pConfig->SetMagnetAction( eMagnetNothing, eChatMagnet );
								}
							}
							
							if ( dialog->RadioButton_SEARCH->isChecked() )
							{
								DCHubSearch * hubsearch = new DCHubSearch( g_pConnectionManager->GetWorkspace() );
								hubsearch->SetSearchForFile( tth, eftHASH );
								hubsearch->show();
								hubsearch->StartSearchWithPrompt();
							}
						}
						
						delete dialog;
						
						break;
					}
					case eMagnetSearch:
					{
						DCHubSearch * hubsearch = new DCHubSearch( g_pConnectionManager->GetWorkspace() );
						hubsearch->SetSearchForFile( tth, eftHASH );
						hubsearch->show();
						hubsearch->StartSearchWithPrompt();
						break;
					}
					case eMagnetDownload:
						/* not supported */
						break;
					case eMagnetNothing:
						break;
					default:
						break;
				}
			}
			else
			{
				g_pConfig->OpenURL(pressedLink);
			}
		}
	}
}

/** */
void DCChat::slotRightButtonClickedChatOutput( const QPoint& )
{
	int para;
	QString pressedPara, pressedNick, userCommand;
	CString cPressedNick;
	QPoint p;
	int x,y, nickStart, nickLen;
	int id;
	bool isNickOnline;
	QPopupMenu *m,*muser,*mslot;
	QString s;	
	QMap<int, DC_UserMenuCommand*> addedcommands;
	DC_UserMenuCommand * umc = 0;
	
	// get the paragraph of text where the right click was
	p = m_pTextEdit_CHATOUTPUT->mapFromGlobal(QCursor::pos());
	m_pTextEdit_CHATOUTPUT->viewportToContents(p.x(),p.y(),x,y);
	p.setX(x);
	p.setY(y);
	para = m_pTextEdit_CHATOUTPUT->paragraphAt(p);
	pressedPara = m_pTextEdit_CHATOUTPUT->text(para);
	
	// cut out the nick
	nickStart = 4 + pressedPara.find("&lt;");
	nickLen = pressedPara.find("&gt;") - nickStart;
	
	// sanity check
	if ( (nickStart == 3) || (nickLen < 0) )
	{
		if (m_bPrivateChat)
		{
			pressedNick = m_sNick;
		}
		else
		{
			// assume a /me line
			// to make things fun, the HTML we have to parse
			// is not the same as the HTML we generate
			
			QRegExp re;
			re.setCaseSensitive(false);
			re.setMinimal(true);
			re.setPattern("<span.*color:"+g_pConfig->GetChatColor(eccPUBLICCHATMENICK)+".*>(.*)</span>");
			if ( re.search(pressedPara) != -1 )
			{
				pressedNick = re.cap(1);
			}
			else
			{
				if ( g_pConfig->GetTimeStamp(etsHUBCHAT) )
				{
					re.setPattern("<span.*color:"+g_pConfig->GetChatColor(eccPUBLICCHATTIMESTAMP)+".*>(.*)</span>\\s");
					pressedPara.remove(re);
				}
				
				int nickend = pressedPara.find(" ");
				pressedNick = pressedPara.mid( 0, nickend );
				
				if ( pressedNick == "*" )
				{
					++nickend;
					pressedNick = pressedPara.mid( nickend, pressedPara.find(" ", nickend) - nickend );
				}
			}
		}
	}
	else
	{
		pressedNick = pressedPara.mid(nickStart, nickLen);
	}
	
	// check for joins / parts message
	if ( pressedNick == "VALKNUT" )
	{
		int lenJoin = tr("Joins: ").length();
		int lenPart = tr("Parts: ").length();
		int joinPos = pressedPara.find(tr("Joins: "));
		int partPos = pressedPara.find(tr("Parts: "));
		
		if ( joinPos == -1 )
		{
			nickStart = lenPart + partPos;
		}
		else
		{
			nickStart = lenJoin + joinPos;	
		}
		
		if ( (joinPos != -1) || (partPos != -1) )
		{
			nickLen = pressedPara.findRev("</span>") - nickStart;
			pressedNick = pressedPara.mid(nickStart, nickLen);
		}
	}
	
	// check for private chat
	if ( (pressedNick == "VALKNUT") && m_bPrivateChat )
	{
		pressedNick = m_sNick;
	}
	
	//printf("Debug pressedNick: %s\n",pressedNick.ascii());
	
	cPressedNick = pressedNick.ascii();
	isNickOnline = m_pClient->UserList()->IsUserOnline(cPressedNick);
	
	// select the user in the userlist
	if (isNickOnline)
	{
		m_pClient->jumpToNick( pressedNick );
	}
	
	m = new QPopupMenu(this);
	
	muser = DCMenuHandler::InsertMenu( m , emisCHAT_RIGHTCLICK_USER, true, pressedNick );
	DCMenuHandler::InsertMenu( muser, emiPRIVATE_CHAT, isNickOnline );
	DCMenuHandler::InsertMenu( muser, emiADD_FRIEND,  isNickOnline );
	DCMenuHandler::InsertMenu( muser, emiBROWSE_USER_FILES, isNickOnline );
	DCMenuHandler::InsertMenu( muser, emiSEPARATOR );
	
	mslot = DCMenuHandler::InsertMenu( muser, emiUPLOAD_SLOT, true);
	DCMenuHandler::InsertMenu( mslot, emiADD, true);
	DCMenuHandler::InsertMenu( mslot, emiADD_PERMANENT, true);
	DCMenuHandler::InsertMenu( mslot, emiREMOVE, true);
	
	DCMenuHandler::InsertMenu( muser, emiSEPARATOR );
	DCMenuHandler::InsertMenu( muser, emiKICK, (isNickOnline && m_pClient->UserList()->IsAdmin(m_pClient->GetNick())) );
	DCMenuHandler::InsertMenu( muser, emiFORCE_MOVE, (isNickOnline && m_pClient->UserList()->IsAdmin(m_pClient->GetNick())) );
	
	addedcommands = m_pClient->AddMenuCommands( muser, euccChat );
	
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	
	DCMenuHandler::InsertMenu( m, emiINSERTSMILEY, g_pConfig->GetEnableEmoticons() );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiCOPY, m_pTextEdit_CHATOUTPUT->hasSelectedText() );
	DCMenuHandler::InsertMenu( m, emiCLEAR, true );
	DCMenuHandler::InsertMenu( m, emiSELECT_ALL, true );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	
	DCMenuHandler::InsertMenu( m, emiZOOM_IN );
	DCMenuHandler::InsertMenu( m, emiZOOM_OUT );
	
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiTRANSLATE, m_pTextEdit_CHATOUTPUT->hasSelectedText() );
	DCMenuHandler::InsertMenu( m, emiTRANSLATOR );

	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiREFRESH );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );

	if ( m_bPrivateChat )
	{
		// insert request/close secure chat
		if ( m_eSecureState == esecsNONE )
			DCMenuHandler::InsertMenu( m, emiREQUEST_SECURE_CHAT, dclibSupportsSSL() );
		else
			DCMenuHandler::InsertMenu( m, emiCLOSE_SECURE_CHAT );
	}

	DCMenuHandler::InsertMenu( m, emiSAVE );

	if ( parentWidget() == 0 )
		DCMenuHandler::InsertMenu( m, emiDOCK );
	else
		DCMenuHandler::InsertMenu( m, emiUNDOCK );

	if ( m_bPrivateChat )
	{
		DCMenuHandler::InsertMenu( m, emiHIDE, m_bPrivateChat && !g_pConfig->GetShowChatInTab() );
		DCMenuHandler::InsertMenu( m, emiCLOSE, m_bPrivateChat );
	}
	
	if ( !m_bPrivateChat )
	{
		QMap<int, DC_UserMenuCommand*> addedcommands2 = m_pClient->AddMenuCommands( m, euccHub );
		
		// QT3 has no QMap::unite() method
		
		QValueList<int> addedkeys2 = addedcommands2.keys();
		
		for ( unsigned int i = 0; i < addedkeys2.size() ; i++ )
		{
			addedcommands.insert( addedkeys2[i], addedcommands2[addedkeys2[i]] );
		}
	}
	
	id = m->exec(QCursor::pos());

	delete m;

	if ( id == -1 )
	{
		return;
	}
	else if ( id == emiINSERTSMILEY )
	{
		int x,y;
		QPtrList<DC_EmoticonObject> * elist = g_pConfig->EmoticonList();

		if ( elist )
		{
			CDialogPictureMap * dialog = new CDialogPictureMap(this);
			QPixmap p(g_pConfig->GetEmoticonImage());
			dialog->SetPixmap(p);

			if ( dialog->exec() == QDialog::Accepted )
			{
				dialog->GetXY(x,y);

				DC_EmoticonObject * EmoticonObject = 0;

				QPtrListIterator<DC_EmoticonObject> it( *elist );
				while ( (EmoticonObject = it.current()) != 0 )
				{
					if ( ((EmoticonObject->left<x) && (EmoticonObject->right>x)) &&
					     ((EmoticonObject->top<y)  && (EmoticonObject->bottom>y)) )
					{
						// damn hacks
						QString smiley = EmoticonObject->m_Text;
						smiley.replace( "&lt;", "<" );
						smiley.replace( "&gt;", ">" );
						smiley.replace( "&amp;", "&" );
						smiley.replace( "&apos;", "\'" );
						smiley.replace( "&quot;", "\"" );
						
						smiley += " ";
						
						TextEdit_CHATINPUT->insert(smiley);
						break;
					}
					++it;
				}
			}

			delete dialog;
		}
	}
	else if ( id == emiTRANSLATE )
	{
		QString sc;
		QClipboard * cb = QApplication::clipboard();
		
		// save old text
		sc = cb->text();
		// copy to cb
		m_pTextEdit_CHATOUTPUT->copy();
		// get new text
		s = cb->text();
		// save old text
		cb->setText(sc);
		
		s.remove("<!--StartFragment-->");
		g_pTranslator->Translate( this, m_sLanguage, s );
	}
	else if ( id == emiTRANSLATOR )
	{
		m_sLanguage = g_pTranslator->SelectLanguage( m_sLanguage, this );
	}
	else if ( id == emiCOPY )
	{
		m_pTextEdit_CHATOUTPUT->copy();
	}
	else if ( id == emiCLEAR )
	{
		m_pTextEdit_CHATOUTPUT->clear();
		AddStatus( tr("Chat Cleared.") );
	}
	else if ( id == emiSELECT_ALL )
	{
		m_pTextEdit_CHATOUTPUT->selectAll();
	}
	else if ( id == emiREQUEST_SECURE_CHAT )
	{
		m_pClient->SendPrivateMessage( m_pClient->GetNick(), m_sNick.ascii(), "<request secchannel>" );
	}
	else if ( id == emiCLOSE_SECURE_CHAT )
	{
		m_pClient->SendPrivateMessage( m_pClient->GetNick(), m_sNick.ascii(), "<close secchannel>" );
	}
	else if ( (id == emiDOCK) || (id == emiUNDOCK) )
	{
		if ( parentWidget() == 0 )
		{
			DCChat::reparent(m_pParent,QPoint(),true);

			if ( !m_bPrivateChat || g_pConfig->GetShowChatInTab() )
			{
				QTabWidget * p = (QTabWidget*)m_pParent;

				p->insertTab(this,m_sLabel);
				p->showPage(this);
			}

			show();
		}
		else
		{
			if ( !m_bPrivateChat || g_pConfig->GetShowChatInTab() )
			{
				QTabWidget * p = (QTabWidget*)m_pParent;

				m_sLabel = p->tabLabel(this);
				p->removePage(this);
			}

			DCChat::reparent(0,QPoint(),true);

			move(0,0);

			// set default icon
			setIcon( g_pIconLoader->GetPixmap(eiGLOBE) );
		}
	}
	else if ( id == emiSAVE )
	{
		QString s = QFileDialog::getSaveFileName(
			QString(),
			"HTML (*.html *.htm)",
			this,
			"save file dialog",
			tr("Choose a filename to save under") );

		if ( !s.isEmpty() )
		{
			QFile f(s);

			if ( f.open(IO_WriteOnly) )
			{
				f.writeBlock(m_pTextEdit_CHATOUTPUT->text().ascii(),strlen(m_pTextEdit_CHATOUTPUT->text().ascii()));
				f.close();
			}
		}
	}
	else if ( id == emiREFRESH )
	{
		if ( CFileManager::Instance()->CreateShareList() )
			s = tr("Refresh share in progress.");
		else
			s = tr("Refresh share already in progress.");
		AddStatus(s);
	}
	else if ( id == emiCLOSE )
	{
		close();
	}
	else if ( id == emiHIDE )
	{
		setEnabled(false);
		hide();
	}
	else if ( id == emiPRIVATE_CHAT )
	{
		m_pClient->DC_PrivateChat( pressedNick, QString(), QString(), true );
	}
	else if ( id == emiADD_FRIEND )
	{
		g_pUsersList->AddFriend(
			pressedNick,
			QString::fromAscii(m_pClient->GetHubName().Data()),
			QString::fromAscii(m_pClient->GetHost().Data()),
			QString()
		);
	}
	else if ( id == emiBROWSE_USER_FILES )
	{
		CString empty;
		g_pTransferView->DLM_QueueAdd( cPressedNick, m_pClient->GetHubName(),
				m_pClient->GetHost(),
				DC_USER_FILELIST, DC_USER_FILELIST, empty, empty, eltBUFFER,
				0, 0, 0, empty );
	}
	else if ( id == emiKICK )
	{
		QString kickMessage;
		
		if ( !m_pClient->GetOPKickMessage(kickMessage,this) )
		{
			return;
		}
		m_pClient->OPKick( pressedNick, kickMessage );
		
	}
	else if ( id == emiFORCE_MOVE )
	{
		QString host, message;
		if ( !m_pClient->GetOPForceMoveMessage(message,host,this) )
		{
			return;
		}
		m_pClient->OPForceMove(pressedNick, message, host);
	}
	else if ( id == emiADD )
	{
		g_pTransferView->DLM_AddUserSlot(cPressedNick, m_pClient->GetHubName(), 1);
	}
	else if ( id == emiADD_PERMANENT )
	{
		g_pTransferView->DLM_AddUserSlot(cPressedNick, m_pClient->GetHubName(), 0, true);
	}
	else if ( id == emiREMOVE )
	{
		g_pTransferView->DLM_AddUserSlot(cPressedNick, m_pClient->GetHubName(), 0);
	}
	else if ( id == emiZOOM_IN )
	{
		m_pTextEdit_CHATOUTPUT->zoomIn();
	}
	else if ( id == emiZOOM_OUT )
	{
		m_pTextEdit_CHATOUTPUT->zoomOut();
	}
	else if ( addedcommands.contains( id ) )
	{
		umc = addedcommands[id];
		
		QString origcommand = umc->m_sCommand;
		userCommand = m_pClient->replaceCommandTags( origcommand, pressedNick );
		
		if ( !userCommand.isEmpty() )
		{
			AddStatus(userCommand);
			m_pClient->SendString(userCommand.ascii());
		}
	}
}

/** */
bool DCChat::close( bool alsoDelete )
{
	// re-set the wflags ...
	setWFlags(WDestructiveClose);
	return QWidget::close(alsoDelete);
}

/** */
void DCChat::slotSendChat()
{
	SendChat();
}

/** */
void DCChat::SetNick( QString nick, QString hubname )
{
	m_sNick = nick;

	if ( nick.isEmpty() )
	{
		setCaption(tr("Chat:")+" ["+hubname+"]");
	}
	else
	{
		setCaption(tr("Private Chat:")+" "+nick+" ["+hubname+"]");
	}
}

/** */
QString DCChat::GetTimeStamp()
{
	QString ts;
	
	if ( m_bPrivateChat )
	{
		if ( g_pConfig->GetTimeStamp(etsPRIVATECHAT) )
		{
			ts  = "<font color=\"";
			ts += g_pConfig->GetChatColor(eccCHATTIMESTAMP);
			ts += "\">";
			ts += QTime::currentTime().toString("[hh:mm:ss]");
			ts += "</font> ";
		}
	}
	else if ( g_pConfig->GetTimeStamp(etsHUBCHAT) )
	{
		ts  = "<font color=\"";
		ts += g_pConfig->GetChatColor(eccPUBLICCHATTIMESTAMP);
		ts += "\">";
		ts += QTime::currentTime().toString("[hh:mm:ss]");
		ts += "</font> ";
	}
	
	return ts;
}

/** */
void DCChat::AddOutput( QString message )
{
	bool bscroll;

	if ( m_pTextEdit_CHATOUTPUT->verticalScrollBar()->maxValue() == m_pTextEdit_CHATOUTPUT->verticalScrollBar()->value() )
	{
		bscroll = true;
	}
	else
	{
		bscroll = false;
	}

	// convert newline
	message.replace( "\n", "<br />" );
	
	// convert tabs to spaces
	message.replace( "\t", "&nbsp;&nbsp;&nbsp;&nbsp;" );

	if ( (g_pConfig->GetChatMaxParagraph()!= 0) && (m_pTextEdit_CHATOUTPUT->paragraphs() > (g_pConfig->GetChatMaxParagraph()+5)) )
	{
		m_pTextEdit_CHATOUTPUT->removeParagraph(0);
		m_pTextEdit_CHATOUTPUT->removeParagraph(0);
		m_pTextEdit_CHATOUTPUT->removeParagraph(0);
		m_pTextEdit_CHATOUTPUT->removeParagraph(0);
		m_pTextEdit_CHATOUTPUT->removeParagraph(0);
	}
	
	m_pTextEdit_CHATOUTPUT->append( message );

	if ( bscroll )
	{
		m_pTextEdit_CHATOUTPUT->scrollToBottom();
		m_pTextEdit_CHATOUTPUT->moveCursor( QTextEdit::MoveEnd, false );
	}

	message += "<br />";

	// save to logfile
	if ( g_pConfig->GetLogChatOption(elcoENABLELOGGING) )
	{
		if ( !(g_pConfig->GetLogChatOption(elcoDISABLEPUBLICCHAT) && !m_bPrivateChat) )
		{
			if ( g_pConfig->CheckLogChatNickNameFilter(m_sNick) )
			{
				QString s = m_sNick;

				if ( g_pConfig->GetLogChatOption(elcoAPPENDHUBNAME) )
				{
					s += "_";
					s += QString::fromAscii(m_pClient->GetHubName().Data());
				}

				if ( g_pConfig->GetLogChatOption(elcoAPPENDHUBHOST) )
				{
					s += "_";
					s += QString::fromAscii(m_pClient->GetIP().Data());
				}

				if ( g_pConfig->GetLogChatOption(elcoAPPENDDATE) )
				{
 					s += "_";
					s += m_sTimeStamp;
				}

				// damn hacking ;-)
				s.remove('/');
				s.remove('\\');
				s.remove(':');

				s = QString::fromAscii(g_pConfig->GetChatLogPath().Data()) + s + ".html";

				QFile f(s);

				if ( f.open(IO_WriteOnly | IO_Append) )
				{
					// add newline
					message += "\n";

					f.writeBlock(message.ascii(),strlen(message.ascii()));
					f.close();
				}
			}
		}
	}
}

/** */
void DCChat::AddStatus( QString message, bool show )
{
	if ( g_pConfig->GetShowStatusMessage() || (show) )
	{
		// convert special chars ... damn hacking ;-)
		message.replace( "<", "&lt;" );
		message.replace( ">", "&gt;" );
		
		QString html = GetTimeStamp();

		if ( m_bPrivateChat )
		{
			html += "<font color=\"";
			html += g_pConfig->GetChatColor(eccCHATSTATUSNICK);
			html += "\"><b>&lt;VALKNUT&gt;</b> </font><font color=\"";
			html += g_pConfig->GetChatColor(eccCHATSTATUSTEXT);
			html += "\">";
			html += message;
			html += "</font>";
		}
		else
		{
			html += "<font color=\"";
			html += g_pConfig->GetChatColor(eccPUBLICCHATSTATUSNICK);
			html += "\"><b>&lt;VALKNUT&gt;</b> </font><font color=\"";
			html += g_pConfig->GetChatColor(eccPUBLICCHATSTATUSTEXT);
			html += "\">";
			html += message;
			html += "</font>";
		}
		
		AddOutput( html );
	}
}

/** */
void DCChat::AddMessage( CMessagePrivateChat * msg, bool bremote, bool forward )
{
	if ( msg->m_eSecureState != m_eSecureState )
	{
		SetCrypt(msg->m_eSecureState);
	}

	if ( msg->m_sSrcNick.IsEmpty() )
		AddMessage( QString::fromAscii(msg->m_sSrcNick.Data()), QString::fromAscii(msg->m_sMessage.Data()), bremote, forward );
	else
		AddMessage( QString::fromAscii(msg->m_sMultiSrcNick.Data()), QString::fromAscii(msg->m_sMessage.Data()), bremote, forward );
}

/** */
void DCChat::AddMessage( QString nick, QString message, bool bremote, bool forward )
{
	QString msg,s,s1;
	int index;
	int i1,i2;
	bool b1,b2;
	bool bSay = false;
	bool bMe = false;
	bool isOP = m_pClient->UserList()->IsAdmin( nick.ascii() );
	QString clientnick = QString::fromAscii( m_pClient->GetNick().Data() );

	//printf("ADD MESSAGE '%s'\n",message.Data());

	if ( message.isEmpty() )
	{
		return;
	}

	if ( CheckForData(message) )
	{
		return;
	}

	// check for nick - but avoid string searching if not enabled
	if ( !m_bPrivateChat && g_pConfig->GetSoundEnabled(eusNICKMENTIONED) )
	{
		if ( message.find(clientnick) != -1 )
		{
			g_pConfig->PlaySound(eusNICKMENTIONED);
		}
	}

	// convert special chars ... damn hacking ;-)
	message.replace( "<", "&lt;" );
	message.replace( ">", "&gt;" );

	// convert special chars in nick ... damn hacking ;-)
	nick.replace( "<", "&lt;" );
	nick.replace( ">", "&gt;" );

	s1  = clientnick;
	s1 += ": ";

	if ( ( message.left(4).upper() == "/ME " ) || ( message.left(4).upper() == "+ME " ) )
	{
		bMe    = true;
		message = message.right( message.length() - 4 );
	}

	if ( message.startsWith(s1) )
	{
		bSay    = true;
		// message = message.right( message.length() - s1.length() );
	}

	msg = message;
	s1 = "";

	b1 = b2 = true;

	while ( !msg.isEmpty() )
	{
		if (b1)
		{
			if ( (i1 = FindFirstLink(msg)) == -1 )
			{
				b1 = false;
			}
		}
		else
		{
			i1 = -1;
		}

		if (b2)
		{
			if ( (i2 = FindFirstEmoticon(msg)) == -1 )
			{
				b2 = false;
			}
		}
		else
		{
			i2 = -1;
		}

		if ( ((i1 < i2) || (i2 == -1)) && (i1 != -1) )
		{
			// convert link
			s1 += msg.left(i1);
			msg = msg.right(msg.length()-i1);

			index = ConvertLinks( msg, s );
		}
		else if ( ((i2 < i1) || (i1 == -1)) && (i2 != -1) )
		{
			// convert emot
			s1 += msg.left(i2);
			msg = msg.right(msg.length()-i2);

			index = ConvertEmoticons( msg, s );
		}
		else
		{
			// no link & no emot found
			s1 += msg;
			msg = "";
			break;
		}

		if ( index > 1 )
		{
			s1 += s;
			msg = msg.right(msg.length()-index);
			continue;
		}

		if ( index == 1 )
		{
			s1 += msg.left(1);
		}

		msg = msg.right(msg.length()-index);
	}

	message = s1;

	QString html = GetTimeStamp();

	if ( !nick.isEmpty() )
	{
		html += "<font color=\"";
		
		if ( m_bPrivateChat )
		{
			if ( bMe )
			{
				html += g_pConfig->GetChatColor(eccPUBLICCHATMENICK);
			}
			else if ( nick != clientnick )
			{
				if ( isOP )
				{
					html += g_pConfig->GetChatColor(eccOPNICK);
				}
				else
				{
					html += g_pConfig->GetChatColor(eccCHATREMOTENICK);
				}
			}
			else
			{
				html += g_pConfig->GetChatColor(eccCHATLOCALNICK);
			}
			
			html += "\"><b>";
			html += nick;
			if ( bMe )
			{
				html += "</b> </font>";
			}
			else
			{
				html += "</b>: </font>";
			}
		}
		else
		{
			if ( forward )
			{
				html += g_pConfig->GetChatColor(eccPUBLICPRIVATECHATNICK);
				if ( bMe )
				{
					html += "\"><b>";
					html += nick;
					html += "</b> </font>";
				}
				else
				{
					html += "\"><b>&lt;";
					html += nick;
					html += "&gt;</b> </font>";
				}
			}
			else if ( bMe )
			{
				html += g_pConfig->GetChatColor(eccPUBLICCHATMENICK);
				html += "\"><b>";
				html += nick;
				html += "</b> </font>";
			}
			else if ( nick != clientnick )
			{
				if ( isOP )
				{
					html += g_pConfig->GetChatColor(eccOPNICK);
				}
				else
				{
					html += g_pConfig->GetChatColor(eccPUBLICCHATREMOTENICK);
				}
				
				html += "\"><b>&lt;";
				html += nick;
				html += "&gt;</b> </font>";
			}
			else
			{
				html += g_pConfig->GetChatColor(eccPUBLICCHATLOCALNICK);
				html += "\"><b>&lt;";
				html += nick;
				html += "&gt;</b> </font>";
			}
		}
	}

	html += "<font color=\"";
	
	if ( m_bPrivateChat )
	{
		if ( bremote && (nick != clientnick) )
		{
			if ( bSay )
			{
				html += g_pConfig->GetChatColor(eccCHATSAY);
			}
			else
			{
				html += g_pConfig->GetChatColor(eccCHATREMOTETEXT);
			}
		}
		else
		{
			html += g_pConfig->GetChatColor(eccCHATLOCALTEXT);
		}
	}
	else
	{
		if ( forward )
		{
			html += g_pConfig->GetChatColor(eccPUBLICPRIVATECHATTEXT);
		}
		else if ( bremote && (nick != clientnick) )
		{
			if ( bSay )
			{
				html += g_pConfig->GetChatColor(eccCHATSAY);
			}
			else
			{
				html += g_pConfig->GetChatColor(eccPUBLICCHATREMOTETEXT);
			}
		}
		else
		{
			html += g_pConfig->GetChatColor(eccPUBLICCHATLOCALTEXT);
		}
	}

	html += "\">";
	html += message;
	html += "</font>";

	AddOutput( html );

	if ( bremote && m_bPrivateChat )
	{
		g_pConfig->PlaySound(eusRECEIVE);

		// send away message
		if ( g_pConfig->GetAwayMode() == euamAWAY )
		{
			CString awaymsg = g_pConfig->GetAwayMessage();

			if ( !(awaymsg.IsEmpty()) && (m_bSendAway) )
			{
				m_pClient->SendPrivateMessage( m_pClient->GetNick(), m_sNick.ascii(), awaymsg );
				m_bSendAway = false;
				AddStatus(tr("Sent away message: ") + QString::fromAscii(awaymsg.Data()));
			}
		}
		else
		{
			m_bSendAway = true;
		}
	}
}

/** */
void DCChat::SendChat( QString message )
{
	int err;
	CString s;
	bool local = false;
	bool cmd = false;

	if ( !m_pClient )
	{
		return;
	}

	if ( message.isEmpty() )
	{
		message = TextEdit_CHATINPUT->text();
	}
	else
	{
		local = true;
	}

	if ( !message.isEmpty() )
	{
		if ( local == false )
		{
			// add history
			AddHistory(message);
			m_sHistoryTempString = QString();
			if ( message.at(0) == '/' )
			{
				cmd = CheckForCommand();
			}
		}

		if ( cmd == false )
		{
			// convert the newlines
			// any windows to unix then all unix to windows
			// so windows do not become double newlines
			message.replace( "\r\n", "\n" );
			message.replace( "\n", "\r\n" );

			if ( m_bPrivateChat )
				err = m_pClient->SendPrivateMessage( m_pClient->GetNick(), m_sNick.ascii(), message.ascii() );
			else
				err = m_pClient->SendChat( m_pClient->GetNick(), message.ascii() );

			if ( err == 0 )
			{
				if ( m_bPrivateChat )
					AddMessage( QString::fromAscii(m_pClient->GetNick().Data()), message, false );
				g_pConfig->PlaySound(eusSEND);
			}
			else
			{
				AddStatus( tr("Message not sent!") );
			}
		}

		if ( local == false )
		{
			TextEdit_CHATINPUT->clear();
		}
	}
}

/** */
int DCChat::FindFirstLink( QString msg )
{
	int current = -1;
	int first = -1;

	for ( unsigned int i = 0; i < g_LinkList.size(); ++i )
	{
		current = msg.find(g_LinkList[i],0,false);
		if ( current != -1 )
		{
			if ( (first == -1) || (first > current) )
			{
				first = current;
			}
		}
	}

	return first;
}

/** */
int DCChat::ConvertLinks( QString msg, QString & s )
{
	int ret = 1;
	int i1,i2,i3;

	// convert links
	for ( unsigned int i = 0; i < g_LinkList.size(); ++i )
	{
		if ( msg.startsWith(g_LinkList[i],false) )
		{
			// link end at space or newline
			i1 = msg.find(' ',0,false);
			i2 = msg.find('\xa',0,false);
			i3 = msg.find('\xd',0,false);

			if ( i1 == -1 )
				i1 = msg.length();
			if ( (i1 > i2) && (i2 != -1) )
				i1 = i2;
			if ( (i1 > i3) && (i3 != -1) )
				i1 = i3;

			s = msg.left(i1);
			s = "<a title=\""+s+"\" name=\""+s+"\" href=\""+s+"\">"+s+"</a>";

			ret = i1;

			break;
		}
	}

	return ret;
}

/** */
int DCChat::FindFirstEmoticon( QString msg )
{
	int i1 = -1, i2 = -1;
	QPtrList<DC_EmoticonObject> * elist = g_pConfig->EmoticonList();

	if ( g_pConfig->GetEnableEmoticons() && (elist != 0) )
	{
		DC_EmoticonObject * EmoticonObject = 0;

		QPtrListIterator<DC_EmoticonObject> it( *elist );
		while ( ( EmoticonObject = it.current() ) != 0 )
		{
			++it;
			if ( (i1 = msg.find( EmoticonObject->m_Text, 0 )) != -1 )
			{
				if ( (i2 == -1) || (i2 > i1) )
				{
					i2 = i1;
				}
			}
		}
	}

	return i2;
}

/** */
int DCChat::ConvertEmoticons( QString msg, QString & s )
{
	int ret = 1;
	QPtrList<DC_EmoticonObject> * elist = g_pConfig->EmoticonList();

	// convert emoticons
	if ( g_pConfig->GetEnableEmoticons() && (elist != 0) )
	{
		DC_EmoticonObject * EmoticonObject = 0;
		QPtrListIterator<DC_EmoticonObject> it( *elist );
		
		while ( ( EmoticonObject = it.current() ) != 0 )
		{
			if ( msg.startsWith(EmoticonObject->m_Text) )
			{
				s   = "<img alt=\"";
				s  += EmoticonObject->m_Text;
				s  += "\" title=\"";
				s  += EmoticonObject->m_Text;
				s  += "\" align=\"center\" source=\"emoticon";
				s  += QString().setNum(EmoticonObject->m_nID);
				s  += "\" />";
				ret = EmoticonObject->m_Text.length();
				break;
			}
			++it;
		}
	}

	return ret;
}

/** */
bool DCChat::CheckForCommand()
{
	bool res = false;
	QString s;
	
	QString cmd = g_pConfig->ReplaceUserChatCommands( TextEdit_CHATINPUT->text() );
	
	if ( cmd.isEmpty() )
	{
		return res;
	}
	
	if ( cmd.at(0) == '/' )
	{
		/*
		 * Extract command and remaining arguments.
		 * More complicated commands need to
		 * extract their arguments themselves but
		 * most commands just take a nick.
		 *
		 * Commands whose first argument is
		 * not a nick should not use theRest
		 * because it gets set to the nick
		 * if private chat.
		 */
		
		int firstspace = cmd.find( ' ' );
		
		QString cmdOnly;
		QString theRest;
		
		if ( firstspace == -1 )
		{
			cmdOnly = cmd;
			if ( m_bPrivateChat )
			{
				theRest = m_sNick;
			}
		}
		else
		{
			cmdOnly = cmd.left( firstspace );
			theRest = cmd.mid( firstspace + 1 );
		}
		
		// AddStatus( "cmdOnly ='" + cmdOnly + "'", true );
		// AddStatus( "theRest = '" + theRest + "'", true );
		
		// help
		if ( cmdOnly == "/dchelp" )
		{
			s  = tr("Help:");
			s += "\n";
			s += tr("/clear - clears the chat window");
			s += "\n";
			s += tr("/mode - shows the current mode");
			s += "\n";
			s += tr("/refresh - refresh share");
			s += "\n";
			s += tr("/rebuild - remove no longer present files from hash database");
			s += "\n";
			s += tr("/validate - check hash database for errors");
			s += "\n";
			s += tr("/slots &lt;N&gt; - set number of upload slots to N");
			s += "\n";
			s += tr("/ts - switch time display in chat on/off");
			s += "\n";
			s += tr("/adv - send an advertisment to the hub");
			s += "\n";
			s += tr("/join &lt;address&gt; - disconnect from currently connected hub and connect to another hub");
			s += "\n";
			s += tr("/msg &lt;nick&gt; &lt;message&gt; - send private message");
			s += "\n";
			s += tr("/away &lt;message&gt; - set automatic response for private messages; if the message is empty, disable it");
			s += "\n";
			s += tr("/bye &lt;message&gt; - disconnect from channel with an optional message");
			s += "\n";
			s += tr("/uptime [show] - show valknut uptime [to other users]");
			s += "\n";
			s += tr("/ls &lt;nick&gt; - get share list from user");
			s += "\n";
			s += tr("/grant &lt;nick&gt; - grant user a slot");
			s += "\n";
			s += tr("/grantp &lt;nick&gt; - grant user a permanent slot");
			s += "\n";
			s += tr("/friend &lt;nick&gt; - add user to friend list");
			s += "\n";
			s += tr("/fav - add hub to bookmark list");
			s += "\n";
			s += tr("/info &lt;nick&gt; - show user info");
			s += "\n";
			s += tr("/now - send current time to the chat");
			s += "\n";
			s += tr("/sh &lt;command&gt; &lt;args...&gt; - send output of shell command to chat");
			s += "\n";
			s += tr("/raw - send raw message");
			s += "\n";
			s += tr("/disablesorting - disable sorting of the user list");
			s += "\n";
			s += tr("/enablesorting - enable sorting of the user list");
			s += "\n";
			s += tr("/ignore &lt;nick&gt; - do not show chat messages from the user");
			s += "\n";
			s += tr("/unignore &lt;nick&gt; - show chat messages from the user again");
			s += "\n";
			s += tr("/ratio [show] - show share ratio [to other users]");
			s += "\n";
			s += tr("/newlog - start a new logfile");
			s += "\n";
			AddStatus( s, true );
			res = true;
		}
		// /clear - clear the chat window
		else if ( cmdOnly == "/clear" )
		{
			m_pTextEdit_CHATOUTPUT->clear();

			AddStatus( tr("Chat Cleared."), true );
			res = true;
		}
		// /info - show user info
		else if ( cmdOnly == "/info" )
		{
			CMessageMyInfo myinfo;
			
			if ( theRest.isEmpty() )
			{
				s = tr("No nick, try /dchelp");
			}
			else if (m_pClient->UserList()->GetUserMyInfo( theRest.ascii(), &myinfo ) == false)
			{
				s = tr("Get info failed.");
			}
			else
			{
				s  = tr("Info:");
				s += "\n";
				s += tr("Nick:");
				s += ' ';
				s += QString::fromAscii(myinfo.m_sNick.Data());
				s += "\n";
				s += tr("Operator:");
				s += ' ';

				if ( myinfo.m_bOperator )
				{
					s += tr("yes");
				}
				else
				{
					s += tr("no");
				}

				s += "\n";

				s += tr("Comment:");
				s += ' ';
				s += QString::fromAscii(myinfo.m_sComment.Data());
				s += "\n";
				s += tr("Speed:");
				s += ' ';
				s += QString::fromAscii(myinfo.m_sUserSpeed.Data());
				s += "\n";
				s += tr("EMail:");
				s += ' ';
				s += QString::fromAscii(myinfo.m_sEMail.Data());
				s += "\n";
				s += tr("Shared:");
				s += ' ';
				s += QString().setNum(myinfo.m_nShared);
				s += ' ';
				s += tr("Bytes");
				s += "\n";

				s += tr("Away:");
				s += ' ';

				switch(myinfo.m_eAwayMode)
				{
					case euamAWAY:
						s += tr("on");
						break;
					default:
						s += tr("off");
						break;
				}

				s += "\n";

				s += tr("Version:");
				s += ' ';

				switch(myinfo.m_eClientVersion)
				{
					case eucvDCPP:
						s += "DC++";
						break;
					case eucvDCGUI:
						s += "Valknut";
						break;
					case eucvMICRODC:
						s += "microdc";
						break;
					case eucvSHAKESPEER:
						s += "ShakesPeer";
						break;
					default:
 						s += tr("Unknown");
						break;
				}

				s += "\n";

				s += tr("Tag:");
				s += ' ';
				s += QString::fromAscii(myinfo.m_sVerComment.Data());
				s += "\n";
				s += tr("Supports encryption:");
				s += ' ';
				if ( myinfo.m_bTLSFlag )
				{
					s += tr("yes");
				}
				else
				{
					s += tr("no");
				}
				s += "\n";
			}

			AddStatus( s, true );
			res = true;
		}
		// /uptime - show valknut uptime
		else if ( cmdOnly == "/uptime" )
		{
			int d,h,m;
			int i = time(0)-dclibUptime();

			d = i/(60*60*24);
			i = i%(60*60*24);
			h = i/(60*60);
			i = i%(60*60);
			m = i/60;

			s  = tr("Valknut uptime");
			s += ": ";

			if ( d > 0 )
			{
				s += QString().setNum(d);
				s += ' ';
				if ( d == 1 )
					s += tr("day");
				else
					s += tr("days");
				s += ',';
			}
			if ( h > 0 )
			{
				s += QString().setNum(h);
				s += ' ';
				if ( h == 1 )
					s += tr("hour");
				else
					s += tr("hours");
				s += ',';
			}
			s += QString().setNum(m);
			s += ' ';
			if ( m == 1 )
				s += tr("minute");
			else
				s += tr("minutes");
			
			if ( cmd.mid( 8 ).lower() == "show" )
				SendChat(s);
			else
				AddStatus( s, true );

			res = true;
		}
		// /mode - show current mode
		else if ( cmdOnly == "/mode" )
		{
			s  = tr("Current mode: ");

			if ( g_pConfig->GetMode() == ecmACTIVE )
				s += tr("active");
			else
				s += tr("passive");

			// get listen manager status
			if ( CListenManager::Instance() )
			{
				CString lms = CListenManager::Instance()->GetSocketErrorMsg();
			
				if ( !(lms.IsEmpty()) )
				{
					s += " (";
					s += QString::fromAscii(lms.Data());
					s += ')';
				}	
			}
			
			s += '.';
			
			AddStatus( s, true );
			res = true;
		}
		// /refresh - refresh share 
		else if ( cmdOnly == "/refresh" )
		{
			if ( CFileManager::Instance()->CreateShareList() )
			{
				s = tr("Refresh share in progress.");
			}
			else
			{
				s = tr("A share operation is already in progress.");
			}

			AddStatus( s, true );
			res = true;
		}
		// /rebuild - clean up share databases
		else if ( cmdOnly == "/rebuild" )
		{
			if ( CFileManager::Instance()->RebuildLists() )
			{
				s = tr("Rebuild share in progress.");
			}
			else
			{
				s = tr("A share operation is already in progress.");
			}
			
			AddStatus( s, true );
			res = true;
		}
		else if ( cmdOnly == "/validate" )
		{
			if ( CFileManager::Instance()->ValidateLeaves() )
			{
				s = tr("Validate hash database in progress.");
			}
			else
			{
				s = tr("A share operation is already in progress.");
			}
			
			AddStatus( s, true );
			res = true;
		}
		// /slots - set number of slots
		else if ( cmdOnly == "/slots" )
		{
			s = cmd.mid(7);
			if ( s.isEmpty() )
			{
				AddStatus( tr("Upload slots: ") + QString().setNum(g_pConfig->GetMaxUpload()), true );
			}
			else
			{
				bool valid = false;
				int i = s.toInt(&valid);
				
				if ( valid && (i >= 0) && (i <= 99) )
				{
					g_pConfig->SetMaxUpload(i);
					AddStatus( tr("Setting upload slots to") + " " + QString::number(i), true );
				}
				else
				{
					AddStatus( tr("Invalid number of upload slots specified."), true );
				}
			}
			
			res = true;
		}
		// /ts - switch time display in chat on/off 
		else if ( cmdOnly == "/ts" )
		{
			bool b;

			if ( m_bPrivateChat )
			{
				b = !g_pConfig->GetTimeStamp(etsPRIVATECHAT);
				g_pConfig->SetTimeStamp(etsPRIVATECHAT,b);
			}
			else
			{
				b = !g_pConfig->GetTimeStamp(etsHUBCHAT);
				g_pConfig->SetTimeStamp(etsHUBCHAT,b);
			}

			s = tr("Switch timestamp ");

			if ( b )
				s += tr("on.");
			else
				s += tr("off.");

			AddStatus( s, true );
			res = true;
		}
		// /adv - send an advertisment to the hub
		else if ( cmdOnly == "/adv" )
		{
			AddStatus( tr("Send advertisment."), true );

			s = tr("\nDownload Valknut from http://wxdcgui.sourceforge.net\nIt does everything !\nWorks on almost any platform you can think of and makes you coffee.\n");

			SendChat(s);

			res = true;
		}
		// /now - send current time to the chat
		else if ( cmdOnly == "/now" )
		{
			s = QTime::currentTime().toString("hh:mm:ss");
			
			if ( cmd.length() > 5 )
			{
				s += " ";
				s += cmd.mid(5,cmd.length()-5);
			}

			SendChat(s);

			res = true;
		}
		// /raw - send raw message
		else if ( cmdOnly == "/raw" )
		{
			s = cmd.mid( 5 );
			
			if ( !(s.isEmpty()) )
			{
				AddStatus( cmd, true );
				m_pClient->SendString(s.ascii());
			}
			else
			{
				AddStatus( tr("Wrong parameter for '/raw'"), true );
			}

			res = true;
		}
#ifdef ENABLE_PRETEND_TESTING_COMMAND
		else if ( cmdOnly == "/pretend" )
		{
			AddStatus( cmd, true );
			
			/* no attempt is made to make encoding correct */
			QCString cs = cmd.mid(9).local8Bit();
			m_pClient->DataAvailable(cs,cs.length());
			
			res = true;
		}
#endif
		// /join <address> - disconnect from currently connected hub and connect to another hub 
		else if ( cmdOnly == "/join" )
		{
			QString address = cmd.mid(6,cmd.length()-6);

			if ( address.isEmpty() )
			{
				s = tr("Wrong parameter for '/join'");
			}
			else
			{
				s = tr("Join to: ") + address;

				CString caddress = address.ascii();

				m_pClient->SetHubName(caddress);
				m_pClient->setCaption(address);
				m_pClient->Connect(caddress);
			}

			AddStatus( s, true );
			res = true;
		}
		// /ls - get share list from user
		else if ( cmdOnly == "/ls" )
		{
			QString nick;

			if ( theRest.isEmpty() )
			{
				s = tr("No nick, try /dchelp");
			}
			else
			{
				/** add transfer to the waitlist */
				CString empty;
				g_pTransferView->DLM_QueueAdd( theRest.ascii(), m_pClient->GetHubName(), m_pClient->GetHost(),
							DC_USER_FILELIST, DC_USER_FILELIST, empty, empty, eltBUFFER,
							0, 0, 0, empty );

				s  = tr("Download sharelist from");
				s += " '";
				s += theRest;
				s += '\'';
			}

			AddStatus( s, true );
			res = true;
		}
		// /grantp - grant perm. slot to user
		else if ( cmdOnly == "/grantp" )
		{
			if ( theRest.isEmpty() )
			{
				s = tr("No nick, try /dchelp");
			}
			else
			{
				g_pTransferView->DLM_AddUserSlot( theRest.ascii(), m_pClient->GetHubName(), 0, true );

				s  = tr("Permanent slot added for");
				s += " '";
				s += theRest;
				s += '\'';
			}

			AddStatus( s, true );
			res = true;
		}
		// /grant - grant slot to user
		else if ( cmdOnly == "/grant" )
		{
			if ( theRest.isEmpty() )
			{
				s = tr("No nick, try /dchelp");
			}
			else
			{
				g_pTransferView->DLM_AddUserSlot( theRest.ascii(), m_pClient->GetHubName(), 1 );

				s  = tr("Slot added for");
				s += " '";
				s += theRest;
				s += '\'';
			}

			AddStatus( s, true );
			res = true;
		}
		// /friend - add user to friend list
		else if ( cmdOnly == "/friend" )
		{
			if ( theRest.isEmpty() )
			{
				s = tr("No nick, try /dchelp");
			}
			else
			{
				g_pUsersList->AddFriend( theRest, QString::fromAscii(m_pClient->GetHubName().Data()), QString::fromAscii(m_pClient->GetHost().Data()), "" );

				s  = tr("Added to friend list ");
				s += " '";
				s += theRest;
				s += '\'';
			}

			AddStatus( s, true );
			res = true;
		}
		// /fav - add hub to bookmarks
		else if ( cmdOnly == "/fav" )
		{
			QString qhubname = QString::fromAscii( m_pClient->GetHubName().Data() );
			g_pHubListManager->AddBookmark( qhubname, QString::fromAscii(m_pClient->GetHost().Data()), QString() );

			s  = tr("Add bookmark hub");
			s += " '";
			s += qhubname;
			s += '\'';

			AddStatus( s, true );
			res = true;
		}
		// /bye <msg> - disconnect from channel
		else if ( cmdOnly == "/bye" )
		{
			s = cmd.mid( 5 );
			if ( !(s.isEmpty()) )
			{
				SendChat(s);
			}

			m_pClient->Disconnect(true);

			res = true;
		}
		// /msg <nick> <message> - send private message 
		else if ( cmdOnly == "/msg" )
		{
			QString nick,message;
			
			AddStatus( cmd, true );

			int i = theRest.find(' ');
			if ( i != -1 )
			{
				nick    = theRest.left(i);
				message = theRest.mid(i+1);
			}

			if ( nick.isEmpty() || message.isEmpty() )
			{
				s = tr("Wrong parameter for '/msg'");
			}
			else
			{
				if ( m_pClient->SendPrivateMessage( m_pClient->GetNick(), nick.ascii(), message.ascii() ) == 0 )
				{
					s = tr("Private message sent.");
					g_pConfig->PlaySound(eusSEND);
				}
				else
				{
					s = tr("Private message not sent!");
				}
			}

			AddStatus( s, true );
			res = true;

		}
		// /away <message> - set automatic response for private messages; if the message is empty, disable it 
		else if ( cmdOnly == "/away" )
		{
			s = cmd.mid(6);

			g_pConfig->SetAwayMessage(s.ascii());

			if ( s.isEmpty() )
				s = tr("Away message disabled.");
			else
				s = tr("Away message set.");

			AddStatus( s, true );
			res = true;
		}
		else if ( cmdOnly == "/sh" )
		{
			// display the command the user entered
			AddStatus( cmd, true );
			
			QString shellargs = cmd.mid(4, cmd.length() - 4);
			
			if ( shellargs.isEmpty() )
			{
				AddStatus( tr("No command entered."), true );
				return true;
			}
			
			if ( g_pConfig->GetThreadsForShellCommands() )
			{
				DCShellCommandRunner * runner = new DCShellCommandRunner( this, shellargs );
				m_pShellRunners->append(runner);
				runner->start();
				
			}
			else
			{
				QProcess shellproc( this );
				
				/* QT3 seems somewhat unhelpful and this is necessary */
				shellproc.addArgument( "sh" );
				shellproc.addArgument( "-c" );
				shellproc.addArgument( shellargs );
				
				shellproc.setCommunication( QProcess::Stdout );
				
				if ( shellproc.launch( QByteArray() ) )
				{
					// wait up to n seconds for the process to finish
					int i = 0;
					while ( (shellproc.isRunning()) && (i < g_pConfig->GetShellCommandTimeout()) )
					{
#ifdef WIN32
						Sleep(1000);
#else
						sleep(1);
#endif
						i++;
					}
					
					// kill the process if it's still running
					if ( shellproc.isRunning() )
					{
						AddStatus( tr("Process still running after") + " " + QString().setNum(g_pConfig->GetShellCommandTimeout()) + " " + tr("seconds, killing process..."), true );
						shellproc.tryTerminate();
#ifdef WIN32
						Sleep(1000);
#else
						sleep(1);
#endif
						if ( shellproc.isRunning() )
						{
							shellproc.kill();
						}
					}
					
					if ( shellproc.normalExit() )
					{
						int exitcode = shellproc.exitStatus();
						if ( exitcode == 0 )
						{
							s = QString( shellproc.readStdout() ).stripWhiteSpace();
							if ( s.isEmpty() )
							{
								AddStatus( tr("Command produced no visible output."), true );
							}
							else
							{
								SendChat(s);
							}
						}
						else
						{
							AddStatus( tr("Process exited with status") + " " + QString::number(exitcode), true);
						}
					}
					else
					{
						AddStatus( tr("Process was killed or crashed."), true);
					}
				}
				else
				{
					AddStatus( tr("Failed to start shell command."), true );
				}
			}
			
			res = true;
		}
		else if ( cmdOnly == "/sendphoto" )
		{
			if ( !m_bPrivateChat )
			{
				s = tr("Only allowed in private chat.");
			}
			else if ( m_pClient->UserList()->GetUserClientVersion(m_sNick.ascii()) != eucvDCGUI )
			{
				s = tr("The user is using an old version or not using Valknut");
			}
			else
			{
				CString fn = g_pConfig->GetUserPhotoFileName().ascii();
				CByteArray bas,bad;

				if ( CDir().IsFile( fn, false ) )
				{
					if ( (bas.LoadFromFile(fn) == false) || (bas.Size() == 0) )
					{
						s = tr("Can't read your photo.");
					}
					else
					{
						CBase64::Encode( &bad, &bas );

						if ( bad.Size() == 0 )
						{
							s = tr("Photo encode error.");
						}
						else
						{
							CString se;

							se.Set((char*)bad.Data(),bad.Size());
							se  = "<photo data=\"" + se + "\">";

							if ( m_pClient->SendPrivateMessage( m_pClient->GetNick(), m_sNick.ascii(), se ) != 0 )
							{
								s = tr("Photo not sent.");
							}
							else
							{
								s = tr("Photo sent.");
							}
						}
					}
				}
				else
				{
					s = tr("Photo not found.");
				}
			}

			AddStatus( s, true );
			res = true;
		}
		else if ( cmdOnly == "/disablesorting" )
		{
			m_pClient->ListView_USERLIST->setSortColumn(-1);
			s = tr("Disabled user list sorting.");
			AddStatus( s, true );
			res = true;
		}
		else if ( cmdOnly == "/enablesorting" )
		{
			m_pClient->ListView_USERLIST->setSortColumn(0);
			s = tr("Enabled user list sorting.");
			AddStatus( s, true );
			res = true;
		}
		else if ( cmdOnly == "/ignore" )
		{
			if ( theRest.isEmpty() )
			{
				s = tr("No nick, try /dchelp");
			}
			else
			{
				if (g_pUsersList->isNickInList(theRest) == false)
				{
					g_pUsersList->AddFriend( theRest, QString::fromAscii(m_pClient->GetHubName().Data()), QString::fromAscii(m_pClient->GetHost().Data()), QString() );
				}
				
				g_pUsersList->setIgnore( theRest, true );
				s  = tr("Ignoring chat from ");
				s += theRest;
			}
			
			AddStatus( s, true );
			res = true;
			
		}
		else if ( cmdOnly == "/unignore" )
		{
			if ( theRest.isEmpty() )
			{
				s = tr("No nick, try /dchelp");
			}
			else
			{
				if ( g_pUsersList->ignoreNick( theRest ) )
				{
					g_pUsersList->setIgnore( theRest, false );
					s  = tr("Showing chat from ");
					s += theRest;
				}
				else
				{
					s  = theRest;
					s += tr(" is not in ignore list");
				}
				
			}
			
			AddStatus( s, true );
			res = true;
		}
		else if ( cmdOnly == "/ratio" )
		{
			ulonglong up = CSocket::m_Traffic.GetTraffic(ettTX);
			ulonglong down = CSocket::m_Traffic.GetTraffic(ettRX);
			
			double ratio = 0;
			
			if ( down > 0 )
			{
				ratio = (double) up / (double) down;
			}
			
			s  = tr("ratio: overall: ");
			s += QString().setNum( ratio, 'f', 2 );
			s += " (";
			s += tr("uploads");
			s += ": ";
			s += DCGuiUtils::GetSizeString( up );
			s += " | ";
			s += tr("downloads");
			s += ": ";
			s += DCGuiUtils::GetSizeString( down );
			s += ")\n";
			
			up -= g_pConfig->GetStartUploaded();
			down -= g_pConfig->GetStartDownloaded();
			
			if ( down > 0 )
			{
				ratio = (double) up / (double) down;
			}
			else
			{
				ratio = 0;
			}
			
			s += tr("this session: ");
			s += QString().setNum( ratio, 'f', 2 );
			s += " (";
			s += tr("uploads");
			s += ": ";
			s += DCGuiUtils::GetSizeString( up );
			s += " | ";
			s += tr("downloads");
			s += ": ";
			s += DCGuiUtils::GetSizeString( down );
			s += ")";
			
			if ( cmd.mid(7).lower() == "show" )
			{
				SendChat( s );
			}
			else
			{
				AddStatus( s, true );
			}
			res = true;
		}
		else if ( cmdOnly == "/newlog" )
		{
			m_sTimeStamp  = QDateTime::currentDateTime().toString("yyyyMMdd-hhmmss");
			
			s  = tr("New log timestamp: ");
			s += m_sTimeStamp;
			
			if ( g_pConfig->GetLogChatOption(elcoAPPENDDATE) == false )
			{
				s += "\n";
				s += tr("Warning: log timestamp not enabled.");
			}
			
			AddStatus( s, true );
			res = true;
		}
		else if ( g_pConfig->GetSendUnknownCommandsAsChat() == false )
		{
			bool isAllowed = false;
			
			QStringList allowed = QStringList::split( ",", g_pConfig->GetAllowedUnknownCommands() );
			
			for ( unsigned int i = 0; i < allowed.count(); i++ )
			{
				// e.g. allow "/me" and "/me waves" but not "/mediqjdoh"
				QString tail = cmd.mid(allowed[i].length(),1);
				if ( cmd.startsWith(allowed[i]) &&
				     ((tail == " ") || (tail.isEmpty())) )
				{
					isAllowed = true;
					break;
				}
			}
			
			if ( !isAllowed )
			{
				AddStatus( cmd );
				AddStatus( tr("Unknown command, try /dchelp") );
				res = true;
			}
		}
	}
	else
	{
		/* A custom chat command was replaced with a message, send it */
		AddStatus( TextEdit_CHATINPUT->text(), true );
		SendChat( cmd );
		res = true;
	}
	
	return res;
}

/** */
bool DCChat::CheckForData( QString message )
{
	bool res = false;

	// now we check for special private messages
	if ( m_bPrivateChat )
	{
		// remote send a photo
		if ( message.startsWith("<photo data=\"") && message.endsWith("\">") )
		{
			CString s;
			CByteArray ba;

			//printf("CheckForData message='%s'\n",message.ascii());

			s = message.mid(13,message.length()-15).ascii();
			
			//printf("CheckForData s='%s'\n",s.Data());

			if ( CBase64::Decode( &ba, &s ) > 0 )
			{
				g_pUsersList->AddFriendPhoto(m_sNick,&ba);
				AddStatus(tr("Photo received."), true );

				res = true;
			}
		}
	}

	return res;
}

/** */
void DCChat::NickCompletion()
{
	// shameles stolen from ksirc
	int start, end;
	int p,i;
	QString s;
	TextEdit_CHATINPUT->getCursorPosition(&p,&i);

	if ( m_nTabPressed > 0 )
	{
		s     = m_sTabSaved;
		start = m_nTabStart;
		end   = m_nTabEnd;
	}
	else
	{
		s = TextEdit_CHATINPUT->text(p);
		
		/* remove trailing tab character */
		s = s.left( s.length() - 1 );
		
		m_sTabSaved = s;
		if ( i == 0 )
			end = 0;
		else
			end = i - 1;
		start = s.findRev(QRegExp("\\s"), end);
		m_nTabStart = start;
		m_nTabEnd = end;
	}

	if ( s.isEmpty() && (m_sTabNick.isEmpty() == false) )
	{
		// use the saved nick
		QString line = m_sTabNick + ": ";
		TextEdit_CHATINPUT->removeParagraph(p);
		TextEdit_CHATINPUT->append(line);
		TextEdit_CHATINPUT->setCursorPosition( p,line.length() );
	}
	else
	{
		int cpos = 1;

		if ( start == -1 )
		{
			m_sTabNick = m_pClient->findNick( s.mid(0, end+1), m_nTabPressed );

			if ( m_sTabNick.isNull() )
			{
				m_nTabPressed = 0;
				m_sTabNick = m_pClient->findNick( s.mid(0, end+1), m_nTabPressed );
			}

			s.replace(0, end + 1, m_sTabNick + ": " );
			cpos += 2;
		}
		else
		{
			m_sTabNick = m_pClient->findNick( s.mid(start + 1, end - start), m_nTabPressed );

			if ( m_sTabNick.isNull() )
			{
				m_nTabPressed = 0;
				m_sTabNick = m_pClient->findNick( s.mid(start + 1, end - start), m_nTabPressed );
			}

			s.replace(start + 1, end - start, m_sTabNick);
		}

		if ( m_sTabNick.isEmpty() == false )
		{
			TextEdit_CHATINPUT->removeParagraph(p);
			TextEdit_CHATINPUT->append(s);

			TextEdit_CHATINPUT->setCursorPosition(p,start + m_sTabNick.length() + cpos);

			m_nTabPressed++;
		}
	}
}

/** insert double clicked word into chat input box */
void DCChat::slotDoubleClickedChatOutput( int, int )
{
	// double clicking already selects the text
	QString selected = m_pTextEdit_CHATOUTPUT->selectedText();
	// printf( "selected=%s\n", selected.ascii() );
	
	// convert emoticon image tags to plain text
	QRegExp imgtag;
	imgtag.setMinimal(true);
	imgtag.setPattern("<img align=center alt=(.*) source=emoticon.* title=.* >");
	
	selected.replace( imgtag, "\\1" );
	
	// do noting on no selection
	if ( selected.isEmpty() )
	{
		return;
	}
	
	// remove all html tags
	QRegExp re;
	re.setMinimal(true);
	re.setPattern("<.*>");
	selected.remove( re );
	
	// "NICK> " gets selected, trim "> " off nick
	if ( selected.right(5) == "&gt; " )
	{
		selected = selected.left( selected.length() - 5 );
	}
	
	// trim white space off ends
	selected = selected.simplifyWhiteSpace();
	
	// add space on end
	selected += " ";
	
	if ( TextEdit_CHATINPUT->text().isEmpty() )
	{
		TextEdit_CHATINPUT->insert( selected );
	}
	else
	{
		if ( TextEdit_CHATINPUT->text().right(1) != " " )
		{
			selected.prepend(" ");
		}
		
		// not using TextEdit_CHATINPUT->append(), that would add a new line
		TextEdit_CHATINPUT->insert( selected );
	}
}

/** */
void DCChat::slotBGColorChanged()
{
	if ( g_pConfig->GetChatBackgroundColorEnabled() )
	{
		QBrush paper = m_pTextEdit_CHATOUTPUT->paper();
		paper.setColor(g_pConfig->GetChatBackgroundColor());
		m_pTextEdit_CHATOUTPUT->setPaper(paper);
		
		paper = TextEdit_CHATINPUT->paper();
		paper.setColor(g_pConfig->GetChatBackgroundColor());
		TextEdit_CHATINPUT->setPaper(paper);
	}
	else
	{
		QBrush paper = QTextEdit().paper();
		m_pTextEdit_CHATOUTPUT->setPaper(paper);
		TextEdit_CHATINPUT->setPaper(paper);
	}
}
