/***************************************************************************
                          dcclient.cpp  -  description
                             -------------------
    begin                : Thu Oct 4 2001
    copyright            : (C) 2001-2005 by Mathias Küster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "dcclient.h"

#include <stdlib.h>

#include <qapplication.h>
#include <qtextedit.h>
#include <qlineedit.h>
#include <qlistview.h>
#include <qlistbox.h>
#include <qcheckbox.h>
#include <qregexp.h>
#include <qcombobox.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qfile.h>
#include <qmessagebox.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qtabwidget.h>
#include <qinputdialog.h>
#include <qfiledialog.h>
#include <qclipboard.h>
#include <qlayout.h>
#include <qsplitter.h>
#include <qtoolbutton.h>
#include <qtooltip.h>
#include <qstringlist.h>
#include <qworkspace.h>
#include <qgrid.h>

/* this is QByteArray's header in QT3 */
#include <qcstring.h>

#include "dcevent.h"
#include "dcspy.h"
#include "dctransferview.h"
#include "dcconfig.h"
#include "dcchat.h"
#include "dcmenuhandler.h"
#include "dcwidget.h"
#include "dcuserslist.h"
#include "dcfriendobject.h"
#include "dciconloader.h"

#include "dcconnectionmanager.h"
#include "dcguiutils.h"

#include <dclib/cfilemanager.h>
#include <dclib/dcos.h>

#include "ui/DCDialogForceMove.h"
#include "ui/DCDialogUserCommandLines.h"

/** */
DCClient::DCClient(QWidget* parent, const char *name, int wflags, CString remoteEncoding) : DCDialogClient(parent, name, wflags), CClient(remoteEncoding), m_UserItemDict(1499)
{
	int idx;

	// set default icon
	setIcon( g_pIconLoader->GetPixmap(eiNOTCONNECTED) );

	// set the ColumnWidthMode to manual, we will take care of this task
	for( idx = 0; idx < ListView_USERLIST->columns(); idx++ )
	{
		ListView_USERLIST->setColumnWidthMode( idx, QListView::Manual );
	}

	ListView_USERLIST->setColumnAlignment(5,Qt::AlignRight);

	m_pMessageQueue = new QPtrQueue<CDCMessage>();

	m_UserItemDict.setAutoDelete(true);

	m_bUseTabWidget = g_pConfig->GetShowChatInTab();

	m_nChatFloodCounter = 0;
	
	stripper.setPattern("\\[.*\\]");
	stripper.setMinimal(true);
	
	InitDocument();
}

/** */
DCClient::~DCClient()
{
	m_Timer.stop();

	m_pHubChat->removeEventFilter(this);
	delete m_pHubChat;

	if ( g_pUsersList )
	{
		// force recheck all friend state
		DCFriendObject m_pFriendObject;
		QApplication::postEvent( g_pUsersList, new DC_FriendEvent( &m_pFriendObject ) );
	}

	Disconnect();

	m_Mutex.lock();

	if ( m_pMessageQueue )
	{
		QPtrQueue<CDCMessage> * tmp = m_pMessageQueue;
		m_pMessageQueue = 0;
		
		while ( !tmp->isEmpty() )
		{
			delete tmp->dequeue();
		}
		
		delete tmp;
	}

	m_Mutex.unlock();

	m_ChatMap.clear();

	m_UserItemDict.clear();
	
	for ( uint i = 0; i < hubcommands.count(); i++ )
	{
		delete hubcommands.at(i);
	}
	hubcommands.clear();
}

/** */
void DCClient::InitDocument()
{
	StringMap * map;
	QValueList<int> frames;

	SetCrypt(false);

	// restore settings
	if ( g_pConfig->GetMap("CLIENTVIEW",map) )
	{
		if ( ((*map)["WIDTH"].toInt() > 0) && ((*map)["HEIGHT"].toInt() > 0) )
		{
			if ( !isMinimized() && !isMaximized() )
			{
				setGeometry( x(), y(), (*map)["WIDTH"].toInt(), (*map)["HEIGHT"].toInt() );
			}
		}

		ListView_USERLIST->setSorting( (*map)["SORTCOLUMN"].toInt() );
		ListView_USERLIST->setSortOrder( DCGuiUtils::SortOrderFromName((*map)["SORTORDER"]) );
	}

	// add hubchat widget
	m_pHubChat = new DCChat( TabWidget_CHAT, "HUB-Chat", WDestructiveClose, this, false );
	m_pHubChat->installEventFilter(this);
	m_pHubChat->SetNick(QString(),tr("Hub"));

	TabWidget_CHAT->insertTab( m_pHubChat, tr("Hub") );

	// remove chat list tab
	if ( m_bUseTabWidget )
	{
		TabWidget_CHAT->removePage(Tab_CHATUSERLIST);
		delete Tab_CHATUSERLIST;
		Tab_CHATUSERLIST = 0;
		ListView_CHATUSERLIST = 0;

		TabWidget_CHAT->setCurrentPage(0);
		
		TabWidget_CHAT->setCornerWidget( new QPushButton( QIconSet( g_pIconLoader->GetPixmap(eiFILECLOSE) ), QString(), TabWidget_CHAT ) );
		QToolTip::add( TabWidget_CHAT->cornerWidget(), tr("Close chat tab") );
		TabWidget_CHAT->cornerWidget()->setEnabled(false);
	}
	else
	{
		// set width mode
		ListView_CHATUSERLIST->setColumnWidthMode( 0, QListView::Maximum );
		// hide the column header
		ListView_CHATUSERLIST->header()->hide();

		connect( ListView_CHATUSERLIST,SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(slotDoubleClickedChatUserList(QListViewItem*)) );

		TabWidget_CHAT->setTabLabel( TabWidget_CHAT->page(0), tr("Chat List")+" ("+QString().setNum(ListView_CHATUSERLIST->childCount())+")" );

		TabWidget_CHAT->setCurrentPage(1);
	}

	if ( g_pConfig->GetUserListRightAlignment() )
	{
		Splitter_CLIENT->moveToFirst(Frame_CHATLIST);

		Frame_CHATLIST->adjustSize();
		Frame_USERLIST->adjustSize();
	}

	if ( g_pConfig->GetMap("USERLISTVIEW",map) )
	{
		if ( (*map)[QString().setNum(eclcNICK)].toInt() > 0 )
			g_pConfig->SetClientColumnWidth( eclcNICK, (*map)[QString().setNum(eclcNICK)].toInt() );
		if ( (*map)[QString().setNum(eclcCOMMENT)].toInt() > 0 )
			g_pConfig->SetClientColumnWidth( eclcCOMMENT, (*map)[QString().setNum(eclcCOMMENT)].toInt() );
		if ( (*map)[QString().setNum(eclcTAG)].toInt() > 0 )
			g_pConfig->SetClientColumnWidth( eclcTAG, (*map)[QString().setNum(eclcTAG)].toInt() );
		if ( (*map)[QString().setNum(eclcEMAIL)].toInt() > 0 )
			g_pConfig->SetClientColumnWidth( eclcEMAIL, (*map)[QString().setNum(eclcEMAIL)].toInt() );
		if ( (*map)[QString().setNum(eclcSPEED)].toInt() > 0 )
			g_pConfig->SetClientColumnWidth( eclcSPEED, (*map)[QString().setNum(eclcSPEED)].toInt() );
		if ( (*map)[QString().setNum(eclcSHARE)].toInt() > 0 )
			g_pConfig->SetClientColumnWidth( eclcSHARE, (*map)[QString().setNum(eclcSHARE)].toInt() );
		if ( (*map)[QString().setNum(eclcIP)].toInt() > 0 )
			g_pConfig->SetClientColumnWidth( eclcIP, (*map)[QString().setNum(eclcIP)].toInt() );
		if ( (*map)[QString().setNum(eclcLOCKPK)].toInt() > 0 )
			g_pConfig->SetClientColumnWidth( eclcLOCKPK, (*map)[QString().setNum(eclcLOCKPK)].toInt() );
		if ( (*map)[QString().setNum(eclcSUPPORTS)].toInt() > 0 )
			g_pConfig->SetClientColumnWidth( eclcSUPPORTS, (*map)[QString().setNum(eclcSUPPORTS)].toInt() );
		if ( (*map)["WIDTH"].toInt() > 0 )
		{
			frames = Splitter_CLIENT->sizes();
			if (frames.size() == 2)
			{
				if ( g_pConfig->GetUserListRightAlignment() )
				{
					frames[0] = (width() - Splitter_CLIENT->handleWidth() - 1) - (*map)["WIDTH"].toInt();
					frames[1] = (*map)["WIDTH"].toInt() + 1;
				} else {
					frames[0] = (*map)["WIDTH"].toInt() + 1;
					frames[1] = (width() - Splitter_CLIENT->handleWidth() - 1) - (*map)["WIDTH"].toInt();
				}
				Splitter_CLIENT->setSizes(frames);
			}
		}
	}

	// set viewable columns
	ResizeListViewColumn();

	// load column order from config
	// it does not matter that this code used to use the
	// horribly broken eClientColumn enum for the column numbers
	// because it never did any conversions between the broken enum
	// and the actual column numbers
	if ( g_pConfig->GetMap("USERLISTCOLUMNORDER",map) )
	{
		for ( int i = 0; i < ListView_USERLIST->columns(); ++i )
		{
			if ( map->contains(QString::number(i)) )
			{
				ListView_USERLIST->header()->moveSection((*map)[QString::number(i)].toInt(),i);
			}
		}
	}
	
	connect( ListView_USERLIST,SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(slotDoubleClickedUserList(QListViewItem*)) );
	connect( ListView_USERLIST,SIGNAL(contextMenuRequested( QListViewItem *, const QPoint &, int )), this, SLOT(slotRightButtonClickedUserList(QListViewItem*, const QPoint &, int )) );
	connect( ListView_USERLIST->header(),SIGNAL(sizeChange(int, int, int)), this, SLOT(slotSaveColumnWidth(int, int, int)) );
	connect( TabWidget_CHAT,SIGNAL(currentChanged(QWidget*)), this, SLOT(slotTabWidgetChatCurrentChange(QWidget*)) );
	connect( ToolButton_CONNECT,SIGNAL(clicked()), this, SLOT(slotHubConnect()) );
	connect( ToolButton_SSL,SIGNAL(clicked()), this, SLOT(slotSSLInfo()) );
	
	if (m_bUseTabWidget)
	{
		connect( TabWidget_CHAT->cornerWidget(), SIGNAL(clicked()), this, SLOT(slotTabCornerCloseWidgetClicked()) );
	}

	// set connect disconnect button pixmap
	SetConnection(false);

	TabWidget_CHAT->installEventFilter(this);
	ListView_USERLIST->installEventFilter(this);
	
	connect( &m_Timer, SIGNAL(timeout()), this, SLOT(timerDone()) );
	
	m_Timer.start( 500, true );
}

/** */
void DCClient::DeInitDocument()
{
	StringMap * map;

	// save client view settings
	g_pConfig->GetMap("CLIENTVIEW",map);

	(*map)["WIDTH"]      = QString().setNum(width());
	(*map)["HEIGHT"]     = QString().setNum(height());
	
#if QT_VERSION >= 0x030100
	(*map)["SORTCOLUMN"] = QString().setNum(ListView_USERLIST->sortColumn());
	(*map)["SORTORDER"]  = DCGuiUtils::SortOrderName( ListView_USERLIST->sortOrder() );
#endif

	g_pConfig->GetMap("USERLISTVIEW",map);
	(*map)[QString().setNum(eclcNICK)] = QString().setNum(g_pConfig->GetClientColumnWidth( eclcNICK ));
	(*map)[QString().setNum(eclcCOMMENT)] = QString().setNum(g_pConfig->GetClientColumnWidth( eclcCOMMENT ));
	(*map)[QString().setNum(eclcTAG)] = QString().setNum(g_pConfig->GetClientColumnWidth( eclcTAG ));
	(*map)[QString().setNum(eclcEMAIL)] = QString().setNum(g_pConfig->GetClientColumnWidth( eclcEMAIL ));
	(*map)[QString().setNum(eclcSPEED)] = QString().setNum(g_pConfig->GetClientColumnWidth( eclcSPEED ));
	(*map)[QString().setNum(eclcSHARE)] = QString().setNum(g_pConfig->GetClientColumnWidth( eclcSHARE ));
	(*map)[QString().setNum(eclcIP)] = QString().setNum(g_pConfig->GetClientColumnWidth( eclcIP ));
	(*map)[QString().setNum(eclcLOCKPK)] = QString().setNum(g_pConfig->GetClientColumnWidth( eclcLOCKPK ));
	(*map)[QString().setNum(eclcSUPPORTS)] = QString().setNum(g_pConfig->GetClientColumnWidth( eclcSUPPORTS ));
	
	/* FIXME fix this properly! +1 ??? */
	(*map)["WIDTH"] = QString().setNum(Frame_USERLIST->width()+1);

	g_pConfig->GetMap("USERLISTCOLUMNORDER",map);
	
	/*
	 * again because before it didn't convert between enum and column number
	 * it doesn't matter, you could start at "columns() - 1" and decrement "i" if you wanted
	 */
	for ( int i = 0; i < ListView_USERLIST->columns(); ++i )
	{
		(*map)[QString::number(i)] = QString::number(ListView_USERLIST->header()->mapToLogical(i));
	}
	
	if ( !m_ChatMap.isEmpty() )
	{
		QMap<QString, DCChat*>::const_iterator it = m_ChatMap.constBegin();
		while( it != m_ChatMap.constEnd() )
		{
			it.data()->removeEventFilter(this);
			it.data()->close();
			++it;
		}
	}
}

/** */
void DCClient::SetCrypt( bool b )
{
	QToolTip::remove(ToolButton_SSL);
	ToolButton_SSL->setEnabled(b);

	if ( b )
	{
		QToolTip::add(ToolButton_SSL, tr("Line is encrypted."));
		ToolButton_SSL->setPixmap( g_pIconLoader->GetPixmap(eiSSL_YES));
	}
	else
	{
		QToolTip::add(ToolButton_SSL, tr("Line is not encrypted."));
		ToolButton_SSL->setPixmap( g_pIconLoader->GetPixmap(eiSSL_NO));
	}
}

/** */
void DCClient::SetConnection( bool b )
{
	QToolTip::remove(ToolButton_CONNECT);

	if ( b )
	{
		QToolTip::add(ToolButton_CONNECT, tr("Disconnect."));
		ToolButton_CONNECT->setPixmap( g_pIconLoader->GetPixmap(eiCONNECT_NO));
	}
	else
	{
		QToolTip::add(ToolButton_CONNECT, tr("Connect."));
		ToolButton_CONNECT->setPixmap( g_pIconLoader->GetPixmap(eiCONNECT_CREATING));
	}
}

/** */
void DCClient::slotSSLInfo()
{
	/* a non blocking info dialog, deleted when closed or when the hub is closed */
	QMessageBox * mb = new QMessageBox(
		QString::fromAscii(GetHubName().Data()),
		QString::fromAscii((GetSSLVersion() + '\n' + GetSSLCipher() + "\n\n" + VerifyPeerCertificate()).Data()),
		QMessageBox::Information,
		QMessageBox::Ok,
		QMessageBox::NoButton,
		QMessageBox::NoButton,
		this,
		"SSLInfo",
		false, /* non modal */
		Qt::WStyle_DialogBorder | Qt::WDestructiveClose
	);
	mb->show();
}

/** event filter */
bool DCClient::eventFilter(QObject* object, QEvent* event)
{
	if((event->type() == QEvent::Close)&&((DCClient*)object!=this))
	{
		QCloseEvent* e=(QCloseEvent*)event;

		if ( m_pHubChat == object )
		{
			e->ignore();
			return true;
		}

		DCChat* pView=(DCChat*)object;

		CloseChat(pView);
	}
	else if ( event->type() == QEvent::KeyPress )
	{
		QKeyEvent * e = (QKeyEvent*)event;

		// reconnect
		if ( (e->state() == AltButton) && (e->key() == Key_R) )
		{
			slotHubConnect();
		}
		
		if ( ListView_USERLIST->hasFocus() )
		{
			// TODO: add userlist shortcuts
		}
	}
	else if ( (event->type() == QEvent::ContextMenu) && (object == TabWidget_CHAT) )
	{
		/*
		 * In QT3 we do not get any events on right or other clicks
		 * on any of the tabs, only on right clicking the empty space
		 * after the tabs.
		 */
		if ( (TabWidget_CHAT->currentPage() != 0) &&
		     (TabWidget_CHAT->currentPage() != Tab_CHATUSERLIST) )
		{
			((DCChat*)TabWidget_CHAT->currentPage())->slotRightButtonClickedChatOutput(QPoint());
		}
	}

	return QWidget::eventFilter( object, event );    // standard event processing
}

/** current tab widget change slot */
void DCClient::slotTabWidgetChatCurrentChange( QWidget * w )
{
	DCChat * chat = (DCChat*)w;
	
	if ( ! TabWidget_CHAT->tabIconSet(w).isNull() )
	{
		TabWidget_CHAT->setTabIconSet( w, QIconSet() );
		
		// reset nick color
		SetNickColor( chat->GetNick(), ListView_USERLIST->colorGroup().text() );
	}
	
	if (m_bUseTabWidget)
	{
		if (TabWidget_CHAT->currentPage() == m_pHubChat)
		{
			TabWidget_CHAT->cornerWidget()->setEnabled(false);
		}
		else
		{
			TabWidget_CHAT->cornerWidget()->setEnabled(true);
		}
	}
}

/** overridden so that the columns are resized on show() */
void DCClient::show()
{
	QWidget::show();

	ResizeListViewColumn();
}

/** resize event handler */
void DCClient::resizeEvent( QResizeEvent * /* event */ )
{
	ResizeListViewColumn();
}

/** close event handler - remove client from connection manager before accepting */
void DCClient::closeEvent( QCloseEvent * event )
{
	g_pConnectionManager->HubClosing( this );
	
	DeInitDocument();
	
	event->accept();
}

/** */
void DCClient::UpdateClientColumn( bool enabled, int index, int width )
{
	if ( enabled == false )
	{
		ListView_USERLIST->setColumnWidth( index, 0 );
		ListView_USERLIST->header()->setResizeEnabled(false, index);
	}
	else
	{
		if ( width < 10 ) { width = 10; } // otherwise newly-enabled columns remain hidden
		ListView_USERLIST->header()->setResizeEnabled(true, index);
		ListView_USERLIST->setColumnWidth( index, width );
	}
}

/** resize the ListView columns */
void DCClient::ResizeListViewColumn()
{
	int width,i;

//	printf("%d\n",g_pConfig->GetClientColumnWidth( eclcNICK ));
	if ( g_pConfig->GetClientColumnWidth( eclcNICK ) < 30 )
	{
		i     = 36;
		width = ListView_USERLIST->width();

		if ( ListView_USERLIST->verticalScrollBar()->isVisible() )
			width -= (ListView_USERLIST->verticalScrollBar()->size().width()+4);

		if ( width > 0 )
		{
			if ( g_pConfig->GetClientColumn( eclcCOMMENT ) == false )
				i -= 5;
			if ( g_pConfig->GetClientColumn( eclcTAG ) == false )
				i -= 4;
			if ( g_pConfig->GetClientColumn( eclcSPEED ) == false )
				i -= 2;
			if ( g_pConfig->GetClientColumn( eclcEMAIL ) == false )
				i -= 3;
			if ( g_pConfig->GetClientColumn( eclcSHARE ) == false )
				i -= 4;
			if ( g_pConfig->GetClientColumn( eclcIP ) == false )
				i -= 4;
			if ( g_pConfig->GetClientColumn( eclcLOCKPK ) == false )
				i -= 4;
			if ( g_pConfig->GetClientColumn( eclcSUPPORTS ) == false )
				i -= 4;

			g_pConfig->SetClientColumnWidth( eclcCOMMENT,  (width*5)/i );
			g_pConfig->SetClientColumnWidth( eclcTAG,      (width*4)/i );
			g_pConfig->SetClientColumnWidth( eclcSPEED,    (width*2)/i );
			g_pConfig->SetClientColumnWidth( eclcEMAIL,    (width*3)/i );
			g_pConfig->SetClientColumnWidth( eclcSHARE,    (width*4)/i );
			g_pConfig->SetClientColumnWidth( eclcIP,       (width*4)/i );
			g_pConfig->SetClientColumnWidth( eclcLOCKPK,   (width*4)/i );
			g_pConfig->SetClientColumnWidth( eclcSUPPORTS, (width*4)/i );
			g_pConfig->SetClientColumnWidth( eclcNICK,     (width*6)/i );
		}
	}

	if ( g_pConfig->GetClientColumnWidth( eclcNICK ) > 0 )
	{
		UpdateClientColumn( true,                                       0, g_pConfig->GetClientColumnWidth( eclcNICK ) );
		UpdateClientColumn( g_pConfig->GetClientColumn( eclcCOMMENT ),  1, g_pConfig->GetClientColumnWidth( eclcCOMMENT ) );
		UpdateClientColumn( g_pConfig->GetClientColumn( eclcTAG ),      2, g_pConfig->GetClientColumnWidth( eclcTAG ) );
		UpdateClientColumn( g_pConfig->GetClientColumn( eclcSPEED ),    3, g_pConfig->GetClientColumnWidth( eclcSPEED ) );
		UpdateClientColumn( g_pConfig->GetClientColumn( eclcEMAIL ),    4, g_pConfig->GetClientColumnWidth( eclcEMAIL ) );
		UpdateClientColumn( g_pConfig->GetClientColumn( eclcSHARE ),    5, g_pConfig->GetClientColumnWidth( eclcSHARE ) );
		UpdateClientColumn( g_pConfig->GetClientColumn( eclcIP ),       6, g_pConfig->GetClientColumnWidth( eclcIP ) );
		UpdateClientColumn( g_pConfig->GetClientColumn( eclcLOCKPK ),   7, g_pConfig->GetClientColumnWidth( eclcLOCKPK ) );
		UpdateClientColumn( g_pConfig->GetClientColumn( eclcSUPPORTS ), 8, g_pConfig->GetClientColumnWidth( eclcSUPPORTS ) );
	}
}

/** */
void DCClient::UpdateCaption()
{
	QString s;

	switch(GetConnectionState())
	{
		case estCONNECTED:
				s = "[+]";
				break;
		default:
				s = "[-]";
				break;
	}

	s += " ";
	s += QString::fromAscii(GetHubName().Data());

	if ( GetHubTopic().NotEmpty() )
	{
		s += " (";
		s += QString::fromAscii(GetHubTopic().Data());
		s += ")";
	}
	
	if (s.length() > 50)
	{
		s = s.left(50) + "...";
	}

	setCaption(s);
	g_pConnectionManager->CaptionChanged( this );
}

/** */
void DCClient::UpdateStatusBar()
{
	TextLabel_USERCOUNT->setText( QString().setNum(ListView_USERLIST->childCount()) );
	TextLabel_SHARESIZE->setText( DCGuiUtils::GetSizeString(UserList()->ShareSize()) );
}

/** */
void DCClient::timerDone()
{
	CDCMessage *DCMsg;
	int i;

	for(i=0;i<200;i++)
	{
		if ( m_Mutex.tryLock() == false )
		{
			break;
		}

		if ( m_pMessageQueue )
		{
			/* returns 0 if queue is empty */
			DCMsg = m_pMessageQueue->dequeue();
		}
		else
		{
			DCMsg = 0;
		}

		m_Mutex.unlock();

		if ( DCMsg == 0 )
		{
			break;
		}

		switch ( DCMsg->m_eType )
		{
			case DC_MESSAGE_CONNECTION_STATE:
			{
				CMessageConnectionState *msg = (CMessageConnectionState*) DCMsg;

				if ( msg->m_eState == estCONNECTED )
				{
					m_nChatFloodCounter = 0;
					m_bUserPassword     = false;

					UpdateCaption();
					setIcon( g_pIconLoader->GetPixmap(eiCONNECTED) );
					// repaint new icon in minimized mode
					if ( isMinimized() )
						showMinimized();

					m_pHubChat->AddStatus(tr("Connected"));

					// init userlist
					DC_NickList(0);

					g_pConfig->PlaySound(eusCONNECT);

					SetConnection(true);

					SetCrypt(false);
				}
				if ( msg->m_eState == estSSLCONNECTED )
				{
					SetCrypt(true);
				}
				else if ( msg->m_eState == estCONNECTIONTIMEOUT )
				{
					m_pHubChat->AddStatus(tr("Connection timeout"));

					SetConnection(false);

					SetCrypt(false);
				}
				else if ( msg->m_eState == estSOCKETERROR )
				{
					QString s;
					s  = tr("Error: '");
					s += QString::fromAscii(msg->m_sMessage.Data());
					s += "'";
					m_pHubChat->AddStatus(s);

					SetConnection(false);

					SetCrypt(false);
				}
				else if ( msg->m_eState == estDISCONNECTED )
				{
					m_nChatFloodCounter = 0;

					UpdateCaption();
					setIcon( g_pIconLoader->GetPixmap(eiNOTCONNECTED) );
					// repaint new icon in minimized mode
					if ( isMinimized() )
						showMinimized();

					m_pHubChat->AddStatus(tr("Disconnected"));
					g_pConfig->PlaySound(eusDISCONNECT);

					if ( g_pUsersList )
					{
						// force recheck all friend state
						DCFriendObject m_pFriendObject;
						QApplication::postEvent( g_pUsersList, new DC_FriendEvent( &m_pFriendObject ) );
					}

					// reset all chats
					QMap<QString, DCChat*>::const_iterator it = m_ChatMap.constBegin();
					
					while ( it != m_ChatMap.constEnd() )
					{
						it.data()->SetCrypt(esecsNONE);
						it.data()->AddStatus(tr("We left the hub."));
						++it;
        				}

					SetConnection(false);

					SetCrypt(false);
				}

				break;
			}

			case DC_MESSAGE_VALIDATEDENIDE:
			{
				m_pHubChat->AddStatus(tr("Your Nick is already in use."));
				Disconnect();
				break;
			}

			case DC_MESSAGE_NICKLIST:
			{
				DC_NickList( (CMessageNickList *) DCMsg );
				break;
			}

			case DC_MESSAGE_OPLIST:
			{
				DC_OpList( (CMessageOpList *) DCMsg );
				break;
			}

			case DC_MESSAGE_REVCONNECTTOME:
			{
				if ( GetMode() == ecmPASSIVE )
				{
					if ( g_pConfig->GetSendMessageOnActiveModeRequest() )
					{
						CMessageRevConnectToMe * msg = (CMessageRevConnectToMe*) DCMsg;
						SendPrivateMessage( GetNick(), msg->m_sDstNick, tr("Sorry, you will not be able to download from me, because we are both in passive mode. (automated message)").ascii() );
					}
				}

				break;
			}

			case DC_MESSAGE_CHAT:
			{
				CMessageChat * msg = (CMessageChat *) DCMsg;
				
				if ( m_ChatFloodMessage.m_sMessage == msg->m_sMessage && m_ChatFloodMessage.m_sNick == msg->m_sNick )
				{
					// increment flood counter
					m_nChatFloodCounter++;
				}
				else
				{
					// update message and reset counter
					m_ChatFloodMessage.m_sMessage = msg->m_sMessage;
					m_ChatFloodMessage.m_sNick    = msg->m_sNick;
					m_nChatFloodCounter = 0;
				}

				// if flood detect enabled ?
				if ( g_pConfig->GetFloodCount() > 0 )
				{
					if ( m_nChatFloodCounter >= g_pConfig->GetFloodCount() )
					{
						// only one flood message
						if ( m_nChatFloodCounter == g_pConfig->GetFloodCount() )
						{
							m_pHubChat->AddStatus(tr("Flood Detected")+" ("+QString().setNum(m_nChatFloodCounter)+")");

							// now the admin kick the user if enabled
							if ( g_pConfig->GetFloodOpKick() && (!UserList()->IsAdmin(msg->m_sNick)) && (UserList()->IsAdmin(GetNick())) )
							{
								//if OP kick the user for Flooding
								CString kickmessage=g_pConfig->GetFloodOpKickMessage();
								SendPrivateMessage( GetNick(), msg->m_sNick, ("You are being kicked because: " + kickmessage));
								SendChat( GetNick(), GetNick() + " is kicking " + msg->m_sNick + " because: " + kickmessage );
								SendKick(msg->m_sNick);
							}
						}

						// don't show the flood messages
						msg = 0;
					}
				}

				if ( msg )
				{
					DC_Chat( msg );
				}

				break;
			}

			case DC_MESSAGE_MYINFO:
			{
				DC_MyInfo( (CMessageMyInfo *) DCMsg );
				break;
			}

			case DC_MESSAGE_QUIT:
			{
				DC_Quit( (CMessageQuit *) DCMsg );
				break;
			}

			case DC_MESSAGE_SEARCH_FILE:
			{
				if ( g_pSpy )
					g_pSpy->DC_Search( (CMessageSearchFile *) DCMsg );
				break;
			}

			case DC_MESSAGE_HELLO:
			{
				DC_Hello( (CMessageHello *) DCMsg );

				break;
			}

			case DC_MESSAGE_FORCEMOVE:
			{
				CMessageForceMove * msg = (CMessageForceMove*) DCMsg;

				if ( g_pConfig->GetForceMoveEnabled() )
				{
					UpdateCaption();

					m_pHubChat->AddStatus( tr("Redirect to ")+ QString::fromAscii(msg->m_sHost.Data()) + ":" + QString().setNum(msg->m_nPort) );
				}
				else
				{
					m_pHubChat->AddStatus( tr("Redirect disabled ")+ QString::fromAscii(msg->m_sHost.Data()) + ":" + QString().setNum(msg->m_nPort) );
				}

				break;
			}

			case DC_MESSAGE_HUBNAME:
			{
				CMessageHubName * msg = (CMessageHubName*) DCMsg;
				QString qhubname = QString::fromAscii( msg->m_sHubName.Data() );

				if ( (msg->m_sOldHubName.NotEmpty()) && (msg->m_sOldHubName != msg->m_sHubName) )
				{
					m_pHubChat->AddStatus( tr("Hubname change:")+QString(" '")+QString::fromAscii(msg->m_sOldHubName.Data())+QString("' -> '")+qhubname+QString("'") );
				}

				UpdateCaption();

				// update caption from the main chat
				m_pHubChat->SetNick(QString(),qhubname);

				// update caption from all private chats
				if ( !m_ChatMap.isEmpty() )
				{
					QMap<QString, DCChat*>::const_iterator it = m_ChatMap.constBegin();

					while( it != m_ChatMap.constEnd() )
					{
						it.data()->SetNick( it.data()->GetNick(), qhubname );
						++it;
					}
				}

				break;
			}

			case DC_MESSAGE_HUB_TOPIC:
			{
				UpdateCaption();
				break;
			}

			case DC_MESSAGE_PRIVATECHAT:
			{
				DC_PrivateChat( (CMessagePrivateChat*) DCMsg );

				break;
			}

			case DC_MESSAGE_GETPASS:
			{
				bool ok = false;
				DCConfigHubProfile pConfigHubProfile;

				if ( m_bUserPassword == false )
				{
					// get password from profile
					if ( g_pConfig->GetBookmarkHubProfile( GetHubName(), GetHost(), &pConfigHubProfile ) )
					{
						if ( pConfigHubProfile.m_sPassword.NotEmpty() )
						{
							m_pHubChat->AddStatus(tr("Use password from profile"));
							SendPass(pConfigHubProfile.m_sPassword);
							SetUsedPassword(true);
							m_bUserPassword = true;
							break;
						}
						else
						{
							m_pHubChat->AddStatus(tr("Password from profile is empty"));
						}
					}
					else
					{
						m_pHubChat->AddStatus(tr("No profile found for this hub"));
					}
				}

				QString text = QInputDialog::getText(
					tr( "Password - ")+GetHubName().Data(),
					tr( "Please enter your password" ),
					QLineEdit::Password, QString::null, &ok, this );

				if ( ok && !text.isEmpty() )
				{
					SendPass(text.ascii());
					SetUsedPassword(true);
				}
				else
				{
					Disconnect();
				}

				break;
			}

			case DC_MESSAGE_BADPASS:
			{
				m_pHubChat->AddStatus(tr("Wrong password"));
				SetUsedPassword(false);
				break;
			}

			case DC_MESSAGE_USERIP:
			{
				CMessageUserIP * msg = (CMessageUserIP*) DCMsg;
				
				std::list<CString>::const_iterator nick_it = msg->m_lNicks.begin();
				std::list<CString>::const_iterator ip_it = msg->m_lIPs.begin();
				QString qnick, qip;
				DCUserItem * pui = 0;
				
				while ( (nick_it != msg->m_lNicks.end()) && (ip_it != msg->m_lIPs.end()) )
				{
					qnick = QString::fromAscii( nick_it->Data() );
					qip = QString::fromAscii( ip_it->Data() );
					
					pui = m_UserItemDict.find( qnick );
					
					if ( pui )
					{
						// fill in the IP address column for the given nick
						pui->pItem->setText( 6, qip );
						pui->MyInfo.m_sTransferHost = *ip_it;
					}
					
					if ( *nick_it == GetNick() )
					{
						m_pHubChat->AddStatus(tr("Hub gave us our IP: ") + qip);
						
						if ( g_pConfig->GetUserIP2Enabled() == false )
						{
							m_pHubChat->AddStatus(tr("Enable") + " \"" + tr("Use IP address from the hub") + "\" " + tr("in Options -> Connection -> Mode to use this value."));
						}
					}
					
					++nick_it;
					++ip_it;
				}
				
				break;
			}

			case DC_MESSAGE_USER_COMMAND:
			{
				DC_UserCommand( (CMessageUserCommand*)DCMsg );
				
				break;
			}

			case DC_MESSAGE_LOG:
			{
				m_pHubChat->AddStatus(QString::fromAscii(((CMessageLog*)DCMsg)->sMessage.Data()));
			}

			default:
			{
				break;
			}
		}

		if ( DCMsg )
			delete DCMsg;
	}

	m_Timer.start( 500, true );
}

/** */
int DCClient::DC_CallBack( CDCMessage * DCMessage )
{
	int err = -1;

	m_Mutex.lock();

	if ( DCMessage && m_pMessageQueue )
	{
		m_pMessageQueue->enqueue(DCMessage);
		err = 0;
	}

	m_Mutex.unlock();

	return err;
}

/** chat */
void DCClient::DC_Chat( CMessageChat * MessageChat )
{
	QString qnick = QString::fromAscii(MessageChat->m_sNick.Data());
	QString qmessage = QString::fromAscii(MessageChat->m_sMessage.Data());
		
	// check for ignored nick
	if ( g_pUsersList->ignoreNick( qnick ) )
	{
		//printf("Ignoring \"%s\" from \"%s\"\n", MessageChat->m_sMessage.Data(), MessageChat->m_sNick.Data());
		return;
	}
	
	m_pHubChat->AddMessage(qnick,qmessage,true);

	if ( !m_pHubChat->isVisible() )
	{
		TabWidget_CHAT->setTabIconSet( m_pHubChat, QIconSet( g_pIconLoader->GetPixmap(eiMESSAGE) ) );
	}
	
	// send chat event
	g_pConnectionManager->HubEvent(this);
	
	// check for auto responses
	if ( g_pConfig->GetAutoResponderEnabled() )
	{
		doAutoResponse( qnick, qmessage, m_pHubChat );
	}
}

/** myinfo */
void DCClient::DC_MyInfo( CMessageMyInfo * MessageMyInfo )
{
	QPixmap pm;
	
	if ( MessageMyInfo->m_sNick.IsEmpty() )
	{
		m_pHubChat->AddStatus(tr("Cannot display user with empty nick."));
		return;
	}
	
	QString qnick = QString::fromAscii(MessageMyInfo->m_sNick.Data());

	DCUserItem * pui = m_UserItemDict.find( qnick );

	if ( pui == 0 )
	{
		pui = new DCUserItem();
		m_UserItemDict.insert( qnick, pui );

#ifndef WIN32
		pm = *g_pConfig->GetUserIcon(MessageMyInfo);
		pui->pItem = new DC_UserListItem( ListView_USERLIST, qnick, pm );
#else
		pui->pItem = new DC_UserListItem( ListView_USERLIST, qnick );
#endif
		pui->pItem->m_sNickLC = qnick.lower();
		pui->pItem->m_sNickStripped = qnick.lower().remove( stripper );
		
		// user not in the list, add it
		addUser( qnick );
	}
	else
	{
		if ( (pui->MyInfo.m_eAwayMode != MessageMyInfo->m_eAwayMode) ||
		     (pui->MyInfo.m_eUserSpeed != MessageMyInfo->m_eUserSpeed) ||
		     (pui->MyInfo.m_bOperator != MessageMyInfo->m_bOperator) ||
		     (pui->MyInfo.m_eClientMode != MessageMyInfo->m_eClientMode) ||
		     (pui->MyInfo.m_eClientVersion != MessageMyInfo->m_eClientVersion) ||
		     (pui->MyInfo.m_bFireballFlag != MessageMyInfo->m_bFireballFlag) ||
		     (pui->MyInfo.m_bServerFlag != MessageMyInfo->m_bServerFlag) ||
		     (pui->MyInfo.m_bTLSFlag != MessageMyInfo->m_bTLSFlag) )
		{
#ifndef WIN32
			pm = *g_pConfig->GetUserIcon(MessageMyInfo);
			pui->pItem->setPixmap(0,pm);
#endif
		}
	}

	pui->MyInfo = *MessageMyInfo;

	QString qcomment = QString::fromAscii(MessageMyInfo->m_sComment.Data());
	pui->pItem->setText(1,qcomment);
	pui->pItem->setText(2,MessageMyInfo->m_sVerComment.Data());
	pui->pItem->setText(3,MessageMyInfo->m_sUserSpeed.Data());
	pui->pItem->setText(4,QString::fromAscii(MessageMyInfo->m_sEMail.Data()));

	pui->pItem->myvalue = MessageMyInfo->m_nShared;
	pui->pItem->mycol   = 5;

	pui->pItem->setText(5,DCGuiUtils::GetSizeString(MessageMyInfo->m_nShared));

	pui->pItem->setText(6,MessageMyInfo->m_sTransferHost.Data());
	pui->pItem->setText(7,MessageMyInfo->m_MessageLock.m_sData.Data());
	pui->pItem->setText(8,MessageMyInfo->m_MessageSupports.m_sContent.Data());
	
	// send friend event
	if ( g_pUsersList )
	{
		DCFriendObject m_pFriendObject;

		m_pFriendObject.m_sName        = qnick;
		m_pFriendObject.m_sDescription = qcomment;
		m_pFriendObject.m_eAwayMode    = MessageMyInfo->m_eAwayMode;
		m_pFriendObject.m_sHubName     = QString::fromAscii(GetHubName().Data());
		m_pFriendObject.m_sHubHost     = QString::fromAscii(GetHost().Data());
		QApplication::postEvent( g_pUsersList, new DC_FriendEvent( &m_pFriendObject ) );
	}
	
	UpdateStatusBar();
}

/** hello */
void DCClient::DC_Hello( CMessageHello * MessageHello )
{
	if ( MessageHello->m_sNick.IsEmpty() )
	{
		m_pHubChat->AddStatus(tr("Cannot display user with empty nick."));
		return;
	}
	
	QString qnick = QString::fromAscii(MessageHello->m_sNick.Data());
	DCUserItem * pui = m_UserItemDict.find( qnick );
	QPixmap pm;

	if ( pui == 0 )
	{
		pui = new DCUserItem();
		m_UserItemDict.insert( qnick, pui );

#ifndef WIN32
		pm = *g_pConfig->GetUserIcon(&pui->MyInfo);
		pui->pItem = new DC_UserListItem( ListView_USERLIST, qnick, pm );
#else
		pui->pItem = new DC_UserListItem( ListView_USERLIST, qnick );
#endif
		pui->pItem->m_sNickLC = qnick.lower();
		pui->pItem->m_sNickStripped = qnick.lower().remove( stripper );
		
		pui->MyInfo.m_sNick = MessageHello->m_sNick;
		pui->MyInfo.m_eAwayMode = euamNORMAL;
		
		addUser( qnick );
	}

	pui->pItem->mycol = 5;

	UpdateStatusBar();
}

/** quit */
void DCClient::DC_Quit( CMessageQuit * MessageQuit )
{
	if ( MessageQuit->m_sNick.IsEmpty() )
	{
		return;
	}
	
	QString qnick = QString::fromAscii(MessageQuit->m_sNick.Data());
	DCUserItem * pui = m_UserItemDict.find( qnick );

	if ( pui != 0 )
	{
		delete pui->pItem;
		m_UserItemDict.remove( qnick );
	}

	if ( m_ChatMap.contains(qnick) )
	{
		DCChat * chat = m_ChatMap[qnick];
		chat->SetCrypt(esecsNONE);
		chat->AddStatus(tr("User left the hub."));
        }

	// send friend event
	if ( g_pUsersList )
	{
		DCFriendObject m_pFriendObject;

		m_pFriendObject.m_sName     = qnick;
		m_pFriendObject.m_eAwayMode = euamOFFLINE;
		m_pFriendObject.m_sHubName  = QString::fromAscii(GetHubName().Data());
		m_pFriendObject.m_sHubHost  = QString::fromAscii(GetHost().Data());
		QApplication::postEvent( g_pUsersList, new DC_FriendEvent( &m_pFriendObject ) );
	}

	if ( g_pConfig->GetChatShowJoinsAndParts() )
	{
		if ( g_pConfig->GetChatShowJoinsAndPartsOnlyFav() )
		{
			if ( g_pUsersList->isNickInList( qnick ) )
			{
				m_pHubChat->AddStatus( tr("Parts: ") + qnick );
			}
		}
		else
		{
			m_pHubChat->AddStatus( tr("Parts: ") + qnick );
		}
	}

	UpdateStatusBar();
}

/** nicklist */
void DCClient::DC_NickList( CMessageNickList * MessageNickList )
{
	CString * Nick = 0;
	QString qnick;
	QPixmap pm;
	DCUserItem * pui;

	ListView_USERLIST->setUpdatesEnabled(false);
	ListView_USERLIST->clear();

	m_UserItemDict.clear();

	if ( MessageNickList )
	{
		while ( (Nick=MessageNickList->m_NickList.Next(Nick)) != 0 )
		{
			if ( Nick->IsEmpty() )
			{
				m_pHubChat->AddStatus(tr("Cannot display user with empty nick."));
				continue;
			}
			
			qnick = QString::fromAscii(Nick->Data());
			pui = m_UserItemDict.find( qnick );
			if ( pui == 0 )
			{
				pui = new DCUserItem();
				m_UserItemDict.insert( qnick, pui );
	
#ifndef WIN32
				pm = *g_pConfig->GetUserIcon(&pui->MyInfo);
				pui->pItem = new DC_UserListItem( ListView_USERLIST, qnick, pm );
#else
				pui->pItem = new DC_UserListItem( ListView_USERLIST, qnick );
#endif
				pui->pItem->m_sNickLC = qnick.lower();
				pui->pItem->m_sNickStripped = qnick.lower().remove( stripper );
				
				pui->pItem->mycol = 5;
				pui->MyInfo.m_sNick = (*Nick);
				pui->MyInfo.m_eAwayMode = euamNORMAL;
			}

			// send friend event
			if ( g_pUsersList )
			{
				DCFriendObject m_pFriendObject;

				m_pFriendObject.m_sName     = qnick;
				m_pFriendObject.m_eAwayMode = euamONLINE;
				m_pFriendObject.m_sHubName  = QString::fromAscii(GetHubName().Data());
				m_pFriendObject.m_sHubHost  = QString::fromAscii(GetHost().Data());
				QApplication::postEvent( g_pUsersList, new DC_FriendEvent( &m_pFriendObject ) );
			}
		}
	}

	ListView_USERLIST->setUpdatesEnabled(true);
	ListView_USERLIST->triggerUpdate();

	UpdateStatusBar();
}

/** oplist */
void DCClient::DC_OpList( CMessageOpList * MessageOpList )
{
	CString * Nick = 0;
	QString qnick;
	QPixmap pm;
	DCUserItem * pui;

	ListView_USERLIST->setUpdatesEnabled(false);

	while ( (Nick=MessageOpList->m_NickList.Next(Nick)) != 0 )
	{
		if ( Nick->IsEmpty() )
		{
			m_pHubChat->AddStatus(tr("Cannot display operator with empty nick."));
			continue;
		}
		
		qnick = QString::fromAscii(Nick->Data());
		pui = m_UserItemDict.find( qnick );
		if ( pui == 0 )
		{
			pui = new DCUserItem();
			m_UserItemDict.insert( qnick, pui );

			// user not in the list, add it
			addUser( qnick );
			pui->MyInfo.m_bOperator  = true;

#ifndef WIN32
			pm = *g_pConfig->GetUserIcon(&pui->MyInfo);
			pui->pItem = new DC_UserListItem( ListView_USERLIST, qnick, pm );
#else
			pui->pItem = new DC_UserListItem( ListView_USERLIST, qnick );
#endif
			pui->pItem->m_sNickLC = qnick.lower();
			pui->pItem->m_sNickStripped = qnick.lower().remove( stripper );
			
			pui->pItem->mycol = 5;
			pui->MyInfo.m_sNick = (*Nick);
		}
		else
		{
			if ( pui->MyInfo.m_bOperator == false )
			{
				pui->MyInfo.m_bOperator  = true;
#ifndef WIN32
				pm = *g_pConfig->GetUserIcon(&pui->MyInfo);
				pui->pItem->setPixmap( 0, pm );
#endif
			}
		}

		
		pui->MyInfo.m_eAwayMode  = euamNORMAL;
		pui->pItem->m_bSortTop = true;
	}

	ListView_USERLIST->setUpdatesEnabled(true);
	ListView_USERLIST->triggerUpdate();

	UpdateStatusBar();
}

void DCClient::DC_PrivateChat( QString nick, QString fromnick, QString message, bool bLocal )
{
	CMessagePrivateChat msg;

	msg.m_sSrcNick      = nick.ascii();
	msg.m_sMultiSrcNick = fromnick.ascii();
	msg.m_sMessage      = message.ascii();

	DC_PrivateChat( &msg, bLocal );
}

/** */
void DCClient::DC_PrivateChat( CMessagePrivateChat * msg, bool bLocal )
{
	QWidget * focuswidget = 0;
	bool bShowChat = true;

	if ( msg->m_sSrcNick.IsEmpty() )
	{
		return;
	}
	
	QString qnick = QString::fromAscii(msg->m_sSrcNick.Data());
	QString qmessage = QString::fromAscii(msg->m_sMessage.Data());
	
	// also check for ignored nick for PMs
	if ( g_pUsersList->ignoreNick( qnick ) )
	{
		//printf("Ignoring PM \"%s\" from \"%s\"\n", msg->m_sMessage.Data(), msg->m_sSrcNick.Data() );
		return;
	}

	DCChat * chat = 0;

	// create a new chat window
	if ( !m_ChatMap.contains(qnick) )
	{
		focuswidget = qApp->focusWidget();

		if ( m_bUseTabWidget )
		{
			chat = new DCChat( TabWidget_CHAT, "chat", WDestructiveClose, this );
		}
		else
		{
			chat = new DCChat( g_pConnectionManager->GetWorkspace(), "chat", WDestructiveClose, this );
			new QListViewItem(ListView_CHATUSERLIST,qnick);
			TabWidget_CHAT->setTabLabel( TabWidget_CHAT->page(0), tr("Chat List")+" ("+QString().setNum(ListView_CHATUSERLIST->childCount())+")" );
		}

		// fix hide if chat maximized
		chat->show();
		chat->hide();

		chat->installEventFilter(this);
		chat->SetNick(qnick,QString::fromAscii(GetHubName().Data()));

		m_ChatMap.insert( qnick, chat );

		if ( (g_pConfig->GetOpenPrivateChatWindow() == false) && (bLocal == false) )
		{
			bShowChat = false;
		}
		// check for max lines and set chat invisible
		else if ( (g_pConfig->GetChatMessageMaxLines() > 0) &&
			  (QString(msg->m_sMessage.Data()).contains("\n") > g_pConfig->GetChatMessageMaxLines()) )
		{
			bShowChat = false;
		}
		
		// check for open chat windows nick exceptions
		DCConfigHubProfile pConfigHubProfile;
		QString nickregexp;
		
		if ( g_pConfig->GetBookmarkHubProfile( GetHubName(), GetHost(), &pConfigHubProfile ) )
		{
			nickregexp = pConfigHubProfile.m_sSuppressedNicks.Data();
		}
		
		if ( nickregexp.isEmpty() )
		{
			nickregexp = g_pConfig->GetSuppressedNicks();
		}
		
		if ( !nickregexp.isEmpty() )
		{
			QRegExp rx(nickregexp);
			rx.setCaseSensitive(false);
			if (rx.search(qnick) != -1)
			{
				bShowChat = (!g_pConfig->GetOpenPrivateChatWindow()) || bLocal;
			}
		}
		
		if ( m_bUseTabWidget )
		{
			TabWidget_CHAT->insertTab( chat, qnick );
		}

		if ( bShowChat || bLocal )
		{
			if ( m_bUseTabWidget )
				TabWidget_CHAT->showPage(chat);
			else
				chat->setEnabled(true);
			chat->show();
		}
		else
		{
			if ( !m_bUseTabWidget )
			{
				chat->setEnabled(false);
				chat->hide();
			}

			// restore focus
			if ( focuswidget )
				focuswidget->setFocus();
		}

		g_pConfig->PlaySound(eusFIRSTRECEIVE);
	}
	else
	{
		chat = m_ChatMap[qnick];
		
		if ( bLocal ) // enable all local (user input)
		{
			if ( chat->isVisible() == false )
			{
				if ( m_bUseTabWidget )
					TabWidget_CHAT->showPage(chat);
				else
					chat->setEnabled(true);
				chat->show();
			}
			SetNickColor( qnick, ListView_USERLIST->colorGroup().text() );
		}
	}

	if ( msg->m_sMessage.NotEmpty() )
	{
		chat->AddMessage( msg );

		if ( chat->isVisible() == false )
		{
			SetNickColor( qnick, Qt::red );

			if ( m_bUseTabWidget )
				TabWidget_CHAT->setTabIconSet( chat, QIconSet( g_pIconLoader->GetPixmap(eiMESSAGE) ) );
		}
		else
		{
			SetNickColor( qnick, ListView_USERLIST->colorGroup().text() );
		}
	}

	if ( (chat->isVisible() == false) &&
	     (g_pConfig->GetSendHidePrivateChatToPublicChat()) )
	{
		QString nick;

		// send private chat to public chat
		if ( (msg->m_sSrcNick != msg->m_sMultiSrcNick) && (msg->m_sMultiSrcNick.NotEmpty()) )
			nick = qnick + "::" + QString::fromAscii(msg->m_sMultiSrcNick.Data());
		else
			nick = qnick;
		m_pHubChat->AddMessage( nick, qmessage, true, true );
	}
	
	// send chat event
	g_pConnectionManager->HubEvent(this);
	
	if ( g_pConfig->GetAutoResponderEnabledForPM() )
	{
		doAutoResponse( qnick, qmessage, chat );
	}	
}

/** */
void DCClient::DC_UserCommand( CMessageUserCommand * msg )
{
	DC_UserMenuCommand * umc = 0;
	
	QString qname = QString::fromAscii(msg->name.Data());
	QString qcommand = QString::fromAscii(msg->command.Data());
	
	if ( msg->type == euctClear )
	{
		if ( msg->context == euccMask ) // clear all
		{
			for ( uint i = 0; i < hubcommands.count(); i++ )
			{
				delete hubcommands.at(i);
			}
			hubcommands.clear();
			return;
		}
		
		for ( uint i = 0; i < hubcommands.count(); i++ )
		{
			umc = hubcommands.at(i);
			
			if ( (msg->context & umc->m_nContext & euccHub) == euccHub )
			{
				umc->m_nContext -= euccHub;
			}
			if ( (msg->context & umc->m_nContext & euccChat) == euccChat )
			{
				umc->m_nContext -= euccChat;
			}
			if ( (msg->context & umc->m_nContext & euccSearch) == euccSearch )
			{
				umc->m_nContext -= euccSearch;
			}
			if ( (msg->context & umc->m_nContext & euccFilelist) == euccFilelist )
			{
				umc->m_nContext -= euccFilelist;
			}
			
			if ( umc->m_nContext == 0 )
			{
				delete umc;
				hubcommands.remove(i);
				i--;
			}
		}
	}
	else if ( msg->type == euctRemove )
	{
		for ( uint i = 0; i < hubcommands.count(); i++ )
		{
			umc = hubcommands.at(i);
			
			if ( umc->m_sName == qname )
			{
				if ( (msg->context & umc->m_nContext & euccHub) == euccHub )
				{
					umc->m_nContext -= euccHub;
				}
				if ( (msg->context & umc->m_nContext & euccChat) == euccChat )
				{
					umc->m_nContext -= euccChat;
				}
				if ( (msg->context & umc->m_nContext & euccSearch) == euccSearch )
				{
					umc->m_nContext -= euccSearch;
				}
				if ( (msg->context & umc->m_nContext & euccFilelist) == euccFilelist )
				{
					umc->m_nContext -= euccFilelist;
				}
				
				if ( umc->m_nContext == 0 )
				{
					delete umc;
					hubcommands.remove(i);
				}

				return; // only remove one command
			}
		}
	}
	else if ( msg->type == euctSeparator ) // separators might not have names
	{
		umc = new DC_UserMenuCommand();
		
		umc->m_nType    = msg->type;
		umc->m_nContext = msg->context;
		umc->m_sName    = qname;
		umc->m_sCommand = qcommand;
		
		hubcommands.append(umc);
	}
	else if ( (msg->type == euctRaw) || (msg->type == euctRawOnce) )
	{
		// update existing command
		for ( uint i = 0; i < hubcommands.count(); i++ )
		{
			umc = hubcommands.at(i);
			
			if ( umc->m_sName == qname )
			{
				umc->m_nType    = msg->type;
				umc->m_nContext = umc->m_nContext | msg->context;
				umc->m_sCommand = qcommand;
				return;
			}
		}
		
		umc = new DC_UserMenuCommand();
		umc->m_nType    = msg->type;
		umc->m_nContext = msg->context;
		umc->m_sName    = qname;
		umc->m_sCommand = qcommand;
		
		hubcommands.append(umc);
	}
	else
	{
		printf("Unknown type %d for command '%s'\n", msg->type, msg->name.Data());
	}
}

/** handling if user join the hub */
void DCClient::addUser( QString nick )
{
	// show joins
	if ( g_pConfig->GetChatShowJoinsAndParts() )
	{
		if ( g_pConfig->GetChatShowJoinsAndPartsOnlyFav() )
		{
			if ( g_pUsersList->isNickInList( nick ) )
			{
				m_pHubChat->AddStatus( tr("Joins: ") + nick );
			}
		}
		else
		{
			m_pHubChat->AddStatus( tr("Joins: ") + nick );
		}
	}

	// show rejoin on private chat
	if ( m_ChatMap.contains(nick) )
	{
		DCChat * chat = m_ChatMap[nick];
		chat->AddStatus(tr("User rejoin the hub."));
	}

	// send event to friendlist
	if ( g_pUsersList )
	{
		DCFriendObject m_pFriendObject;

		m_pFriendObject.m_sName     = nick;
		m_pFriendObject.m_eAwayMode = euamONLINE;
		m_pFriendObject.m_sHubName  = QString::fromAscii(GetHubName().Data());
		m_pFriendObject.m_sHubHost  = QString::fromAscii(GetHost().Data());
		QApplication::postEvent( g_pUsersList, new DC_FriendEvent( &m_pFriendObject ) );
	}
}

/** */
void DCClient::SetNickColor( QString nick, QColor color )
{
	if ( nick.isEmpty() )
	{
		return;
	}
	
	DCUserItem * pui = m_UserItemDict.find( nick );

	if ( pui != 0 )
	{
		if ( pui->pItem->colorText != color )
		{
			pui->pItem->colorText = color;
			pui->pItem->repaint();
		}
	}
}

/** */
void DCClient::CloseChat( DCChat * chat )
{
	chat->removeEventFilter(this);

	m_ChatMap.remove(chat->GetNick());

	if ( m_bUseTabWidget )
	{
		TabWidget_CHAT->removePage(chat);
	}
	else
	{
		QListViewItem* item;
		if ( (item = ListView_CHATUSERLIST->findItem(chat->GetNick(),0)) != 0 )
		{
			delete item;
			TabWidget_CHAT->setTabLabel( TabWidget_CHAT->page(0), tr("Chat List")+" ("+QString().setNum(ListView_CHATUSERLIST->childCount())+")" );
		}
	}

	chat->close();
}

/** */
void DCClient::CloseAllChats( bool onlyOffline )
{
	if ( onlyOffline )
	{
		QPtrList<DCChat> toclose;
		
		for ( QMap<QString, DCChat*>::const_iterator it = m_ChatMap.constBegin(); it != m_ChatMap.constEnd(); ++it )
		{
			if ( m_UserItemDict.find( it.key() ) == 0 )
			{
				toclose.append(it.data());
			}
		}
		
		for ( unsigned int i = 0; i < toclose.count(); ++i )
		{
			CloseChat( toclose.at(i) );
		}
	}
	else
	{
		while( !m_ChatMap.isEmpty() )
		{
			CloseChat(m_ChatMap.begin().data());
        	}
	}
}

/** */
void DCClient::slotDoubleClickedUserList( QListViewItem * item )
{
	if ( item == 0 )
	{
		return;
	}

	if ( g_pConfig->GetDoubleClickAction() == 1 )
	{
		CString empty;
		g_pTransferView->DLM_QueueAdd(
			item->text(0).ascii(),
			GetHubName(),
			GetHost(),
			DC_USER_FILELIST,
			DC_USER_FILELIST,
			empty,
			empty,
			eltBUFFER,
			0,
			0,
			0,
			empty
		);
	}
	else
	{
		DC_PrivateChat( item->text(0), QString(), QString(), true );
	}
}

/** */
void DCClient::slotDoubleClickedChatUserList( QListViewItem * item )
{
	QString nick;

	if ( item == 0 )
	{
		return;
	}

	nick = item->text(0);

	if ( nick.isEmpty() )
	{
		return;
	}

	if ( m_ChatMap.contains(nick) )
	{
		DCChat * chat = m_ChatMap[nick];
		
		if ( !m_bUseTabWidget )
			chat->setEnabled(true);

		// take widget to top
		if ( chat->isMinimized() )
		{
			chat->showNormal();
		}
		else if ( chat->isVisible() )
		{
			// TODO: dont work fix it
			chat->show();
			chat->raise();
		}
		else
		{
			chat->show();
		}

		SetNickColor( nick, ListView_USERLIST->colorGroup().text() );
	}
}

/** */
void DCClient::slotHubConnect()
{
	if ( GetConnectionState() == estNONE )
	{
		SetConnection(true);
		Connect();
		UpdateReconnect( ersNONE, 0 );
	}
	else if ( GetConnectionState() == estCONNECTED )
	{
		SetConnection(false);
		Disconnect();
	}
}

/** */
void DCClient::slotRightButtonClickedUserList( QListViewItem *, const QPoint &, int column )
{
	QMap<int, DC_UserMenuCommand*> addedcommands;

	QStringList selectedNicks;
	
	QListViewItemIterator lvit( ListView_USERLIST );
	
	while ( lvit.current() )
	{
		if ( lvit.current()->isSelected() )
		{
			selectedNicks.append( lvit.current()->text(0) );
		}
		
		++lvit;
	}

	const int numSelected = selectedNicks.count();

	QPopupMenu * m = new QPopupMenu(this);

	if ( numSelected == 1 )
	{
		DCMenuHandler::InsertMenu( m, emiUSER_CAPTION, false, tr("User: ") + selectedNicks.first() );
	}
	else
	{
		DCMenuHandler::InsertMenu( m, emiUSER_CAPTION, false, QString::number(numSelected) + tr(" users") );
	}

	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiPRIVATE_CHAT, (numSelected > 0) );
	DCMenuHandler::InsertMenu( m, emiADD_FRIEND, (numSelected > 0) );
	DCMenuHandler::InsertMenu( m, emiBROWSE_USER_FILES, (numSelected > 0) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	
	QPopupMenu * mslot = DCMenuHandler::InsertMenu( m, emiUPLOAD_SLOT, true );
	DCMenuHandler::InsertMenu( mslot, emiADD, (numSelected > 0) );
	DCMenuHandler::InsertMenu( mslot, emiADD_PERMANENT, (numSelected > 0) );
	DCMenuHandler::InsertMenu( mslot, emiREMOVE, (numSelected > 0) );

	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	
	QPopupMenu * mcols = new QPopupMenu( m );
	m->insertItem( tr("Columns"), mcols );
	int columns[9];
	
	/* Since the column order can be changed the menu order must reflect that */
	for ( int i = 0; i < 9; ++i )
	{
		int index = ListView_USERLIST->header()->mapToSection(i);
		columns[i] = mcols->insertItem( ListView_USERLIST->columnText( index ) );
		if ( index == 0 )
		{
			mcols->setItemEnabled( columns[i], false );
		}
		mcols->setItemChecked( columns[i], ListView_USERLIST->columnWidth( index ) > 0 );
	}

	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	if ( GetSupportsNoGetInfo() == false )
	{
		DCMenuHandler::InsertMenu( m, emiUPDATE_USER, (numSelected > 0) );
	}
	DCMenuHandler::InsertMenu( m, emiRELOAD_USERLIST, (GetConnectionState() == estCONNECTED) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiCOPY_COLUMN_TO_CLIPBOARD, (numSelected == 1) );
	DCMenuHandler::InsertMenu( m, emiCOPY_ROW_TO_CLIPBOARD, (numSelected == 1) );

	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiCHECK_CLIENT_VERSION, numSelected > 0 );

	if ( UserList()->IsAdmin(GetNick()) )
	{
		DCMenuHandler::InsertMenu( m, emiKICK, (numSelected > 0) );
		DCMenuHandler::InsertMenu( m, emiFORCE_MOVE, (numSelected > 0) );
	}

	addedcommands = AddMenuCommands( m, euccChat );
	
	QValueList<int> addedkeys = addedcommands.keys();
	
	for ( unsigned int i = 0; i < addedkeys.size(); i++ )
	{
		m->setItemEnabled( addedkeys[i], numSelected > 0 );
	}
	
	int id = m->exec(QCursor::pos());

	delete m;

	if ( id == -1 )
	{
		return;
	}
	else if ( id == emiPRIVATE_CHAT )
	{
		for ( QStringList::const_iterator it = selectedNicks.constBegin(); it != selectedNicks.constEnd(); ++it )
		{
			DC_PrivateChat( *it, QString::null, QString::null, true );
		}
	}
	else if ( id == emiADD_FRIEND )
	{
		QString qhubname = QString::fromAscii(GetHubName().Data());
		QString qhubhost = QString::fromAscii(GetHost().Data());
		for ( QStringList::const_iterator it = selectedNicks.constBegin(); it != selectedNicks.constEnd(); ++it )
		{
			g_pUsersList->AddFriend( *it, qhubname, qhubhost, QString::null );
		}
	}
	else if ( id == emiBROWSE_USER_FILES )
	{
		if ( GetMode() == ecmNONE )
		{
			return;
		}

		CString empty;
		for ( QStringList::const_iterator it = selectedNicks.constBegin(); it != selectedNicks.constEnd(); ++it )
		{
			/** add transfer to the waitlist */
			g_pTransferView->DLM_QueueAdd( (*it).ascii(), GetHubName(), GetHost(),
						DC_USER_FILELIST, DC_USER_FILELIST, empty, empty, eltBUFFER,
						0, 0, 0, empty );
		}
	}
	else if ( id == emiCHECK_CLIENT_VERSION )
	{
		if ( GetMode() == ecmNONE )
		{
			return;
		}

		CString empty;
		for ( QStringList::const_iterator it = selectedNicks.constBegin(); it != selectedNicks.constEnd(); ++it )
		{
			/** add transfer to the waitlist */
			g_pTransferView->DLM_QueueAdd( (*it).ascii(), GetHubName(), GetHost(),
						"client check", "client check", empty, empty, eltCLIENTVERSION,
						0, 0, 0, empty );
		}
	}
	else if ( id == emiUPDATE_USER )
	{
		for ( QStringList::const_iterator it = selectedNicks.constBegin(); it != selectedNicks.constEnd(); ++it )
		{
			SendGetInfo( (*it).ascii(), GetNick() );
		}
	}
	else if ( id == emiCOPY_COLUMN_TO_CLIPBOARD )
	{
		QListViewItem * item1 = ListView_USERLIST->currentItem();

		if ( item1 )
		{
			QApplication::clipboard()->setText( item1->text(column) );
		}
	}
	else if ( id == emiCOPY_ROW_TO_CLIPBOARD )
	{
		QString s;
		QListViewItem * item1 = ListView_USERLIST->currentItem();

		if ( item1 )
		{
			for ( int idx = 0; idx < ListView_USERLIST->columns(); ++idx )
			{
				int col = ListView_USERLIST->header()->mapToSection(idx);
				if ( ListView_USERLIST->columnWidth(col) > 0 )
				{
					s += item1->text(col);
					s += " ";
				}
			}
			s = s.stripWhiteSpace();
			QApplication::clipboard()->setText(s);
		}
	}
	else if ( id == emiADD )
	{
		for ( QStringList::const_iterator it = selectedNicks.constBegin(); it != selectedNicks.constEnd(); ++it )
		{
			g_pTransferView->DLM_AddUserSlot( (*it).ascii(), GetHubName(), 1 );
		}
	}
	else if ( id == emiADD_PERMANENT )
	{
		for ( QStringList::const_iterator it = selectedNicks.constBegin(); it != selectedNicks.constEnd(); ++it )
		{
			g_pTransferView->DLM_AddUserSlot( (*it).ascii(), GetHubName(), 0, true );
		}
	}
	else if ( id == emiREMOVE )
	{
		for ( QStringList::const_iterator it = selectedNicks.constBegin(); it != selectedNicks.constEnd(); ++it )
		{
			g_pTransferView->DLM_AddUserSlot( (*it).ascii(), GetHubName(), 0 );
		}
	}
	else if ( id == emiKICK )
	{
		QString kickmessage;

		if ( !GetOPKickMessage(kickmessage,this) )
		{
			return;
		}
		
		for ( QStringList::const_iterator it = selectedNicks.constBegin(); it != selectedNicks.constEnd(); ++it )
		{
			OPKick( *it, kickmessage );
		}
	}
	else if ( id == emiFORCE_MOVE )
	{
		QString host,message;

		if ( !GetOPForceMoveMessage(message,host,this) )
		{
			return;
		}
				
		for ( QStringList::const_iterator it = selectedNicks.constBegin(); it != selectedNicks.constEnd(); ++it )
		{
			OPForceMove( *it, message, host );
		}
	}
	else if ( id == emiRELOAD_USERLIST )
	{
		RequestNickList();
	}
	else if ( addedcommands.contains( id ) )
	{
		DC_UserMenuCommand * umc = addedcommands[id];
		
		QString origUserCommand = umc->m_sCommand;
		
		QString usercommand;
		
		for ( QStringList::const_iterator it = selectedNicks.constBegin(); it != selectedNicks.constEnd(); ++it )
		{
			usercommand = replaceCommandTags( origUserCommand, *it );
			
			if ( !usercommand.isEmpty() )
			{
				m_pHubChat->AddStatus(usercommand);
				SendString(usercommand.ascii());
			}
		}
	}
	else
	{
		for ( int i = 0; i < 9; ++i )
		{
			if ( id == columns[i] )
			{
				int index = ListView_USERLIST->header()->mapToSection(i);
				int width;
				if ( ListView_USERLIST->columnWidth( index ) > 0 )
				{
					width = 0;
					UpdateClientColumn( false, index, width );
				}
				else
				{
					int visible = 0;
					for ( int j = 0; j < 9; ++j )
					{
						if ( ListView_USERLIST->columnWidth(j) > 0 )
						{
							++visible;
						}
					}
					
					if ( visible > 0 )
					{
						width = ListView_USERLIST->width()/visible;
					}
					else
					{
						width = 40;
					}
					
					UpdateClientColumn( true, index, width );
				}
				
				/* column widths appear to be saved automatically since a slot is connected to the header */
				eClientColumn mapped_section;
				if ( index == 0 )
				{
					mapped_section = eclcNICK;
				}
				else
				{
					mapped_section = eClientColumn( index - 1 );
				}
				
				/* speed and email are the wrong way around in the enum */
				if ( mapped_section == eclcSPEED )
				{
					mapped_section = eclcEMAIL;
				}
				else if ( mapped_section == eclcEMAIL )
				{
					mapped_section = eclcSPEED;
				}
				
				g_pConfig->SetClientColumn( mapped_section, width > 0 );
						
				break;
			}
		}
	}
}

void DCClient::slotSaveColumnWidth(int section, int /*oldSize*/, int newSize)
{
	eClientColumn mapped_section;

	if (section == 0)
		mapped_section = eclcNICK;
	else
		mapped_section = eClientColumn(section - 1);
	
	/*
	 * Speed and email are also the wrong way around.
	 * Fix the enum already?
	 */
	if ( mapped_section == eclcEMAIL )
	{
		mapped_section = eclcSPEED;
	}
	else if ( mapped_section == eclcSPEED )
	{
		mapped_section = eclcEMAIL;
	}
	
	g_pConfig->SetClientColumnWidth( mapped_section, newSize );
}

/** */
bool DCClient::GetOPKickMessage( QString & message, QWidget * parent )
{
	bool ok = false;

	message = QInputDialog::getText(
		tr("OP Kick"),
		tr("Please enter a reason"),
		QLineEdit::Normal, QString::null, &ok, parent );

	return ok;
}

/** */
bool DCClient::GetOPForceMoveMessage( QString & message, QString & host, QWidget * parent )
{
	bool ret = false;
	
	DCDialogForceMove * dialog = new DCDialogForceMove(parent);
	
	if ( dialog->exec() == QDialog::Accepted )
	{
		host = dialog->LineEdit_HOST->text();
		message = dialog->LineEdit_MESSAGE->text();
		ret = true;
	}
	
	delete dialog;
	
	return ret;
}

/** */
bool DCClient::OPKick( QString nick, QString message )
{
	bool res = false;

	SendPrivateMessage( GetNick(), nick.ascii(), ("You are being kicked because: " + message).ascii());
	SendChat( GetNick(), GetNick() + " is kicking " + nick.ascii() + " because: " + message.ascii() );
	SendKick(nick.ascii());

	return res;
}

/** */
bool DCClient::OPForceMove( QString nick, QString message, QString host )
{
	bool res = false;

	SendOpForceMove( nick.ascii(), host.ascii(), message.ascii() );

	return res;
}

/** */
void DCClient::slotDoubleClickedUserChat( QListBoxItem * item )
{
	QString nick;

	if ( item == 0 )
	{
		return;
	}

	nick = item->text();

	if ( nick.isEmpty() )
	{
		return;
	}

	if ( m_ChatMap.contains(nick) )
	{
		DCChat * chat = m_ChatMap[nick];
		
		if ( !m_bUseTabWidget )
			chat->setEnabled(true);
		chat->show();
		SetNickColor( nick, ListView_USERLIST->colorGroup().text() );
        }
}

/** */
QString DCClient::findNick( QString part, uint which )
{
/*
	for(uint i=0; i < nicks->count(); i++)
	{
		if (matches.contains(nicks->text(i)))
			continue;
		if(qstrlen(nicks->text(i)) >= part.length())
		{
			if(qstrnicmp(part, nicks->text(i), part.length()) == 0)
			{
				QString qsNick = ksopts->nick;
				if(qstrcmp(nicks->text(i), qsNick) != 0)
				{ // Don't match your own nick
					matches.append(nicks->text(i));
				}
			}
		}
	}
*/	
	QStringList matches;
	
	QString tlpart = part.stripWhiteSpace().lower();
	
	if ( tlpart == lastPart )
	{
		matches = lastMatches;
	}
	else
	{
		matches = matchNicks( tlpart );
		lastMatches = matches;
		lastPart = tlpart;
	}
	
	//printf("matches='%s'\n", matches.join(",").ascii());

	if(matches.count() > 0)
	{
		if(which < matches.count())
			return *(matches.at(which));
		else
			return QString::null;
	}

	return part;
}

void DCClient::slotTabCornerCloseWidgetClicked()
{
	CloseChat( (DCChat*) TabWidget_CHAT->currentPage() );
}

/** Replaces the %[tag] s in the command with the proper values */
QString DCClient::replaceCommandTags(QString & command, QString remotenick)
{
	QString usercommand = command;
	
	usercommand.replace( "%[mynick]", GetNick().Data() );
	usercommand.replace( "%[myNI]", GetNick().Data() );
	
	QString comtag = GetComment().Data();
	QString comment = comtag.mid(0, comtag.find('<'));
	QString tag = comtag.mid(comtag.find('<'));
	
	usercommand.replace( "%[mytag]", tag );
	usercommand.replace( "%[myTAG]", tag );
	
	usercommand.replace( "%[mydescription]", comment );
	usercommand.replace( "%[myDE]", comment );
	
	usercommand.replace( "%[myemail]", GetEMail().Data() );
	usercommand.replace( "%[myEM]", GetEMail().Data() );
	
	const QString myss = QString::number( GetShareSize() );
	usercommand.replace( "%[myshare]", myss );
	usercommand.replace( "%[mySS]", myss );
	
	const QString myssshort = DCGuiUtils::GetSizeString(GetShareSize());
	usercommand.replace( "%[myshareshort]", myssshort );
	usercommand.replace( "%[mySSshort]", myssshort );
	
	const QString myip = GetExternalIP( false ).Data();
	usercommand.replace( "%[myip]", myip );
	usercommand.replace( "%[myI4]", myip );
	
	usercommand.replace( "%[hub]", GetHubName().Data() );
	usercommand.replace( "%[hubNI]", GetHubName().Data() );
	
	usercommand.replace( "%[hubURL]", GetHost().Data() );
	
	usercommand.replace( "%[hubIP]", GetResolvedIP().Data() );
	usercommand.replace( "%[hubI4]", GetResolvedIP().Data() );
	
	usercommand.replace( "%[nick]", remotenick );
	usercommand.replace( "%[userNI]", remotenick );
	
	DC_UserListItem * item = 0;
	DCUserItem * pui = m_UserItemDict.find(remotenick);
	if ( pui )
	{
		item = pui->pItem;
	}
	
	if ( item != 0 )
	{
		usercommand.replace( "%[userTAG]", item->text(2) );
		usercommand.replace( "%[tag]", item->text(2) );
		
		usercommand.replace( "%[userDE]", item->text(1) );
		usercommand.replace( "%[description]", item->text(1) );
		
		usercommand.replace( "%[userSS]", QString().setNum( item->myvalue ) );
		usercommand.replace( "%[share]", QString().setNum( item->myvalue ) );
		
		usercommand.replace( "%[userSSshort]", item->text(5) );
		usercommand.replace( "%[shareshort]", item->text(5) );
		
		usercommand.replace( "%[userEM]", item->text(4) );
		usercommand.replace( "%[email]", item->text(4) );
		
		usercommand.replace( "%[userI4]", item->text(6) );
		usercommand.replace( "%[ip]", item->text(6) );
	}
	
	if ( usercommand.find("%[line:") != -1 )
	{
		DCDialogUserCommandLines * dialog = new DCDialogUserCommandLines(this);
		dialog->LineEdit_NICK->setText( remotenick );
		dialog->LineEdit_HUB->setText( QString::fromAscii(GetHubName().Data()) );
		
		QMap<QString, QLineEdit*> lineeditmap;
		dialog->ScrollView->setResizePolicy( QScrollView::AutoOneFit );
		QGrid * grid = new QGrid( 2, dialog->ScrollView->viewport() );
		grid->setMargin( 6 );
		grid->setSpacing( 6 );
		
		int loc = 0, end = 0;
		
		for ( ; ; )
		{
			loc = usercommand.find( "%[line:", loc );
			end = usercommand.find( "]", loc );
			
			if ( (loc == -1) || (end == -1) )
			{
				break;
			}
			
			// 7 is length of "%[line:"
			QString reason = usercommand.mid(loc + 7, end - (loc + 7));
			
			QString key = "%[line:" + reason + "]";
			
			if ( !lineeditmap.contains(key) )
			{
				new QLabel( reason, grid );
				
				lineeditmap[key] = new QLineEdit( grid );
			}
			
			loc = end + 1;
		}
		
		dialog->ScrollView->addChild(grid);
		
		if ( dialog->exec() == QDialog::Accepted )
		{
			for ( QMap<QString, QLineEdit*>::const_iterator it = lineeditmap.constBegin(); it != lineeditmap.constEnd(); ++it )
			{
				QString text = it.data()->text();
				text.replace( "$", "&#36;" );
				text.replace( "|", "&#124;" );
				
				usercommand.replace( it.key(), text );
				
				if ( dialog->CheckBox_ALL_NICKS->isChecked() )
				{
					command.replace( it.key(), text );
				}
			}
			delete dialog;
		}
		else
		{
			delete dialog;
			return QString();
		}
	}
	
	time_t ti = time(0);
	struct tm * t = localtime(&ti);
	QByteArray timebuf;
	timebuf.resize( usercommand.local8Bit().length() + 512 );
	
	/*
	 * If strftime returns 0 either the buffer was not big enough
	 * or the actual output is zero characters.
	 */
	for ( int tries = 0; tries < 10; ++tries )
	{
		if ( strftime( timebuf.data(), timebuf.size(), usercommand.ascii(), t ) > 0 )
		{
			usercommand = QString::fromAscii( timebuf );
			break;
		}
		else
		{
			timebuf.resize( timebuf.size() + 128 );
		}
	}
	
	return usercommand;
}

void DCClient::doAutoResponse( QString nick, QString message, DCChat * chatobject )
{
	long delay = g_pConfig->GetAutoResponseDelay();
	long now = QDateTime::currentDateTime().toTime_t();
	
	if ( (delay > 0) &&
	     (nick == lastAutoNick) &&
	     (now < delay + lastAutoResponseTime)
	   )
	{
		// avoid spamming
		//printf("Auto response spam protection!\n");
		return;
	}
	
	// null pointer check
	if (chatobject == 0)
	{
		printf("Error: DCClient::doAutoResponse called with NULL chatobject!\n");
		return;
	}
	
	// do not respond to ourself
	if (nick == QString::fromAscii(GetNick().Data()))
	{
		//printf("Not auto-responding to ourself.\n");
		return;
	}
	
	// check for ignored nicks
	QString ignores = g_pConfig->GetAutoResponseIgnores();
	if ( !ignores.isEmpty() )
	{
		QRegExp ignore_re;
		ignore_re.setPattern(ignores);
		ignore_re.setCaseSensitive(false);
		if (ignore_re.search(nick) != -1)
		{
			//printf("Not auto responding to %s\n", nick.ascii());
			return;
		}
	}
	
	QPtrList<DC_AutoResponseObject> arlist;
	arlist.setAutoDelete(true);
	
	g_pConfig->GetAutoResponses(&arlist);
	
	QRegExp re;
	
	for ( DC_AutoResponseObject * aro = arlist.first(); aro; aro = arlist.next() )
	{
		re.setPattern(aro->m_sTrigger);
		re.setCaseSensitive(aro->m_bCaseSensitive);
		if (re.search(message) != -1) // pattern matches
		{
			QString origResponse = aro->m_sResponse;
			QString chat = replaceCommandTags( origResponse, nick );
			
			if ( chat.isEmpty() )
			{
				break;
			}
			
			//SendChat( GetNick(), chat.ascii() );
			chatobject->TextEdit_CHATINPUT->setText(chat);
			chatobject->SendChat();
			
			// update last nick and time for anti-spam
			lastAutoNick = nick;
			lastAutoResponseTime = QDateTime::currentDateTime().toTime_t();
			
			// stop searching list of triggers
			break;
		}
	}
	
	arlist.clear();
}

/** */
QStringList DCClient::matchNicks( const QString & part) const
{
	QStringList matches;
	QListViewItemIterator it( ListView_USERLIST );
	
	if ( g_pConfig->GetTabCompleteMatchMode() == 0 ) // containing
	{
		if ( g_pConfig->GetTabCompleteStripTags() )
		{
			while ( it.current() )
			{
				if ( ((DC_UserListItem*)it.current())->m_sNickStripped.contains(part) > 0 )
				{
					matches << it.current()->text(0);
				}
				++it;
			}
		}
		else
		{
			while ( it.current() )
			{
				if ( ((DC_UserListItem*)it.current())->m_sNickLC.contains(part) > 0 )
				{
					matches << it.current()->text(0);
				}
				++it;
			}
		}
	}
	else // starts with
	{
		if ( g_pConfig->GetTabCompleteStripTags() )
		{
			while ( it.current() )
			{
				if ( ((DC_UserListItem*)it.current())->m_sNickStripped.startsWith(part) )
				{
					matches << it.current()->text(0);
				}
				++it;
			}
		}
		else
		{
			while ( it.current() )
			{
				if ( ((DC_UserListItem*)it.current())->m_sNickLC.startsWith(part) )
				{
					matches << it.current()->text(0);
				}
				++it;
			}
		}
	}
	
	return matches;
}

/** */
QMap<int, DC_UserMenuCommand*> DCClient::AddMenuCommands( QPopupMenu * menu, int context )
{
	QMap<int, DC_UserMenuCommand*> added;
	QValueList<DC_UserMenuCommand*> allcommands;
	QMap<QString, QPopupMenu*> createdmenus;
	QPopupMenu * prevmenu = 0;
	
	if ( !menu )
	{
		return added;
	}
	
	if ( g_pConfig->GetEnableUserCommand() )
	{
		for ( uint i = 0; i < hubcommands.count(); i++ )
		{
			allcommands.append(hubcommands.at(i));
		}
	}
	
	// int key is NOT the same as added key!
	QMap<int, DC_UserMenuCommand*> usercommands;
	
	g_pConfig->GetUserMenuCommandsDirect(&usercommands);
	
	allcommands += usercommands.values();
	
	usercommands.clear();
	
	if ( g_pConfig->GetUserCommandSubmenu() )
	{
		menu = DCMenuHandler::InsertMenu(menu, emisUSER_COMMANDS);
	}
	
	for ( uint i = 0; i < allcommands.size(); i++ )
	{
		if ( (allcommands[i]->m_nContext & context) == context )
		{
			if ( !(allcommands[i]->m_sHubIP.isEmpty()) )
			{
				QString hubiplc = allcommands[i]->m_sHubIP.lower();
				if ( (hubiplc == "op") && (UserList()->IsAdmin(GetNick()) == false) )
				{
					//printf("Skipping command '%s' because we are not OP here\n",allcommands[i]->m_sName.Data());
					continue;
				}
				else if ( (hubiplc != QString::fromAscii(GetHost().ToLower().Data())) && (hubiplc != QString::fromAscii(GetIP().ToLower().Data())) )
				{
					//printf("Skipping command '%s' because host/IP does not match (required=%s our host=%s its ip=%s)\n",allcommands[i]->m_sHubIP.Data(),GetHost().Data(),GetIP().Data());
					continue;
				}
			}
			
			QPopupMenu * submenu = menu;
			QString name = allcommands[i]->m_sName;
			
			if ( name.find('\\') != -1 )
			{
				QStringList subMenuNames = QStringList::split( '\\', name, false );
				name = subMenuNames.last(); // remove and store command name
				subMenuNames.pop_back();
				
				QString subMenuKey = subMenuNames.first();
				QPopupMenu * parentMenu = menu;
				
				for ( uint j = 0; j < subMenuNames.size(); j++ )
				{
					if ( createdmenus.contains(subMenuKey) )
					{
						parentMenu = createdmenus[subMenuKey];
						submenu = parentMenu;
					}
					else
					{
						submenu = new QPopupMenu(parentMenu);
						createdmenus[subMenuKey] = submenu;
						parentMenu->insertItem( subMenuNames[j], submenu );
						parentMenu = submenu;
					}
					
					subMenuKey += '\\';
					subMenuKey += subMenuNames[j];
				}
			}
			
			if ( allcommands[i]->m_nType == euctSeparator )
			{
				// the protocol was badly designed, separators do not have names
				// and so cannot be removed from menus
				// whatever docs exist on random forum posts etc.
				// also say separators are supposed to be a "vertical bar"
				// valknut currently uses horizontal bars
				if ( allcommands[i]->m_sName.isEmpty() && prevmenu )
				{
					prevmenu->insertSeparator();
				}
				else
				{
					submenu->insertSeparator();
				}
			}
			else if ( (allcommands[i]->m_nType == euctRaw) || (allcommands[i]->m_nType == euctRawOnce) )
			{
				added.insert( submenu->insertItem(name), allcommands[i] );
				prevmenu = submenu;
			}
			//else
			//{
			//	printf("AddMenuCommands: wrong command type %d in lists for command '%s'\n", allcommands[i]->m_nType, allcommands[i]->m_sName.Data());
			//}
		}
	}
	
	if ( g_pConfig->GetUserCommandSubmenu() && (added.size() == 0) )
	{
		menu->setEnabled(false);
	}
	
	return added;
}

/** */
bool DCClient::jumpToNick( const QString & nick )
{
	if ( nick.isEmpty() )
	{
		return false;
	}
	
	DCUserItem * pui = m_UserItemDict.find( nick );
	
	if ( pui != 0 )
	{
		ListView_USERLIST->clearSelection();
		ListView_USERLIST->setSelected(pui->pItem, true);
		ListView_USERLIST->setCurrentItem(pui->pItem);
		ListView_USERLIST->ensureItemVisible(pui->pItem);
		return true;
	}
	else
	{
		return false;
	}
}
