/***************************************************************************
                          dcconnectionmanager.cpp  -  description
                             -------------------
    begin                : Mon Oct 1 2001
    copyright            : (C) 2001-2005 by Mathias Küster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "dcconnectionmanager.h"

#include <stdlib.h>

#include <qtoolbar.h>
#include <qworkspace.h>

#include "dcclient.h"
#include "dciconloader.h"
#include "dcconfig.h"

#include <dclib/dcos.h>

DCConnectionManager * g_pConnectionManager = 0;

/** */
DCConnectionManager::DCConnectionManager( QObject * parent, const char *name) : QObject(parent, name)
{
	m_pWorkspace = (QWorkspace*)parent;

	m_pMessageQueue = new QPtrQueue<CDCMessage>();

	InitDocument();

	g_pConnectionManager = this;
}

/** */
DCConnectionManager::~DCConnectionManager()
{
	CClient * Client = 0;

	g_pConnectionManager = NULL;

	m_Mutex.Lock();

	while( (Client=m_pClientList->Next(0)) != 0 )
	{
		m_pClientList->Remove(Client);
		Client->Disconnect();
	}

	if ( m_pMessageQueue )
	{
		QPtrQueue<CDCMessage> * tmp = m_pMessageQueue;
		m_pMessageQueue = 0;
		
		while ( !tmp->isEmpty() )
		{
			delete tmp->dequeue();
		}
		
		delete tmp;
	}

	m_Mutex.UnLock();
}

/** */
void DCConnectionManager::InitDocument()
{
	m_pTabBar = 0;
	m_nSelectedTab = -1;
	m_bDisableTabSelected = false;
	
	connect( &m_Timer, SIGNAL(timeout()), this, SLOT(timerDone()) );

	connect( m_pWorkspace, SIGNAL(windowActivated(QWidget *)), this, SLOT(slotWorkspaceWindowActivated( QWidget *)) );
	
	m_Timer.start( 500, true );
}

/** */
void DCConnectionManager::InitTabBar( QToolBar * toolbar )
{
	m_pTabBar = new QTabBar(toolbar);
	toolbar->setStretchableWidget(m_pTabBar);
	connect(m_pTabBar, SIGNAL(selected(int)), this, SLOT(slotTabSelected(int)));
	// hide empty tabbar
	m_pTabBar->hide();
}

/** */
void DCConnectionManager::resetTabIcon( DCClientTab * ct )
{
	if ( !ct )
		return;

	ct->setIconSet( QIconSet() );
	// little trick to force a repaint/resize
	ct->setText(ct->text());
}

/** */
void DCConnectionManager::slotTabSelected( int id )
{
	if ( m_bDisableTabSelected )
	{
		m_bDisableTabSelected = false;
	}
	else if ( (id != -1) && (m_nSelectedTab != id) )
	{
		DCClientTab * ct = (DCClientTab*) m_pTabBar->tab(id);
		
		if ( ct )
		{
			resetTabIcon(ct);
			if ( ct->m_pClient )
				ct->m_pClient->showNormal();
		}
	}
	
	m_nSelectedTab = id;
}

/** widget from workspace activated */
void DCConnectionManager::slotWorkspaceWindowActivated( QWidget * w )
{
	DCClientTab * ct = 0;
	int id = m_pTabBar->currentTab();
			
	if ( id != -1 )
		ct = (DCClientTab*) m_pTabBar->tab(id);

	if ( ct && w )
	{
		// only if tab change
		if ( ct->m_pClient != w )
		{
			// set current tab
			int i;
			
			for(i=0;i<m_pTabBar->count();i++)
			{
				DCClientTab * ct = (DCClientTab *)m_pTabBar->tabAt(i);

				if ( ct && ct->m_pClient == w )
				{
					m_bDisableTabSelected = true;
					m_pTabBar->setCurrentTab(ct);
					resetTabIcon(ct);

					break;
				}
			}
		}
	}
}

/** */
void DCConnectionManager::HubEvent( DCClient * client )
{
	int i;

	if ( !client )
		return;
		
	for(i=0;i<m_pTabBar->count();i++)
	{
		DCClientTab * ct = (DCClientTab *)m_pTabBar->tabAt(i);

		if ( ct->m_pClient == client )
		{
			DCClientTab * ctw = 0;
			int id = m_pTabBar->currentTab();
			
			if ( id != -1 )
				ctw = (DCClientTab*) m_pTabBar->tab(id);

			if ( ct != ctw )
			{
				ct->setIconSet( QIconSet( g_pIconLoader->GetPixmap(eiMESSAGE) ) );
				// little trick to force a repaint/resize
				ct->setText(ct->text());
			}

			break;
		}	
	}
}

/** */
void DCConnectionManager::CaptionChanged( DCClient * client )
{
	if ( client )
	{
		DCClientTab * ct;
		for ( int i = 0; i < m_pTabBar->count(); ++i )
		{
			ct = (DCClientTab*) m_pTabBar->tabAt(i);
			
			if ( ct->m_pClient == client )
			{
				QString s = QString::fromAscii( client->GetHubName().Data() );
				
				if ( s.length() > 20 )
				{
					s = s.left(20) + "...";
				}
				
				ct->setText( s );
				
				return;
			}
		}
	}
}

/** */
void DCConnectionManager::HubClosing( DCClient * client )
{
	if ( client )
	{
		DCClientTab * ct;
		for ( int i = 0; i < m_pTabBar->count(); ++i )
		{
			ct = (DCClientTab*) m_pTabBar->tabAt(i);
			
			if ( ct->m_pClient == client )
			{
				m_pTabBar->removeTab( ct );
				
				if ( m_pTabBar->count() == 0 )
				{
					m_pTabBar->hide();
				}
				else if ( m_pTabBar->count() == 1 )
				{
					DCClientTab * ct2 = (DCClientTab*) m_pTabBar->tabAt(0);
					if ( ct2 )
					{
						resetTabIcon(ct2);
					}
				}
				
				break;
			}
		}
		
		RemoveHub( client );
	}
}

/** callback function */
int DCConnectionManager::DC_CallBack( CDCMessage * DCMessage )
{
	int err = -1;

	m_Mutex.Lock();

	if ( DCMessage && m_pMessageQueue )
	{
		m_pMessageQueue->enqueue(DCMessage);
		err = 0;
	}

	m_Mutex.UnLock();

	return err;
}

/** */
void DCConnectionManager::timerDone()
{
	CDCMessage *DCMsg;
	int i;

	for(i=0;i<50;i++)
	{
		if ( m_Mutex.TryLock() == false )
		{
			break;
		}

		if ( m_pMessageQueue )
		{
			/* returns 0 if queue is empty */
			DCMsg = m_pMessageQueue->dequeue();
		}
		else
		{
			DCMsg = 0;
		}

		m_Mutex.UnLock();

		if ( DCMsg == 0 )
		{
			break;
		}

		switch ( DCMsg->m_eType )
		{
			case DC_MESSAGE_CONNECT_CLIENT:
			{
				DCMessageConnectClient *msg = (DCMessageConnectClient*)DCMsg;

				Connect( msg->m_sHubName, msg->m_sHubHost );

				break;
			}

			default:
			{
				break;
			}
		}

		if ( DCMsg )
		{
			delete DCMsg;
		}
	}

	m_Timer.start( 500, true );
}

/** */
void DCConnectionManager::AutoConnect()
{
	DCConfigHubItem * hubitem = 0;
	CList<DCConfigHubItem> list;
	DCConfigHubProfile ConfigHubProfile;

	g_pConfig->GetBookmarkHubList(&list);

	while( (hubitem=list.Next(hubitem)) != 0 )
	{
		// check for autoconnect
		if ( g_pConfig->GetHubProfile( hubitem->m_sProfile, &ConfigHubProfile ) )
		{
			if ( ConfigHubProfile.m_bAutoConnect )
			{
				Connect( hubitem->m_sName, hubitem->m_sHost );
			}
		}
	}
}

/** */
void DCConnectionManager::Connect( CString hubname, CString server, bool sslconnect )
{
	if ( server.IsEmpty() )
	{
		return;
	}

	if ( hubname.IsEmpty() )
	{
		hubname = server;
	}

	m_Mutex.Lock();

	DCClient * client = dynamic_cast<DCClient*>(GetHub(hubname,server));

	if ( client )
	{
		client->setFocus();
	}

	m_Mutex.UnLock();

	if ( client )
	{
		return;
	}

	int i = server.Find(':');
	QString s;
	if ( i >= 0 )
	{
		s = QString::fromAscii( server.Left(i).Data() );
	}
	else
	{
		s = QString::fromAscii( server.Data() );
	}

	client = new DCClient(m_pWorkspace,s.ascii(),WDestructiveClose,g_pConfig->GetRemoteEncoding(hubname,server));

	// create tab
	DCClientTab * ct = new DCClientTab();
	
	ct->m_pClient = client;
	s = QString::fromAscii( client->GetHubName().Data() );
	if ( s.length() > 20 )
		s = s.left(20) + "...";
	ct->setText(s);

	if ( m_pTabBar->count() == 0 )
		m_pTabBar->show();
	m_pTabBar->addTab(ct);

	client->setCaption(hubname.Data());

	if ( g_pConfig->GetOpenClientWindows() == 0 )
	{
		client->showMinimized();
	}
	else if ( g_pConfig->GetOpenClientWindows() == 2 )
	{
		client->showMaximized();
		// update tabbar
		slotWorkspaceWindowActivated(client);
	}
	else
	{
		client->show();
		// update tabbar
		slotWorkspaceWindowActivated(client);
	}

	CConnectionManager::Connect(hubname,server,client,sslconnect);
}

/** */
void DCConnectionManager::OpenPrivateChat( QString hubname, QString hubhost, QString nick )
{
	m_Mutex.Lock();

	DCClient * Client = dynamic_cast<DCClient*>(GetHub(hubname.ascii(),hubhost.ascii()));

	if ( Client )
	{
		Client->DC_PrivateChat(nick,QString(),QString(),true);
	}

	m_Mutex.UnLock();
}

/** */
void DCConnectionManager::CloseAllChats( bool onlyOffline )
{
	CClient * Client = 0;

	m_Mutex.Lock();

	while( (Client=m_pClientList->Next(Client)) != 0 )
	{
		DCClient * dcc = dynamic_cast<DCClient*>(Client);
		if ( dcc )
		{
			dcc->CloseAllChats( onlyOffline );
		}
	}

	m_Mutex.UnLock();
}

/** */
void DCConnectionManager::DisconnectAllClients()
{
	CClient * Client = 0;

	m_Mutex.Lock();

	while( (Client=m_pClientList->Next(Client)) != 0 )
	{
		m_Mutex.UnLock();
		Client->Disconnect(true);
		m_Mutex.Lock();
	}

	m_Mutex.UnLock();
}

/** */
void DCConnectionManager::CloseDisconnectedHubs()
{
	CClient * Client = 0, * oClient = 0;

	m_Mutex.Lock();

	while( (Client=m_pClientList->Next(Client)) != 0 )
	{
		if ( Client->GetConnectionState() == estNONE )
		{
			m_Mutex.UnLock();
			DCClient * dcc = dynamic_cast<DCClient*>(Client);
			if ( dcc )
			{
				dcc->close();
			}
			m_Mutex.Lock();

			Client = oClient;
		}
		else
		{
			oClient = Client;
		}
	}

	m_Mutex.UnLock();
}

/** */
void DCConnectionManager::OPKick( QString hubname, QString hubhost, QString nick, QString message )
{
	m_Mutex.Lock();

	DCClient * Client = dynamic_cast<DCClient*>(GetHub(hubname.ascii(),hubhost.ascii()));

	if ( Client )
	{
		Client->OPKick( nick, message );
	}

	m_Mutex.UnLock();
}

/** */
void DCConnectionManager::OPForceMove( QString hubname, QString hubhost, QString nick, QString message, QString host )
{
	m_Mutex.Lock();

	DCClient * Client = dynamic_cast<DCClient*>(GetHub(hubname.ascii(),hubhost.ascii()));

	if ( Client )
	{
		Client->OPForceMove( nick, message, host );
	}

	m_Mutex.UnLock();
}

/** */
DCClient * DCConnectionManager::GetClientForHub( CString hubname, CString hubhost )
{
	/* using dynamic_cast because there could be non gui CClients
	 * present if doing bookmark/public hub search
	 */
	return dynamic_cast<DCClient*>(GetHub( hubname, hubhost ));
}
