/***************************************************************************
                          dcdebug.cpp  -  description
                             -------------------
    begin                : Don Sep 25 2003
    copyright            : (C) 2003 by Mathias Küster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "dcdebug.h"

#ifndef WIN32
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#endif

#include <qprocess.h>
#include <qpushbutton.h>
#include <qfile.h>
#include <qprocess.h>
#include <qtextedit.h>
#include <qmessagebox.h>
#include <qfiledialog.h>

#include "dcconfig.h"
#include <dclib/dcos.h>
#include <dclib/dcobject.h>
#include <dclib/dclib.h>
#include <dclib/chttp.h>
#include <dclib/core/cstring.h>
#include <dclib/core/cxml.h>

#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif

CString DCDebug::arg_0 = CString();
CString DCDebug::startup_dir = CString();

/** */
DCDebug::DCDebug( QWidget *parent, const char *name ) : DCDialogDebug(parent,name)
{
	m_pDebug = 0;
	m_pHttp = 0;

	connect( PushButton_EXIT, SIGNAL(clicked()), this, SLOT(accept()) );
	connect( PushButton_SEND, SIGNAL(clicked()), this, SLOT(slotSendBacktrace()) );
	connect( PushButton_SAVE, SIGNAL(clicked()), this, SLOT(slotSaveBacktrace()) );

	// no server available to send the info to
	PushButton_SEND->setEnabled(false);

	// enable timer
	connect( &m_Timer, SIGNAL(timeout()), this, SLOT(timerDone()) );
}

/** */
DCDebug::~DCDebug()
{
	m_Timer.stop();

	if ( m_pHttp )
	{
		delete m_pHttp;
	}
	
	if ( m_pDebug )
	{
		disconnect( m_pDebug, SIGNAL(processExited()), this, SLOT(slotDebuggerFinished()) );
		m_pDebug->kill();
		
		delete m_pDebug;
		m_pDebug = 0;
	}
}

/** */
bool DCDebug::Init( QString configpath, QString debugopt )
{
#ifndef WIN32
	TextEdit_LOG->setText(tr("Getting information about the crash..."));
	
	QString s = "thread apply all bt full\nq\n";
	QString filename = configpath+"/"+"gdbrc";
	QFile f(filename);
	QString s1,s2,s3;

	s1 = debugopt.section(",",0,0);
	s2 = debugopt.section(",",1,1);
	s3 = debugopt.section(",",2,2);

	printf("Init debug:\nConfigpath: '%s'\n%s %s %s\n",
			configpath.ascii(),
			s1.ascii(),
			s2.ascii(),
			s3.ascii() );

	f.open( IO_WriteOnly | IO_Truncate );
	f.writeBlock(s,s.length());
	f.close();

	m_pDebug = new QProcess( this );
	m_pDebug->setCommunication( QProcess::Stdout | QProcess::Stderr );
	m_pDebug->addArgument("gdb");
	m_pDebug->addArgument("--nw");
	m_pDebug->addArgument("--nx");
	m_pDebug->addArgument("--quiet");
	m_pDebug->addArgument("--batch");
	m_pDebug->addArgument("-x");
	m_pDebug->addArgument(filename);
	m_pDebug->addArgument(s3);
	m_pDebug->addArgument(s1);

	connect( m_pDebug, SIGNAL(processExited()), this, SLOT(slotDebuggerFinished()) );

	if ( m_pDebug->launch( QString() ) == false )
	{
		disconnect( m_pDebug, SIGNAL(processExited()), this, SLOT(slotDebuggerFinished()) );
		slotDebuggerFinished();
	}
	
	return true;
#else
	return false;
#endif
}

/** */
void DCDebug::slotDebuggerFinished()
{
	QString s;
	
	// get version
	QFile f1("/proc/version");
	
	if ( f1.open( IO_ReadOnly ) )
	{
		s += f1.readAll();
		f1.close();
		s += "\n";
	}
	
	// create output text
	s += "Valknut: ";
	s += VERSION;
	s += " (";
	s += VALKNUT_BUILD_INFO;
	s += ")\nDCLIB: ";
	s += dclibVersion();
	s += " (";
	s += dclibBuildInfo();
	s += ")\nQT(R) compiled: ";
	s += QT_VERSION_STR;
	s += "\nQT(R) used: ";
	s += qVersion();
	s += "\nLIBXML compiled: ";
	s += CXml::Libxml2CompiledVersion();
	s += "\nLIBXML used: ";
	s += CXml::Libxml2RunningVersion();
	s += "\n\n";
	
	if ( m_pDebug->normalExit() )
	{
		QString buffer = m_pDebug->readStdout();
		buffer += "\n";
		buffer += m_pDebug->readStderr();
		buffer += "\n";

		TextEdit_LOG->setText( s + buffer );
	}
	else
	{
		QString failed = tr("Failed to get more information about the crash.");
		failed += "\n";
		failed += tr("The %1 program is required to get the information.").arg("gdb");
		failed += "\n";
		failed += tr("Exit code: %1").arg(m_pDebug->exitStatus());
		failed += "\n";
		failed += tr("Error messages:");
		failed += "\n";
		failed += m_pDebug->readStderr();
		TextEdit_LOG->setText( s + failed );
	}
	
	delete m_pDebug;
	m_pDebug = 0;
}

/** */
void DCDebug::timerDone()
{
	if ( m_bTransfer == false )
	{
		if ( m_pHttp )
		{
			if ( m_pHttp->GetHttpError() != 200 )
			{
				QMessageBox::critical( this, tr("Send error"),
		                	tr("Could not send the data. Error '%1'.").arg(m_pHttp->GetHttpError()) );

				PushButton_SEND->setEnabled(true);
			}
			else
			{
				QMessageBox::critical( this, tr("Send"),
		                	tr("Thanks for your help.") );
			}
		}

		PushButton_SAVE->setEnabled(true);
		PushButton_EXIT->setEnabled(true);
	}
	else
	{
		m_Timer.start( 500, true );
	}
}

/** */
void DCDebug::slotSendBacktrace()
{
	QString text;

	if ( m_pHttp )
		delete m_pHttp;

	PushButton_SAVE->setEnabled(false);
	PushButton_SEND->setEnabled(false);
	PushButton_EXIT->setEnabled(false);

	m_bTransfer = true;
	m_Timer.start( 500, true );

	m_pHttp = new CHttp();
	m_pHttp->SetCallBackFunction( new CCallback1<DCDebug, CDCMessage*>( this, &DCDebug::HttpCallBack ) );

	text  = "--- Comment\n\n";
	text += TextEdit_COMMENT->text();
	text += "\n--- Backtrace\n\n";
	text += TextEdit_LOG->text();
	text += "\n--- End\n";

	m_pHttp->GetUrl("http://dcgui.berlios.de/crash.php",text.ascii());
}

/** */
void DCDebug::slotSaveBacktrace()
{
	QString s = QFileDialog::getSaveFileName(
                    QString(),
                    "Text (*.txt)",
                    this,
                    "save file dialog"
                    "Choose a filename to save under" );

	if ( s.isEmpty() )
	{
		return;
	}

	QFile *file = new QFile(s);

	if ( !file->open( IO_WriteOnly ) )
	{
        	QMessageBox::critical( this, tr("Save error"),
                	tr("Can't open file '%1' for writing.").arg(s) );
	}
	else
	{
		file->writeBlock( TextEdit_LOG->text().ascii(), TextEdit_LOG->text().length() ); 
		file->close();
	}

	delete file;
}

/** http callback function */
int DCDebug::HttpCallBack( CDCMessage * dcmsg )
{
	switch ( dcmsg->m_eType )
	{
		case DC_MESSAGE_CONNECTION_STATE:
		{
			CMessageConnectionState *msg = (CMessageConnectionState*)dcmsg;

			if ( msg->m_eState == estDISCONNECTED )
			{
				m_bTransfer = false;
			}

			break;
		}

		case DC_MESSAGE_TRANSFER:
		{
			break;
		}

		default:
		{
			break;
		}
	}

	if ( dcmsg )
	{
		delete dcmsg;
	}

	return 0;
}

#ifndef WIN32
/** */
void print_signal_name( const int signum )
{
	switch ( signum )
	{
#ifdef SIGPIPE
		case SIGPIPE:
		{
			DPRINTF("SIGPIPE");
			break;
		}
#endif

#ifdef SIGTERM
		case SIGTERM:
		{
			DPRINTF("SIGTERM");
			break;
		}
#endif

#ifdef SIGHUP
		case SIGHUP:
		{
			DPRINTF("SIGHUP");
			break;
		}
#endif

#ifdef SIGINT
		case SIGINT:
		{
			DPRINTF("SIGINT");
			break;
		}
#endif

#ifdef SIGQUIT
		case SIGQUIT:
		{
			DPRINTF("SIGQUIT");
			break;
		}
#endif

#ifdef SIGSEGV
		case SIGSEGV:
		{
			DPRINTF("SIGSEGV");
			break;
		}
#endif

#ifdef SIGFPE
		case SIGFPE:
		{
			DPRINTF("SIGFPE");
			break;
		}
#endif

#ifdef SIGILL
		case SIGILL:
		{
			DPRINTF("SIGILL");
			break;
		}
#endif

#ifdef SIGABRT
		case SIGABRT:
		{
			DPRINTF("SIGABRT");
			break;
		}
#endif

#ifdef SIGTRAP
		case SIGTRAP:
		{
			DPRINTF("SIGTRAP");
			break;
		}
#endif

		default:
		{
			DPRINTF("signal %d", signum);
			break;
		}
	}
}

/** */
bool can_we_handle_signal( const int signum )
{
	bool ok = false;
	
	struct sigaction query_action;
	
	if ( sigaction(signum,NULL,&query_action) == -1 )
	{
		DPRINTF("sigaction returned -1 ");
	}
	else if ( query_action.sa_handler == SIG_IGN )
	{
		DPRINTF("signal is being ignored ");
	}
	else if ( query_action.sa_handler == SIG_DFL )
	{
		// print_signal_name(signum);
		// DPRINTF(" query_action.sa_handler=%d SIG_DFL=%d\n", query_action.sa_handler, SIG_DFL);
		ok = true;
	}
	else
	{
		DPRINTF("signal already handled ");
	}
	
	if ( ok == false )
	{
		DPRINTF("not handling signal ");
		print_signal_name(signum);
		DPRINTF("\n");
	}
	
	return ok;
}

/*!
 *\brief	this handler will probably evolve into 
 *		something better.
 */
static void crash_handler(int sig)
{
	pid_t pid;
	static volatile unsigned long crashed_ = 0;
                             
	/*
	 * let's hope startup_dir and argv0 aren't trashed.
	 * both are defined in main.c.
	 */
	if (sig < 0 || sig > 32) { /* what's that ? */
//		char *buf = g_strdup_printf(
//				_("Caught strange signal (%d). This is probably\n"
//				  "due to a broken compilation; Exiting."), sig);
//		ay_do_error( _("Error"), buf );
//		g_free(buf);
//		printf("SIGNAL_HANDLER: unknown signal %d\n",sig);
		_exit(EXIT_FAILURE);
	}
	/*
	 * besides guarding entrancy it's probably also better 
	 * to mask off signals
	 */
	if (crashed_) _exit(EXIT_FAILURE);

	crashed_++;

//	printf("CRASH CRASH CRASH \n");

	if (0 == (pid = fork()))
	{
		char buf[256];
		char *args[6];
		CString cp = g_pConfig->GetConfigPath();

		args[0] = DCDebug::arg_0.Data(); 
		args[1] = "-c";
		args[2] = cp.Data();
		args[3] = "-C";
		snprintf(buf, sizeof(buf), "%d,%d,%s", getppid(), sig, DCDebug::arg_0.Data());
		args[4] = buf;
		args[5] = NULL;

		chdir(DCDebug::startup_dir.Data());
		setgid(getgid());
		setuid(getuid());
		execvp(DCDebug::arg_0.Data(), args);
	} else {
		waitpid(pid, NULL, 0);
		_exit(EXIT_FAILURE);
	}

	_exit(EXIT_FAILURE);
}

/*!
 *\brief	install crash handlers
 */
void crash_install_handlers(void)
{
	bool segv = false, fpe = false, ill = false, abrt = false, trap = false;
	
	struct sigaction new_action;
	new_action.sa_handler = crash_handler;
	sigemptyset( &new_action.sa_mask );
	new_action.sa_flags = 0;
	
#ifdef SIGSEGV
	segv = can_we_handle_signal( SIGSEGV );
	if ( segv )
	{
		sigaddset( &new_action.sa_mask, SIGSEGV );
	}
#endif
	
#ifdef SIGFPE
	fpe = can_we_handle_signal( SIGFPE );
	if ( fpe )
	{
		sigaddset( &new_action.sa_mask, SIGFPE );
	}
#endif

#ifdef SIGILL
	ill = can_we_handle_signal( SIGILL );
	if ( ill )
	{
		sigaddset( &new_action.sa_mask, SIGILL );
	}
#endif

#ifdef SIGABRT
	abrt = can_we_handle_signal( SIGABRT );
	if ( abrt )
	{
		sigaddset( &new_action.sa_mask, SIGABRT );
	}
#endif

#ifdef SIGTRAP
	trap = can_we_handle_signal( SIGTRAP );
	if ( trap )
	{
		sigaddset( &new_action.sa_mask, SIGTRAP );
	}
#endif

#ifdef SIGSEGV
	if ( segv )
	{
		if ( sigaction(SIGSEGV,&new_action,NULL) == -1 )
		{
			printf("Error installing SIGSEGV handler\n");
		}
	}
#endif

#ifdef SIGFPE
	if ( fpe )
	{
		if ( sigaction(SIGFPE,&new_action,NULL) == -1 )
		{
			printf("Error installing SIGFPE handler\n");
		}
	}
#endif

#ifdef SIGILL
	if ( ill )
	{
		if ( sigaction(SIGILL,&new_action,NULL) == -1 )
		{
			printf("Error installing SIGILL handler\n");
		}
	}
#endif

#ifdef SIGABRT
	if ( abrt )
	{
		if ( sigaction(SIGABRT,&new_action,NULL) == -1 )
		{
			printf("Error installing SIGABRT handler\n");
		}
	}
#endif

#ifdef SIGTRAP
	if ( trap )
	{
		if ( sigaction(SIGTRAP,&new_action,NULL) == -1 )
		{
			printf("Error installing SIGTRAP handler\n");
		}
	}
#endif
	//sigprocmask(SIG_UNBLOCK, &mask, 0);
}
#endif

