/***************************************************************************
                          dcfilebrowser.cpp  -  description
                             -------------------
    begin                : Fre Nov 29 2002
    copyright            : (C) 2002 by Mathias Küster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "dcfilebrowser.h"

#include <qapplication.h>
#include <qheader.h>
#include <qlistview.h>
#include <qlabel.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qfiledialog.h>
#include <qtoolbutton.h>
#include <qcombobox.h>
#include <qtooltip.h>
#include <qclipboard.h>
#include <qmessagebox.h>
#include <qworkspace.h>
#include <qptrlist.h>

#include "dcmenuhandler.h"
#include "dcconfig.h"
#include "dciconloader.h"
#include "dcfilebrowseritems.h"
#include "dcfilelistdecompressor.h"
#include "dcevent.h"

// for user menu commands
#include "dcclient.h"
#include "dcconnectionmanager.h"

#include "dcguiutils.h"

// for searching by TTH
#include "dchubsearch.h"

#include "dcfiletool.h"

#include <dclib/dcos.h>
#include <dclib/core/cstring.h>
#include <dclib/core/cdir.h>
#include <dclib/core/cxml.h>

// for file type icons
#include <dclib/cfilemanager.h>

/** */
DCFileBrowser::DCFileBrowser( QWidget* parent, const char *name, int wflags, bool allowopen )
	: DCDialogFileBrowser( parent, name, wflags )
{
	m_bAllowOpen = allowopen;
	m_bAllowDownload = true;
	
	// set caption
	setCaption(tr("Filebrowser"));
	setIcon( g_pIconLoader->GetPixmap(eiVIEW_SIDETREE) );
	
	// set width mode
	ListView_DIRECTORY->setColumnWidthMode( 0, QListView::Maximum );
	ListView_FILES->setColumnWidthMode( 0, QListView::Maximum );
	ListView_FILES->setColumnWidthMode( 1, QListView::Maximum );
	ListView_FILES->setColumnWidthMode( 2, QListView::Maximum );
	ListView_FILES->setColumnWidthMode( 3, QListView::Maximum );
	ListView_FILES->setColumnWidthMode( 4, QListView::Maximum );
	
	// set right alignment on size columns
	ListView_DIRECTORY->setColumnAlignment( 1, Qt::AlignRight );
	ListView_FILES->setColumnAlignment( 1, Qt::AlignRight );
	ListView_FILES->setColumnAlignment( 2, Qt::AlignRight );
	
	// hide the column header - only if not using size column
	if ( g_pConfig->GetFolderSizesInLeftPane() == false )
	{
		ListView_DIRECTORY->header()->hide();
	}

	m_nShareSize = 0;
	m_nFileCount = 0;

	m_pDecomp = 0;

	InitDocument();
}

/** */
DCFileBrowser::~DCFileBrowser()
{
	ClearView();
	
	if ( m_pDecomp != 0 )
	{
		/* there is no way to stop CBZ::Decompress so we are a bit stuck */
		printf("~DCFileBrowser: waiting for thread to finish\n");
		m_pDecomp->wait();
		printf("~DCFileBrowser: thread finished, cleaning up\n");
		if ( m_pDecomp->getData() != 0 )
		{
			delete m_pDecomp->getData();
		}
		if ( m_pDecomp->recompressed != 0 )
		{
			delete m_pDecomp->recompressed;
		}
		delete m_pDecomp;
	}
}
                                                                                   
void DCFileBrowser::InitDocument()
{
	connect( ListView_DIRECTORY, SIGNAL(selectionChanged()), this, SLOT(slotCurrentChangedDirectory()) );
	connect( ListView_DIRECTORY, SIGNAL(expanded(QListViewItem*)), this, SLOT(slotExpandedDirectory(QListViewItem*)) );
	connect( ListView_DIRECTORY, SIGNAL(collapsed(QListViewItem*)), this, SLOT(slotCollapsedDirectory(QListViewItem*)) );
	connect( ListView_DIRECTORY, SIGNAL(contextMenuRequested( QListViewItem *, const QPoint &, int )), this, SLOT(slotRightButtonClickedDirectory(QListViewItem*, const QPoint &, int )) );
	connect( ListView_FILES, SIGNAL(contextMenuRequested( QListViewItem *, const QPoint &, int )), this, SLOT(slotRightButtonClickedFiles(QListViewItem*, const QPoint &, int )) );
	connect( ListView_FILES, SIGNAL(doubleClicked( QListViewItem *, const QPoint &, int)), this, SLOT(slotItemDoubleClicked( QListViewItem *, const QPoint &, int )) );

 	connect( ToolButton_OPEN, SIGNAL(clicked()), this, SLOT(slotFileOpen()) );
 	connect( ToolButton_SAVE, SIGNAL(clicked()), this, SLOT(slotFileSave()) );
	connect( ToolButton_GOTO_USER, SIGNAL(clicked()), this, SLOT(slotGotoUser()) );
	
	ToolButton_OPEN->setEnabled(m_bAllowOpen);
	ToolButton_GOTO_USER->setEnabled(false);
}

void DCFileBrowser::customEvent( QCustomEvent * event )
{
	if ( event->type() == EVENT_FILELIST_DECOMPRESSED )
	{
		if ( ((DC_FilelistDecompressedEvent*)event)->ok )
		{
			if ( m_pDecomp->recompressed != 0 )
			{
				m_ListData.SetSize(0);
				m_ListData.Append( m_pDecomp->recompressed->Data(), m_pDecomp->recompressed->Size() );
				delete m_pDecomp->recompressed;
			}
			ContinueInitTree( m_pDecomp->getData() );
		}
		else
		{
			QMessageBox::critical(
				this,
				tr("Failed to load filelist"),
				tr("Unable to load") + " " + m_sFileName
			);
			setCaption(tr("Filebrowser"));
			EnableGuiElements();
		}
		delete m_pDecomp;
		m_pDecomp = 0;
	}
	else if ( event->type() == EVENT_OPEN_FILELIST_FOLDER )
	{
		DCFileBrowserListItem * item = ((DC_OpenFilelistFolderEvent*)event)->m_pItem;
		
		ListView_DIRECTORY->clearSelection();
		ListView_DIRECTORY->ensureItemVisible( item );
		ListView_DIRECTORY->setSelected( item, true );
		ListView_DIRECTORY->setCurrentItem( item );
	}
	else
	{
		QWidget::customEvent( event );
	}
}

/** */
void DCFileBrowser::ClearView()
{
	ListView_FILES->clear();

	QListViewItemIterator it( ListView_DIRECTORY );

	for ( ; it.current(); it++ )
	{
		((DCFileBrowserListItem*)it.current())->m_pFileList.clear();
	}

	ListView_DIRECTORY->clear();
}

/** */
void DCFileBrowser::slotFileOpen()
{
	QString file;
	if ( m_sFileName.isEmpty() )
	{
		file = QString::fromAscii(g_pConfig->GetFileListPath().Data());
	}
	else
	{
		file = m_sFileName;
	}
	
	QString s = QFileDialog::getOpenFileName(
		file,
		tr("Modern XML Filelists") + " (*.xml.bz2);;" +
		tr("Modern XML Filelists (uncompressed)") + " (*.xml);;" +
		tr("Old BZ2 Text Filelists") + " (*.bz2);;" +
		tr("Very old HE3 Text Filelists") + " (*.DcLst);;" +
		tr("Old Valknut Filelists") + " (*.filelist);;" +
		tr("All files") + " (*)",
		this,
		tr("open filelist dialog"),
		tr("Choose a file") );

	if ( !s.isEmpty() )
	{
		InitTree(QString::null,QString::null,QString::null,s);
	}
}

/** */
void DCFileBrowser::slotFileSave()
{
	if ( m_ListData.Size() == 0 )
	{
		QMessageBox::critical(
			this,
			tr("Cannot save file"),
			tr("No data available to save.")
		);
		return;
	}
	
	QString s = QFileDialog::getSaveFileName(
                    m_sFileName,
                    QString(),
                    this,
                    tr("save file dialog"),
                    tr("Choose a filename to save under") );

	if ( s.isEmpty() )
	{
		return;
	}
	
	if ( m_ListData.SaveToFile(s.ascii()) == false )
	{
		QMessageBox::critical(
			this,
			tr("Error saving file"),
			tr("Failed to save") + " " + s
		);
	}
}

/** */
void DCFileBrowser::slotCurrentChangedDirectory()
{
	DCFileItem * FileItem;
	QString s;
	QPtrList<QListViewItem> selitems;
	ulonglong size=0;
	int items = 0;
	int citems = 0;
	QListViewItem * item = 0;

	// clear filelist
	ListView_FILES->clear();

	// check if only one dir selected
	if ( DCGuiUtils::SelectedItems( ListView_DIRECTORY, selitems ) == 1 )
	{
		item = selitems.first();

		for ( FileItem = ((DCFileBrowserListItem*)item)->m_pFileList.first(); FileItem; FileItem = ((DCFileBrowserListItem*)item)->m_pFileList.next() )
		{
			if ( (g_pConfig->GetFoldersInRightPane() == false) && FileItem->m_bIsDir )
			{
				// do not show it
				continue;
			}
			
			size += FileItem->m_nSize;

			DC_FBListViewItem *item1 = new DC_FBListViewItem(ListView_FILES);
			item1->myvalue = FileItem->m_nSize;
			item1->mycol   = 1;
			item1->m_bSortTop = false;
			item1->pDirItem = ((DCFileBrowserListItem*)item);
			
			if ( FileItem->m_bIsDir )
			{
				s = tr("Folder");
				item1->setPixmap(0,m_FolderPixmap);
				
				if ( g_pConfig->GetFoldersInRightPaneOnTop() )
				{
					item1->m_bSortTop = true;
				}
			}
			else
			{
				s = CDir::Extension(FileItem->m_sName.ascii()).Data();

				if ( !s.isEmpty() )
				{
					s += " ";
					s += tr("file");
				}
			
				eFileTypes type = CFileManager::Instance()->GetFileType(FileItem->m_sName.ascii());
				QPixmap icon;
				
				switch (type)
				{
					case eftMP3:
						icon = g_pIconLoader->GetPixmap(eiFILETYPE_MP3);
						break;
					case eftARCHIVE:
						icon = g_pIconLoader->GetPixmap(eiFILETYPE_ARCHIVE);
						break;
					case eftDOCUMENT:
						icon = g_pIconLoader->GetPixmap(eiFILETYPE_DOCUMENT);
						break;
					case eftAPPLICATION:
						icon = g_pIconLoader->GetPixmap(eiFILETYPE_APPLICATION);
						break;
					case eftPICTURE:
						icon = g_pIconLoader->GetPixmap(eiFILETYPE_PICTURE);
						break;
					case eftVIDEO:
						icon = g_pIconLoader->GetPixmap(eiFILETYPE_VIDEO);
						break;
					default:
						icon = g_pIconLoader->GetPixmap(eiFILETYPE_UNKNOWN);
						break;
				}
				
				item1->setPixmap( 0, icon );
				
				items++;
			}

			item1->setText(0,FileItem->m_sName);
			item1->setText(1,DCGuiUtils::GetSizeString(FileItem->m_nSize));
			item1->setText(2,QString::number(FileItem->m_nSize));
			item1->setText(3,s);
			item1->setText(4,FileItem->m_sHash);
		}
	}

	if ( item )
	{
		citems = item->childCount();
	}

	TextLabel_STATUS->setText( QString().setNum(items+citems) + " " + tr("Items") + " - " + 
					QString().setNum(items) + " " + tr("Files") +
					" (" + DCGuiUtils::GetSizeString(size) + " " + tr("Total") + ") - " +
					QString().setNum(citems) + " " + tr("Directories") );
}

/** */
void DCFileBrowser::slotExpandedDirectory( QListViewItem * item )
{
	if ( item->parent() != NULL )
		item->setPixmap(0,g_pIconLoader->GetPixmap(eiFOLDER_BLUE_OPEN));
}

/** */
void DCFileBrowser::slotCollapsedDirectory( QListViewItem * item )
{
	if ( item->parent() != NULL )
		item->setPixmap(0,g_pIconLoader->GetPixmap(eiFOLDER_BLUE));
}

/** */
void DCFileBrowser::slotRightButtonClickedDirectory( QListViewItem * item, const QPoint &, int column )
{
	CreateMenu(item,true,column);
}

/** */
void DCFileBrowser::slotRightButtonClickedFiles( QListViewItem * item, const QPoint &, int column )
{
	CreateMenu(item,false,column);
}

/** */
void DCFileBrowser::AddDirectory( QListViewItem * item, QString name )
{
	((DCFileBrowserListItem*)item)->m_pFileList.setAutoDelete(true);
	((DCFileBrowserListItem*)item)->m_sName = name;

	item->setText(0,name);
	item->setPixmap(0,m_FolderPixmap);
	
	((DCFileBrowserListItem*)item)->m_nBytes = 0;
}

/** */
void DCFileBrowser::AddFile( QListViewItem * item, DCFileItem * fileitem )
{
	// increment filecounter
	if ( fileitem->m_bIsDir == false )
	{
		m_nFileCount++;
		m_nShareSize += fileitem->m_nSize;
	}

	// add fileitem to filelist
	((DCFileBrowserListItem*)item)->m_pFileList.append(fileitem);
	
	// add to folder contents size
	((DCFileBrowserListItem*)item)->m_nBytes += fileitem->m_nSize;
	
	DCFileItem * pFileItem = ((DCFileBrowserListItem*)item)->m_pFileItem;
	
	if ( pFileItem != 0 )
	{
		pFileItem->m_nSize = ((DCFileBrowserListItem*)item)->m_nBytes;
	}
}

/** */
void DCFileBrowser::InitTree( QString nick, QString hubname, QString hubhost, QString filename, QString jumpto, QStringList dirs )
{
	if ( filename.isEmpty() )
	{
		return;
	}
	
	if ( m_pDecomp != 0 )
	{
		QMessageBox::critical(
			this,
			tr("Cannot open list now"),
			tr("Filelist browser is busy")
		);
		return;
	}
	
	// store values
	m_sNick     = nick;
	m_sHubName  = hubname;
	m_sHubHost  = hubhost;
	m_sFileName = filename;
	m_sJumpTo   = jumpto;
	m_lDownloadDirs = dirs;
	
	// first check if filename is our own list
	if ( filename == "/dev/null/ownfilelist" )
	{
		m_bAllowDownload = false;
		GetOwnList();
		
		//printf("Size of own list = %lu\n", m_ListData.Size());
		
		if ( (m_sFileName == "myfiles.xml.bz2") || (m_sFileName == "myfiles.txt.bz2") || (m_sFileName == "myfiles.DcLst") )
		{
			// nothing, it all got moved into the thread
		}
		else
		{
			QMessageBox::critical(
				this,
				tr("Error opening own filelist"),
				tr("Failed to open your own filelist")
			);
			
			return;
		}
		
		DisableGuiElements();
		m_pDecomp = new DCFilelistDecompressor( this, &m_ListData );
		m_pDecomp->start();
		return;
	}
	else
	{
		m_bAllowDownload = true;
		if ( !QFile::exists(filename) )
		{
			QMessageBox::critical(
				this,
				tr("Error opening filelist"),
				tr("File") + " " + filename + " " + tr("does not exist.")
			);
			
			return;
		}
		
		DisableGuiElements();
		m_pDecomp = new DCFilelistDecompressor( this, filename.ascii() );
		m_pDecomp->start();
		return;
	}
}

/** */
void DCFileBrowser::ContinueInitTree( CString * data )
{
	/* A bz2 file that decompressed to zero bytes probably... */
	if ( (data == 0) || (data->Length() == 0) )
	{
		QMessageBox::critical(
			this,
			tr("Error opening filelist"),
			m_sFileName + " " + tr("is likely not a valid filelist")
		);
		setCaption(tr("Filebrowser"));
		EnableGuiElements();
		return;
	}

	QListViewItem * item;
	
	ClearView();
	
	// reset count values
	m_nShareSize = 0;
	m_nFileCount = 0;

	// get folder pixmap
	m_FolderPixmap = g_pIconLoader->GetPixmap(eiFOLDER_BLUE);

	// disable updates
	ListView_DIRECTORY->setUpdatesEnabled(false);
	
	// parse filename
	
	QFileInfo fi(m_sFileName);
	QString fname = fi.fileName();
	
	if ( (m_sNick.isEmpty()) || (m_sHubName.isEmpty()) || (m_sHubHost.isEmpty()) )
	{
		int i = fname.find('@');
		if ( i != -1 )
		{
			m_sNick = fname.left(i);
			m_sHubHost = fname.mid(i+1);
			m_sHubHost.remove(".xml.bz2");
			m_sHubHost.remove(".bz2");
			m_sHubHost.remove(".xml");
			m_sHubHost.remove(".DcLst");
			m_sHubHost.remove(".filelist");
			
			// replace _ with : before port number
			int i2 = m_sHubHost.findRev('_');
			if ( i2 != -1 )
			{
				bool ok;
				m_sHubHost.mid(i2+1).toInt(&ok);
				if ( ok )
				{
					m_sHubHost = m_sHubHost.left(i2) + ":" + m_sHubHost.mid(i2+1);
				}
			}
			
			if ( m_sHubName.isEmpty() )
			{
				m_sHubName = m_sHubHost;
			}
		}
	}
	
	// set dialog caption
	
	if ( m_sNick.isEmpty() )
	{
		fname.remove(".xml.bz2");
		fname.remove(".bz2");
		fname.remove(".xml");
		fname.remove(".DcLst");
		fname.remove(".filelist");
		m_sNick = fname;
	}
	
	if ( m_sHubName.isEmpty() )
	{
		setCaption( m_sNick + " - " + tr("Filebrowser") );
	}
	else
	{
		setCaption( m_sNick + " - " + tr("Filebrowser") + " [" + m_sHubName + "]" );
	}
	
	// create root item
	item = new DCFileBrowserListItem( ListView_DIRECTORY );
	item->setText(0,tr("Root Directory"));
	item->setPixmap(0,g_pIconLoader->GetPixmap(eiFOLDER_RED));
	((DCFileBrowserListItem*)item)->m_pFileItem = 0;
	((DCFileBrowserListItem*)item)->m_pFileList.setAutoDelete(true);
	
	if ( data->Left(9).Find("<?xml") != -1 )
		InitXMLTree(item,data);
	else
		InitTXTTree(item,data);

	delete data;

	// update summary
	TextLabel_SUMMARY->setText( QString().setNum(m_nFileCount) + " " + tr("Files") + " (" + 
					DCGuiUtils::GetSizeString(m_nShareSize) + " " + tr("Total") + ")" );

	// update tooltip
	QToolTip::add( TextLabel_SUMMARY, QString().setNum(m_nShareSize) + " B" );

	// update view
	ListView_DIRECTORY->setUpdatesEnabled(true);
	ListView_DIRECTORY->triggerUpdate();

	// open first tree
	ListView_DIRECTORY->setOpen(ListView_DIRECTORY->firstChild(),true);
	
	EnableGuiElements();
	
	/* download any directories */
	for ( QStringList::const_iterator it = m_lDownloadDirs.constBegin(); it != m_lDownloadDirs.constEnd(); ++it )
	{
		DownloadDirectory( *it );
	}
	m_lDownloadDirs.clear();
	
	/* jump to target file or folder */
	if ( !m_sJumpTo.isEmpty() )
	{
		JumpTo( m_sJumpTo );
	}
}

/** */
void DCFileBrowser::DisableGuiElements()
{
	setCaption(tr("Processing filelist..."));
	ToolButton_OPEN->setEnabled(false);
	ToolButton_SAVE->setEnabled(false);
	ToolButton_GOTO_USER->setEnabled(false);
	ListView_DIRECTORY->setEnabled(false);
	ListView_FILES->setEnabled(false);
}

void DCFileBrowser::EnableGuiElements()
{
	ToolButton_OPEN->setEnabled(m_bAllowOpen);
	ToolButton_SAVE->setEnabled(true);
	ListView_DIRECTORY->setEnabled(true);
	ListView_FILES->setEnabled(true);
	
	if ( !m_bAllowOpen || m_sNick.isEmpty() || m_sHubHost.isEmpty() )
	{
		ToolButton_GOTO_USER->setEnabled(false);
	}
	else
	{
		ToolButton_GOTO_USER->setEnabled(true);
	}
}

/** */
void DCFileBrowser::InitTXTTree( QListViewItem * item, CString * data )
{
	QString s1,sname,slength;
	long i,j,d;
	int depth;

	i = j = 0;

	while((i=data->Find("\xD\xA",j))>=0)
	{
		s1 = data->Mid(j,i-j).Data();

		depth=0;
		while( (s1.find("\x9",depth)) != -1 )
			depth++;
		s1.remove( "\x9" );

		if (item)
			while ( depth < item->depth() )
				if ( (item=item->parent()) == 0 )
					break;

		if ( s1.isEmpty() )
			s1 = "\\";

		if ( !s1.isEmpty() )
		{
			d = s1.findRev("|");

			if ( d != -1 ) // parse file entry
			{
				DCFileItem * FileItem = new DCFileItem();

				FileItem->m_sName = s1.mid(0,d);
				FileItem->m_nSize = s1.right(s1.length()-d-1).toULongLong();
				FileItem->m_bIsDir = false;
				
				AddFile( item, FileItem );
			}
			else // parse dir entry
			{
				DCFileItem * FileItem = new DCFileItem();
				
				FileItem->m_sName = s1;
				FileItem->m_nSize = 0;
				FileItem->m_bIsDir = true;
				
				AddFile( item, FileItem );
				
				if ( item == 0 )
					item = new DCFileBrowserListItem( ListView_DIRECTORY );
				else
					item = new DCFileBrowserListItem( item );
				
				((DCFileBrowserListItem*)item)->m_pFileItem = FileItem;

				AddDirectory( item, s1 );
			}
		}

		j = i+2;
	}
	
	// fixup the folder sizes
	QListViewItem * cur = ListView_DIRECTORY->firstChild();
	while ( cur )
	{
		CalcDirSize( cur );
		
		cur = cur->nextSibling();
	}
}

/** */
void DCFileBrowser::InitXMLTree( QListViewItem * item, CString * data )
{
	CXml * xml = new CXml();
	
	// QString::length() is not the length of the UTF-8 data
	// but CString::Length() is
	if ( xml->ParseMemory(data->Data(),data->Length()) && xml->DocFirstChild() )
	{
		// maybe we should just parse the first FileListing?
		do
		{
			if ( (xml->Name() == "FileListing") && xml->FirstChild() )
			{
				ParseXMLTree(xml,item);
				xml->Parent();
			}
		}
		while ( xml->NextNode() );
	}

	delete xml;
}

/** */
void DCFileBrowser::ParseXMLTree( CXml * xml, QListViewItem * item )
{
	CString s;

	do
	{
		if ( xml->Name() == "File" )
		{
			DCFileItem * FileItem = new DCFileItem();
			s = xml->Prop("Name");
			FileItem->m_sName = QString::fromUtf8(s.Data());
			FileItem->m_nSize = xml->Prop("Size").asULL();
			FileItem->m_sHash = xml->Prop("TTH").Data();
			FileItem->m_bIsDir = false;
			
			AddFile(item,FileItem);
		}
		else if ( xml->Name() == "Directory" )
		{
			QListViewItem * item1;
			
			item1 = new DCFileBrowserListItem( item );
			((DCFileBrowserListItem*)item1)->m_pFileItem = 0;
			
			s = xml->Prop("Name");
			AddDirectory(item1,QString::fromUtf8(s.Data()));
			
			if ( xml->FirstChild() )
			{
				ParseXMLTree(xml,item1);
				xml->Parent();
			}
			
			DCFileItem * FileItem = new DCFileItem();
			FileItem->m_sName = QString::fromUtf8(s.Data());
			FileItem->m_nSize = ((DCFileBrowserListItem*)item1)->m_nBytes;
			FileItem->m_bIsDir = true;
			AddFile(item,FileItem);
			((DCFileBrowserListItem*)item1)->m_pFileItem = FileItem;
						
			if ( g_pConfig->GetFolderSizesInLeftPane() )
			{
				item1->setText( 1, DCGuiUtils::GetSizeString(((DCFileBrowserListItem*)item1)->m_nBytes) );
			}
		}
	}
	while ( xml->NextNode() );
}

/** */
QString DCFileBrowser::CreateRemotePath( QListViewItem * item )
{
	QString s;
	QListViewItem * pitem;
	
	if ( !item )
	{
		return s;
	}

	pitem = item;
	s = ((DCFileBrowserListItem*)pitem)->m_sName;
	
	while ( (pitem=pitem->parent()) != 0 )
	{
		// check for root entry
		if ( pitem->parent() != 0 )
		{
			s = ((DCFileBrowserListItem*)pitem)->m_sName + "\\" + s;
		}
	}

	s.replace( "\\\\\\\\", "\\" );

	return s;
}

/** */
void DCFileBrowser::CreateMenu( QListViewItem *, bool direntry, int column )
{
	int id,i;
	QString s;
	ulonglong size;
	QString rootPath  = QString::null;
	QString localName = QString::null;
	QPtrList<QListViewItem> selitems;
	QListViewItem * curitem;
	bool hasFolderItems = false;
	DCClient * client = 0;
	QMap<int, DC_UserMenuCommand*> addedcommands;
	
	if ( direntry )
	{
		i = DCGuiUtils::SelectedItems( ListView_DIRECTORY, selitems );
	}
	else
	{
		i = DCGuiUtils::SelectedItems( ListView_FILES, selitems );
		
		QPtrListIterator<QListViewItem> plit ( selitems );
		
		QListViewItem * cur = 0;
		
		while ( (cur = plit.current()) != 0 )
		{
			++plit;
			
			if ( cur->text(3) == tr("Folder") )
			{
				hasFolderItems = true;
				break;
			} 
		}
	}

	// nothing on no selection
	if ( i == 0 )
	{
		return;
	}
	else
	{
		curitem = selitems.first();
	}

/* TODO readadd test
	if ( direntry && (item->parent() == 0) )
	{
		return;
	}
*/
	QPopupMenu * m = new QPopupMenu(this);

	DCMenuHandler::InsertMenu(m, emiDOWNLOAD, m_bAllowDownload);
	DCMenuHandler::InsertMenu(m, emiDOWNLOAD_TO, m_bAllowDownload);
	// disable 'download as' on directories
	DCMenuHandler::InsertMenu(m, emiDOWNLOAD_AS, m_bAllowDownload && ( !direntry && !hasFolderItems && (i == 1) ));
	DCMenuHandler::InsertMenu(m, emiDOWNLOAD_IN, m_bAllowDownload && ( !direntry && !hasFolderItems) );
	DCMenuHandler::InsertMenu(m, emiSEPARATOR);
	int search_tth = m->insertItem( QIconSet(g_pIconLoader->GetPixmap(eiFILEFIND)), tr("Search by TTH") );
	m->setItemEnabled( search_tth, m_bAllowOpen && !direntry && curitem && !(curitem->text(4).isEmpty()) );
	DCMenuHandler::InsertMenu(m, emiSEPARATOR);
	DCMenuHandler::InsertMenu(m, emiCOPY_COLUMN_TO_CLIPBOARD, !direntry );
	DCMenuHandler::InsertMenu(m, emiCOPY_ROW_TO_CLIPBOARD, !direntry );
	DCMenuHandler::InsertMenu(m, emiCOPYMAGNETLINK, (!direntry && !hasFolderItems) );
	DCMenuHandler::InsertMenu(m, emiSEPARATOR);

	client = g_pConnectionManager->GetClientForHub( m_sHubName.ascii(), m_sHubHost.ascii() );
	
	if ( client != 0 )
	{
		addedcommands = client->AddMenuCommands( m, euccFilelist );
	}
	
	id = m->exec(QCursor::pos());

	delete m;
	
	if ( id == emiCOPY_COLUMN_TO_CLIPBOARD )
	{
		QString s;

                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			s += curitem->text(column);
			s += "\n";
		}

		s = s.stripWhiteSpace();
		QApplication::clipboard()->setText(s);
	}
	else if ( id == emiCOPY_ROW_TO_CLIPBOARD )
	{
		int idx;
		QString s;
		
                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			for( idx = 0; idx < ListView_FILES->columns(); idx++ )
			{
				s += curitem->text(idx);
				s += " ";
			}

			s += "\n";
		}

		s = s.stripWhiteSpace();
		QApplication::clipboard()->setText(s);
	}
	else if ( id == emiCOPYMAGNETLINK )
	{
		//magnet:?xt=urn:tree:tiger:EOSA5AGTL5SD3VWCF3R2OH2WMGXV3S3R7SYN4YA&xl=708780032&dn=FC-6-i386-disc1.iso
		QString all;
		
		for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
		{
			// add tth
			all += "magnet:?xt=urn:tree:tiger:";
			all += curitem->text(4);
			
			// add filesize
			all += "&xl=";
			all += curitem->text(2);
			
			// add filename
			all += "&dn=";
			QString filename = curitem->text(0);
			
			QUrl::encode(filename);
			
			//fixme DC++ replaces spaces with + character
			//filename = filename.replace(' ','+');
			
			filename.replace("%20","+");
			
			all += filename;
			
			all += "\n";
		}
		
		// remove trailing "\n"
		all = all.stripWhiteSpace();
		
		QApplication::clipboard()->setText(all);
	}
	// Check if the user wants a non-default destination
	else if ( id == emiDOWNLOAD_TO )
	{
		rootPath = QFileDialog::getExistingDirectory(QString(), this, "dldia",
                                                     tr("Select destination"));
		if ( rootPath.isEmpty() )
		{
			// If the user cancel, then we don't download
			return;
		}
	}
	// Should the file be renamed
	else if(id == emiDOWNLOAD_AS)
	{
		rootPath = QFileDialog::getSaveFileName(selitems.first()->text(0), QString(), this, "dldia",
                                                tr("Select a filename"));
		if ( rootPath.isEmpty() )
		{
			return;
		}

		QFileInfo fi(rootPath);
		rootPath = fi.dirPath();
		localName = fi.fileName();

		if ( rootPath.isEmpty() || localName.isEmpty() )
		{
			return;
		}
	}
	else if ( id == emiDOWNLOAD_IN )
	{
		QString localrootpath;
		QString localname;
		QString localpath;
		QString remotePath;

		if ( !(curitem = selitems.first()) )
			return;

		size = ((DC_QListViewItem *)curitem)->myvalue;
		QString tth = curitem->text(4);

		// all files need equal size and TTH
                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			if ( size != ((DC_QListViewItem *)curitem)->myvalue )
				return;
			if ( tth != curitem->text(4) )
				return;
		}

		if ( DCFileTool::SelectFileSource( this, size, tth, localname, localrootpath, localpath ) == false )
		{
			return;
		}

		QString remotePathAndFile;
                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			s = curitem->text(0);
			
			// get the full remote path
			remotePath = CreateRemotePath(GetDirItem(curitem));
			
			remotePathAndFile  = remotePath;
			remotePathAndFile += "\\";
			remotePathAndFile += s;
					    
			// add transfer to the waitlist
			DCFileTool::CheckFile( this, m_sNick.ascii(), m_sHubName.ascii(), m_sHubHost.ascii(),
				  remotePathAndFile.ascii(), localname.ascii(), localpath.ascii(), localrootpath.ascii(), eltFILE,
		  		  size, curitem->text(4).ascii(), true);
		}
	}
	else if ( id == search_tth )
	{
		DCHubSearch * hubsearch = new DCHubSearch( g_pConnectionManager->GetWorkspace() );
		hubsearch->SetSearchForFile( curitem->text(4), eftHASH );
		hubsearch->show();
		hubsearch->StartSearchWithPrompt();
	}
	else if ( addedcommands.contains( id ) )
	{
		DC_UserMenuCommand * umc = addedcommands[ id ];
		
		QString usercommand = umc->m_sCommand;
		QString origUserCommand = usercommand;
		QString fullfilename, filesize, filesizeshort, filetth, filetype;
		
		int c = 0;
		for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
		{
			// filelist browsers are only for the one nick
			if ( (umc->m_nType == euctRawOnce) && (c > 0) )
			{
				break;
			}
			
			curitem = selitems.at(c);
			
			usercommand = client->replaceCommandTags( origUserCommand, m_sNick );
			
			if ( usercommand.isEmpty() )
			{
				// had a %[line:reason] but dialog was cancelled
				continue;
			}
			
			if ( direntry )
			{
				fullfilename = CreateRemotePath(curitem);
				filesize.setNum(((DCFileBrowserListItem*)curitem)->m_nBytes);
				filesizeshort = DCGuiUtils::GetSizeString( ((DCFileBrowserListItem*)curitem)->m_nBytes );
				filetth = tr("None");
				filetype = tr("Directory");
			}
			else
			{
				fullfilename = CreateRemotePath(GetDirItem(curitem));
				filesize = curitem->text(2);
				filesizeshort = curitem->text(1);
				filetth = curitem->text(4);
				if ( filetth.isEmpty() )
				{
					filetth = tr("None");
				}
				
				if ( curitem->text(3) == tr("Folder") )
				{
					filetype = tr("Directory");
				}
				else
				{
					filetype = tr("File");
					fullfilename += "\\";
					fullfilename += curitem->text(0);
				}
			}
			
			usercommand.replace( "%[file]", fullfilename );
			usercommand.replace( "%[fileFN]", fullfilename );
			
			usercommand.replace( "%[filesize]", filesize );
			usercommand.replace( "%[fileSI]", filesize );
			
			usercommand.replace( "%[filesizeshort]", filesizeshort );
			usercommand.replace( "%[fileSIshort]", filesizeshort );
			
			usercommand.replace( "%[tth]", filetth );
			usercommand.replace( "%[fileTR]", filetth );
			
			usercommand.replace( "%[type]", filetype );
			
			//client->m_pHubChat->AddStatus( usercommand.ascii() );
			client->SendString( usercommand.ascii() );
			
			c++;
		}
	}

	// Is this a download?
	if ( (id == emiDOWNLOAD) || (id == emiDOWNLOAD_TO) || (id == emiDOWNLOAD_AS) )
	{
                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			// directory download from left pane
			if ( direntry )
			{
				s = curitem->text(0);
					
				DownloadPath( rootPath, s, QString::null, curitem );
			}
			else if ( curitem->text(3) == tr("Folder") ) // directory download from right pane
			{
				QListViewItem * diritem = GetDirItem( curitem );
				
				if ( diritem == 0 )
				{
					QMessageBox::critical( this, tr("Cannot download"), tr("Could not find the path for ") + curitem->text(0) );
				}
				else
				{
					DownloadPath( rootPath, diritem->text(0), QString::null, diritem );
				}
			}
			// Single file download
			else
			{
				QString remotePath;
				QString remoteName;
				
				// set localname
				if ( id != emiDOWNLOAD_AS )
				{
					localName = curitem->text(0);
				}

				// get the full remote path
				remotePath = CreateRemotePath(GetDirItem(curitem));
				
				remoteName = curitem->text(0);
				
				DownloadFile( rootPath, QString(), localName, remotePath, remoteName, ((DC_QListViewItem *)curitem)->myvalue, curitem->text(4) );
			}
		}
	}
}

/** */
void DCFileBrowser::DownloadPath( QString rootPath, QString localPath, QString localName, QListViewItem * item )
{
	QString remotePath;
	DCFileItem * FileItem = 0;
	QString s;
	QString remoteFile;

	if ( !item )
	{
		return;
	}
	
	// get the full remote path
	remotePath = CreateRemotePath(item);
	
	for ( FileItem = ((DCFileBrowserListItem*)item)->m_pFileList.first(); FileItem; FileItem = ((DCFileBrowserListItem*)item)->m_pFileList.next() )
	{
		if ( FileItem->m_bIsDir == false )
		{
		    	remoteFile = FileItem->m_sName;
			DownloadFile( rootPath, localPath, localName, remotePath, 
				remoteFile, FileItem->m_nSize, FileItem->m_sHash );
		}
	}

	item = item->firstChild();

	while( item )
	{
		s = item->text(0);
			
		DownloadPath(rootPath,localPath + "\\" + s,localName,item);
		item = item->nextSibling();
	}
}

/** */
void DCFileBrowser::DownloadFile( QString rootPath, QString localPath, QString localName, 
				  QString remotePath, QString remoteName, ulonglong size,
				  QString hash )
{
	// append filename to the remote path
	remotePath += "\\";
	remotePath += remoteName;
			    
	if ( localName == QString::null )
	{
		    localName = remoteName;
	}

	
	DCFileTool::CheckFile(this, m_sNick.ascii(),
		m_sHubName.ascii(),
		m_sHubHost.ascii(),
		remotePath.ascii(),
		(const char*)localName.ascii(),
		(const char*)localPath.ascii(),
		(const char*)rootPath.ascii(), eltFILE,
		size, hash.ascii() );
}

/** Double click to download the item, copied from DCFileBrowser::CreateMenu */
void DCFileBrowser::slotItemDoubleClicked( QListViewItem * item, const QPoint &, int )
{
	QString rootPath = QString::null;
	QString localName = QString::null;
	
	if ( item == 0 ) // null pointer check, just in case
	{
		return;
	}
	
	QListViewItem * diritem = GetDirItem( item );
	
	if ( diritem == 0 )
	{
		QMessageBox::critical( this, tr("Cannot download"), tr("Could not find the path for ") + item->text(0) );
		return;
	}
	
	if ( item->text(3) == tr("Folder") )
	{
		// we cannot open the folder since that would delete the existing
		// QListViewItems, which is not allowed in this slot and causes
		// random crashes
		
		/* QListViewItem * diritem = ListView_DIRECTORY->currentItem();
		
		if ( diritem == 0 )
		{
			printf("Double click a folder needs a ListView_DIRECTORY->currentItem()\n");
			return;
		}
		
		for ( diritem = diritem->firstChild(); diritem != 0; diritem = diritem->nextSibling() )
		{
			if ( diritem->text(0) == item->text(0) )
			{
				ListView_DIRECTORY->clearSelection();
				ListView_DIRECTORY->ensureItemVisible( diritem );
				ListView_DIRECTORY->setSelected( diritem, true );
				ListView_DIRECTORY->setCurrentItem( diritem );
				break;
			}
		} */
		
		if ( g_pConfig->GetOpenFoldersInRightPane() )
		{
			QApplication::postEvent( this, new DC_OpenFilelistFolderEvent( (DCFileBrowserListItem*)diritem ) );
		}
		else if ( m_bAllowDownload )
		{
			// ask for confirmation since this is unusual behaviour
			int confirm = QMessageBox::question(
				this,
				tr("Download contents?"),
				tr("Download the contents of \"") + diritem->text(0) + "\" ?",
				QMessageBox::Yes,
				QMessageBox::No
			);
			
			if ( confirm == QMessageBox::Yes )
			{
				DownloadPath( rootPath, diritem->text(0), localName, diritem );
			}
		}
	}
	else if ( m_bAllowDownload )
	{
		QString remotePath;
		QString remoteName;
		
		// get the full remote path
		remotePath = CreateRemotePath(diritem);
		
		remoteName = item->text(0);
		
		DownloadFile( rootPath, QString(), localName, remotePath, remoteName, ((DC_QListViewItem *)item)->myvalue, item->text(4) );
	}
}

/** Only needed for non-xml filelists */
ulonglong DCFileBrowser::CalcDirSize( QListViewItem * item )
{
	ulonglong total = 0;

	DCFileBrowserListItem * cur = (DCFileBrowserListItem*) item;
	
	QPtrListIterator<DCFileItem> it( cur->m_pFileList );
	for ( ; it.current(); ++it)
	{
		if ( it.current()->m_bIsDir )
		{
			QListViewItem * child = item->firstChild();
			
			while ( child )
			{
				if ( child->text(0) == it.current()->m_sName )
				{
					total += CalcDirSize( child );
					break;
				}
				
				child = child->nextSibling();
			}
		}
		else
		{
			total += it.current()->m_nSize;
		}
	}
	
	DCFileItem * pFileItem = cur->m_pFileItem;
	if ( pFileItem != 0 )
	{
		pFileItem->m_nSize = total;
	}
	
	cur->m_nBytes = total;
	
	if ( g_pConfig->GetFolderSizesInLeftPane() )
	{
		cur->setText( 1, DCGuiUtils::GetSizeString( total ) );
	}
	
	//printf( "CalcDirSize for %s = %lld\n", item->text(0).ascii(), total );
	
	return total;
}

/** */
QListViewItem * DCFileBrowser::GetDirItem( QListViewItem * item )
{
	if ( item == 0 )
	{
		return 0;
	}

	QListViewItem * diritem = ((DC_FBListViewItem*)item)->pDirItem;

	if ( item->text(3) != tr("Folder") )
	{
		return diritem;
	}
	
	if ( diritem == 0 )
	{
		return 0;
	}
	
	for ( diritem = diritem->firstChild(); diritem != 0; diritem = diritem->nextSibling() )
	{
		if ( diritem->text(0) == item->text(0) )
		{
			return diritem;
		}
	}
	
	return 0;
}

/** */
void DCFileBrowser::GetOwnList()
{
	m_ListData.SetSize(0);
	
	if ( CFileManager::Instance()->GetShareBuffer( esbtXMLBZ, &m_ListData, false ) == 0 )
	{
		m_sFileName = "myfiles.xml.bz2";
	}
	else if ( CFileManager::Instance()->GetShareBuffer( esbtBZ, &m_ListData, false ) == 0 )
	{
		m_sFileName = "myfiles.txt.bz2";
	}
	else if ( CFileManager::Instance()->GetShareBuffer( esbtHE3, &m_ListData, false ) == 0 )
	{
		m_sFileName = "myfiles.DcLst";
	}
	else
	{
		m_sFileName = "";
		m_ListData.SetSize(0);
	}
}

/** */
bool DCFileBrowser::DownloadDirectory( QString dir )
{
	bool res = false;
	bool found = false;
	QStringList dirs = QStringList::split( "\\", dir, false );
	
	qApp->processEvents();
	
	QListViewItem * curitem = ListView_DIRECTORY->firstChild();
	
	if ( !curitem )
	{
		return res;
	}
	
	for ( QStringList::const_iterator it = dirs.constBegin(); it != dirs.constEnd(); ++it )
	{
		curitem = curitem->firstChild();
		
		while ( curitem )
		{
			if ( curitem->text(0) == *it )
			{
				found = true;
				break;
			}
			
			curitem = curitem->nextSibling();
		}
		
		if ( !found )
		{
			return res;
		}
		else
		{
			found = false;
		}
	}
	
	if ( curitem )
	{
		DownloadPath( QString::null, curitem->text(0), QString::null, curitem );
		res = true;
	}
	
	return res;
}

/** */
void DCFileBrowser::JumpTo( QString target )
{
	QStringList parts = QStringList::split( "\\", target, false );
	bool found = false;
	
	qApp->processEvents();
	
	QListViewItem * curitem = ListView_DIRECTORY->firstChild();
	QListViewItem * parent;
	
	QStringList::const_iterator it = parts.constBegin();
	for ( ; it != parts.constEnd(); ++it )
	{
		parent = curitem;
		curitem = curitem->firstChild();
		
		while ( curitem )
		{
			if ( curitem->text(0) == *it )
			{
				found = true;
				break;
			}
			
			curitem = curitem->nextSibling();
		}
		
		if ( !found )
		{
			curitem = parent;
			break;
		}
		else
		{
			found = false;
		}
	}
	
	if ( curitem )
	{
		ListView_DIRECTORY->clearSelection();
		ListView_DIRECTORY->ensureItemVisible( curitem );
		ListView_DIRECTORY->setSelected( curitem, true );
		ListView_DIRECTORY->setCurrentItem( curitem );
		
		if ( it != parts.constEnd() )
		{
			qApp->processEvents();
			
			curitem = ListView_FILES->firstChild();
			while ( curitem )
			{
				if ( curitem->text(0) == *it )
				{
					ListView_FILES->clearSelection();
					ListView_FILES->ensureItemVisible( curitem );
					ListView_FILES->setSelected( curitem, true );
					ListView_FILES->setCurrentItem( curitem );
					break;
				}
				curitem = curitem->nextSibling();
			}
		}
	}
}

/** */
void DCFileBrowser::slotGotoUser()
{
	DCClient * client = 0;
	if ( m_sNick.isEmpty() || m_sHubHost.isEmpty() )
	{
		client = 0;
	}
	else
	{
		client = g_pConnectionManager->GetClientForHub( m_sHubName.ascii(), m_sHubHost.ascii() );
	}
	
	if ( client )
	{
		if ( client->jumpToNick( m_sNick ) )
		{
			client->raise();
			client->setFocus();
		}
		else
		{
			QMessageBox::information(
				this,
				tr("Go to user"),
				tr("Cannot find user")
			);
		}
	}
	else
	{
		QMessageBox::information(
			this,
			tr("Go to user"),
			tr("Cannot find hub")
		);
	}
}
