/***************************************************************************
                          dchublistmanager.cpp  -  description
                             -------------------
    begin                : Mon Oct 1 2001
    copyright            : (C) 2001-2004 by Mathias Küster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "dchublistmanager.h"

#include <stdlib.h>

#include <qapplication.h>
#include <qobject.h>
#include <qlistview.h>
#include <qtoolbutton.h>
#include <qlineedit.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qtabwidget.h>
#include <qcombobox.h>
#include <qcheckbox.h>
#include <qmessagebox.h>
#include <qclipboard.h>
#include <qprogressbar.h>
#include <qregexp.h>
#include <vector>

#include "dcmenuhandler.h"
#include "dchubprofile.h"
#include "dchubfilter.h"
#include "dciconloader.h"
#include "dcconnectionmanager.h"
#include "dcconfig.h"
#include "dchublistitems.h"
#include "dcguiutils.h"

#include <dclib/dcos.h>
#include <dclib/dcobject.h>

#include "dceditserver.h"
#include "cdialogmessage.h"

DCHubListManager * g_pHubListManager = 0;

/** */
DCHubListManager::DCHubListManager(QWidget* parent, const char *name) : DCDialogHubListManager(parent, name)
{
	int idx;
	
	/* icons moved out of .ui files so that they can be themed */
	ToolButton_ADDFILTER->setIconSet( QIconSet( g_pIconLoader->GetPixmap(eiEDITADD) ) );
	ToolButton_EDITFILTER->setIconSet( QIconSet( g_pIconLoader->GetPixmap(eiEDIT) ) );
	ToolButton_DELFILTER->setIconSet( QIconSet( g_pIconLoader->GetPixmap(eiEDITDELETE) ) );
	ToolButton_RELOADPUBLICHUBLIST->setIconSet( QIconSet( g_pIconLoader->GetPixmap(eiRELOAD) ) );
	ToolButton_UPDATEPUBLICHUBLIST->setIconSet( QIconSet( g_pIconLoader->GetPixmap(eiUPDATE) ) );
	ToolButton_CONNECT->setIconSet( QIconSet( g_pIconLoader->GetPixmap(eiCONNECT) ) );
	
	lastPublicHubsWidth = -1;
	lastBookmarkHubsWidth = -1;

	// set default icon
	setIcon( g_pIconLoader->GetPixmap(eiSERVER) );

	// set the ColumnWidthMode to manual, we have to do this
	// because using the QListView resizeMode AllColumns means that it is
	// impossible to change the size of the last column
	for( idx = 0; idx < ListView_PUBLIC->columns(); idx++ )
	{
		ListView_PUBLIC->setColumnWidthMode( idx, QListView::Manual );
	}
	for( idx = 0; idx < ListView_BOOKMARKS->columns(); idx++ )
	{
		ListView_BOOKMARKS->setColumnWidthMode( idx, QListView::Manual );
	}

	m_pMessageQueue = new QPtrQueue<CDCMessage>();

	bookmarksLastSortColumn = 0;
	bookmarksLastSortOrder = Qt::Ascending;

	InitDocument();

	g_pHubListManager = this;
}

/** */
DCHubListManager::~DCHubListManager()
{
	g_pHubListManager = NULL;

	m_HLMMutex.lock();

	if ( m_pMessageQueue )
	{
		QPtrQueue<CDCMessage> * tmp = m_pMessageQueue;
		m_pMessageQueue = 0;
		
		while ( !tmp->isEmpty() )
		{
			delete tmp->dequeue();
		}
		
		delete tmp;
	}

	m_HLMMutex.unlock();
	
	// we might have removed public hubs
	if ( g_pConfig->GetRemovePublicHubEnabled() )
	{
		g_pConfig->SaveDCPublicHub();
	}
	
	// bookmarks may have been sorted and the new order needs to be saved
	updateBookmarkOrder();
	
	/* delete all hub filter objects */
	for ( QMap<QString, DC_HubFilterObject*>::const_iterator it = m_HubFilterMap.constBegin(); it != m_HubFilterMap.constEnd(); ++it )
	{
		delete it.data();
	}
	m_HubFilterMap.clear();
}

/** current tab widget change slot */
void DCHubListManager::slotTabWidgetCurrentChange(QWidget*)
{
	SizeColumnsPreservingRatios();
}

/** overridden so that the column widths are initialized on first show() */
void DCHubListManager::show()
{
	QWidget::show();

	SizeColumnsPreservingRatios();
}

/** resize event handler */
void DCHubListManager::resizeEvent( QResizeEvent * )
{
	SizeColumnsPreservingRatios();
}

/** Initialize or adjust widths of the ListView columns */
void DCHubListManager::SizeColumnsPreservingRatios()
{
	int width;
	
	if ( ListView_PUBLIC->isVisible() )
	{
		width = ListView_PUBLIC->width();
		if ( width > 0 )
		{
			if ( lastPublicHubsWidth == -1 )
			{
				/* set initial widths using default ratios */
				/* also expand the list view since we have lots of columns now */
				ListView_PUBLIC->setColumnWidth( 0, ((width*3)/10) );
				ListView_PUBLIC->setColumnWidth( 1, ((width*3)/10) );
				ListView_PUBLIC->setColumnWidth( 2, ((width*3)/10) );
				ListView_PUBLIC->setColumnWidth( 3, ((width*1)/10) );
				ListView_PUBLIC->setColumnWidth( 4, ((width*2)/10) );
				ListView_PUBLIC->setColumnWidth( 5, ((width*1)/10) );
				ListView_PUBLIC->setColumnWidth( 6, ((width*1)/10) );
				ListView_PUBLIC->setColumnWidth( 7, ((width*3)/10) );
				
				lastPublicHubsWidth = ListView_PUBLIC->width();
			}
			else if ( lastPublicHubsWidth != width )
			{
				DCGuiUtils::AdjustColumnWidths( ListView_PUBLIC, lastPublicHubsWidth );			
				lastPublicHubsWidth = ListView_PUBLIC->width();
			}
		}
	}
	else if ( ListView_BOOKMARKS->isVisible() )
	{
		width = ListView_BOOKMARKS->width();
		if ( width > 0 )
		{
			if ( lastBookmarkHubsWidth == -1 )
			{
				ListView_BOOKMARKS->setColumnWidth( 0, width/3 );
				ListView_BOOKMARKS->setColumnWidth( 1, width/3 );
				ListView_BOOKMARKS->setColumnWidth( 2, width/3 );
				
				lastBookmarkHubsWidth = ListView_BOOKMARKS->width();
			}
			else if ( lastBookmarkHubsWidth != width )
			{
				DCGuiUtils::AdjustColumnWidths( ListView_BOOKMARKS, lastBookmarkHubsWidth );
				lastBookmarkHubsWidth = ListView_BOOKMARKS->width();
			}
		}
	}
}

/** */
void DCHubListManager::SetToolBar( QFrame * frame1, QFrame * frame2, bool vis )
{
	if ( vis )
	{
		frame1->hide();
		frame1->setEnabled(false);
		frame2->show();
		frame2->setEnabled(true);
	}
	else
	{
		frame2->hide();
		frame2->setEnabled(false);
		frame1->show();
		frame1->setEnabled(true);
	}
}

/** */
void DCHubListManager::InitDocument()
{
	StringMap * map;
	bool btb1=true,btb2=true;
	bool bfilter=false;

	DC_HubFilterObject * HubFilterObject;

	// hide progressbar
	ProgressBar_LOADHUBLIST->hide();

	// set bookmark icon
	TabWidget_SERVERLIST->setTabIconSet( TabWidget_SERVERLIST->page(1), QIconSet( g_pIconLoader->GetPixmap(eiBOOKMARK_FOLDER) ) );

	// init the toolbars
	Frame_TOOLBAR1_HIDE->installEventFilter(this);
	Frame_TOOLBAR1_SHOW->installEventFilter(this);
	Frame_TOOLBAR2_HIDE->installEventFilter(this);
	Frame_TOOLBAR2_SHOW->installEventFilter(this);
	
	// restore settings
	if ( g_pConfig->GetMap("HUBVIEW",map) )
	{
		if ( ((*map)["WIDTH"].toInt() > 0) && ((*map)["HEIGHT"].toInt() > 0) )
		{
			setGeometry( (*map)["X"].toInt(), (*map)["Y"].toInt(), (*map)["WIDTH"].toInt(), (*map)["HEIGHT"].toInt() );
		}

		if ( (!((*map)["TOOLBAR1"].isEmpty())) && ((*map)["TOOLBAR1"].toInt() == 0) )
		{
			btb1 = false;
		}

		if ( (!((*map)["TOOLBAR2"].isEmpty())) && ((*map)["TOOLBAR2"].toInt() == 0) )
		{
			btb2 = false;
		}

		// restore active page
		TabWidget_SERVERLIST->setCurrentPage((*map)["ACTIVEPAGE"].toInt());
		// restore linedit
		LineEdit_SERVER->setText((*map)["LE_SERVER"]);
		
		/* restore sorting parameters */
		ListView_PUBLIC->setSortOrder( DCGuiUtils::SortOrderFromName( (*map)["PUBLICSORTORDER"] ) );
		ListView_PUBLIC->setSortColumn( (*map)["PUBLICSORTCOLUMN"].toInt() );
		
		/* returns Qt::Ascending on empty/invalid string */
		bookmarksLastSortOrder = DCGuiUtils::SortOrderFromName( (*map)["BOOKMARKSORTORDER"] );
		
		if ( (!(*map)["BOOKMARKSORTCOLUMN"].isEmpty()) )
		{
			bookmarksLastSortColumn = (*map)["BOOKMARKSORTCOLUMN"].toInt();
		}
		
		if ( (!((*map)["BOOKMARKSORTINGENABLED"].isEmpty())) && ((*map)["BOOKMARKSORTINGENABLED"].toInt() == 0) )
		{
			ListView_BOOKMARKS->setSortColumn(-1);
		}
		else
		{
			ListView_BOOKMARKS->setSortOrder( bookmarksLastSortOrder );
			ListView_BOOKMARKS->setSortColumn( bookmarksLastSortColumn );
		}
		
	}

	SetToolBar( Frame_TOOLBAR1_HIDE, Frame_TOOLBAR1_SHOW, btb1 );
	SetToolBar( Frame_TOOLBAR2_HIDE, Frame_TOOLBAR2_SHOW, btb2 );

	connect( LineEdit_SERVER, SIGNAL(returnPressed()), this, SLOT(slotConnect()) );
	connect( ToolButton_RELOADPUBLICHUBLIST, SIGNAL(clicked()), this, SLOT(slotReloadPublicHubList()) );
	connect( ToolButton_UPDATEPUBLICHUBLIST, SIGNAL(clicked()), this, SLOT(slotUpdatePublicHubList()) );
	connect( ToolButton_CONNECT, SIGNAL(clicked()), this, SLOT(slotConnect()) );

	// public list
	connect( ListView_PUBLIC,SIGNAL(doubleClicked(QListViewItem*, const QPoint &, int)), this, SLOT(slotDoubleClicked(QListViewItem*, const QPoint &, int)) );
	connect( ListView_PUBLIC,SIGNAL(returnPressed(QListViewItem*)), this, SLOT(slotReturnPressed(QListViewItem*)) );
	connect( ListView_PUBLIC,SIGNAL(contextMenuRequested( QListViewItem *, const QPoint &, int )), this, SLOT(slotRightButtonClickedPublicList(QListViewItem*, const QPoint &, int )) );

	// bookmark list
	connect( ListView_BOOKMARKS,SIGNAL(doubleClicked(QListViewItem*, const QPoint &, int)), this, SLOT(slotDoubleClicked(QListViewItem*, const QPoint &, int)) );
	connect( ListView_BOOKMARKS,SIGNAL(returnPressed(QListViewItem*)), this, SLOT(slotReturnPressed(QListViewItem*)) );
	connect( ListView_BOOKMARKS,SIGNAL(contextMenuRequested( QListViewItem *, const QPoint &, int )), this, SLOT(slotRightButtonClickedBookmarkList(QListViewItem*, const QPoint &, int )) );
	connect( ToolButton_BOOKMARK_UP, SIGNAL(clicked()), this, SLOT(slotMoveBookmarkUp()) );
	connect( ToolButton_BOOKMARK_DOWN, SIGNAL(clicked()), this, SLOT(slotMoveBookmarkDown()) );
	connect( ToolButton_BOOKMARK_ENABLE_SORT, SIGNAL(clicked()), this, SLOT(slotEnableBookmarkSorting()) );
	connect( ToolButton_BOOKMARK_SAVE, SIGNAL(clicked()), this, SLOT(slotSaveBookmarks()) );

	connect( TabWidget_SERVERLIST,SIGNAL(currentChanged(QWidget*)), this, SLOT(slotTabWidgetCurrentChange(QWidget*)) );

	// filter
	connect( ToolButton_ADDFILTER, SIGNAL(clicked()), this, SLOT(slotAddFilter()) );
	connect( ToolButton_EDITFILTER, SIGNAL(clicked()), this, SLOT(slotEditFilter()) );
	connect( ToolButton_DELFILTER, SIGNAL(clicked()), this, SLOT(slotDelFilter()) );
	connect( ComboBox_FILTER, SIGNAL(activated( const QString & )), this, SLOT(slotFilterChange( const QString &)) );

	connect( &m_Timer, SIGNAL(timeout()), this, SLOT(timerDone()) );

	// load filter
	g_pConfig->LoadDCHubFilter(&m_HubFilterMap);

	// create default
	if ( m_HubFilterMap.isEmpty() )
	{
		HubFilterObject = new DC_HubFilterObject();

		HubFilterObject->m_sFilterName = "Default";

		m_HubFilterMap[HubFilterObject->m_sFilterName] = HubFilterObject;

		g_pConfig->SaveDCHubFilter(&m_HubFilterMap);
	}

	for ( QMap<QString, DC_HubFilterObject*>::const_iterator it = m_HubFilterMap.constBegin(); it != m_HubFilterMap.constEnd(); ++it )
	{
		HubFilterObject = it.data();
		
		ComboBox_FILTER->insertItem( HubFilterObject->m_sFilterName );

		if ( (*map)["FILTER"] == HubFilterObject->m_sFilterName )
		{
			bfilter = true;
		}
	}

	// restore filter
	if ( bfilter )
	{
		ComboBox_FILTER->setCurrentText((*map)["FILTER"]);
	}
	else
	{
		ComboBox_FILTER->setCurrentText("Default");
	}

	ShowBookmarkList();
	ShowPublicHubList();

	m_Timer.start( 500, true );
}

/** */
void DCHubListManager::DeInitDocument()
{
	StringMap * map;

	// save hub view settings
	g_pConfig->GetMap("HUBVIEW",map);

	(*map)["X"]          = QString().setNum(parentWidget()->x());
	(*map)["Y"]          = QString().setNum(parentWidget()->y());
	(*map)["WIDTH"]      = QString().setNum(width());
	(*map)["HEIGHT"]     = QString().setNum(height());
	(*map)["VISIBLE"]    = QString().setNum(isVisible());
	(*map)["MAXIMIZED"]  = QString().setNum(isMaximized());
	(*map)["MINIMIZED"]  = QString().setNum(isMinimized());
	(*map)["ENABLED"]    = QString().setNum(isEnabled());
	(*map)["ACTIVEPAGE"] = QString().setNum(TabWidget_SERVERLIST->currentPageIndex());
	(*map)["LE_SERVER"]  = LineEdit_SERVER->text();
	(*map)["FILTER"]     = ComboBox_FILTER->currentText();

	if ( !isEnabled() )
	{
		setEnabled(true);
	}

	(*map)["TOOLBAR1"]  = QString().setNum(Frame_TOOLBAR1_SHOW->isEnabled());
	(*map)["TOOLBAR2"]  = QString().setNum(Frame_TOOLBAR2_SHOW->isEnabled());
	
	(*map)["PUBLICSORTCOLUMN"] = QString().setNum( ListView_PUBLIC->sortColumn() );
	(*map)["PUBLICSORTORDER"]  = DCGuiUtils::SortOrderName( ListView_PUBLIC->sortOrder() );
	
	// bookmarks may have been sorted and the new order needs to be saved
	updateBookmarkOrder();
	
	(*map)["BOOKMARKSORTCOLUMN"] = QString().setNum( bookmarksLastSortColumn );
	(*map)["BOOKMARKSORTORDER"]  = DCGuiUtils::SortOrderName( bookmarksLastSortOrder );
	(*map)["BOOKMARKSORTINGENABLED"] = QString().setNum( ListView_BOOKMARKS->sortColumn() != -1 );
}

/** event filter */
bool DCHubListManager::eventFilter(QObject* object, QEvent* event)
{
	if ( event->type() == QEvent::MouseButtonDblClick )
	{
		if ( object == Frame_TOOLBAR1_HIDE )
		{
			SetToolBar( Frame_TOOLBAR1_HIDE, Frame_TOOLBAR1_SHOW, true );
		}
		else if ( object == Frame_TOOLBAR1_SHOW )
		{
			SetToolBar( Frame_TOOLBAR1_HIDE, Frame_TOOLBAR1_SHOW, false );
		}
		else if ( object == Frame_TOOLBAR2_HIDE )
		{
			SetToolBar( Frame_TOOLBAR2_HIDE, Frame_TOOLBAR2_SHOW, true );

			// show progressbar
			if ( m_pHubListUrl != 0 )
				ProgressBar_LOADHUBLIST->show();
		}
		else if ( object == Frame_TOOLBAR2_SHOW )
		{
			SetToolBar( Frame_TOOLBAR2_HIDE, Frame_TOOLBAR2_SHOW, false );
		}
	}

	return QWidget::eventFilter( object, event );    // standard event processing
}

/** callback function */
int DCHubListManager::DC_CallBack( CDCMessage * DCMessage )
{
	int err = -1;

	m_HLMMutex.lock();

	if ( DCMessage && m_pMessageQueue )
	{
		m_pMessageQueue->enqueue(DCMessage);
		err = 0;
	}

	m_HLMMutex.unlock();

	return err;
}

/** */
void DCHubListManager::timerDone()
{
	CDCMessage *DCMsg;
	int i;

	for(i=0;i<50;i++)
	{
		if ( m_HLMMutex.tryLock() == false )
		{
			break;
		}

		if ( m_pMessageQueue )
		{
			/* returns 0 if queue is empty */
			DCMsg = m_pMessageQueue->dequeue();
		}
		else
		{
			DCMsg = 0;
		}

		m_HLMMutex.unlock();

		if ( DCMsg == 0 )
		{
			break;
		}

		switch ( DCMsg->m_eType )
		{
			case DC_MESSAGE_GETHUBLIST:
			{
				DCMessageGetHubList * msg = (DCMessageGetHubList*) DCMsg;

				if ( msg->m_bRun == false )
				{
					// enable buttons
					ToolButton_RELOADPUBLICHUBLIST->setEnabled(true);
					ToolButton_UPDATEPUBLICHUBLIST->setEnabled(true);

					// hide progressbar
					ProgressBar_LOADHUBLIST->hide();

					ShowPublicHubList();
				}
				else
				{
					// disable buttons
					ToolButton_RELOADPUBLICHUBLIST->setEnabled(false);
					ToolButton_UPDATEPUBLICHUBLIST->setEnabled(false);

					// show progressbar
					ProgressBar_LOADHUBLIST->show();
				}

				break;
			}

			case DC_MESSAGE_TRANSFER:
			{
				CMessageTransfer *msg = (CMessageTransfer*)DCMsg;

				ProgressBar_LOADHUBLIST->setTotalSteps(msg->m_nLength);
				ProgressBar_LOADHUBLIST->setProgress(msg->m_nTransfered);

				break;
			}

			default:
			{
				break;
			}
		}

		if ( DCMsg )
		{
			delete DCMsg;
		}
	}

	m_Timer.start( 500, true );
}

/** */
void DCHubListManager::ShowPublicHubList()
{
	DCConfigHubItem * hubitem = 0;
	CList<DCConfigHubItem> PublicHubList;
	DC_HubFilterObject * HubFilterObject = 0;
	QString s;
	bool bshow;
	DCPublicHubListItem * item = 0;
	QString checkstring;
	int x=0;
	int sortColumn = ListView_PUBLIC->sortColumn();
	Qt::SortOrder sortOrder = ListView_PUBLIC->sortOrder();

	// disable filter combo box
	ComboBox_FILTER->setEnabled(false);

	TabWidget_SERVERLIST->setTabLabel( TabWidget_SERVERLIST->page(0), tr("Public")+" (0/0)" );

	ListView_PUBLIC->setUpdatesEnabled(false);
	ListView_PUBLIC->setSortColumn(-1); // disable sorting
	ListView_PUBLIC->clear();

	g_pConfig->GetPublicHubList(&PublicHubList);

	// get the hubfilter
	s = ComboBox_FILTER->currentText();

	if ( !s.isEmpty() )
	{
		QMap<QString, DC_HubFilterObject*>::const_iterator it = m_HubFilterMap.find( s );
		if ( it != m_HubFilterMap.constEnd() )
		{
			HubFilterObject = it.data();
		}
		
		if ( HubFilterObject &&
		     !HubFilterObject->m_bName &&
		     !HubFilterObject->m_bServer &&
		     !HubFilterObject->m_bDescription &&
		     (HubFilterObject->m_nUser==0) )
		{
			HubFilterObject = 0;
		}
	}

	QRegExp filter;
	if ( HubFilterObject )
	{
		filter.setPattern( HubFilterObject->m_sContains );
		filter.setCaseSensitive( false );
	}

	while( (hubitem=PublicHubList.Next(hubitem)) != 0 )
	{
		bshow = true;

		++x;

		if ( x == 100 )
		{
			qApp->processEvents();
			x = 0;
		}

		// use the filter
		if ( HubFilterObject )
		{
			if ( hubitem->m_nUserCount < HubFilterObject->m_nUser )
			{
				bshow = false;
			}
			else if ( HubFilterObject->m_bName || HubFilterObject->m_bServer || HubFilterObject->m_bDescription )
			{
				bshow = false;
				
				if ( HubFilterObject->m_bName )
				{
					checkstring = QString::fromAscii(hubitem->m_sName.Data());
					bshow = (checkstring.find(filter) != -1);
				}
				
				if ( !bshow && HubFilterObject->m_bServer )
				{
					checkstring = QString::fromAscii(hubitem->m_sHost.Data());
					bshow = (checkstring.find(filter) != -1);
				}
				
				if ( !bshow && HubFilterObject->m_bDescription )
				{
					checkstring = QString::fromAscii(hubitem->m_sDescription.Data());
					bshow = (checkstring.find(filter) != -1);
				}
			}
		}

		if ( bshow )
		{
			item = new DCPublicHubListItem(ListView_PUBLIC);

			item->setText(0,QString::fromAscii(hubitem->m_sName.Data()));
			item->setText(1,QString::fromAscii(hubitem->m_sHost.Data()));
			item->setText(2,QString::fromAscii(hubitem->m_sDescription.Data()));
			item->SetUsers(hubitem->m_nUserCount);
			item->setText(4,QString::fromAscii(hubitem->m_sCountry.Data()));
			item->SetShared(hubitem->m_nShared);
			item->SetMinShare(hubitem->m_nMinShare);
			item->setText(7,QString::fromAscii(hubitem->m_sExtra.Data()));
			
		}
	}

	ListView_PUBLIC->setSortColumn(sortColumn); // enable sorting
	ListView_PUBLIC->setSortOrder(sortOrder);
	ListView_PUBLIC->setUpdatesEnabled(true);
	ListView_PUBLIC->triggerUpdate();

	TabWidget_SERVERLIST->setTabLabel( TabWidget_SERVERLIST->page(0), tr("Public")+" ("+QString().setNum(ListView_PUBLIC->childCount())+"/"+QString().setNum(PublicHubList.Count())+")");

	// enable filter combo box
	ComboBox_FILTER->setEnabled(true);
}

/** */
void DCHubListManager::ShowBookmarkList()
{
	int x = 0;
	DCConfigHubItem * hubitem = 0;
	CList<DCConfigHubItem> list;
	int sortColumn = ListView_BOOKMARKS->sortColumn();
	Qt::SortOrder sortOrder = ListView_BOOKMARKS->sortOrder();

	g_pConfig->GetBookmarkHubList(&list);

	ListView_BOOKMARKS->setUpdatesEnabled(false);
	ListView_BOOKMARKS->setSortColumn(-1);
	ListView_BOOKMARKS->clear();

	while( (hubitem=list.Next(hubitem)) != 0 )
	{
		DCBookmarkHubItem * item = new DCBookmarkHubItem( ListView_BOOKMARKS );
		item->setText( 0, hubitem->m_sName.Data() );
		item->setText( 1, hubitem->m_sHost.Data() );
		item->setText( 2, hubitem->m_sDescription.Data() );
		item->position = x;

		// if you add an item to an unsorted QListView, it gets put at the top
		// move it to the bottom
		if ( (sortColumn == -1) && (ListView_BOOKMARKS->childCount() > 1) )
		{
			item->moveItem( ListView_BOOKMARKS->lastItem() );
		}

		x++;

		if ( x == 100 )
		{
			qApp->processEvents();
			x = 0;
		}
	}

	ListView_BOOKMARKS->setSortColumn(sortColumn);
	ListView_BOOKMARKS->setSortOrder(sortOrder);
	ListView_BOOKMARKS->setUpdatesEnabled(true);
	ListView_BOOKMARKS->triggerUpdate();
	
	TabWidget_SERVERLIST->setTabLabel( TabWidget_SERVERLIST->page(1), tr("Bookmarks")+" ("+QString().setNum(ListView_BOOKMARKS->childCount())+")" );
}

/** */
void DCHubListManager::slotReloadPublicHubList()
{
	g_pConfig->ClearPublicHubList();

	GetPublicHubList();
}

/** update public hub list */
void DCHubListManager::slotUpdatePublicHubList()
{
	GetPublicHubList();
}

/** */
void DCHubListManager::slotDoubleClicked( QListViewItem * item, const QPoint &, int )
{
	if ( item )
		g_pConnectionManager->Connect( item->text(0).ascii(), item->text(1).ascii() );
}

/** */
void DCHubListManager::slotReturnPressed( QListViewItem * item )
{
	if ( item )
		g_pConnectionManager->Connect( item->text(0).ascii(), item->text(1).ascii() );
}

/** */
void DCHubListManager::slotConnect()
{
	g_pConnectionManager->Connect( CString(), LineEdit_SERVER->text().ascii() );
}

/** */
void DCHubListManager::slotAddFilter()
{
	DC_HubFilterObject * HubFilterObject;
	DCHubFilter * hf;

	hf = new DCHubFilter(this);

	if ( hf->exec() == QDialog::Accepted )
	{
		HubFilterObject = new  DC_HubFilterObject();

		// get the new data
		hf->Get( HubFilterObject );

		if ( HubFilterObject->m_sFilterName.isEmpty() )
		{
			// todo: error message

			delete HubFilterObject;
		}
		else if ( m_HubFilterMap.contains( HubFilterObject->m_sFilterName ) )
		{
			QMessageBox::information(
				this,
				tr("Add hub filter"),
				tr("A filter with that name already exists")
			);

			delete HubFilterObject;
		}
		else
		{
			m_HubFilterMap[HubFilterObject->m_sFilterName] = HubFilterObject;

			ComboBox_FILTER->insertItem( HubFilterObject->m_sFilterName );
		}

		g_pConfig->SaveDCHubFilter(&m_HubFilterMap);

		delete hf;
	}
}

/** */
void DCHubListManager::slotEditFilter()
{
	QString s;
	DC_HubFilterObject * HubFilterObject;
	DCHubFilter * hf;

	s = ComboBox_FILTER->currentText();

	if ( (!(s.isEmpty())) && (s != "Default") )
	{
		QMap<QString, DC_HubFilterObject*>::const_iterator it = m_HubFilterMap.find( s );
		if ( it != m_HubFilterMap.constEnd() )
		{
			HubFilterObject = it.data();
			
			hf = new DCHubFilter(this);

			hf->Edit( HubFilterObject );

			if ( hf->exec() == QDialog::Accepted )
			{
				// get the new data
				hf->Get( HubFilterObject );

				if ( HubFilterObject->m_sFilterName.isEmpty() )
				{
					HubFilterObject->m_sFilterName = s;

					// todo: error message
				}
				else if ( s != HubFilterObject->m_sFilterName )
				{
					if ( m_HubFilterMap.contains( HubFilterObject->m_sFilterName ) )
					{
						QMessageBox::information(
							this,
							tr("Edit hub filter"),
							tr("A filter with that name already exists")
						);
						HubFilterObject->m_sFilterName = s;
					}
					else
					{
						/* not deleted because it's the same pointer */
						m_HubFilterMap.erase(s);
						m_HubFilterMap[HubFilterObject->m_sFilterName] = HubFilterObject;

						ComboBox_FILTER->changeItem( HubFilterObject->m_sFilterName, ComboBox_FILTER->currentItem() );
					}
				}

				// update view
				slotFilterChange( HubFilterObject->m_sFilterName );

				g_pConfig->SaveDCHubFilter(&m_HubFilterMap);
			}

			delete hf;
		}
	}
}

/** */
void DCHubListManager::slotDelFilter()
{
	QString s = ComboBox_FILTER->currentText();

	if ( (!(s.isEmpty())) && (s != "Default") )
	{
		ComboBox_FILTER->removeItem(ComboBox_FILTER->currentItem());

		QMap<QString, DC_HubFilterObject*>::const_iterator it = m_HubFilterMap.find( s );
		if ( it != m_HubFilterMap.constEnd() )
		{
			delete it.data();
			m_HubFilterMap.erase( s );
			g_pConfig->SaveDCHubFilter(&m_HubFilterMap);
		}
		else
		{
			QMessageBox::warning(
				this,
				tr("Delete hub filter"),
				tr("Filter not found")
			);
		}
	}
}

/** */
void DCHubListManager::slotFilterChange( const QString & s )
{
	if ( !(s.isEmpty()) )
	{
		ShowPublicHubList();
	}
}

/** */
void DCHubListManager::RemoveBookmark( QListViewItem * item )
{
	if ( g_pConfig->RemoveBookmarkHub( item->text(0).ascii(), item->text(1).ascii(), item->text(2).ascii() ) )
	{
		g_pConfig->SaveDCBookHub();

		QListViewItemIterator it( ListView_BOOKMARKS );
		DCBookmarkHubItem * removed = (DCBookmarkHubItem*) item;
		for ( ; (*it); ++it )
		{
			DCBookmarkHubItem * current = (DCBookmarkHubItem*) (*it);
			if ( current->position > removed->position )
			{
				current->position--;
			}
		}

		delete item;

		// update tabbar
		TabWidget_SERVERLIST->setTabLabel( TabWidget_SERVERLIST->page(1), tr("Bookmarks")+" ("+QString().setNum(ListView_BOOKMARKS->childCount())+")" );
	}
}

/** */
void DCHubListManager::AddBookmark( QString hubname, QString hubhost, QString description )
{
	if ( g_pConfig->AddBookmarkHub( hubname.ascii(), hubhost.ascii(), description.ascii() ) )
	{
		g_pConfig->SaveDCBookHub();

		DCBookmarkHubItem * item = new DCBookmarkHubItem( ListView_BOOKMARKS );
		item->setText( 0, hubname );
		item->setText( 1, hubhost );
		item->setText( 2, description );
		item->position = ListView_BOOKMARKS->childCount() - 1;

		// if you add an item to an unsorted QListView, it gets put at the top
		// move it to the bottom
		if ( (ListView_BOOKMARKS->sortColumn() == -1) && (ListView_BOOKMARKS->childCount() > 1) )
		{
			item->moveItem( ListView_BOOKMARKS->lastItem() );
		}

		// update tabbar
		TabWidget_SERVERLIST->setTabLabel( TabWidget_SERVERLIST->page(1), tr("Bookmarks")+" ("+QString().setNum(ListView_BOOKMARKS->childCount())+")" );
	}
}

/** */
void DCHubListManager::UpdateBookmark( QString hubname, QString hubhost, QString description )
{
	if ( g_pConfig->UpdateBookmarkHub( hubname.ascii(), hubhost.ascii(), description.ascii() ) )
	{
		g_pConfig->SaveDCBookHub();
	}
}

/** */
CStringList<CString> * DCHubListManager::GetFilteredHubList()
{
	CStringList<CString> * list = new CStringList<CString>();
	QListViewItemIterator it( ListView_PUBLIC );

	for ( ; it.current(); it++ )
	{
		if ( it.current()->isEnabled() )
		{
			CString addr = it.current()->text(1).ascii();
			CString * string = 0;
			if ( list->Get( addr, &string ) != 0 )
			{
				list->Add( addr, new CString(addr) );
			}
		}
	}

	if ( list->Count() == 0 )
	{
		delete list;
		list = 0;
	}

	return list;
}

/** */
void DCHubListManager::slotRightButtonClickedBookmarkList( QListViewItem * /*item*/, const QPoint &, int )
{
	int id;
	QPopupMenu *m;
	DCEditServer * des;
	QPtrList<QListViewItem> selitems;
	QListViewItem * curitem;
	DCConfigHubProfile pConfigHubProfile;
	DCConfigHubItem ConfigHubItem;

	CString profilename;

	DCGuiUtils::SelectedItems(ListView_BOOKMARKS,selitems);

	m = new QPopupMenu(this);

	DCMenuHandler::InsertMenu( m, emiCONNECT, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiADD );
	DCMenuHandler::InsertMenu( m, emiEDIT, (selitems.count() == 1) );
	DCMenuHandler::InsertMenu( m, emiREMOVE, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiUPDATE_SERVER, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiUPDATE_ALL_SERVER );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiHUB_PROFILE_EDITOR );

	id = m->exec(QCursor::pos());

	delete m;

	if ( id == -1 )
	{
		return;
	}

	if ( id == emiADD )
	{
		des = new DCEditServer(this);
		des->setCaption(tr("Add Bookmark"));

		//reset values
		des->Reset();
		
		// DCEditServer now checks that name and host are not empty
		if ( des->exec() == QDialog::Accepted )
		{
			// add bookmark to the list
			AddBookmark( des->LineEdit_NAME->text(), des->LineEdit_HOST->text(), des->LineEdit_DESCRIPTION->text() );

			// check if profiling possible
			if ( des->CheckBox_PROFILE->isChecked() )
			{
				// get profile from dialog
				des->GetProfile(&pConfigHubProfile);
				
				// add new profile
				g_pConfig->AddHubProfile( &pConfigHubProfile );

				// save profile
				g_pConfig->SaveHubProfile();

				// set bookmark for that hub
				g_pConfig->SetBookmarkHubProfile(des->LineEdit_NAME->text().ascii(),des->LineEdit_NAME->text().ascii());
			}
			else
			{
				// clear bookmark from that hub
				g_pConfig->SetBookmarkHubProfile(des->LineEdit_NAME->text().ascii(),CString());
			}
			
			// update myinfo
			g_pConnectionManager->SendMyInfoToConnectedServers();
		}

		delete des;
	}
	else if ( id == emiEDIT )
	{
		bool prof = false;

		// edit only for one selected item
		curitem = selitems.first();

		des = new DCEditServer(this);
		des->setCaption(tr("Edit Bookmark"));

		des->LineEdit_NAME->setText(curitem->text(0));
		des->LineEdit_HOST->setText(curitem->text(1));
		des->LineEdit_DESCRIPTION->setText(curitem->text(2));

		// hubitem
		if ( g_pConfig->GetBookmarkHub( curitem->text(0).ascii(), &ConfigHubItem ) )
		{
			if ( ConfigHubItem.m_sProfile.NotEmpty() )
			{
				// get profile
				if ( g_pConfig->GetHubProfile( ConfigHubItem.m_sProfile, &pConfigHubProfile ) )
				{
					des->SetProfile(&pConfigHubProfile);

					prof = true;
				}
			}
		}

		// DCEditServer now checks that name and host are not empty
		if ( des->exec() == QDialog::Accepted  )
		{
			RemoveBookmark( curitem );

			if ( prof )
			{
				// remove the profile
				g_pConfig->DelHubProfile( pConfigHubProfile.m_sName );
                	}

			// add as new bookmark
			AddBookmark( des->LineEdit_NAME->text(), des->LineEdit_HOST->text(), des->LineEdit_DESCRIPTION->text() );

			// check if profiling possible
			if ( des->CheckBox_PROFILE->isChecked() )
			{
				// get profile from dialog
				des->GetProfile(&pConfigHubProfile);

				// add new profile
				g_pConfig->AddHubProfile( &pConfigHubProfile );

				// save profile
				g_pConfig->SaveHubProfile();

				// set bookmark for that hub
				g_pConfig->SetBookmarkHubProfile(des->LineEdit_NAME->text().ascii(),des->LineEdit_NAME->text().ascii());
			}
			else
			{
				// remove bookmark from that hub
				g_pConfig->SetBookmarkHubProfile(des->LineEdit_NAME->text().ascii(),CString());
			}

			// update myinfo
			g_pConnectionManager->SendMyInfoToConnectedServers();
		}

		delete des;
	}
	else if ( id == emiREMOVE )
	{
		int i;
		StringMap * map;

		if ( g_pConfig->GetMap("DIALOG_REMOVE_BOOKMARK",map) == false )
		{
			(*map)["SHOW"]    = QString().setNum(1);
			(*map)["DEFAULT"] = QString().setNum(0);
		}

		if ( ((*map)["SHOW"].toInt() == 1) || ((*map)["DEFAULT"].toInt() == 1) )
		{
			CDialogMessage * dlg = new CDialogMessage( this,
				QMessageBox::Warning,
				tr("Remove bookmarks"),
        			tr("You are sure ?"),
				tr("Remove"),
				tr("Cancel") );

			i = dlg->exec();

			(*map)["SHOW"]    = QString().setNum(!dlg->GetCheckBoxStatus());
			(*map)["DEFAULT"] = QString().setNum(i);

			delete dlg;


		}
		else
		{
			i = (*map)["DEFAULT"].toInt();
		}

		switch(i)
		{
	    		case 1: // The user clicked the cancel
        			return;
			default:
				break;
		}

	        for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
	        {
			// remove profile for that entry
			if ( g_pConfig->GetHubProfile( curitem->text(0).ascii(), &pConfigHubProfile ) )
			{
				g_pConfig->DelHubProfile( pConfigHubProfile.m_sName );
			}

			// remove bookmark
			RemoveBookmark(curitem);
		}
		
		// save changes to profile list, some may have been deleted
		g_pConfig->SaveHubProfile();
		
		// update myinfo
		g_pConnectionManager->SendMyInfoToConnectedServers();
	}
	else if ( id == emiCONNECT )
	{
	        for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
	        {
			g_pConnectionManager->Connect( curitem->text(0).ascii(), curitem->text(1).ascii() );
		}
	}
	else if ( id == emiUPDATE_SERVER )
	{
		DCConfigHubItem hubitem;

	        for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
	        {
			if ( g_pConfig->GetPublicHub( curitem->text(0).ascii(), &hubitem ) )
			{
				UpdateBookmark( hubitem.m_sName.Data(), hubitem.m_sHost.Data(), hubitem.m_sDescription.Data() );
				curitem->setText(1,hubitem.m_sHost.Data());
				curitem->setText(2,hubitem.m_sDescription.Data());
			}
		}
	}
	else if ( id == emiUPDATE_ALL_SERVER )
	{
		DCConfigHubItem * hubitem = 0;
		DCConfigHubItem hubitem1;
		CList<DCConfigHubItem> list;

		g_pConfig->GetBookmarkHubList(&list);

		while( (hubitem=list.Next(hubitem)) != 0 )
		{
			if ( g_pConfig->GetPublicHub( hubitem->m_sName, &hubitem1 ) )
			{
				g_pConfig->UpdateBookmarkHub( hubitem1.m_sName, hubitem1.m_sHost, hubitem1.m_sDescription );
			}
		}

		g_pConfig->SaveDCBookHub();

		ShowBookmarkList();
	}
	else if ( id == emiHUB_PROFILE_EDITOR )
	{
		DCHubProfile * dlg = new DCHubProfile(this);

		dlg->exec();

		delete dlg;
		
		// update myinfo
		g_pConnectionManager->SendMyInfoToConnectedServers();
	}
}

/** */
void DCHubListManager::slotRightButtonClickedPublicList( QListViewItem * /*item*/, const QPoint &, int column )
{
	int id;
	QPopupMenu *m;
	QPtrList<QListViewItem> selitems;
	QListViewItem * curitem;
	int numSelected;
	
	numSelected = DCGuiUtils::SelectedItems(ListView_PUBLIC,selitems);
	
	if ( numSelected == 0 )
	{
		return;
	}

	m = new QPopupMenu(this);

	DCMenuHandler::InsertMenu( m, emiCONNECT );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	
	if ( g_pConfig->GetRemovePublicHubEnabled() )
	{
		DCMenuHandler::InsertMenu( m, emiREMOVE );
		DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	}
	
	DCMenuHandler::InsertMenu( m, emiADD_BOOKMARK );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiCOPY_COLUMN_TO_CLIPBOARD, (numSelected == 1) );
	DCMenuHandler::InsertMenu( m, emiCOPY_ROW_TO_CLIPBOARD, (numSelected == 1) );

	id = m->exec(QCursor::pos());

	delete m;

	if ( id == -1 )
	{
		return;
	}

	if ( id == emiCONNECT )
	{
	        for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
	        {
			g_pConnectionManager->Connect( curitem->text(0).ascii(), curitem->text(1).ascii() );
		}
	}
	else if ( id == emiREMOVE )
	{
		for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
		{
			if ( g_pConfig->RemovePublicHub(curitem->text(0).ascii(),curitem->text(1).ascii(),curitem->text(2).ascii()) )
			{
				delete curitem;
			}
			else
			{
				QMessageBox::warning(this,tr("Remove public hub"),
				tr("Error removing hub \"" + curitem->text(0) + "\" (")
				+ curitem->text(1) + ")"
				);
			}
		}
		
		TabWidget_SERVERLIST->setTabLabel( TabWidget_SERVERLIST->page(0), tr("Public")+" ("+QString().setNum(ListView_PUBLIC->childCount())+"/"+QString().setNum(CConfig::Instance()->GetPublicHubListSize())+")");
	}
	else if ( id == emiADD_BOOKMARK )
	{
	        for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
	        {
			AddBookmark( curitem->text(0), curitem->text(1), curitem->text(2) );
		}

		TabWidget_SERVERLIST->setTabLabel( TabWidget_SERVERLIST->page(1), tr("Bookmarks")+" ("+QString().setNum(ListView_BOOKMARKS->childCount())+")" );
	}
	else if ( id == emiCOPY_COLUMN_TO_CLIPBOARD )
	{
		curitem = selitems.first();
		QApplication::clipboard()->setText( curitem->text(column) );
	}
	else if ( id == emiCOPY_ROW_TO_CLIPBOARD )
	{
		int idx;
		QString s;

		curitem = selitems.first();
		
		for( idx = 0; idx < ListView_PUBLIC->columns(); idx++ )
		{
			s += curitem->text(idx) + " ";
		}
		s = s.stripWhiteSpace();
		QApplication::clipboard()->setText(s);
	}
}

/** update CConfig's sorted bookmarks map */
void DCHubListManager::updateBookmarkOrder()
{
	if ( ListView_BOOKMARKS->sortColumn() == -1 )
	{
		//printf("updateBookmarkOrder cancelled, view is not sorted\n");
		return;
	}
	
	const int count = ListView_BOOKMARKS->childCount();
	std::vector<int> newpositions(count);
	
	int i = 0;
	QListViewItemIterator it( ListView_BOOKMARKS );
	for ( ; (*it); ++it )
	{
		DCBookmarkHubItem * item = (DCBookmarkHubItem*) (*it);
		
		//printf("%s %d->%d\n",item->text(0).ascii(),item->position,i);
		if ( (item->position < 0) || (item->position >= count) )
		{
			printf("updateBookmarkOrder: Existing key for %s out of range at %d\n", item->text(0).ascii(), item->position);
			return;
		}
		newpositions[item->position] = i;
		item->position = i;
		
		++i;
	}
	
	//printf("call ReorderBookmarkHubs\n");
	
	g_pConfig->ReorderBookmarkHubs( newpositions );
	g_pConfig->SaveDCBookHub();
	
	bookmarksLastSortColumn = ListView_BOOKMARKS->sortColumn();
	bookmarksLastSortOrder  = ListView_BOOKMARKS->sortOrder();
}

/** */
void DCHubListManager::slotMoveBookmarkUp()
{
	QListViewItem * item = ListView_BOOKMARKS->currentItem();
	
	if ( !item )
	{
		return;
	}
	
	int oldindex = DCGuiUtils::FindRowForItem( ListView_BOOKMARKS, item );
	
	if ( (oldindex < 1) || (oldindex > (ListView_BOOKMARKS->childCount() - 1)) )
	{
		return;
	}
	
	if ( ListView_BOOKMARKS->sortColumn() != -1 )
	{
		updateBookmarkOrder();
		ListView_BOOKMARKS->setSortColumn(-1);
	}
	
	int newindex = oldindex - 1;
	
	((DCBookmarkHubItem*) item)->position = newindex;
	((DCBookmarkHubItem*) item->itemAbove())->position = oldindex;
	
	item->itemAbove()->moveItem( item );
	ListView_BOOKMARKS->setCurrentItem(item);
	
	g_pConfig->MoveBookmarkHub( oldindex, newindex );
	
	g_pConfig->SaveDCBookHub();
}

/** */
void DCHubListManager::slotMoveBookmarkDown()
{
	QListViewItem * item = ListView_BOOKMARKS->currentItem();
	
	if ( !item )
	{
		return;
	}
	
	int oldindex = DCGuiUtils::FindRowForItem( ListView_BOOKMARKS, item );
	
	if ( (oldindex == -1) || (oldindex > (ListView_BOOKMARKS->childCount() - 2)) )
	{
		return;
	}
	
	if ( ListView_BOOKMARKS->sortColumn() != -1 )
	{
		updateBookmarkOrder();
		ListView_BOOKMARKS->setSortColumn(-1);
	}
	
	int newindex = oldindex + 1;
	
	((DCBookmarkHubItem*) item)->position = newindex;
	((DCBookmarkHubItem*) item->itemBelow())->position = oldindex;
	
	item->moveItem( item->itemBelow() );
	ListView_BOOKMARKS->setCurrentItem(item);
	
	g_pConfig->MoveBookmarkHub( oldindex, newindex );
	
	g_pConfig->SaveDCBookHub();
}

/** */
void DCHubListManager::slotEnableBookmarkSorting()
{
	if ( ListView_BOOKMARKS->sortColumn() == -1 )
	{
		ListView_BOOKMARKS->setSortColumn(bookmarksLastSortColumn);
		ListView_BOOKMARKS->setSortOrder(bookmarksLastSortOrder);
		// no updateBookmarkOrder() here
	}
}

/** */
void DCHubListManager::slotSaveBookmarks()
{
	/*
	 * Of couse updateBookmarkOrder() could just have been made a slot, but due to the large
	 * number of problems encountered while attempting to have the code in updateBookmarkOrder
	 * called automatically after the QListView has been sorted ( which appears to be impossible -
	 * you can connect to various QHeader signals but these are called either before
	 * the QListView has been sorted, or worse, during ) so it is not a slot to
	 * indicate that it is not called automatically and updateBookmarkOrder()
	 * must be called manually at various points and on shutdown.
	 */
	if ( ListView_BOOKMARKS->sortColumn() == -1 )
	{
		g_pConfig->SaveDCBookHub();
	}
	else
	{
		updateBookmarkOrder();
	}
}
