/***************************************************************************
                          dchubsearch.cpp  -  description
                             -------------------
    begin                : Fri Mar 15 2002
    copyright            : (C) 2002-2005 by Mathias Küster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "dchubsearch.h"

/** Qt (TM) includes */
#include <qapplication.h>
#include <qlineedit.h>
#include <qtextedit.h>
#include <qradiobutton.h>
#include <qlistview.h>
#include <qcheckbox.h>
#include <qlcdnumber.h>
#include <qcombobox.h>
#include <qspinbox.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qstringlist.h>
#include <qtabwidget.h>
#include <qbuttongroup.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qfiledialog.h>
#include <qclipboard.h>
#include <qprogressbar.h>
#include <qvaluevector.h>
//#include <qdragobject.h>
#include <qfileinfo.h>
#include <qdatetime.h>
#include <qlabel.h>

/** */
#include <dclib/dcos.h>
#include <dclib/cutils.h>
#include <dclib/core/cstringlist.h>

#ifndef WIN32
#include <stdlib.h>
#endif

#include "dcconfig.h"
#include "dcclient.h"
#include "dcconnectionmanager.h"
#include "dchublistmanager.h"
#include "dcfiletool.h"
#include "dctransferview.h"
#include "dcmenuhandler.h"
#include "dcwidget.h"
#include "dciconloader.h"
#include "dcguiutils.h"

#include <dclib/core/cdir.h>
#include <dclib/core/cxml.h>
#include <dclib/chttp.h>

// for file type icons
#include <dclib/cfilemanager.h>

// for saving search results with correct encoding
#include <dclib/core/ciconv.h>

// for loading saved search results
#include <dclib/cmessagehandler.h>

// not in dchubsearch.h any more
#include <dclib/csearchmanager.h>

// all for checking a TTH is valid and so not send out useless searches
#include <dclib/core/cbytearray.h>
#include <dclib/core/cbase32.h>
#include <dclib/cfilehasher.h>

/* column numbers for the search results listview */
#define RESULTS_COLUMN_COUNT		0
#define RESULTS_COLUMN_FILE		1
#define RESULTS_COLUMN_SIZE		2
#define RESULTS_COLUMN_TTH		3
#define RESULTS_COLUMN_NICK		4
#define RESULTS_COLUMN_FREESLOTS	5
#define RESULTS_COLUMN_TOTALSLOTS	6
#define RESULTS_COLUMN_HUB		7
#define RESULTS_COLUMN_PATH		8
#define RESULTS_COLUMN_HOST		9

/** */
DCHubSearch::DCHubSearch(QWidget* parent) : DCDialogHubSearch(parent, "HubSearch", WDestructiveClose), m_GroupDict(1499)
{
	int idx;

	lastFileResultsWidth = -1;
	lastUserResultsWidth = -1;

	m_eCurrentGrouping = egsrtNONE;
	m_nFilteredResults = 0;
	m_pEmptyGroup = 0;

	// set default icon
	setIcon( g_pIconLoader->GetPixmap(eiFIND) );

	// set the ColumnWidthMode to manual, we have to do this
	// because using the QListView resizeMode AllColumns means that it is
	// impossible to change the size of the last column
	for( idx = 0; idx < ListView_SEARCHRESULT->columns(); idx++ )
	{
		ListView_SEARCHRESULT->setColumnWidthMode( idx, QListView::Manual );
	}

	for( idx = 0; idx < ListView_SEARCHRESULTUSER->columns(); idx++ )
	{
		ListView_SEARCHRESULTUSER->setColumnWidthMode( idx, QListView::Manual );
	}

	// right align numeric columns
	ListView_SEARCHRESULT->setColumnAlignment( RESULTS_COLUMN_SIZE, Qt::AlignRight );
	ListView_SEARCHRESULT->setColumnAlignment( RESULTS_COLUMN_FREESLOTS, Qt::AlignRight );
	/* not total slots otherwise it's too close to the hub IP address and it's hard to read */

	// select at least
	ComboBox_SEARCHLIMIT->setCurrentItem(1);

	// count column is hidden unless results have been grouped
	ListView_SEARCHRESULT->hideColumn(RESULTS_COLUMN_COUNT);

	m_pMessageList      = new CList<CDCMessage>();
	m_pSearchResultList = new CList<CMessageSearchResult>();
	m_pSearchQueryList  = 0;
	m_pSearchHistory    = new CList<CDCMessage>();

	InitDocument();

	/*
	 * CSearchManager does not delete the callback function anymore
	 * because multiple search windows will never work with that.
	 */
	m_pOurCallback = new CCallback1<DCHubSearch, CDCMessage*>( this, &DCHubSearch::DC_CallBack );
	
	/* figure out global search state */
	switch ( CSearchManager::Instance()->SearchType() )
	{
		case estyNONE:
			m_eSearchState = egssREADY;
			Label_GLOBAL_SEARCH_STATUS->setText(tr("Ready"));
			SetSearchView(true);
			break;
		case estySINGLE:
		case estyMULTI:
			m_eSearchState = egssOTHER;
			Label_GLOBAL_SEARCH_STATUS->setText(tr("Other search..."));
			SetSearchView(false);
			break;
		case estyEXTERNAL:
			m_eSearchState = egssAUTO;
			Label_GLOBAL_SEARCH_STATUS->setText(tr("Auto search..."));
			SetSearchView(false);
			break;
	}
	
	if ( CSearchManager::Instance()->GetCallBackFunction() == 0 )
	{
		CSearchManager::Instance()->SetCallBackFunction( m_pOurCallback );
	}

	m_Timer.start( 500, true );
}

/** */
DCHubSearch::~DCHubSearch()
{
	DeInitDocument();

	if ( CSearchManager::Instance()->GetCallBackFunction() == m_pOurCallback )
	{
		if ( (m_eSearchState == egssSEARCH) && ((CSearchManager::Instance()->SearchType() == estySINGLE) || (CSearchManager::Instance()->SearchType() == estyMULTI)) )
		{
			CSearchManager::Instance()->StopSearch();
		}
		CSearchManager::Instance()->SetCallBackFunction( 0 );
	}
	
	delete m_pOurCallback;
	m_pOurCallback = 0;

	ClearSearchResults();

	ListView_SEARCH->removeEventFilter(this);
	Combobox_SEARCH->removeEventFilter(this);

	m_Timer.stop();

	SocketCallbackMutex.lock();

	// cleanup all lists
	delete m_pMessageList;
	m_pMessageList = 0;

	delete m_pSearchResultList;
	m_pSearchResultList = 0;

	delete m_pSearchQueryList;
	m_pSearchQueryList = 0;

	delete m_pSearchHistory;
	m_pSearchHistory = 0;
	
	SocketCallbackMutex.unlock();
}

/** */
void DCHubSearch::InitDocument()
{
	StringMap * map;

	// restore settings
	if ( g_pConfig->GetMap("SEARCHVIEW",map) )
	{
		if ( ((*map)["WIDTH"].toInt() > 0) && ((*map)["HEIGHT"].toInt() > 0) )
		{
			resize( (*map)["WIDTH"].toInt(), (*map)["HEIGHT"].toInt() );
		}
		
		if ( !((*map)["MAXTHREADS"].isEmpty()) )
		{
			SpinBox_MAXTHREADS->setValue( (*map)["MAXTHREADS"].toInt() );
		}
		
		if ( !((*map)["GROUPING"].isEmpty()) )
		{
			m_eCurrentGrouping = (eGroupSearchResultType) (*map)["GROUPING"].toInt();
		}
		
		if ( m_eCurrentGrouping != egsrtNONE )
		{
			ListView_SEARCHRESULT->setRootIsDecorated(true);
			ListView_SEARCHRESULT->setColumnWidth( RESULTS_COLUMN_COUNT, ListView_SEARCHRESULT->columnWidth( RESULTS_COLUMN_FREESLOTS ) );
			ListView_SEARCHRESULT->setSorting( RESULTS_COLUMN_COUNT );
		}
	}

	connect( LineEdit_INCLUDE, SIGNAL(returnPressed()), this, SLOT(slotTextFilterResults()) );
	connect( LineEdit_EXCLUDE, SIGNAL(returnPressed()), this, SLOT(slotTextFilterResults()) );
	connect( PushButton_SEARCH, SIGNAL(clicked()), this, SLOT(slotSearchReturnPressed()) );
	connect( PushButton_ADDQUEUE, SIGNAL(clicked()), this, SLOT(slotAddSearchQueue()) );
	connect( PushButton_REFRESHCONNECTEDHUBS, SIGNAL(clicked()), this, SLOT(slotRefreshConnectedHubs()) );
	connect( PushButton_APPLYTEXTFILTER, SIGNAL(clicked()), this, SLOT(slotTextFilterResults()) );
	connect( PushButton_RESETTEXTFILTER, SIGNAL(clicked()), this, SLOT(slotTextFilterReset()) );
 	connect( PushButton_RESETPARAMS, SIGNAL(clicked()), this, SLOT(slotReset()) );
	connect( SpinBox_FREESLOTS, SIGNAL(valueChanged(int)), this, SLOT(slotChangedFreeSlots(int)) );
	connect( CheckBox_SEARCHFILEONLY, SIGNAL(toggled(bool)), this, SLOT(slotToggledSearchFileOnly(bool)) );
	connect( TabWidget_HUBSEARCH,SIGNAL(currentChanged(QWidget*)), this, SLOT(slotTabWidgetCurrentChange(QWidget*)) );
	connect( ListView_SEARCHRESULT,SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(slotDoubleClickedSearchResult(QListViewItem*)) );
	connect( ListView_SEARCHRESULT,SIGNAL(contextMenuRequested( QListViewItem *, const QPoint &, int )), this, SLOT(slotRightButtonClickedSearchResult(QListViewItem*, const QPoint &, int )) );
	connect( ListView_SEARCH,SIGNAL(contextMenuRequested( QListViewItem *, const QPoint &, int )), this, SLOT(slotRightButtonClickedSearch(QListViewItem*, const QPoint &, int )) );
	
	connect( Combobox_SEARCH, SIGNAL(activated(int)), this, SLOT(slotSearchSelected(int)) );
	
	connect( PushButton_PURGEHISTORY, SIGNAL(clicked()), this, SLOT(slotPurgeHistory()) );
	
	connect( ComboBox_SEARCHLIMIT, SIGNAL(activated(int)), this, SLOT(slotSizeLimitChanged(int)) );
	
	connect( &m_Timer, SIGNAL(timeout()), this, SLOT(timerDone()) );

	ListView_SEARCH->installEventFilter(this);
	Combobox_SEARCH->installEventFilter(this);
}

/** */
void DCHubSearch::DeInitDocument()
{
	StringMap * map;

	// save search view settings
	g_pConfig->GetMap("SEARCHVIEW",map);

	(*map)["WIDTH"]     = QString().setNum(width());
	(*map)["HEIGHT"]    = QString().setNum(height());
	(*map)["MAXTHREADS"] = QString().setNum(SpinBox_MAXTHREADS->value());
	(*map)["GROUPING"]   = QString().setNum(m_eCurrentGrouping);
}

/** current tab widget change slot */
void DCHubSearch::slotTabWidgetCurrentChange(QWidget*)
{
	SizeColumnsPreservingRatios();
}

/** overridden so that the column widths are initialized on first show() */
void DCHubSearch::show()
{
	QWidget::show();

	SizeColumnsPreservingRatios();
}

/** resize event handler */
void DCHubSearch::resizeEvent( QResizeEvent * )
{
	SizeColumnsPreservingRatios();
}

/** Initialize or adjust widths of the ListView columns */
void DCHubSearch::SizeColumnsPreservingRatios()
{
	int width;
	
	if ( ListView_SEARCHRESULT->isVisible() )
	{
		width = ListView_SEARCHRESULT->width();
		if ( width > 0 )
		{
			if ( lastFileResultsWidth == -1 )
			{
				ListView_SEARCHRESULT->setColumnWidth( RESULTS_COLUMN_FILE, ((width*7)/37) );
				ListView_SEARCHRESULT->setColumnWidth( RESULTS_COLUMN_SIZE, ((width*3)/37) );
				ListView_SEARCHRESULT->setColumnWidth( RESULTS_COLUMN_TTH, ((width*4)/37) );
				ListView_SEARCHRESULT->setColumnWidth( RESULTS_COLUMN_NICK, ((width*4)/37) );
				ListView_SEARCHRESULT->setColumnWidth( RESULTS_COLUMN_FREESLOTS, ((width*3)/37) );
				ListView_SEARCHRESULT->setColumnWidth( RESULTS_COLUMN_TOTALSLOTS, ((width*3)/37) );
				ListView_SEARCHRESULT->setColumnWidth( RESULTS_COLUMN_HUB, ((width*4)/37) );
				ListView_SEARCHRESULT->setColumnWidth( RESULTS_COLUMN_PATH, ((width*6)/37) );
				ListView_SEARCHRESULT->setColumnWidth( RESULTS_COLUMN_HOST, ((width*3)/37) );
				
				lastFileResultsWidth = ListView_SEARCHRESULT->width();
			}
			else if ( lastFileResultsWidth != width )
			{
				DCGuiUtils::AdjustColumnWidths( ListView_SEARCHRESULT, lastFileResultsWidth );
				lastFileResultsWidth = ListView_SEARCHRESULT->width();
			}
		}
	}
	else if ( ListView_SEARCHRESULTUSER->isVisible() )
	{
		width = ListView_SEARCHRESULTUSER->width();
		if ( width > 0 )
		{
			if ( lastUserResultsWidth == -1 )
			{
				ListView_SEARCHRESULTUSER->setColumnWidth( 0, width/2 );
				ListView_SEARCHRESULTUSER->setColumnWidth( 1, width/2 );
				
				lastUserResultsWidth = ListView_SEARCHRESULTUSER->width();
			}
			else if ( lastUserResultsWidth != width )
			{
				DCGuiUtils::AdjustColumnWidths( ListView_SEARCHRESULTUSER, lastUserResultsWidth );
				lastUserResultsWidth = ListView_SEARCHRESULTUSER->width();
			}
		}
	}
}

/** event filter */
bool DCHubSearch::eventFilter( QObject * object, QEvent * event )
{
	if ((event->type() == QEvent::KeyPress)&&((QListView*)object==ListView_SEARCH))
	{
		QKeyEvent * e = (QKeyEvent*)event;

		if ( e->key() == Key_Delete )
		{
			RemoveSelectedSearch(NULL);
		}
	}
	else if ((event->type() == QEvent::KeyPress)&&((QComboBox*)object==Combobox_SEARCH))
	{
		QKeyEvent * e = (QKeyEvent*)event;
		if((e->key() == Key_Enter) || (e->key() == Key_Return ))
		{
			slotSearchReturnPressed();
		}
	}

	return QWidget::eventFilter( object, event );    // standard event processing
}

/** */
void DCHubSearch::slotSearchSelected(int sel)
{
	CMessageSearchUser *msu;
	CMessageSearchFile *msf;
	int i;
	CDCMessage *hmsg=0;

	i = 0;

	while ( (hmsg=(CDCMessage *)m_pSearchHistory->Next((CDCMessage*)hmsg)) != 0 )
	{
		if ( i == sel )
			break;
		i++;
	}

	if ( hmsg == 0 )
		return;

	if ( hmsg->m_eType == DC_MESSAGE_SEARCH_USER )
	{
		msu = (CMessageSearchUser *)hmsg;

		LineEdit_SEARCHSIZE->setText("0");
		ComboBox_SEARCHUNIT->setCurrentItem(0);
		ComboBox_SEARCHLIMIT->setCurrentItem(1);
		slotSizeLimitChanged(1);
		ComboBox_SEARCHTYPE->setCurrentItem(0);
	}
	else
	{
		msf = (CMessageSearchFile *)hmsg;
		
		LineEdit_SEARCHSIZE->setText( QString().setNum(msf->m_nSize) );
		ComboBox_SEARCHUNIT->setCurrentItem(0);
		
		if ( msf->m_bSizeLimit )
		{
			switch(msf->m_eSizeType)
			{
				case esstATLEAST:
					ComboBox_SEARCHLIMIT->setCurrentItem(1);
					break;
				case esstATMOST:
					ComboBox_SEARCHLIMIT->setCurrentItem(2);
					break;
				default:
					break;
			}
		}
		else
		{
			ComboBox_SEARCHLIMIT->setCurrentItem(0);
		}
		
		slotSizeLimitChanged(ComboBox_SEARCHLIMIT->currentItem());
		
		ComboBox_SEARCHTYPE->setCurrentItem(msf->m_eFileType);
	}
}

/** */
void DCHubSearch::addHistory( CDCMessage * msg )
{
	CMessageSearchUser *msu1,*msu2;
	CMessageSearchFile *msf1,*msf2;
	CString s1,s2;
	CDCMessage *hmsg = 0;
	int i = 0;
	bool found = false;
	
	if ( !msg )
		msg = GetSearchObject();
	if ( !msg)
		return;

	if ( msg->m_eType == DC_MESSAGE_SEARCH_USER )
	{
		msu1 = (CMessageSearchUser *)msg;
		s1 = msu1->m_sNick;
	}
	else
	{
		msf1 = (CMessageSearchFile *)msg;
		s1 = msf1->m_sString;
	}

	while ( (hmsg=(CDCMessage *)m_pSearchHistory->Next((CDCMessage*)hmsg)) != 0 )
	{
		if ( msg->m_eType == hmsg->m_eType )
		{
			if ( msg->m_eType == DC_MESSAGE_SEARCH_USER )
			{
				msu2 = (CMessageSearchUser *)hmsg;
				s2   = msu2->m_sNick;
			}
			else
			{
				msf2 = (CMessageSearchFile *)hmsg;
				s2   = msf2->m_sString;
			}

			if ( s1 == s2 )
			{
				Combobox_SEARCH->removeItem(i);
				Combobox_SEARCH->insertItem(s1.Data());
				m_pSearchHistory->Del(hmsg);
				m_pSearchHistory->Add(msg);
				found = true;
				break;
			}
		}
		
		i++;
	}
	
	if ( !found )
	{
		m_pSearchHistory->Add(msg);
		Combobox_SEARCH->insertItem(s1.Data());
	}
	
	Combobox_SEARCH->setCurrentItem(Combobox_SEARCH->count()-1);
}

/** */
void DCHubSearch::SetSearchView( bool enabled )
{
	if ( enabled )
	{
		PushButton_SEARCH->setText(tr("Start"));
		PushButton_SEARCH->setEnabled(true);
	}
	else
	{
		PushButton_SEARCH->setText(tr("Stop"));
	}

	//LineEdit_SEARCH->setEnabled(enabled);
	Combobox_SEARCH->setEnabled(enabled);
	ButtonGroup_HUBS->setEnabled(enabled);
	SpinBox_MAXTHREADS->setEnabled(enabled);
	PushButton_ADDQUEUE->setEnabled(enabled);
	CheckBox_ENABLETAG->setEnabled(enabled);
	CheckBox_MULTISEARCH->setEnabled(enabled);
	PushButton_RESETPARAMS->setEnabled(enabled);
	PushButton_PURGEHISTORY->setEnabled(enabled);
	TextLabel_TYPE->setEnabled(enabled);
	ComboBox_SEARCHTYPE->setEnabled(enabled);
	ComboBox_SEARCHLIMIT->setEnabled(enabled);
	LineEdit_SEARCHSIZE->setEnabled(enabled);
	ComboBox_SEARCHUNIT->setEnabled(enabled);
	Label_MAX_RESULTS->setEnabled(enabled);
	SpinBox_MAXRESULT->setEnabled(enabled);
}

/** */
int DCHubSearch::DC_CallBack( CDCMessage * DCMsg )
{
	SocketCallbackMutex.lock();

	int err = -1;

	if ( (DCMsg != 0) && (m_pMessageList != 0) )
	{
		m_pMessageList->Add( DCMsg );
		err = 0;
	}

	SocketCallbackMutex.unlock();

	return err;
}

/** */
void DCHubSearch::timerDone()
{
	int i,t;

	if ( m_eSearchState == egssREADY )
	{
		switch ( CSearchManager::Instance()->SearchType() )
		{
			case estyNONE:
				break;
			case estySINGLE:
			case estyMULTI:
				if ( CSearchManager::Instance()->GetCallBackFunction() != m_pOurCallback )
				{
					m_eSearchState = egssOTHER;
					Label_GLOBAL_SEARCH_STATUS->setText(tr("Other search..."));
					SetSearchView(false);
				}
				else
				{
					AddLogMessage("Bug: someone else started our search (state=ready type=single/multi)\n");
					Label_GLOBAL_SEARCH_STATUS->setText(tr("Search..."));
					m_eSearchState = egssSEARCH;
					SetSearchView(false);
				}
				break;
			case estyEXTERNAL:
				m_eSearchState = egssAUTO;
				Label_GLOBAL_SEARCH_STATUS->setText(tr("Auto search..."));
				SetSearchView(false);
				
				/* show log message */
				ShowResults(false);
				
				break;
		}
	}
	else if ( m_eSearchState == egssOTHER )
	{
		switch ( CSearchManager::Instance()->SearchType() )
		{
			case estyNONE:
				m_eSearchState = egssREADY;
				Label_GLOBAL_SEARCH_STATUS->setText(tr("Ready"));
				SetSearchView(true);
				break;
			case estySINGLE:
			case estyMULTI:
				/* other search is still running */
				break;
			case estyEXTERNAL:
				m_eSearchState = egssAUTO;
				Label_GLOBAL_SEARCH_STATUS->setText(tr("Auto search..."));
				SetSearchView(false);
				break;
		}
	}
	else if ( m_eSearchState == egssAUTO )
	{
		switch ( CSearchManager::Instance()->SearchType() )
		{
			case estyNONE:
				/* show any more log messages */
				ShowResults(false);
				
				AddLogMessage(tr("Auto search finished"));
				
				m_eSearchState = egssREADY;
				Label_GLOBAL_SEARCH_STATUS->setText(tr("Ready"));
				SetSearchView(true);
				break;
			case estySINGLE:
			case estyMULTI:
				if ( CSearchManager::Instance()->GetCallBackFunction() != m_pOurCallback )
				{
					m_eSearchState = egssOTHER;
					Label_GLOBAL_SEARCH_STATUS->setText(tr("Other search..."));
					SetSearchView(false);
				}
				else
				{
					AddLogMessage("Bug: someone else started our search (state=auto type=single/multi)\n");
					Label_GLOBAL_SEARCH_STATUS->setText(tr("Search..."));
					m_eSearchState = egssSEARCH;
					SetSearchView(false);
				}
				break;
			case estyEXTERNAL:
				/* auto search still running */
				/* show any more log messages */
				ShowResults(false);
				break;
		}
	}
	else if ( m_eSearchState == egssSEARCH )
	{
		ShowResults(false);
		qApp->processEvents();
		
		switch ( CSearchManager::Instance()->SearchType() )
		{
			case estyNONE:
				m_eSearchState = egssREADY;
				Label_GLOBAL_SEARCH_STATUS->setText(tr("Ready"));
				SetSearchView( true );
				setCaption( tr("%1 - %2 Results").arg(GetSearchQueryString()).arg(m_pSearchResultList->Count()) );
				AddLogMessage( tr("Search ended with %1 results").arg(m_pSearchResultList->Count()) );
				
				if ( m_nFilteredResults > 0 )
				{
					AddLogMessage( tr("%1 results were filtered").arg(m_nFilteredResults) );
				}
				break;
			case estySINGLE:
			case estyMULTI:
				if ( CSearchManager::Instance()->GetCallBackFunction() == m_pOurCallback )
				{
					LineEdit_LOGHUBS->setText( 
						QString().setNum(CSearchManager::Instance()->HubIndex()) +
						"/" +
						QString().setNum(CSearchManager::Instance()->HubCount()) +
						" (" +
						QString().setNum(CSearchManager::Instance()->HubError()) +
						")" );

					i=0;
					if (CSearchManager::Instance()->HubCount()>0)
						i = ((CSearchManager::Instance()->HubIndex()*100)/CSearchManager::Instance()->HubCount());
					ProgressBar_LOGHUBS->setProgress(i);

					i = time(0)-CSearchManager::Instance()->StartTime();

					if ( CSearchManager::Instance()->HubIndex() > 0 )
						t = ((i*CSearchManager::Instance()->HubCount())/CSearchManager::Instance()->HubIndex());
					else
						t = 0;

					LineEdit_LOGTIME->setText( (CUtils::GetTimeString(i) + "/" + CUtils::GetTimeString(t)).Data() );

				}
				else
				{
					m_eSearchState = egssOTHER;
					Label_GLOBAL_SEARCH_STATUS->setText(tr("Other search..."));
					
					setCaption( tr("%1 - %2 Results").arg(GetSearchQueryString()).arg(m_pSearchResultList->Count()) );
					AddLogMessage( tr("Search ended with %1 results").arg(m_pSearchResultList->Count()) );
					
					if ( m_nFilteredResults > 0 )
					{
						AddLogMessage( tr("%1 results were filtered").arg(m_nFilteredResults) );
					}
				}
				break;
			case estyEXTERNAL:
				m_eSearchState = egssAUTO;
				Label_GLOBAL_SEARCH_STATUS->setText(tr("Auto search..."));
				
				setCaption( tr("%1 - %2 Results").arg(GetSearchQueryString()).arg(m_pSearchResultList->Count()) );
				AddLogMessage( tr("Search ended with %1 results").arg(m_pSearchResultList->Count()) );
				
				if ( m_nFilteredResults > 0 )
				{
					AddLogMessage( tr("%1 results were filtered").arg(m_nFilteredResults) );
				}
				break;
		}
	}

	// restart timer
	m_Timer.start( 500, true );
}

/** */
void DCHubSearch::ShowResults( bool bClearList )
{
	CDCMessage *DCMsg;
	bool bupdate;

	if ( bClearList == true )
	{
		ClearSearchResults();
	}

	if ( SocketCallbackMutex.tryLock() == false )
	{
		return;
	}

	bupdate = false;

	if ( m_pMessageList != 0 )
	{
		while( (DCMsg = m_pMessageList->Next(0)) != 0 )
		{
			m_pMessageList->Remove(DCMsg);

			switch ( DCMsg->m_eType )
			{
				case DC_MESSAGE_SEARCHRESULT:
				{
					bool b;
					b = DC_SearchResult( (CMessageSearchResult *) DCMsg );

					if ( b )
					{
						DCMsg  = 0;
						bupdate = true;
						ListView_SEARCHRESULT->setUpdatesEnabled(false);
					}

					break;
				}

				case DC_MESSAGE_SEARCHRESULT_USER:
				{
					CMessageSearchResultUser * msg = (CMessageSearchResultUser*) DCMsg;

					QListViewItem *item = new QListViewItem(ListView_SEARCHRESULTUSER);
					item->setText(0,msg->m_sNick.Data());
					item->setText(1,msg->m_sHubName.Data());
					break;
				}

				case DC_MESSAGE_LOG:
				{
					DC_LogMessage( (CMessageLog *) DCMsg );

					break;
				}

				default:
				{
					break;
				}
			}

			if ( DCMsg )
				delete DCMsg;
		}
	}

	SocketCallbackMutex.unlock();

	if ( bupdate )
	{
		ListView_SEARCHRESULT->setUpdatesEnabled(true);
		ListView_SEARCHRESULT->triggerUpdate();
	}
}

/** search result */
bool DCHubSearch::DC_SearchResult( CMessageSearchResult * MessageSearchResult )
{
	bool res = false;

	if ( CSearchManager::Instance()->SearchType() == estyNONE )
	{
		return res;
	}

	if ( SpinBox_MAXRESULT->value() > 0 )
	{
		if ( m_pSearchResultList->Count() >= SpinBox_MAXRESULT->value() )
		{
			++m_nFilteredResults;
			return res;
		}
	}

	/* SRs will have already been adjusted by CClient if in passive mode */
	if ( (g_pConfig->GetAdjustSearchResultHubNames()) &&
	     (g_pConfig->GetMode() == ecmACTIVE) )
	{
		if ( g_pConnectionManager->GetConnectedHubCount() == 1 )
		{
			if ( RadioButton_CONNECTEDHUBS->isChecked() ||
			     RadioButton_CONNECTEDSINGLEHUB->isChecked() )
			{
				std::map<CString, CString> * hubmap = g_pConnectionManager->GetConnectedHubServerMap();
				
				if ( hubmap != 0 )
				{
					std::map<CString, CString>::const_iterator it = hubmap->begin();
					if ( it != hubmap->end() )
					{
						MessageSearchResult->m_sHubName = it->first;
						MessageSearchResult->m_sHubHost = it->second;
						// printf( "Adjusted hub name to %s\n", it->first.Data() );
						// printf( "Adjusted hub host to %s\n", it->second.Data() );
					}
					
					delete hubmap;
				}
			}
		}
		else
		{
			if ( RadioButton_CONNECTEDSINGLEHUB->isChecked() )
			{
				QString both = ComboBox_CONNECTEDHUBS->currentText();
				if ( !both.isEmpty() )
				{
					MessageSearchResult->m_sHubName = both.left( both.findRev('(') ).ascii();
					int start = both.findRev('(') + 1;
					MessageSearchResult->m_sHubHost = both.mid( start, both.length() - (start+1) ).ascii();
				}
			}
		}
	}

	ShowSearchResult( MessageSearchResult, ListView_SEARCHRESULT );

	res = true;

	m_pSearchResultList->Add(MessageSearchResult);

	LCDNumber_RESULTS->display( (int) m_pSearchResultList->Count() );

	return res;
}

/** */
QListViewItem * DCHubSearch::ShowSearchResult( CMessageSearchResult * MessageSearchResult, QListView * parent )
{
	DC_ListResult * item;
	
	if ( m_eCurrentGrouping == egsrtNONE )
	{
		item = new DC_ListResult(parent);
		AddSearchResult( MessageSearchResult, item );
	}
	else
	{
		CDir cdir;
		CString filename, pathname;
		QString key;
		int column;
		DC_ListResult * groupitem;
		
		switch ( m_eCurrentGrouping )
		{
			case egsrtFILE:
				column = RESULTS_COLUMN_FILE;
				cdir.SplitPathFile(MessageSearchResult->m_sFile,pathname,filename);
				key = QString::fromAscii(filename.Data());
				break; 
			case egsrtSIZE:
				column = RESULTS_COLUMN_SIZE;
				key.setNum(MessageSearchResult->m_nSize);
				break;
			case egsrtNICK:
				column = RESULTS_COLUMN_NICK;
				key = QString::fromAscii(MessageSearchResult->m_sNick.Data());
				break;
			case egsrtSLOTS_FREE:
				column = RESULTS_COLUMN_FREESLOTS;
				key.setNum(MessageSearchResult->m_nFreeSlot);
				break;
			case egsrtSLOTS_TOTAL:
				column = RESULTS_COLUMN_TOTALSLOTS;
				key.setNum(MessageSearchResult->m_nMaxSlot);
				break;
			case egsrtHUB:
				column = RESULTS_COLUMN_HUB;
				key = QString::fromAscii(MessageSearchResult->m_sHubName.Data());
				break;
			case egsrtPATH:
				column = RESULTS_COLUMN_PATH;
				cdir.SplitPathFile(MessageSearchResult->m_sFile,pathname,filename);
				key = QString::fromAscii(pathname.Data());
				break;
			case egsrtHOST:
				column = RESULTS_COLUMN_HOST;
				key = QString::fromAscii(MessageSearchResult->m_sHubHost.Data());
				break;
			case egsrtHASH:
				column = RESULTS_COLUMN_TTH;
				key = QString::fromAscii(MessageSearchResult->m_sHash.Data());
				break;
			default:
				column = RESULTS_COLUMN_FILE;
				cdir.SplitPathFile(MessageSearchResult->m_sFile,pathname,filename);
				key = QString::fromAscii(filename.Data());
				break;
		}
		
		if ( key.isEmpty() )
		{
			if ( m_pEmptyGroup == 0 )
			{
				m_pEmptyGroup = new DC_ListResult( ListView_SEARCHRESULT );
				m_pEmptyGroup->setSelectable( false );
				m_pEmptyGroup->mycol = RESULTS_COLUMN_COUNT;
				m_pEmptyGroup->setOpen( true );
			}
			
			groupitem = m_pEmptyGroup;
		}
		else
		{
			groupitem = m_GroupDict.find(key);
			
			if ( groupitem == 0 )
			{
				groupitem = new DC_ListResult( ListView_SEARCHRESULT );
				groupitem->setSelectable( false );
				groupitem->mycol = RESULTS_COLUMN_COUNT;
				groupitem->setOpen( true );
				
				groupitem->setText( column, key );
				m_GroupDict.insert( key, groupitem );
			}
		}
		
		item = new DC_ListResult( groupitem );
		AddSearchResult( MessageSearchResult, item );
		
		groupitem->myvalue = groupitem->childCount();
		groupitem->setText( RESULTS_COLUMN_COUNT, QString().setNum(groupitem->myvalue) );
	}

	return item;
}

/** */
QListViewItem * DCHubSearch::ShowSearchResult( CMessageSearchResult * MessageSearchResult, DC_ListResult * parent )
{
	DC_ListResult *item = new DC_ListResult(parent);

	AddSearchResult( MessageSearchResult, item );

	return item;
}

/** */
bool DCHubSearch::AddSearchResult( CMessageSearchResult * MessageSearchResult, DC_ListResult * item )
{
	bool res = false;
	bool visible = true;
	int i;
	QString s;
	CString spath,sname;

	if ( MessageSearchResult->m_nFreeSlot < (unsigned int)SpinBox_FREESLOTS->value() )
	{
		visible = false;
	}

	s = MessageSearchResult->m_sFile.Data();

	if (CheckBox_SEARCHFILEONLY->isChecked())
	{
	        // Checks if s matches the search pattern
		QString search_pattern;
		//for(i=0;i<=LineEdit_SEARCH->text().contains(" ");i++)
		for(i=0;i<=Combobox_SEARCH->currentText().contains(" ");i++)
		{
		        //search_pattern = LineEdit_SEARCH->text().section(" ",i,i);
		        search_pattern = Combobox_SEARCH->currentText().section(" ",i,i);

			if ( s.find(search_pattern,0,false) == -1 )
			{
				visible = false;
			}
		}
	}

	// split filename and path
	s = MessageSearchResult->m_sFile.Data();

	CDir().SplitPathFile(s.ascii(),spath,sname);

	item->myvalue = MessageSearchResult->m_nSize;
	item->mycol   = RESULTS_COLUMN_SIZE;

	item->setText(RESULTS_COLUMN_FILE,QString::fromAscii(sname.Data()));
	if (MessageSearchResult->m_bFolder)
	{
		item->setText(RESULTS_COLUMN_SIZE,tr("Folder"));
		
		item->setPixmap(RESULTS_COLUMN_FILE,g_pIconLoader->GetPixmap(eiFOLDER_BLUE));
	}
	else
	{
		item->setText(RESULTS_COLUMN_SIZE,DCGuiUtils::GetSizeString(MessageSearchResult->m_nSize));
		
		eFileTypes type = CFileManager::Instance()->GetFileType(sname);
		QPixmap icon;
		
		switch (type)
		{
			case eftMP3:
				icon = g_pIconLoader->GetPixmap(eiFILETYPE_MP3);
				break;
			case eftARCHIVE:
				icon = g_pIconLoader->GetPixmap(eiFILETYPE_ARCHIVE);
				break;
			case eftDOCUMENT:
				icon = g_pIconLoader->GetPixmap(eiFILETYPE_DOCUMENT);
				break;
			case eftAPPLICATION:
				icon = g_pIconLoader->GetPixmap(eiFILETYPE_APPLICATION);
				break;
			case eftPICTURE:
				icon = g_pIconLoader->GetPixmap(eiFILETYPE_PICTURE);
				break;
			case eftVIDEO:
				icon = g_pIconLoader->GetPixmap(eiFILETYPE_VIDEO);
				break;
			default:
				icon = g_pIconLoader->GetPixmap(eiFILETYPE_UNKNOWN);
				break;
		}
		
		item->setPixmap(RESULTS_COLUMN_FILE,icon);
	}
	item->setText(RESULTS_COLUMN_TTH,MessageSearchResult->m_sHash.Data());
	item->setText(RESULTS_COLUMN_NICK,QString::fromAscii(MessageSearchResult->m_sNick.Data()));
	item->setText(RESULTS_COLUMN_FREESLOTS,QString().setNum(MessageSearchResult->m_nFreeSlot));
	item->setText(RESULTS_COLUMN_TOTALSLOTS,QString().setNum(MessageSearchResult->m_nMaxSlot));
	item->setText(RESULTS_COLUMN_HUB,QString::fromAscii(MessageSearchResult->m_sHubName.Data()));
	item->setText(RESULTS_COLUMN_PATH,QString::fromAscii(spath.Data()));
	item->setText(RESULTS_COLUMN_HOST,MessageSearchResult->m_sHubHost.Data());
	
	item->setVisible(visible);
	item->setEnabled(visible);

	return res;
}

/** */
void DCHubSearch::GroupSearchResults( eGroupSearchResultType type )
{
	if ( m_eCurrentGrouping == type )
	{
		return;
	}
	
	CMessageSearchResult * msg = 0;

	ListView_SEARCHRESULT->setUpdatesEnabled(false);
	
	ClearSearchResults();

	ListView_SEARCHRESULT->setRootIsDecorated((type != egsrtNONE));

	m_eCurrentGrouping = type;

	if ( type == egsrtNONE )
	{
		ListView_SEARCHRESULT->hideColumn(RESULTS_COLUMN_COUNT);
		while ( (msg = m_pSearchResultList->Next(msg)) != 0 )
		{
			AddSearchResult( msg, new DC_ListResult( ListView_SEARCHRESULT ) );
		}
	}
	else
	{
		ListView_SEARCHRESULT->setColumnWidth( RESULTS_COLUMN_COUNT, ListView_SEARCHRESULT->columnWidth( RESULTS_COLUMN_FREESLOTS ) );
		
		while ( (msg = m_pSearchResultList->Next(msg)) != 0 )
		{
			ShowSearchResult( msg, ListView_SEARCHRESULT );
		}
		
		ListView_SEARCHRESULT->setSorting( RESULTS_COLUMN_COUNT );
		ListView_SEARCHRESULT->sort();
	}

	ListView_SEARCHRESULT->setUpdatesEnabled(true);
	ListView_SEARCHRESULT->triggerUpdate();
}

/** */
void DCHubSearch::DC_LogMessage( CMessageLog * MessageLog )
{
	AddLogMessage( QString::fromAscii(MessageLog->sMessage.Data()) );
}

void DCHubSearch::AddLogMessage( QString message )
{
	bool bscroll;

	if ( TextEdit_LOG->verticalScrollBar()->maxValue() == TextEdit_LOG->verticalScrollBar()->value() )
	{
		bscroll = true;
	}
	else
	{
		bscroll = false;
	}
	
	message = QDateTime::currentDateTime().toString( "yyyy-MM-dd hh:mm.ss " ) + message;
	
	TextEdit_LOG->append(message);
	
	if ( bscroll )
	{
		TextEdit_LOG->scrollToBottom();
		TextEdit_LOG->moveCursor( QTextEdit::MoveEnd, false );
	}
}

QString DCHubSearch::GetSearchQueryString()
{
	if ( (ListView_SEARCH->childCount() == 1) && (ListView_SEARCH->firstChild() != 0) )
	{
		return ListView_SEARCH->firstChild()->text(1);
	}
	else
	{
		return tr("%1 items").arg( ListView_SEARCH->childCount() );
	}
}

/** */
CDCMessage * DCHubSearch::GetSearchObject()
{
	CMessageSearchUser * MessageSearchUser = 0;
	CMessageSearchFile * MessageSearchFile = 0;

	//if ( LineEdit_SEARCH->text().isEmpty() )
	if ( Combobox_SEARCH->currentText().isEmpty() )
	{
		return 0;
	}

	// set searchmode
	if ( ComboBox_SEARCHTYPE->currentItem() == 0 )
	{
		MessageSearchUser = new CMessageSearchUser();
		//MessageSearchUser->m_sNick = LineEdit_SEARCH->text().ascii();
		MessageSearchUser->m_sNick = Combobox_SEARCH->currentText().simplifyWhiteSpace().ascii();

		return MessageSearchUser;
	}

	MessageSearchFile = new CMessageSearchFile();

	MessageSearchFile->m_eFileType = eFileTypes(ComboBox_SEARCHTYPE->currentItem());
	
	MessageSearchFile->m_sString = Combobox_SEARCH->currentText().simplifyWhiteSpace().ascii();

	if ( MessageSearchFile->m_eFileType == eftHASH )
	{
		// fix tth
		MessageSearchFile->m_sString = MessageSearchFile->m_sString.ToUpper();
		
		// cannot check TTH valid here because it comes up twice
	}
	
	MessageSearchFile->m_bLocal = (g_pConfig->GetMode() == ecmPASSIVE);
	/* CClient handles setting source to the correct nick or IP */
	
	MessageSearchFile->m_nSize = LineEdit_SEARCHSIZE->text().toULongLong();

	if ( ComboBox_SEARCHUNIT->currentItem() == 1 )
	{
		MessageSearchFile->m_nSize *= 1024;
	}
	else if ( ComboBox_SEARCHUNIT->currentItem() == 2 )
	{
		MessageSearchFile->m_nSize *= 1024*1024;
	}
	else if ( ComboBox_SEARCHUNIT->currentItem() == 3 )
	{
		MessageSearchFile->m_nSize *= 1024*1024*1024;
	}

	if ( ComboBox_SEARCHLIMIT->currentItem() == 1 )
	{
		/* disable size limit if search is "at least 0 bytes" */
		MessageSearchFile->m_bSizeLimit = (MessageSearchFile->m_nSize > 0);
		
		MessageSearchFile->m_eSizeType = esstATLEAST;
	}
	else if ( ComboBox_SEARCHLIMIT->currentItem() == 2 )
	{
		MessageSearchFile->m_bSizeLimit = true;
		MessageSearchFile->m_eSizeType = esstATMOST;
	}
	else if ( ComboBox_SEARCHLIMIT->currentItem() == 0 )
	{
		MessageSearchFile->m_bSizeLimit = false;
		MessageSearchFile->m_eSizeType = esstATLEAST;
	}

	return MessageSearchFile;
}

/** */
void DCHubSearch::slotRefreshConnectedHubs()
{
	ComboBox_CONNECTEDHUBS->clear();

	std::map<CString, CString> * hubmap = g_pConnectionManager->GetConnectedHubServerMap();
	if ( hubmap != 0 )
	{
		for ( std::map<CString, CString>::const_iterator it = hubmap->begin(); it != hubmap->end(); ++it )
		{
			// add hubname with hubhost
			ComboBox_CONNECTEDHUBS->insertItem( (it->first + " (" + it->second +")").Data() );
		}

		delete hubmap;
	}
}

/** */
void DCHubSearch::UpdateHidden( bool SearchFileOnly, int FreeSlots, bool TextFilter )
{
	QListViewItemIterator iter( ListView_SEARCHRESULT );

	ListView_SEARCHRESULT->setUpdatesEnabled(false);

	for ( ; iter.current(); iter++ )
	{
		bool visible = true;
		QString path_file;

		path_file = iter.current()->text(RESULTS_COLUMN_PATH) + iter.current()->text(RESULTS_COLUMN_FILE);

		if ( SearchFileOnly )
		{
			// Checks if iter.current() path+filename matches the search pattern
			QString search_pattern;
			//for(int j=0; j<=LineEdit_SEARCH->text().contains(" "); j++)
			for(int j=0; j<=Combobox_SEARCH->currentText().contains(" "); j++)
                        {
				//search_pattern = LineEdit_SEARCH->text().section(" ",j,j);
				search_pattern = Combobox_SEARCH->currentText().section(" ",j,j);
				if ( path_file.find(search_pattern,0,false) == -1 )
                                {
					visible = false;
                                }
			}
		}

		// filtering with include
		if (TextFilter && (LineEdit_INCLUDE->text().isEmpty() == false))
		{
			QString search_pattern;
			for(int i=0; search_pattern = LineEdit_INCLUDE->text().section(" ",i,i,QString::SectionSkipEmpty); i++)
			{
				if ( path_file.find(search_pattern,0,false) == -1 )
				{
					visible = false;
				}
			}
		}
		// filtering with exclude
		if (TextFilter && (LineEdit_EXCLUDE->text().isEmpty() == false))
		{
			QString search_pattern;
			for(int i=0; search_pattern = LineEdit_EXCLUDE->text().section(" ",i,i,QString::SectionSkipEmpty); i++)
			{
				if ( path_file.find(search_pattern,0,false) != -1 )
				{
					visible = false;
				}
			}
		}
		
		if ( visible )
		{
			if ( iter.current()->text(RESULTS_COLUMN_FREESLOTS).toInt() < FreeSlots )
			{
				visible = false;
			}
		}

		iter.current()->setVisible(visible);
		iter.current()->setEnabled(visible);
	}

	ListView_SEARCHRESULT->setUpdatesEnabled(true);
	ListView_SEARCHRESULT->triggerUpdate();
}

/** */
void DCHubSearch::slotTextFilterResults()
{
	UpdateHidden( CheckBox_SEARCHFILEONLY->isOn(), SpinBox_FREESLOTS->value(), true );
}

/** */
void DCHubSearch::slotTextFilterReset()
{
	UpdateHidden( CheckBox_SEARCHFILEONLY->isOn(), SpinBox_FREESLOTS->value(), false );
}

/** */
void DCHubSearch::slotToggledSearchFileOnly( bool chkstate )
{
	UpdateHidden( chkstate, SpinBox_FREESLOTS->value() );
}

/** */
void DCHubSearch::slotChangedFreeSlots( int free_slots )
{
	UpdateHidden( CheckBox_SEARCHFILEONLY->isOn(), free_slots );
}

/** */
void DCHubSearch::slotSearchReturnPressed()
{
	if ( m_eSearchState == egssREADY )
	{	
		startSearch();
	}
	else
	{
		CSearchManager::Instance()->StopSearch();
		PushButton_SEARCH->setEnabled(false);
	}
}

/** */
void DCHubSearch::startSearch()
{
	CDCMessage * DCMessage;
	CStringList<CString> * hublist = 0;

	eSearchMode searchmode;
	eSearchType searchtype;
	
	if ( CSearchManager::Instance()->SearchType() != estyNONE )
	{
		QMessageBox::critical( this, tr("Hub Search Error"),
			tr("Another search is already running!") );
		return;
	}

	// clear messagelist
	SocketCallbackMutex.lock();

	/* FIXME why not just clear the list? CList has auto-delete */
	while( (DCMessage = m_pMessageList->Next(0)) != 0 )
	{
		m_pMessageList->Del(DCMessage);
	}

	m_pSearchResultList->Clear();

	SocketCallbackMutex.unlock();

	// clear searchresults
	ClearSearchResults();
	ListView_SEARCHRESULTUSER->clear();

	// reset counter
	LCDNumber_RESULTS->display(0);
	ProgressBar_LOGHUBS->setProgress(0);
	m_nFilteredResults = 0;

	// reset
	LineEdit_LOGTIME->clear();
	LineEdit_LOGHUBS->setText("0/0");

	// clear the list on new search if only one search in the list
	if ( (m_pSearchQueryList != 0) && (m_pSearchQueryList->Count() == 1) )
	{
		m_pSearchQueryList->Clear();
		ListView_SEARCH->clear();
	}

	// add a new search into empty list
	if ( (m_pSearchQueryList == 0) || (m_pSearchQueryList->Count() == 0) )
	{
		slotAddSearchQueue();
	}
	
	addHistory();
	
	if ( (m_pSearchQueryList == 0) || (m_pSearchQueryList->Count() == 0) )
	{
		QMessageBox::critical( this, tr("Hub Search Error"),
				       tr("Please add a search!") );
		return;
	}

	// single or multi search
	if ( CheckBox_MULTISEARCH->isChecked() )
	{
		searchtype = estyMULTI;
	}
	else
	{
		searchtype = estySINGLE;
	}

	if ( (g_pConfig->GetMode() == ecmPASSIVE) && (searchtype == estyMULTI) )
	{
		QMessageBox::critical( this, tr("Hub Search Error"),
			tr("Multi Search only work in active mode!") );
		return;
	}

	if ( CSearchManager::Instance()->GetCallBackFunction() != m_pOurCallback )
	{
		CSearchManager::Instance()->SetCallBackFunction( m_pOurCallback );
	}

	// set searchmode
	if ( RadioButton_CONNECTEDSINGLEHUB->isChecked() )
	{
		searchmode = esmCONNECTEDSINGLE;
	}
	else if ( RadioButton_CONNECTEDHUBS->isChecked() )
	{
		searchmode = esmCONNECTEDALL;
	}
	else if ( RadioButton_FILTEREDHUBS->isChecked() )
	{
		hublist = g_pHubListManager->GetFilteredHubList();

		searchmode = esmPUBLIC;
	}
	else if ( RadioButton_AVAILABLEHUBS->isChecked() )
	{
		searchmode = esmPUBLIC;
	}
	else
	{
		searchmode = esmBOOKMARK;
	}

	// set max clients
	CSearchManager::Instance()->MaxClients(SpinBox_MAXTHREADS->value());
	// enable/disable tag
	CSearchManager::Instance()->EnableTag(CheckBox_ENABLETAG->isChecked());

	if ( searchmode == esmCONNECTEDSINGLE )
	{
		CString hubname;

		// get current hubname
		hubname = ComboBox_CONNECTEDHUBS->currentText().ascii();
		// extract hubhost
		hubname = hubname.Mid( hubname.FindRev('(')+1, hubname.Length()-hubname.FindRev('(')-2 );

		// check for empty hubhost
		if ( hubname.IsEmpty() )
		{
			QMessageBox::critical( this, tr("Hub Search Error"),
				tr("Please select a connected hub.") );

			return;
		}
		else
		{
			// add hubhost to the hublist
			hublist = new CStringList<CString>();
			hublist->Add( hubname, new CString(hubname) );
			
			AddLogMessage( tr("Search for \"%1\" on %2").arg(GetSearchQueryString(),ComboBox_CONNECTEDHUBS->currentText()) );
			
			if ( CSearchManager::Instance()->StartSearch(searchmode,searchtype,m_pSearchQueryList,hublist) != 0 )
			{
				QMessageBox::critical( this, tr("Hub Search Error"),
					tr("No connected hubs found.") );

				return;
			}
		}
	}
	else if ( searchmode == esmCONNECTEDALL )
	{
		AddLogMessage( tr("Search for \"%1\" on %2").arg(GetSearchQueryString(),tr("all connected hubs")) );
		
		if ( CSearchManager::Instance()->StartSearch(searchmode,searchtype,m_pSearchQueryList,hublist) != 0 )
		{
			QMessageBox::critical( this, tr("Hub Search Error"),
				tr("No connected hubs found.") );

			return;
		}
	}
	else
	{
		if ( searchmode == esmPUBLIC )
		{
			QString whichhubs;
			if ( hublist )
			{
				whichhubs = tr("%1 filtered public hubs").arg(hublist->Count());
			}
			else
			{
				whichhubs += tr("all public hubs");
			}
			
			AddLogMessage( tr("Search for \"%1\" on %2").arg(GetSearchQueryString(),whichhubs) );
			
			if ( CSearchManager::Instance()->StartSearch(searchmode,searchtype,m_pSearchQueryList,hublist) != 0 )
			{
				delete hublist;

				QMessageBox::critical( this, tr("Hub Search Error"),
					tr("No hubs found.") );

				return;
			}
		}
		else
		{
			AddLogMessage( tr("Search for \"%1\" on %2").arg(GetSearchQueryString(),tr("all bookmark hubs")) );
			if ( CSearchManager::Instance()->StartSearch(searchmode,searchtype,m_pSearchQueryList,hublist) != 0 )
			{
				QMessageBox::critical( this, tr("Hub Search Error"),
					tr("No hubs found.") );

				return;
			}
		}
	}

	// update view
	SetSearchView( false );
	m_eSearchState = egssSEARCH;
	Label_GLOBAL_SEARCH_STATUS->setText(tr("Search..."));
	
	setCaption( tr("%1 - Searching").arg(GetSearchQueryString()) );
	
	return;
}

/** */
void DCHubSearch::slotAddSearchQueue()
{
	if ( eFileTypes(ComboBox_SEARCHTYPE->currentItem()) == eftHASH )
	{
		CString tth = Combobox_SEARCH->currentText().simplifyWhiteSpace().upper().ascii();
		CByteArray dst;
		if ( CBase32::Decode( &dst, &tth ) != CFileHasher::HashSize() )
		{
			if ( QMessageBox::warning(
				this,
				tr("Add search"),
				tr("Invalid TTH"),
				QMessageBox::Ignore,
				QMessageBox::Cancel | QMessageBox::Default | QMessageBox::Escape ) == QMessageBox::Cancel )
			{
				return;
			}
		}
	}
	
	CDCMessage * msg, *msg1;
	
	CMessageSearchUser * MessageSearchUser = 0;
	CMessageSearchFile * MessageSearchFile = 0;

	addHistory();

	if ( m_pSearchQueryList == 0 )
	{
		m_pSearchQueryList = new CList<CDCMessage>();
	}

	if ( (msg = GetSearchObject()) == 0 )
	{
		return;
	}

	if ( msg->m_eType == DC_MESSAGE_SEARCH_USER )
	{
		MessageSearchUser = (CMessageSearchUser *)msg;
	}
	else
	{
		MessageSearchFile = (CMessageSearchFile *)msg;
	}

	msg1 = 0;

	while ( (msg1 = m_pSearchQueryList->Next(msg1)) != 0 )
	{
		if ( msg1->m_eType == msg->m_eType )
		{
			//TODO: compare if search already in the list
			if ( MessageSearchUser )
			{
				if ( ((CMessageSearchUser*)msg1)->m_sNick == MessageSearchUser->m_sNick )
				{
					return;
				}
			}
			else
			{
				if ( ((CMessageSearchFile*)msg1)->m_sString == MessageSearchFile->m_sString )
				{
					return;
				}
			}
		}
	}

	DC_QSearchListViewItem *item = new DC_QSearchListViewItem(ListView_SEARCH);

	item->p_msg = msg;

	if ( MessageSearchUser )
	{
		item->setText(0,tr("USER"));
		item->setText(1,MessageSearchUser->m_sNick.Data());
	}
	else
	{
		item->setText(0,tr("FILE"));
		item->setText(1,MessageSearchFile->m_sString.Data());
	}

	m_pSearchQueryList->Add(msg);
}

/** */
void DCHubSearch::slotReset()
{
	Combobox_SEARCH->clearEdit();
	LineEdit_SEARCHSIZE->setText("0");
	ComboBox_SEARCHUNIT->setCurrentItem(0);
	ComboBox_SEARCHLIMIT->setCurrentItem(1);
	slotSizeLimitChanged(1);
	ComboBox_SEARCHTYPE->setCurrentItem(1);
	SpinBox_FREESLOTS->setValue(0);
	ListView_SEARCH->clear();

	if ( m_pSearchQueryList != 0 )
	{
		m_pSearchQueryList->Clear();
	}
}

/** */
void DCHubSearch::slotDoubleClickedSearchResult( QListViewItem * /*item*/ )
{
	QListViewItem * curitem = ListView_SEARCHRESULT->currentItem();
	
	/* group parents are not selectable */
	if ( !curitem || (curitem->isSelectable() == false) )
	{
		return;
	}
	
	if ( curitem->text(RESULTS_COLUMN_SIZE) == tr("Folder") )
	{
		DownloadFolder( curitem );
	}
	else
	{
		// add file as new source
		DCFileTool::AddFileSource( this, curitem->text(RESULTS_COLUMN_NICK).ascii(),
				curitem->text(RESULTS_COLUMN_HUB).ascii(),
				curitem->text(RESULTS_COLUMN_HOST).ascii(),
				(curitem->text(RESULTS_COLUMN_PATH)+curitem->text(RESULTS_COLUMN_FILE)).ascii(),
				curitem->text(RESULTS_COLUMN_FILE).ascii(),
				CString(),
				CString(),
				eltFILE,
				((DC_QListViewItem*)curitem)->myvalue,
				curitem->text(RESULTS_COLUMN_TTH).ascii() );
	}

	return;
/*
	int i;
	CString s;
	CByteArray cba;
	QByteArray qba;

	DC_ListResult *item1 = (DC_ListResult*)item;

	QStoredDrag *sd = new QStoredDrag( "dchubsearch/searchresult", this );

	s = item1->text(0).ascii();
	if ( s.Data() != 0 )
		cba.Append( s.Data(), s.Length() );
	cba.Append("\0",1);

	s = s.setNum(item1->myvalue);
	if ( s.Data() != 0 )
		cba.Append( s.Data(), s.Length() );
	cba.Append("\0",1);

	for(i=2;i<=6;i++)
	{
		s = item1->text(i).ascii();
		if ( s.Data() != 0 )
			cba.Append( s.Data(), s.Length() );
		cba.Append("\0",1);
	}

	qba.assign((const char*)cba.Data(),(unsigned int)cba.Size());
	sd->setEncodedData(qba);

	QDragObject *d = sd;
	d->dragCopy();
*/
}

/** */
void DCHubSearch::slotRightButtonClickedSearchResult( QListViewItem * /*item*/, const QPoint &, int column )
{
	int id;
	ulonglong size;
        QPtrList<QListViewItem> selitems;
        QListViewItem * curitem, * item;
	bool isadmin = false, allowDL = true, addUserCommands = true;
	CXml xml;
	DCClient * client = 0;
	QMap<int, DC_UserMenuCommand*> addedcommands;

	// user have select items ?
	DCGuiUtils::SelectedItems( ListView_SEARCHRESULT, selitems );
	
	// cannot download folders yet
	// check only one hub is selected for user commands
	
	QString hubhost;
	if ( selitems.count() > 0 )
	{
		hubhost = selitems.first()->text(RESULTS_COLUMN_HOST);
	}
	else
	{
		addUserCommands = false;
	}
	
	for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
	{
		if ( curitem->text(RESULTS_COLUMN_SIZE) == tr("Folder") )
		{
			allowDL = false;
		}
		
		if ( curitem->text(RESULTS_COLUMN_HOST) != hubhost )
		{
			addUserCommands = false;
		}
	}

	// check if admin for admin-menu
	for ( isadmin = false, curitem = selitems.first(); curitem; curitem = selitems.next() )
	{
		if ( g_pConnectionManager->IsAdmin(curitem->text(RESULTS_COLUMN_HUB).ascii(),curitem->text(RESULTS_COLUMN_HOST).ascii()) == false )
		{
			isadmin = false;
			break;
		}
		else
		{
			isadmin = true;
		}
	}
	
	QPopupMenu *m,*msort;

	m = new QPopupMenu(this);

	DCMenuHandler::InsertMenu( m, emiDOWNLOAD, (selitems.count() > 0) && allowDL );
	DCMenuHandler::InsertMenu( m, emiDOWNLOAD_TO, (selitems.count() > 0) && allowDL );
	DCMenuHandler::InsertMenu( m, emiDOWNLOAD_AS, (selitems.count() > 0) && allowDL );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiDOWNLOAD_FOLDER, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiDOWNLOAD_REPAIR, (selitems.count() > 0) && allowDL );
	DCMenuHandler::InsertMenu( m, emiDOWNLOAD_BIN_SECTORS, (selitems.count() > 0) && allowDL );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiDOWNLOAD_IN, (selitems.count() > 0) && allowDL );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiBROWSE_USER_FILES, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiCOPY_COLUMN_TO_CLIPBOARD, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiCOPY_ROW_TO_CLIPBOARD, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiCOPYDCFILELINK, (selitems.count() == 1) );
	DCMenuHandler::InsertMenu( m, emiCOPYMAGNETLINK, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiCONNECT_TO_HUB, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiADD_BOOKMARK, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );

	// insert submenu
	msort = DCMenuHandler::InsertMenu( m, emisGROUP, true );
	DCMenuHandler::InsertMenu( msort, emiDISABLE_GROUP, true );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_FILE, true );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_SIZE, true );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_NICK, true );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_HASH, true );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_SLOTS_FREE, true );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_SLOTS_TOTAL, true );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_HUB, true );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_PATH, true );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_HOST, true );

	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiKICK, isadmin && (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiFORCE_MOVE, isadmin && (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiLOAD );
	DCMenuHandler::InsertMenu( m, emiSAVE, (m_pSearchResultList->Count() > 0) );

	if ( addUserCommands )
	{
		curitem = selitems.first();
		client = g_pConnectionManager->GetClientForHub( curitem->text(RESULTS_COLUMN_HUB).ascii(), curitem->text(RESULTS_COLUMN_HOST).ascii() );
		
		if ( client != 0 )
		{
			addedcommands = client->AddMenuCommands( m, euccSearch );
		}
	}

	id = m->exec(QCursor::pos());

	delete m;

	if ( id == emiCOPYDCFILELINK )
	{
		CString s;

                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			s += "DCFILE://";
			s += xml.ToUTF8(curitem->text(RESULTS_COLUMN_HOST).ascii());
			s += "?file=";
			s += CHttp::Encode((curitem->text(RESULTS_COLUMN_PATH) + curitem->text(RESULTS_COLUMN_FILE)).ascii());
			s += "&size=";
			s += CString::number(((DC_QListViewItem*)curitem)->myvalue);
			s += "&nick=";
			s += CHttp::Encode(curitem->text(RESULTS_COLUMN_NICK).ascii());
			s += "&hub=";
			s += CHttp::Encode(curitem->text(RESULTS_COLUMN_HUB).ascii());
			
			if ( !(curitem->text(RESULTS_COLUMN_TTH).isEmpty()) )
			{
				s += "&hash=";
				s += CHttp::Encode(curitem->text(RESULTS_COLUMN_TTH).ascii());
			}
			s += "\n";
		}

		QApplication::clipboard()->setText(QString::fromAscii(s.Data()).stripWhiteSpace());
	}
	else if ( id == emiCOPYMAGNETLINK )
	{
		//magnet:?xt=urn:tree:tiger:EOSA5AGTL5SD3VWCF3R2OH2WMGXV3S3R7SYN4YA&xl=708780032&dn=FC-6-i386-disc1.iso
		QString text;
		
		for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
		{
			text += "magnet:?xt=urn:tree:tiger:";
			text += curitem->text(RESULTS_COLUMN_TTH);
			
			// get size from myvalue used for numeric sorting
			text += "&xl=";
			text += QString().setNum(((DC_QListViewItem*)curitem)->myvalue);
			
			text += "&dn=";
			QString filename = curitem->text(RESULTS_COLUMN_FILE);
			QUrl::encode(filename);
			filename.replace("%20","+");
			text += filename;
			
			text += "\n";
		}
		
		// remove trailing "\n"
		text = text.stripWhiteSpace();
		
		QApplication::clipboard()->setText(text);
	}
	else if ( (id == emiDOWNLOAD) || (id == emiDOWNLOAD_AS) || (id == emiDOWNLOAD_TO)
		|| (id == emiDOWNLOAD_REPAIR) || (id == emiDOWNLOAD_BIN_SECTORS) )
	{
		QString localrootpath;
		QString localname;

		// select downloadfolder for all selected files
		if ( id == emiDOWNLOAD_TO )
		{
			localrootpath = QFileDialog::getExistingDirectory( QString(), this, "bdf", tr("Select download folder"), true );

			if ( localrootpath.isEmpty() )
				return;
		}

                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			size = ((DC_QListViewItem *)curitem)->myvalue;
			localname = curitem->text(RESULTS_COLUMN_FILE);

			if ( (id == emiDOWNLOAD_AS) || (id == emiDOWNLOAD_REPAIR) || (id == emiDOWNLOAD_BIN_SECTORS) )
			{
				localrootpath = QFileDialog::getSaveFileName( localname, QString(), this, "bdf", tr("Select file for")+" "+localname );

				if ( localrootpath.isEmpty() )
					return;

				QFileInfo fi(localrootpath);
				localrootpath = fi.dirPath();
				localname     = fi.fileName();

				if ( (localrootpath.isEmpty()) || (localname.isEmpty()) )
					return;
			}

			// add transfer to the waitlist
			eRepairType repair;
			switch( id )
				{
				case emiDOWNLOAD_REPAIR:
					repair = ertBYTES;
					break;

				case emiDOWNLOAD_BIN_SECTORS:
					repair = ertSECTORS;
					break;

				default:
					repair = ertNONE;
					break;
				}

			DCFileTool::CheckFile( this, curitem->text(RESULTS_COLUMN_NICK).ascii(), curitem->text(RESULTS_COLUMN_HUB).ascii(), curitem->text(RESULTS_COLUMN_HOST).ascii(),
				(curitem->text(RESULTS_COLUMN_PATH)+curitem->text(RESULTS_COLUMN_FILE)).ascii(), localname.ascii(), CString(), localrootpath.ascii(), eltFILE,
				size, curitem->text(RESULTS_COLUMN_TTH).ascii(), false, repair );
		}

		// redraw the list
		ListView_SEARCHRESULT->triggerUpdate();
	}
	else if ( id == emiDOWNLOAD_IN )
	{
		QString localrootpath;
		QString localname;
		QString localpath;

		if ( !(curitem = selitems.first()) )
			return;

		size = ((DC_QListViewItem *)curitem)->myvalue;
		QString tth = curitem->text(RESULTS_COLUMN_TTH);

		// all files need equal size and TTH
                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			if ( size != ((DC_QListViewItem *)curitem)->myvalue )
				return;
			if ( tth != curitem->text(RESULTS_COLUMN_TTH) )
				return;
		}

		if ( DCFileTool::SelectFileSource( this, size, tth, localname, localrootpath, localpath ) == false )
		{
			return;
		}

                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			// add transfer to the waitlist
			DCFileTool::CheckFile( this, curitem->text(RESULTS_COLUMN_NICK).ascii(), curitem->text(RESULTS_COLUMN_HUB).ascii(), curitem->text(RESULTS_COLUMN_HOST).ascii(),
					  (curitem->text(RESULTS_COLUMN_PATH)+curitem->text(RESULTS_COLUMN_FILE)).ascii(), localname.ascii(), localpath.ascii(), localrootpath.ascii(), eltFILE,
					  size, curitem->text(RESULTS_COLUMN_TTH).ascii(), true);
		}

		// redraw the list
		ListView_SEARCHRESULT->triggerUpdate();
	}
	else if ( id == emiBROWSE_USER_FILES)
	{
		/** add transfer to the waitlist */
		CString empty;
                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			CString hubname(curitem->text(RESULTS_COLUMN_HUB).ascii());
			CString hubhost(curitem->text(RESULTS_COLUMN_HOST).ascii());

			if ( g_pConnectionManager->IsHubOnline(hubname,hubhost) == ehsNONE )
			{
				switch( QMessageBox::warning( this, tr("Filelist download"),
							      tr("Not connected to required hub!"),
							      tr("Connect"),
							      tr("Cancel"), 0, 0, 1 ) )
				{
					case 0:
						 g_pConnectionManager->Connect(hubname,hubhost);
						 break;
					case 1:
						break;
				}
			}

			g_pTransferView->DLM_QueueAdd( curitem->text(RESULTS_COLUMN_NICK).ascii(), hubname, hubhost,
						DC_USER_FILELIST, DC_USER_FILELIST,
						(curitem->text(RESULTS_COLUMN_PATH)+curitem->text(RESULTS_COLUMN_FILE)).ascii(),
						empty,
						eltBUFFER,
						0, 0, 0, empty );
		}
	}
	else if ( id == emiDOWNLOAD_FOLDER )
	{
		for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			DownloadFolder( curitem );
		}
	}
	else if ( id == emiCOPY_COLUMN_TO_CLIPBOARD )
	{
		QString s;

                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			s += curitem->text(column);
			s += "\n";
		}

		s = s.stripWhiteSpace();
		QApplication::clipboard()->setText(s);
	}
	else if ( id == emiCOPY_ROW_TO_CLIPBOARD )
	{
		int idx;
		QString s;

                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			for( idx = 0; idx < ListView_SEARCHRESULT->columns(); idx++ )
			{
				s += curitem->text(idx);
				s += " ";
			}

			s += "\n";
		}

		s = s.stripWhiteSpace();
		QApplication::clipboard()->setText(s);
	}
	else if ( id == emiCONNECT_TO_HUB )
	{
		// connect to the hub
                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			g_pConnectionManager->Connect( curitem->text(RESULTS_COLUMN_HUB).ascii(), curitem->text(RESULTS_COLUMN_HOST).ascii() );
		}
	}
	else if ( id == emiADD_BOOKMARK )
	{
		// bookmark this hub
                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			g_pHubListManager->AddBookmark( curitem->text(RESULTS_COLUMN_HUB), curitem->text(RESULTS_COLUMN_HOST), QString() );
		}
	}
	else if ( id == emiDISABLE_GROUP )
	{
		GroupSearchResults( egsrtNONE );
	}
	else if ( id == emiGROUP_BY_FILE )
	{
		GroupSearchResults( egsrtFILE );
	}
	else if ( id == emiGROUP_BY_SIZE )
	{
		GroupSearchResults( egsrtSIZE );
	}
	else if ( id == emiGROUP_BY_HASH )
	{
		GroupSearchResults( egsrtHASH );
	}
	else if ( id == emiGROUP_BY_NICK )
	{
		GroupSearchResults( egsrtNICK );
	}
	else if ( id == emiGROUP_BY_SLOTS_FREE )
	{
		GroupSearchResults( egsrtSLOTS_FREE );
	}
	else if ( id == emiGROUP_BY_SLOTS_TOTAL )
	{
		GroupSearchResults( egsrtSLOTS_TOTAL );
	}
	else if ( id == emiGROUP_BY_HUB )
	{
		GroupSearchResults( egsrtHUB );
	}
	else if ( id == emiGROUP_BY_PATH )
	{
		GroupSearchResults( egsrtPATH );
	}
	else if ( id == emiGROUP_BY_HOST )
	{
		GroupSearchResults( egsrtHOST );
	}
	else if ( id == emiKICK )
	{
		// TODO: need a patch to kick a user only one time
		QString message;

		if ( ! DCClient::GetOPKickMessage( message, this ) )
		{
			return;
		}

		// kick all user
		while( (item = selitems.first()) )
		{
			// remove this item from the list
			selitems.remove(item);

			// kick this user
			g_pConnectionManager->OPKick( item->text(RESULTS_COLUMN_HUB), item->text(RESULTS_COLUMN_HOST), item->text(RESULTS_COLUMN_NICK), message );

			// search for same entry and remove it
			for ( curitem = selitems.first(); curitem; )
			{
				if ( (curitem->text(RESULTS_COLUMN_NICK) == item->text(RESULTS_COLUMN_NICK)) &&
				     (curitem->text(RESULTS_COLUMN_HUB) == item->text(RESULTS_COLUMN_HUB)) &&
				     (curitem->text(RESULTS_COLUMN_HOST) == item->text(RESULTS_COLUMN_HOST)) )
				{
					selitems.remove(curitem);
					curitem = selitems.current();
				}
				else
				{
					curitem = selitems.next();
				}
			}
		}
	}
	else if ( id == emiFORCE_MOVE )
	{
		// TODO: need a patch to move a user only one time
		QString message, host;

		if ( ! DCClient::GetOPForceMoveMessage( message, host, this ) )
		{
			return;
		}

		// force move all user
		while( (item = selitems.first()) )
		{
			// remove this item from the list
			selitems.remove(item);

			// force move this user
			g_pConnectionManager->OPForceMove( item->text(RESULTS_COLUMN_HUB), item->text(RESULTS_COLUMN_HOST), item->text(RESULTS_COLUMN_NICK), message, host );

			// search for same entry and remove it
			for ( curitem = selitems.first(); curitem; )
			{
				if ( (curitem->text(RESULTS_COLUMN_NICK) == item->text(RESULTS_COLUMN_NICK)) &&
				     (curitem->text(RESULTS_COLUMN_HUB) == item->text(RESULTS_COLUMN_HUB)) &&
				     (curitem->text(RESULTS_COLUMN_HOST) == item->text(RESULTS_COLUMN_HOST)) )
				{
					selitems.remove(curitem);
					curitem = selitems.current();
				}
				else
				{
					curitem = selitems.next();
				}
			}
		}
	}
	else if ( id == emiSAVE )
	{
		QString filename = QFileDialog::getSaveFileName(
			QString(),
			QString(),
			this,
			tr("save file dialog"),
			tr("Choose a filename to save under") );

		if ( !filename.isEmpty() )
		{
			QFile file(filename);

			if ( file.open( IO_WriteOnly ) )
			{
				CIconv ciconv( g_pConfig->GetLocalEncoding(), g_pConfig->GetRemoteEncoding() );
				CString s;
				CMessageSearchResult * msr = 0;
				
				while ( (msr = m_pSearchResultList->Next(msr)) != 0 )
				{
					s  = "$SR ";
					s += ciconv.encode(msr->m_sNick);
					s += " ";
					s += ciconv.encode(msr->m_sFile);
					if ( msr->m_bFolder == false )
					{
						s += 0x05;
						s += CString::number(msr->m_nSize);
					}
					s += " ";
					s += CString::number(msr->m_nFreeSlot);
					s += "/";
					s += CString::number(msr->m_nMaxSlot);
					s += 0x05;
					if ( msr->m_sHash.IsEmpty() )
					{
						s += ciconv.encode(msr->m_sHubName);
					}
					else
					{
						s += "TTH:";
						s += msr->m_sHash;
					}
					s += " (";
					s += msr->m_sHubHost;
					s += ")|\n";
					file.writeBlock(s.Data(),s.Length());
				}

				file.close();
			}
		}
	}
	else if ( id == emiLOAD )
	{
		QString filename = QFileDialog::getOpenFileName(
			QString(),
			QString(),
			this,
			tr("open file dialog"),
			tr("Choose a file to open") );

		if ( !filename.isEmpty() )
		{
			QFile file(filename);
			if ( file.open( IO_ReadOnly ) )
			{
				m_pSearchResultList->Clear();

				ListView_SEARCHRESULT->setUpdatesEnabled(false);
				ClearSearchResults();

				QString s;
				CString m;
				int p;
				CDCMessage * dcmessage = 0;
				CMessageHandler MessageHandler;

				while( file.readLine(s,4096) != -1 )
				{
					m = s.ascii();
					p = 0;

					if ( MessageHandler.Parse(&m,p,&dcmessage) == DC_MESSAGE_SEARCHRESULT )
					{
						m_pSearchResultList->Add( (CMessageSearchResult*) dcmessage );
					}
				}

				file.close();

				eGroupSearchResultType current = m_eCurrentGrouping;
				
				// must change m_eCurrentGrouping to anything else
				// otherwise GroupSearchResults will do nothing
				if ( m_eCurrentGrouping == egsrtNONE )
				{
					m_eCurrentGrouping = egsrtFILE;
				}
				else
				{
					m_eCurrentGrouping = egsrtNONE;
				}
				
				GroupSearchResults( current );

				ListView_SEARCHRESULT->setUpdatesEnabled(true);
				ListView_SEARCHRESULT->triggerUpdate();
				LCDNumber_RESULTS->display( (int) m_pSearchResultList->Count() );
			}
		}
	}
	else if ( addedcommands.contains( id ) )
	{
		DC_UserMenuCommand * umc = addedcommands[ id ];
		
		QString usercommand = umc->m_sCommand;
		QString origUserCommand = usercommand;
		QStringList doneNicks;
		QString fullfilename, filesize, filesizeshort, filetth, filetype;
		
		for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
		{
			// check if already done this nick
			if ( (umc->m_nType == euctRawOnce) && doneNicks.contains(curitem->text(RESULTS_COLUMN_NICK)) )
			{
				continue;
			}
			
			usercommand = client->replaceCommandTags( origUserCommand, curitem->text(RESULTS_COLUMN_NICK) );
			
			if ( usercommand.isEmpty() )
			{
				// had a %[line:reason] but dialog was cancelled
				continue;
			}
			
			fullfilename = curitem->text(RESULTS_COLUMN_PATH);
			if ( (!fullfilename.isEmpty()) && (fullfilename.right(1) != "\\") )
			{
				fullfilename += "\\";
			}
			fullfilename += curitem->text(RESULTS_COLUMN_FILE);
			
			if ( curitem->text(RESULTS_COLUMN_SIZE) == tr("Folder") )
			{
				filesize = tr("Unknown");
				filesizeshort = tr("Unknown");
				filetth = tr("None");
				filetype = tr("Directory");
			}
			else
			{
				filesize.setNum(((DC_QListViewItem*)curitem)->myvalue);
				filesizeshort = curitem->text(RESULTS_COLUMN_SIZE);
				filetth = curitem->text(RESULTS_COLUMN_TTH);
				filetype = tr("File");
			}
			
			usercommand.replace( "%[file]", fullfilename );
			usercommand.replace( "%[fileFN]", fullfilename );
			
			usercommand.replace( "%[filesize]", filesize );
			usercommand.replace( "%[fileSI]", filesize );
			
			usercommand.replace( "%[filesizeshort]", filesizeshort );
			usercommand.replace( "%[fileSIshort]", filesizeshort );
			
			usercommand.replace( "%[tth]", filetth );
			usercommand.replace( "%[fileTR]", filetth );
			
			usercommand.replace( "%[type]", filetype );
			
			client->SendString( usercommand.ascii() );
			
			doneNicks << curitem->text(RESULTS_COLUMN_NICK);
		}
	}
}

/** */
void DCHubSearch::slotRightButtonClickedSearch( QListViewItem *, const QPoint &, int )
{
	// TODO: handle delete,edit searches (multiselection support)
	int id;
        QPtrList<QListViewItem> selitems;

	// user have select items ?
	DCGuiUtils::SelectedItems( ListView_SEARCH, selitems );

	QPopupMenu *m;

	m = new QPopupMenu(this);

	DCMenuHandler::InsertMenu( m, emiREMOVE, selitems.count() > 0 );
	DCMenuHandler::InsertMenu( m, emiCLEAR );

	id = m->exec(QCursor::pos());

	delete m;

	if ( id == emiREMOVE )
	{
		RemoveSelectedSearch(&selitems);
	}
	else if ( id == emiCLEAR )
	{
		if ( m_pSearchQueryList )
			m_pSearchQueryList->Clear();
		ListView_SEARCH->clear();
	}
}

/** */
void DCHubSearch::RemoveSelectedSearch( QPtrList<QListViewItem> * list )
{
        QPtrList<QListViewItem> selitems;
        DC_QSearchListViewItem * curitem;

	if ( !list )
	{
		if ( DCGuiUtils::SelectedItems( ListView_SEARCH, selitems ) == 0 )
		{
			return;
		}

		list = &selitems;
	}

	for ( curitem = (DC_QSearchListViewItem *)list->first(); curitem; curitem = (DC_QSearchListViewItem *)list->next() )
	{
		if ( m_pSearchQueryList )
			m_pSearchQueryList->Del(curitem->p_msg);
		delete curitem;
	}
}

/** Clears the history of search strings */
void DCHubSearch::slotPurgeHistory()
{
	Combobox_SEARCH->clear();
	slotReset();
	m_pSearchHistory->Clear();
	TextEdit_LOG->clear();
}

/** Disable size entry if size limit not used */
void DCHubSearch::slotSizeLimitChanged( int index )
{
	if ( index == 0 )
	{
		LineEdit_SEARCHSIZE->setEnabled(false);
		ComboBox_SEARCHUNIT->setEnabled(false);
	}
	else
	{
		LineEdit_SEARCHSIZE->setEnabled(true);
		ComboBox_SEARCHUNIT->setEnabled(true);
	}
}

/** */
void DCHubSearch::SetSearchForFile( QString file, eFileTypes filetype, int sizepos, ulonglong size )
{
	Combobox_SEARCH->setEditText(file);
	
	ComboBox_SEARCHUNIT->setCurrentItem(0);
	LineEdit_SEARCHSIZE->setText(QString().setNum(size));
	ComboBox_SEARCHLIMIT->setCurrentItem(sizepos);
	slotSizeLimitChanged(sizepos);
	
	ComboBox_SEARCHTYPE->setCurrentItem(filetype);
	
	TabWidget_SEARCH_PARAMETERS->setCurrentPage(0);
}

/** */
void DCHubSearch::StartSearchWithPrompt()
{
	if ( (m_eSearchState == egssREADY) ||
	     (QMessageBox::question(
	     	this,
		tr("Hub Search"),
		tr("Another search is running.\nStop other search?"),
		QMessageBox::Yes,
		QMessageBox::No | QMessageBox::Escape,
		QMessageBox::NoButton
	      ) == QMessageBox::Yes )
	   )
	{
		slotSearchReturnPressed();
	}
}

/** */
void DCHubSearch::ClearSearchResults()
{
	delete m_pEmptyGroup;
	m_pEmptyGroup = 0;
	
	m_GroupDict.clear();
	
	ListView_SEARCHRESULT->clear();
}

/** */
void DCHubSearch::DownloadFolder( QListViewItem * item )
{
	CString hubname(item->text(RESULTS_COLUMN_HUB).ascii());
	CString hubhost(item->text(RESULTS_COLUMN_HOST).ascii());
	
	if ( g_pConnectionManager->IsHubOnline(hubname,hubhost) == ehsNONE )
	{
		int ret = QMessageBox::warning(
			this,
			tr("Folder download"),
			tr("Not connected to required hub!"),
			tr("Connect"),
			tr("Cancel"),
			0,
			0,
			1
		);
		
		if ( ret == 0 )
		{
			g_pConnectionManager->Connect(hubname,hubhost);
		}
	}
	
	CString dir(item->text(RESULTS_COLUMN_PATH).ascii());
	
	if ( item->text(RESULTS_COLUMN_SIZE) == tr("Folder") )
	{
		dir += item->text(RESULTS_COLUMN_FILE).ascii();
	}
	
	g_pTransferView->DLM_QueueAdd(
		item->text(RESULTS_COLUMN_NICK).ascii(),
		hubname,
		hubhost,
		DC_USER_FILELIST,
		DC_USER_FILELIST,
		dir,
		dir,
		eltBUFFER,
		0,
		0,
		0,
		CString()
	);
}
