/***************************************************************************
              userlistmodel.cpp  -  User List Model implementation
                             -------------------
    begin                : Sun Nov 18 2007
    copyright            : (C) 2007 by Edward Sheldrake
    email                : ejs1920@yahoo.co.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "userlistmodel.h"

#include <QtAlgorithms>

/* need to convert shared bytes to size string */
#include "dcguiutils.h"

/** */
UserListModel::UserListModel( QObject * parent ) : QAbstractItemModel( parent )
{
	sortColumn = COLUMN_NICK;
	sortOrder = Qt::AscendingOrder;
	stripper.setPattern("\\[.*\\]");
	stripper.setMinimal(true);
}

/** delete all items in model */
UserListModel::~UserListModel()
{
	clear();
}

/** */
int UserListModel::rowCount( const QModelIndex & /* index */ ) const
{
	return itemList.size();
}

/** */
int UserListModel::columnCount( const QModelIndex & /* index */ ) const
{
	return 9;
}

/** */
QVariant UserListModel::data( const QModelIndex & index, int role ) const
{
	if ( !index.isValid() )
	{
		return QVariant();
	}
	
	UserListItem * item = itemList[index.row()];
	
	if ( item == 0 )
	{
		return QVariant();
	}
	
	if ( role == Qt::DisplayRole ) // text
	{
		switch ( index.column() )
		{
			case COLUMN_NICK:     return item->nick;
			case COLUMN_COMMENT:  return item->comment;
			case COLUMN_TAG:      return item->tag;
			case COLUMN_SPEED:    return item->speed;
			case COLUMN_EMAIL:    return item->email;
			case COLUMN_SHARE:    return DCGuiUtils::GetSizeString( item->shared );
			case COLUMN_IP:       return item->ip;
			case COLUMN_LOCKPK:   return item->lockpk;
			case COLUMN_SUPPORTS: return item->supports;
		}
	}
	else if ( role == Qt::DecorationRole ) // icon
	{
		if ( index.column() == COLUMN_NICK )
		{
			// to check icons are being cached properly
			// printf("row=%d column=%d cacheKey=%llu\n",index.row(),index.column(),item->pixmap.cacheKey());
			return item->pixmap;
		}
	}
	else if ( role == Qt::ToolTipRole ) // tool tip
	{
		if ( index.column() == COLUMN_SHARE )
		{
			return QString::number( item->shared );
		}
	}
	else if ( role == Qt::TextAlignmentRole ) // text alignment
	{
		if ( index.column() == COLUMN_SHARE )
		{
			return Qt::AlignRight;
		}
	}
	else if ( role == Qt::ForegroundRole) // text colour
	{
		if ( (index.column() == COLUMN_NICK) && (item->highlight) )
		{
			return QColor( 255, 0, 0 );
		}
	}
	
	return QVariant();
}

/** */
QVariant UserListModel::headerData( int section, Qt::Orientation orientation, int role ) const
{
	if ( (orientation == Qt::Horizontal) && (role == Qt::DisplayRole) )
	{
		switch ( section )
		{
			case COLUMN_NICK:     return tr("Nick");
			case COLUMN_COMMENT:  return tr("Comment");
			case COLUMN_TAG:      return tr("Tag");
			case COLUMN_SPEED:    return tr("Speed");
			case COLUMN_EMAIL:    return tr("Email");
			case COLUMN_SHARE:    return tr("Share");
			case COLUMN_IP:       return tr("IP");
			case COLUMN_LOCKPK:   return tr("Lock/PK");
			case COLUMN_SUPPORTS: return tr("Supports");
		}
	}
	
	return QVariant();
}

/** */
bool UserListItemNickLessThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->nick_lc < r->nick_lc;
	}
	else
	{
		return l->sorttop;
	}
}

/** */
bool UserListItemCommentLessThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->comment_lc < r->comment_lc;
	}
	else
	{
		return l->sorttop;
	}
}

/** */
bool UserListItemTagLessThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->tag_lc < r->tag_lc;
	}
	else
	{
		return l->sorttop;
	}
}

/** */
bool UserListItemSpeedLessThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->speed_lc < r->speed_lc;
	}
	else
	{
		return l->sorttop;
	}
}

/** */
bool UserListItemEmailLessThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->email_lc < r->email_lc;
	}
	else
	{
		return l->sorttop;
	}
}

/** */
bool UserListItemShareLessThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->shared < r->shared;
	}
	else
	{
		return l->sorttop;
	}
}

/** */
bool UserListItemIPLessThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->ip < r->ip;
	}
	else
	{
		return l->sorttop;
	}
}

/** */
bool UserListItemLockPKLessThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->lockpk < r->lockpk;
	}
	else
	{
		return l->sorttop;
	}
}

/** */
bool UserListItemSupportsLessThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->supports < r->supports;
	}
	else
	{
		return l->sorttop;
	}
}

/** */
bool UserListItemNickGreaterThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->nick_lc > r->nick_lc;
	}
	else
	{
		return r->sorttop;
	}
}

/** */
bool UserListItemCommentGreaterThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->comment_lc > r->comment_lc;
	}
	else
	{
		return r->sorttop;
	}
}

/** */
bool UserListItemTagGreaterThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->tag_lc > r->tag_lc;
	}
	else
	{
		return r->sorttop;
	}
}

/** */
bool UserListItemSpeedGreaterThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->speed_lc > r->speed_lc;
	}
	else
	{
		return r->sorttop;
	}
}

/** */
bool UserListItemEmailGreaterThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->email_lc > r->email_lc;
	}
	else
	{
		return r->sorttop;
	}
}

/** */
bool UserListItemShareGreaterThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->shared > r->shared;
	}
	else
	{
		return r->sorttop;
	}
}

/** */
bool UserListItemIPGreaterThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->ip > r->ip;
	}
	else
	{
		return r->sorttop;
	}
}

/** */
bool UserListItemLockPKGreaterThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->lockpk > r->lockpk;
	}
	else
	{
		return r->sorttop;
	}
}

/** */
bool UserListItemSupportsGreaterThan( const UserListItem * l, const UserListItem * r )
{
	if ( l->sorttop == r->sorttop )
	{
		return l->supports > r->supports;
	}
	else
	{
		return r->sorttop;
	}
}

/** */
void UserListModel::sort( int column, Qt::SortOrder order )
{
	sortColumn = column;
	sortOrder = order;

	if ( column == -1 ) // sorting disabled
	{
		return;
	}

	emit layoutAboutToBeChanged();
	
	if ( order == Qt::AscendingOrder )
	{
		switch ( column )
		{
			case COLUMN_NICK:
				qStableSort( itemList.begin(), itemList.end(), UserListItemNickLessThan );
				break;
			case COLUMN_COMMENT:
				qStableSort( itemList.begin(), itemList.end(), UserListItemCommentLessThan );
				break;
			case COLUMN_TAG:
				qStableSort( itemList.begin(), itemList.end(), UserListItemTagLessThan );
				break;
			case COLUMN_SPEED:
				qStableSort( itemList.begin(), itemList.end(), UserListItemSpeedLessThan );
				break;
			case COLUMN_EMAIL:
				qStableSort( itemList.begin(), itemList.end(), UserListItemEmailLessThan );
				break;
			case COLUMN_SHARE:
				qStableSort( itemList.begin(), itemList.end(), UserListItemShareLessThan );
				break;
			case COLUMN_IP:
				qStableSort( itemList.begin(), itemList.end(), UserListItemIPLessThan );
				break;
			case COLUMN_LOCKPK:
				qStableSort( itemList.begin(), itemList.end(), UserListItemLockPKLessThan );
				break;
			case COLUMN_SUPPORTS:
				qStableSort( itemList.begin(), itemList.end(), UserListItemSupportsLessThan );
				break;
		}
	}
	else if ( order == Qt::DescendingOrder )
	{
		switch ( column )
		{
			case COLUMN_NICK:
				qStableSort( itemList.begin(), itemList.end(), UserListItemNickGreaterThan );
				break;
			case COLUMN_COMMENT:
				qStableSort( itemList.begin(), itemList.end(), UserListItemCommentGreaterThan );
				break;
			case COLUMN_TAG:
				qStableSort( itemList.begin(), itemList.end(), UserListItemTagGreaterThan );
				break;
			case COLUMN_SPEED:
				qStableSort( itemList.begin(), itemList.end(), UserListItemSpeedGreaterThan );
				break;
			case COLUMN_EMAIL:
				qStableSort( itemList.begin(), itemList.end(), UserListItemEmailGreaterThan );
				break;
			case COLUMN_SHARE:
				qStableSort( itemList.begin(), itemList.end(), UserListItemShareGreaterThan );
				break;
			case COLUMN_IP:
				qStableSort( itemList.begin(), itemList.end(), UserListItemIPGreaterThan );
				break;
			case COLUMN_LOCKPK:
				qStableSort( itemList.begin(), itemList.end(), UserListItemLockPKGreaterThan );
				break;
			case COLUMN_SUPPORTS:
				qStableSort( itemList.begin(), itemList.end(), UserListItemSupportsGreaterThan );
				break;
		}
	}
	
	for ( int i = 0; i < itemList.size(); ++i )
	{
		itemList.at(i)->index = i;
	}
	
	emit layoutChanged();
}

/** */
QModelIndex UserListModel::index( int row, int column, const QModelIndex & parent ) const
{
	if ( !hasIndex( row, column, parent ) )
	{
		return QModelIndex();
	}
	
	return createIndex( row, column );
}

/** */
QModelIndex UserListModel::parent( const QModelIndex & /* parent */ ) const
{
	return QModelIndex();
}

/** */
void UserListModel::clear()
{
	emit layoutAboutToBeChanged();
	
	QList<UserListItem*> tmp = itemList;
	
	itemList.clear();
	itemHash.clear();
	
	for ( QList<UserListItem*>::const_iterator it = tmp.constBegin(); it != tmp.constEnd(); ++it )
	{
		delete *it;
	}
	
	emit layoutChanged();
}

/** */
bool UserListModel::hasNick( const QString & nick ) const
{
	return itemHash.contains( nick );
}

/** */
void UserListModel::removeUser( const QString & nick )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item == 0 )
	{
		return;
	}
	
	emit layoutAboutToBeChanged();
	
	itemHash.remove( nick );
	itemList.removeAt( item->index );
	
	// fix indexes
	for( int i = item->index; i < itemList.size(); ++i )
	{
		itemList.at(i)->index--;
	}
	
	// destroy item
	delete item;
	
	emit layoutChanged();
}

/** */
void UserListModel::addUser( const QString nick,
			     const QPixmap icon,
			     const QString comment,
			     const QString tag,
			     const QString speed,
			     const QString email,
			     const ulonglong share,
			     const QString ip,
			     const QString lockpk,
			     const QString supports )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item != 0 )
	{
		item->pixmap   = icon;
		item->comment  = comment;
		item->tag      = tag;
		item->speed    = speed;
		item->email    = email;
		item->shared   = share;
		item->ip       = ip;
		item->lockpk   = lockpk;
		item->supports = supports;
		
		item->nick_lc    = nick.toLower();
		item->nick_stripped = nick.toLower().remove( stripper );
		item->comment_lc = comment.toLower();
		item->tag_lc     = tag.toLower();
		item->speed_lc   = speed.toLower();
		item->email_lc   = email.toLower();
		
		QModelIndex topLeft = createIndex( item->index, COLUMN_NICK );
		QModelIndex bottomRight = createIndex( item->index, COLUMN_SUPPORTS );
		
		emit dataChanged( topLeft, bottomRight );
		
		if ( sortColumn != COLUMN_NICK )
		{
			sort( sortColumn, sortOrder );
		}
		
		return;
	}
	
	emit layoutAboutToBeChanged();
	
	item = new UserListItem();
	
	item->nick     = nick;
	item->pixmap   = icon;
	item->comment  = comment;
	item->tag      = tag;
	item->speed    = speed;
	item->email    = email;
	item->shared   = share;
	item->ip       = ip;
	item->lockpk   = lockpk;
	item->supports = supports;
	
	item->nick_lc    = nick.toLower();
	item->nick_stripped = nick.toLower().remove( stripper );
	item->comment_lc = comment.toLower();
	item->tag_lc     = tag.toLower();
	item->speed_lc   = speed.toLower();
	item->email_lc   = email.toLower();
	
	itemHash[ nick ] = item;
	
	if ( sortColumn == -1 ) // if sorting disabled
	{
		item->index = itemList.size();
		itemList.append( item );
		
		emit layoutChanged();
		
		return; 
	}

	// find position in list for new item
	QList<UserListItem*>::iterator it;
	
	if ( sortOrder == Qt::AscendingOrder )
	{
		switch ( sortColumn )
		{
			case COLUMN_NICK:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemNickLessThan );
				break;
			case COLUMN_COMMENT:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemCommentLessThan );
				break;
			case COLUMN_TAG:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemTagLessThan );
				break;
			case COLUMN_SPEED:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemSpeedLessThan );
				break;
			case COLUMN_EMAIL:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemEmailLessThan );
				break;
			case COLUMN_SHARE:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemShareLessThan );
				break;
			case COLUMN_IP:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemIPLessThan );
				break;
			case COLUMN_LOCKPK:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemLockPKLessThan );
				break;
			case COLUMN_SUPPORTS:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemSupportsLessThan );
				break;
		}
	}
	else if ( sortOrder == Qt::DescendingOrder )
	{
		switch ( sortColumn )
		{
			case COLUMN_NICK:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemNickGreaterThan );
				break;
			case COLUMN_COMMENT:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemCommentGreaterThan );
				break;
			case COLUMN_TAG:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemTagGreaterThan );
				break;
			case COLUMN_SPEED:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemSpeedGreaterThan );
				break;
			case COLUMN_EMAIL:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemEmailGreaterThan );
				break;
			case COLUMN_SHARE:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemShareGreaterThan );
				break;
			case COLUMN_IP:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemIPGreaterThan );
				break;
			case COLUMN_LOCKPK:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemLockPKGreaterThan );
				break;
			case COLUMN_SUPPORTS:
				it = qLowerBound( itemList.begin(), itemList.end(), item, UserListItemSupportsGreaterThan );
				break;
		}
	}
	
	int i;
	if ( it == itemList.begin() )
	{
		i = 0;
	}
	else if ( it == itemList.end() )
	{
		i = itemList.size();
	}
	else
	{
		i = (*it++)->index;
	}
	
	item->index = i;
	
	itemList.insert( i, item );
	
	i++;
	
	// fix remaining indexes
	for ( ; i < itemList.size(); i++ )
	{
		itemList.at(i)->index++;
	}
	
	emit layoutChanged();
}

/** */
void UserListModel::updateIcon( const QString & nick, const QPixmap icon )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item == 0 )
	{
		return;
	}
	
	item->pixmap = icon;
	
	QModelIndex position = createIndex( item->index, COLUMN_NICK );
	
	emit dataChanged( position, position );
}

/** */
void UserListModel::updateComment( const QString & nick, const QString comment )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item == 0 )
	{
		return;
	}
	
	item->comment = comment;
	item->comment_lc = comment.toLower();
	
	QModelIndex position = createIndex( item->index, COLUMN_COMMENT );
	
	emit dataChanged( position, position );
	
	if ( sortColumn == COLUMN_COMMENT )
	{
		sort( sortColumn, sortOrder );
	}
}

/** */
void UserListModel::updateTag( const QString & nick, const QString tag )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item == 0 )
	{
		return;
	}
	
	item->tag = tag;
	item->tag_lc = tag.toLower();
	
	QModelIndex position = createIndex( item->index, COLUMN_TAG );
	
	emit dataChanged( position, position );
	
	if ( sortColumn == COLUMN_TAG )
	{
		sort( sortColumn, sortOrder );
	}
}

/** */
void UserListModel::updateSpeed( const QString & nick, const QString speed )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item == 0 )
	{
		return;
	}
	
	item->speed = speed;
	item->speed_lc = speed.toLower();
	
	QModelIndex position = createIndex( item->index, COLUMN_SPEED );
	
	emit dataChanged( position, position );
	
	if ( sortColumn == COLUMN_SPEED )
	{
		sort( sortColumn, sortOrder );
	}
}

/** */
void UserListModel::updateEmail( const QString & nick, const QString email )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item == 0 )
	{
		return;
	}
	
	item->email = email;
	item->email_lc = email.toLower();
	
	QModelIndex position = createIndex( item->index, COLUMN_EMAIL );
	
	emit dataChanged( position, position );
	
	if ( sortColumn == COLUMN_EMAIL )
	{
		sort( sortColumn, sortOrder );
	}
}

/** */
void UserListModel::updateShare( const QString & nick, const ulonglong share )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item == 0 )
	{
		return;
	}
	
	item->shared = share;
	
	QModelIndex position = createIndex( item->index, COLUMN_SHARE );
	
	emit dataChanged( position, position );
	
	if ( sortColumn == COLUMN_SHARE )
	{
		sort( sortColumn, sortOrder );
	}
}

/** */
void UserListModel::updateIP( const QString & nick, const QString ip )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item == 0 )
	{
		return;
	}
	
	item->ip = ip;
	
	QModelIndex position = createIndex( item->index, COLUMN_IP );
	
	emit dataChanged( position, position );
	
	if ( sortColumn == COLUMN_IP )
	{
		sort( sortColumn, sortOrder );
	}
}

/** */
void UserListModel::updateLockPK( const QString & nick, const QString lockpk )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item == 0 )
	{
		return;
	}
	
	item->lockpk = lockpk;
	
	QModelIndex position = createIndex( item->index, COLUMN_LOCKPK );
	
	emit dataChanged( position, position );
	
	if ( sortColumn == COLUMN_LOCKPK )
	{
		sort( sortColumn, sortOrder );
	}
}

/** */
void UserListModel::updateSupports( const QString & nick, const QString supports )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item == 0 )
	{
		return;
	}
	
	item->supports = supports;
	
	QModelIndex position = createIndex( item->index, COLUMN_SUPPORTS );
	
	emit dataChanged( position, position );
	
	if ( sortColumn == COLUMN_SUPPORTS )
	{
		sort( sortColumn, sortOrder );
	}
}

/** */
void UserListModel::setSortTop( const QString & nick, const bool top )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( !item || (item->sorttop == top) )
	{
		return;
	}
	
	item->sorttop = top;
	
	sort( sortColumn, sortOrder );
}

/** */
void UserListModel::setHighlight( const QString & nick, const bool highlight )
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item == 0 )
	{
		return;
	}
	
	item->highlight = highlight;
	
	QModelIndex position = createIndex( item->index, COLUMN_NICK );
	
	emit dataChanged( position, position );
}

/** */
QModelIndex UserListModel::indexForNick( const QString & nick ) const
{
	UserListItem * item = itemHash.value( nick );
	
	if ( item == 0 )
	{
		return QModelIndex();
	}
	
	return createIndex( item->index, COLUMN_NICK );
}

/** */
QStringList UserListModel::matchNicksContaining( const QString & part, bool stripTags ) const
{
	QStringList matches;
	
	if ( part.isEmpty() )
	{
		return matches;
	}
	
	if ( stripTags )
	{
		for ( QList<UserListItem*>::const_iterator it = itemList.constBegin(); it != itemList.constEnd(); ++it )
		{
			if ( (*it)->nick_stripped.contains(part) )
			{
				matches << (*it)->nick;
			}
		}
	}
	else
	{
		for ( QList<UserListItem*>::const_iterator it = itemList.constBegin(); it != itemList.constEnd(); ++it )
		{
			if ( (*it)->nick_lc.contains(part) )
			{
				matches << (*it)->nick;
			}
		}
	}
	
	return matches;
}

/** */
QStringList UserListModel::matchNicksStartingWith( const QString & part, bool stripTags ) const
{
	QStringList matches;
	
	if ( part.isEmpty() )
	{
		return matches;
	}
	
	if ( stripTags )
	{
		for ( QList<UserListItem*>::const_iterator it = itemList.constBegin(); it != itemList.constEnd(); ++it )
		{
			if ( (*it)->nick_stripped.startsWith(part) )
			{
				matches << (*it)->nick;
			}
		}	
	}
	else
	{
		for ( QList<UserListItem*>::const_iterator it = itemList.constBegin(); it != itemList.constEnd(); ++it )
		{
			if ( (*it)->nick_lc.startsWith(part) )
			{
				matches << (*it)->nick;
			}
		}
	}
	
	return matches;
}
