/*=======================================================================
 *
 * bdf2psf, a utility to convert BDF fonts to PSF version 2 format.
 * Copyright (C) 2007-2008 Zachary T. Smith, fbui@comcast.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This module is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this module; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 *
 * (See the file COPYING in the main directory of this archive for
 * more details.)
 *
 *=====================================================================*/

/* Version 0.4 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>

#include "bdf_font.h"

#define VERSION "0.3"

#define PSF2_MAGIC0     0x72
#define PSF2_MAGIC1     0xb5
#define PSF2_MAGIC2     0x4a
#define PSF2_MAGIC3     0x86
#define PSF2_HAS_UNICODE_TABLE 0x01
#define PSF2_MAXVERSION 0
#define PSF2_SEPARATOR  0xFF
#define PSF2_STARTSEQ   0xFE

struct psf2_header {
	u8 magic[4];
	u32 version;
	u32 headersize;    /* offset of bitmaps in file */
	u32 flags;
	u32 length;        /* number of glyphs */
	u32 charsize;      /* number of bytes for each character */
	u32 height, width; /* max dimensions of glyphs */
};

#define NCHARS (256)

#define RGB_THRESHOLD 0x6B0000


static u8 bit_reversal_array [256] =
{
0x00, 0x80, 0x40, 0xc0, 0x20, 0xa0, 0x60, 0xe0, 0x10, 0x90, 0x50, 0xd0, 0x30, 0xb0, 0x70, 0xf0, 0x08, 0x88, 0x48, 0xc8, 0x28, 0xa8, 0x68, 0xe8, 0x18, 0x98, 0x58, 0xd8, 0x38, 0xb8, 0x78, 0xf8, 0x04, 0x84, 0x44, 0xc4, 0x24, 0xa4, 0x64, 0xe4, 0x14, 0x94, 0x54, 0xd4, 0x34, 0xb4, 0x74, 0xf4, 0x0c, 0x8c, 0x4c, 0xcc, 0x2c, 0xac, 0x6c, 0xec, 0x1c, 0x9c, 0x5c, 0xdc, 0x3c, 0xbc, 0x7c, 0xfc, 0x02, 0x82, 0x42, 0xc2, 0x22, 0xa2, 0x62, 0xe2, 0x12, 0x92, 0x52, 0xd2, 0x32, 0xb2, 0x72, 0xf2, 0x0a, 0x8a, 0x4a, 0xca, 0x2a, 0xaa, 0x6a, 0xea, 0x1a, 0x9a, 0x5a, 0xda, 0x3a, 0xba, 0x7a, 0xfa, 0x06, 0x86, 0x46, 0xc6, 0x26, 0xa6, 0x66, 0xe6, 0x16, 0x96, 0x56, 0xd6, 0x36, 0xb6, 0x76, 0xf6, 0x0e, 0x8e, 0x4e, 0xce, 0x2e, 0xae, 0x6e, 0xee, 0x1e, 0x9e, 0x5e, 0xde, 0x3e, 0xbe, 0x7e, 0xfe, 0x01, 0x81, 0x41, 0xc1, 0x21, 0xa1, 0x61, 0xe1, 0x11, 0x91, 0x51, 0xd1, 0x31, 0xb1, 0x71, 0xf1, 0x09, 0x89, 0x49, 0xc9, 0x29, 0xa9, 0x69, 0xe9, 0x19, 0x99, 0x59, 0xd9, 0x39, 0xb9, 0x79, 0xf9, 0x05, 0x85, 0x45, 0xc5, 0x25, 0xa5, 0x65, 0xe5, 0x15, 0x95, 
0x55, 0xd5, 0x35, 0xb5, 0x75, 0xf5, 0x0d, 0x8d, 0x4d, 0xcd, 0x2d, 0xad, 0x6d, 0xed, 0x1d, 0x9d, 0x5d, 0xdd, 0x3d, 0xbd, 0x7d, 0xfd, 0x03, 0x83, 0x43, 0xc3, 0x23, 0xa3, 0x63, 0xe3, 0x13, 0x93, 0x53, 0xd3, 0x33, 0xb3, 0x73, 0xf3, 0x0b, 0x8b, 0x4b, 0xcb, 0x2b, 0xab, 0x6b, 0xeb, 0x1b, 0x9b, 0x5b, 0xdb, 0x3b, 0xbb, 0x7b, 0xfb, 0x07, 0x87, 0x47, 0xc7, 0x27, 0xa7, 0x67, 0xe7, 0x17, 0x97, 0x57, 0xd7, 0x37, 0xb7, 0x77, 0xf7, 0x0f, 0x8f, 0x4f, 0xcf, 0x2f, 0xaf, 0x6f, 0xef, 0x1f, 0x9f, 0x5f, 0xdf, 0x3f, 0xbf, 0x7f, 0xff, 
};


void
dump_pattern (u32 value)
{
	int i;
	u32 mask=1<<31;
	while (mask) {
		int ch = (mask & value) ? '#' : '.';
		putchar (ch);
		mask >>= 1;
	}
	putchar('\n');
}


int
main(int c, char **v)
{
#define MAXWIDTH 500
#define MAXHEIGHT 500
	static unsigned long bitmap[MAXHEIGHT];
	unsigned short ch;
	FILE *f;
	struct psf2_header hdr;
	int byte_width;
	int font_maxwidth;
	int font_height;
	int i,j,k;

	printf ("This is bdf2psf version %s, a language translator.\n", VERSION);
	printf ("Copyright (C) 2007-2008 by Zachary T Smith.\n");
	printf ("This program is provided AS-IS under the GNU Public License,\n");
	printf ("so use it at your own risk.\n");

	if (c!=3) {
		printf ("Usage: bdftopsf infile outfile\n");
		exit(0);
	}

	/*-------------------------------------------------
	 * Use my old BDF reader to grab the bitmaps.
	 *------------------------------------------------*/
	Font *font = Font_new();
	if (!Font_read_bdf (font, v[1])) {
		printf ("Failed to read %s\n", v[1]);
		exit(0);
	}

	/*-------------------------------------------------
	 * Determine maximal width for all chars, in case
	 * the input font is not fixed width.
	 *------------------------------------------------*/
	j=0;
	i=0;
	k=font->nchars;
	while (k--) {
		if (font->bbw[i] > j) {
			j = font->bbw[i];
		}
		i++;
	}

	font_maxwidth=j;
	font_height = font->ascent + font->descent;

	byte_width = (font_maxwidth + 7) / 8;

	/*-------------------------------------------------
	 * Create output PSF2 file, write header.
	 *------------------------------------------------*/
	f = fopen(v[2], "wb");
	if (!f) {
		perror("fopen");
		exit(0);
	}
	hdr.magic[0] = PSF2_MAGIC0;
	hdr.magic[1] = PSF2_MAGIC1;
	hdr.magic[2] = PSF2_MAGIC2;
	hdr.magic[3] = PSF2_MAGIC3;
	hdr.version = 0;
	hdr.flags = 0 ; /*PSF2_HAS_UNICODE_TABLE;*/
	hdr.length = 256;
#if 0
	// Fudge it. Fit a small font into the non-framebuffer console
	// standard width needed by setfont.
	//
	if (font_maxwidth<8)
		font_maxwidth=8;
#endif
	hdr.width = font_maxwidth;
	hdr.height = font_height;
	hdr.charsize = hdr.height * byte_width;
	hdr.headersize = sizeof(hdr);
	hdr.flags = 0;
	fwrite(&hdr,1,sizeof(hdr),f);

	/*-------------------------------------------------
	 * Print useful info.
	 *------------------------------------------------*/
	printf ("PSF header length %d bytes\n", sizeof(hdr));
	printf ("PSF font data length %d bytes\n", hdr.charsize*256);
	printf ("PSF byte width %d \n", byte_width);
	printf ("BDF font width=%d, height=%d\n", font_maxwidth, font_height);
	int cols = 1024 / font_maxwidth;
	int rows = 768 / font_height;
	printf ("On a 1024x768 framebuffer console, you'd get %d cols x %d rows.\n", cols,rows);
	cols = 800 / font_maxwidth;
	rows = 600 / font_height;
	printf ("On a 800x600 framebuffer console, you'd get %d cols x %d rows.\n", cols,rows);

#if 0
	/*-------------------------------------------------
	 * Create a blank bitmap pattern and write out
	 * any chars before the 1st char.
	 *------------------------------------------------*/
	u8 *buffer=malloc(byte_width*font_height);
	memset(buffer,0,byte_width*font_height);
	for (i=0; i<font->first_char; i++)
		fwrite (buffer, 1,byte_width*font_height, f);
#endif

	/*-------------------------------------------------
	 * Write out all the chars.
	 *------------------------------------------------*/
	for (ch=0; ch < 256; ch++) {
		u32 *bm = font->bitmaps[ch];
		union {
			u8 b[4];
			u32 l;
		} ary;
		int yoff, asc, desc, pattern_height;
		int empty_above, empty_below;
		int right_shift = font->bbxoff[ch];

		yoff= font->bbyoff[ch];
		asc= font->bbh[ch] + yoff;
		desc = -yoff;

		pattern_height = font->bbh[ch];
		empty_above = font->ascent - asc;
		empty_below = font->descent - desc;

		// printf ("=== Char %d, bitwidth=%d, bitheight=%d, desc=%d\n", ch, font->bbw[ch], asc,desc);
		// printf ("\tempty_above %d, empty_below=%d, height=%d\n", empty_above, empty_below, font->bbh[ch]);
		// printf ("font ascent %d, font descent %d\n", font->ascent, font->descent);

		/*-------------------------------------------------
		 * Write out blank area above pattern.
		 *------------------------------------------------*/
		ary.l = 0;
		for (j=0; j< empty_above; j++)
			fwrite (ary.b, 1,byte_width, f);

		/*-------------------------------------------------
		 * Write out the pattern itself.
		 *------------------------------------------------*/
		for (j=0; j<pattern_height; j++) {
			unsigned char tmp,tmp2;
			ary.l = bm[j];

			dump_pattern(ary.l);

			switch (byte_width) {
			case 1:
				ary.b[0] = ary.b[3];
				break;
			case 2:
				ary.b[0] = ary.b[3];
				ary.b[1] = ary.b[2];
				break;
			case 3:
				ary.b[0] = ary.b[3];
				tmp = ary.b[1];
				ary.b[1] = ary.b[2];
				ary.b[2] = tmp;
				break;
			case 4:
				tmp2 = ary.b[0];
				ary.b[0] = ary.b[3];
				tmp = ary.b[1];
				ary.b[1] = ary.b[2];
				ary.b[2] = tmp;
				ary.b[3] = tmp2;
				break;
			}

			if (right_shift) {
				u32 big = ary.b[0];	// MSB
				big <<= 8;
				big |= ary.b[1];
				big <<= 8;
				big |= ary.b[2];
				big <<= 8;
				big |= ary.b[3];	// LSB
				big >>= right_shift;
				ary.b[3] = big & 0xff;
				big >>= 8;
				ary.b[2] = big & 0xff;
				big >>= 8;
				ary.b[1] = big & 0xff;
				big >>= 8;
				ary.b[0] = big & 0xff;
			}
			fwrite (ary.b, 1,byte_width, f);
		}
		putchar('\n');

		/*-------------------------------------------------
		 * Write out blank area below pattern.
		 *------------------------------------------------*/
		ary.l = 0;
		for (j=0; j< empty_below; j++)
			fwrite (ary.b, 1,byte_width, f);
	}

#if 0
	/*-------------------------------------------------
	 * Use formerly created blank bitmap pattern to 
	 * write out any chars after the last encoded char.
	 *------------------------------------------------*/
	for (i=1+font->last_char; i<256; i++)
		fwrite (buffer, 1,byte_width*font_height, f);
	free(buffer);
#endif

	fclose(f);
	Font_free(font);

	return 0;
}


