/***************************************************************************
*   Copyright (C) 2005 by Alexander Nemish  *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <kdialog.h>
#include <klocale.h>
#include <qvariant.h>
#include <qpushbutton.h>
#include <qheader.h>
#include <qtable.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qwhatsthis.h>

#include "bookmarksdlg.h"
/*
 *  Constructs a BookmarksDlg as a child of 'parent', with the
 *  name 'name' and widget flags set to 'f'.
 *
 *  The dialog will by default be modeless, unless you set 'modal' to
 *  TRUE to construct a modal dialog.
 */

BookmarksDlg::BookmarksDlg(const Bookmarks& bms, QWidget* parent,
                           const char* name, bool modal, WFlags fl)
		: QDialog(parent, name, modal, fl),
		m_bookmarks(bms)
{
	if ( !name )
		setName( "BookmarksDlg" );
	setSizeGripEnabled(TRUE);
	BookmarksDlgLayout = new QVBoxLayout( this, 11, 6, "BookmarksDlgLayout");

	layout3 = new QHBoxLayout( 0, 0, 6, "layout3");

	bookmarksTable = new QTable(3, 2, this, "bookmarksList" );
	bookmarksTable->horizontalHeader()->setLabel(0, "Title");
	bookmarksTable->horizontalHeader()->setLabel(1, "Page number");
// 	bookmarksTable->setLeftMargin(0);
	bookmarksTable->setColumnStretchable(0, true);
	bookmarksTable->setSelectionMode(QTable::SingleRow);
	layout3->addWidget( bookmarksTable );

	layout2 = new QVBoxLayout( 0, 0, 6, "layout2");

	addButton = new QPushButton( this, "addButton" );
	layout2->addWidget( addButton );

	deleteButton = new QPushButton( this, "deleteButton" );
	layout2->addWidget( deleteButton );
	Horizontal_Spacing2_2 = new QSpacerItem( 20, 60, QSizePolicy::Minimum, QSizePolicy::Expanding );
	layout2->addItem( Horizontal_Spacing2_2 );
	layout3->addLayout( layout2 );
	BookmarksDlgLayout->addLayout( layout3 );

	Layout1 = new QHBoxLayout( 0, 0, 6, "Layout1");

	buttonHelp = new QPushButton( this, "buttonHelp" );
	buttonHelp->setAutoDefault( TRUE );
	Layout1->addWidget( buttonHelp );
	Horizontal_Spacing2 = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
	Layout1->addItem( Horizontal_Spacing2 );

	buttonOk = new QPushButton( this, "buttonOk" );
	buttonOk->setAutoDefault( TRUE );
	buttonOk->setDefault( TRUE );
	Layout1->addWidget( buttonOk );

	buttonCancel = new QPushButton( this, "buttonCancel" );
	buttonCancel->setAutoDefault( TRUE );
	Layout1->addWidget( buttonCancel );
	BookmarksDlgLayout->addLayout( Layout1 );
	languageChange();
	resize( QSize(400, 267).expandedTo(minimumSizeHint()) );
	clearWState( WState_Polished );

	updateBookmarks();
	// signals and slots connections
	connect(buttonOk, SIGNAL(clicked()), this, SLOT(accept()));
	connect(buttonCancel, SIGNAL(clicked()), this, SLOT(reject()));
	connect(addButton, SIGNAL(clicked()), this, SLOT(addBookmark()));
	connect(deleteButton, SIGNAL(clicked()), this, SLOT(deleteBookmark()));
	connect(bookmarksTable, SIGNAL(valueChanged(int, int)), this, SLOT(valueChanged(int, int)));
}

/*
 *  Destroys the object and frees any allocated resources
 */
BookmarksDlg::~BookmarksDlg()
{
	// no need to delete child widgets, Qt does it all for us
}

/*
 *  Sets the strings of the subwidgets using the current
 *  language.
 */
void BookmarksDlg::languageChange()
{
	setCaption( tr2i18n( "Bookmarks editor" ) );
	addButton->setText( tr2i18n( "Add" ) );
	deleteButton->setText( tr2i18n( "Delete" ) );
	buttonHelp->setText( tr2i18n( "&Help" ) );
	buttonHelp->setAccel( QKeySequence( tr2i18n( "F1" ) ) );
	buttonOk->setText( tr2i18n( "&OK" ) );
	buttonOk->setAccel( QKeySequence( QString::null ) );
	buttonCancel->setText( tr2i18n( "Ca&ncel" ) );
	buttonCancel->setAccel( QKeySequence( tr2i18n( "Alt+N" ) ) );
}

void BookmarksDlg::addBookmark()
{
	m_bookmarks.push_back(Bookmark(2, "new bookmark"));
	std::sort(m_bookmarks.begin(), m_bookmarks.end());
	updateBookmarks();
}

void BookmarksDlg::deleteBookmark()
{
	int row = bookmarksTable->currentRow();
	if (row >= 0 && row < m_bookmarks.size())
	{
		m_bookmarks.erase(m_bookmarks.begin() + row);
		updateBookmarks();
	}
}

void BookmarksDlg::updateBookmarks()
{
	bookmarksTable->setNumRows(0);
	for (Bookmarks::size_type i = 0; i < m_bookmarks.size(); ++i)
	{
		const Bookmark& bm = m_bookmarks[i];
		bookmarksTable->setNumRows(i+1);
		bookmarksTable->setText(i, 0, bm.name());
		bookmarksTable->setText(i, 1, QString::number(bm.page()));
	}	
}

void BookmarksDlg::valueChanged(int row, int col)
{
	if (row >= m_bookmarks.size()) return;
	Bookmark& bm = m_bookmarks[row];
	//Bookmark title
	if (col == 0)
		bm.setName(bookmarksTable->text(row, col));
	//page number
	else if (col == 1)
	{
		bool ok;
		unsigned int page = bookmarksTable->text(row, col).toUInt(&ok);
		if (ok)
			bm.setPage(page);
		else
			//return old value
			bookmarksTable->setText(row, col, QString::number(bm.page()));
	}
}

#include "bookmarksdlg.moc"
