/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#ifndef BIBPARSER_H
#define BIBPARSER_H

#include "c2bUtils.h"
#include "monthDB.h"

#include <QDir>
#include <QObject>
#include <QRegExp>
#include <QStringList>


/**
    Class for Bibliographic reference manipulation

    @author Pere Constans
*/

typedef QHashIterator<QString, QString> bibReferenceIterator;

class bibReference : public QHash<QString, QString>
{

    friend class bibParser;

public:
    bibReference()
    {}

    void clearFields()
    {
        pos = 0;
        bib_fieldList.clear();
        bib_fieldListNB_re.clear();
        bib_fieldList_re.clear();
    }
    void clearReference()
    {
        positionValue = 0;
        keyName.clear();
        typeName.clear();
        rawReference.clear();
        unicodeReference.clear();
        QHash<QString, QString>::clear();
    }
    QString keyName;
    QString rawReference;
    QString typeName;
    QString unicodeReference;
    int positionValue;


private:
    QList<QRegExp> bib_fieldListNB_re;
    QList<QRegExp> bib_fieldList_re;
    QStringList bib_fieldList;
    int pos;

};


class bibParser : public QObject
{

    Q_OBJECT

public:
    bibParser(QObject* parent = 0);
    ~bibParser();

    QStringList bibFieldList;
    QStringList bibFieldSortedList;
    const bibReference wholeReference(const QString& str);
    void initReferenceParsing(const QString& fn, const QStringList& fields, bibReference* ref);

    inline bool hasBibTeX(const QString& str)
    {
        if (bib_begin0_re.indexIn(str) > -1)
            return true;
        else
            return bib_begin1_re.indexIn(str) > -1;
    }

    inline bool referencesIn(const QString& str, bibReference* ref)
    {
        // File parsing for given fields in ref
        ref->clearReference();
        int pos = referenceStarts(str, ref->pos);
        if (pos < 0)
            return false;
        ref->positionValue = pos;
        QString str_ref = referenceAt(str, &pos);
        ref->pos = pos;
        ref->rawReference = str_ref;
        c2bUtils::bibToC2b(str_ref);
        str_ref = str_ref.simplified();
        ref->unicodeReference = str_ref;
        bib_key_re.indexIn(str_ref);
        ref->keyName = bib_key_re.cap(1);
        bib_type_re.indexIn(str_ref);
        ref->typeName = bib_type_re.cap(1).toLower();
        str_ref.replace(str_ref.length() - 1, 1, ",}");
        QString fvalue;
        for (int i = 0; i < ref->bib_fieldList_re.count(); ++i)
        {
            QRegExp* bf = &ref->bib_fieldList_re[i];
            pos = bf->indexIn(str_ref);
            if (pos > 0)
            {
                if (inBraces(pos + bf->matchedLength(), str_ref, &fvalue))
                    (*ref)[ref->bib_fieldList.at(i)] = fvalue.trimmed();
            }
            else
            {
                bf = &ref->bib_fieldListNB_re[i];
                if (bf->indexIn(str_ref) > -1)
                    (*ref)[ref->bib_fieldList.at(i)] = bf->cap(1).trimmed();
            }
        }
        if (ref->contains("file"))
            if (!QDir::isAbsolutePath(ref->value("file")))
                (*ref)["file"] = bib_file_dir + ref->value("file");
        return true;
    }


protected:
    QRegExp field_re;
    QString medlToc2b(const QString&);
    QString setPages(const QString&);
    QStringList TypesList;
    monthDB* MDB;


private:
    QList<QRegExp> bib_fieldListNB_re;
    QList<QRegExp> bib_fieldList_re;
    QRegExp bib_begin0_re;
    QRegExp bib_begin1_re;
    QRegExp bib_begin_re;
    QRegExp bib_key_re;
    QRegExp bib_type_re;
    QString bib_file_dir;
    void setBibTypes();
    void setFields();
    void setRegularExpressions();

    inline int referenceStarts(const QString& str, int pos = 0)
    {
        int i;
        if (pos == 0)
        {
            i = bib_begin0_re.indexIn(str);
            if (i < 0)
                i = bib_begin1_re.indexIn(str, pos);
        }
        else
            i = bib_begin1_re.indexIn(str, pos);
        if (i < 0)
            return i;
        return bib_begin_re.indexIn(str, i);
    }

    inline int referenceEnds(const QString& str, int pos = 0)
    {
        // If referenceStarts call is successful, we know for sure
        // that there is an opening { right after pos.
        // Do not check again here.
        // Checking for brace closure is the safest way for parsing.
        // It will fail, though, for references incorrectly written.
        // Avoid overextending in these cases by checking the
        // start of the next reference.
        int ref_length = referenceStarts(str, pos + 2) - 1;
        if (ref_length < 0)
            ref_length = str.length();
        int brace_pos = str.indexOf('{', pos);
        int open_braces = 1;
        QChar si;
        for (int i = brace_pos + 1; i < ref_length; ++i)
        {
            si = str.at(i);
            if (si == '{')
                open_braces++;
            else if (si == '}')
                open_braces--;
            if (open_braces == 0)
                return i;
        }
        return ref_length - 1;
    }

    inline const QString referenceAt(const QString& str, int* pos)
    {
        // String str contains one or multiple references (file contents)
        int _pos = referenceEnds(str, *pos) + 1;
        QString str_ref = str.mid(*pos, _pos - (*pos));
        *pos = _pos;
        return str_ref;
    }

    inline bool inBraces(const int pos, const QString& str, QString* in)
    {
        if (str.at(pos - 1) == '{') // pos > 0 always
        {
            int open_braces = 1;
            QChar si;
            for (int i = pos; i < str.length(); ++i)
            {
                si = str.at(i);
                if (si == '{')
                    open_braces++;
                else if (si == '}')
                    open_braces--;
                if (open_braces == 0)
                {
                    *in = str.mid(pos, i - pos);
                    return true;
                }
            }
        }
        else // in Quotes
            for (int i = pos; i < str.length() - 1; ++i)
                if (str.at(i) == '"')
                    if (str.at(i + 1) == ',')
                    {
                        *in = str.mid(pos, i - pos);
                        return true;
                    }
        return false;
    }

};

#endif
