/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "c2bCiteIDMaker.h"

#include "c2b.h"
#include "c2bSettings.h"
#include "c2bUtils.h"

#include <QWidget>


c2bCiteIDMaker::c2bCiteIDMaker(QHash<QString, c2bLineEdit*>& fields, QObject* parent) : QObject(parent), bibFields(fields)
{
    c2bMain = c2b::mainWidget();
    m_author = "";
    m_journal = "";
    m_pages = "";
    m_volume = "";
    m_year = "";
    loadSettings();
    connect(c2bSettingsP, SIGNAL(newSettings()), this, SLOT(loadSettings()));
}

c2bCiteIDMaker::~c2bCiteIDMaker()
{}


/** \page citeid_placeholders Predefined cite ID placeholders

    - <tt><<author_all_initials>></tt> Takes capitalized initials of all authors in cite.

    - <tt><<author_first>></tt> Takes first author last name/s.

    - <tt><<journal_initials>></tt> Takes capitalized initials of journal name.

    - <tt><<pages_first>></tt> First page.

    - <tt><<volume>></tt> Volume number.

    - <tt><<year_abbreviated>></tt> Last two digits from year.

    - <tt><<year_full>></tt> All digits from year.


    <b>Note:</b> If <tt>author</tt> is empty, <tt>editor</tt> will be
    considered instead. On conference proceedings or monographs this situation
    is usual.

*/
void c2bCiteIDMaker::makeCiteID()
{
    QString cid = m_citeidpattern;
    cid.replace("<<author_first>>", m_author);
    cid.replace("<<author_all_initials>>", m_author);
    cid.replace("<<journal_initials>>", m_journal);
    cid.replace("<<pages_first>>", m_pages);
    cid.replace("<<volume>>", m_volume);
    cid.replace("<<year_abbreviated>>", m_year);
    cid.replace("<<year_full>>", m_year);
    cid.remove(" ");
    bibFields["id"]->setText(cid);
    if (cid.isEmpty())
        cid = "(none)";
    c2bMain->setWindowTitle(tr("%1 - cb2Bib").arg(cid));
}

void c2bCiteIDMaker::makeCiteID_author_first()
{
    m_author = bibFields["author"]->text();
    if (m_author.isEmpty())
        m_author = bibFields["editor"]->text();
    QRegExp rx("([-'\\s\\w]+)(\\sand|$)");
    rx.setMinimal(true);
    rx.indexIn(m_author);
    m_author = rx.cap(1);
    if (m_author.contains(QRegExp("[^A-Za-z]")))
        m_author = c2bUtils::c2bToAscii(m_author);
    makeCiteID();
}

void c2bCiteIDMaker::makeCiteID_author_all_initials()
{
    m_author = bibFields["author"]->text();
    if (m_author.isEmpty())
        m_author = bibFields["editor"]->text();
    m_author.remove(QRegExp("\\b\\w\\b"));
    m_author.remove(" and ");
    m_author.remove(QRegExp("[a-z]"));
    if (m_author.contains(QRegExp("[^A-Z]")))
        m_author = c2bUtils::c2bToAscii(m_author);
    makeCiteID();
}

void c2bCiteIDMaker::makeCiteID_journal_initials()
{
    m_journal = bibFields["journal"]->text();
    m_journal.remove(QRegExp("[^A-Z]"));
    makeCiteID();
}

void c2bCiteIDMaker::makeCiteID_pages_first()
{
    m_pages = c2bUtils::firstPage(bibFields["pages"]->text());
    makeCiteID();
}

void c2bCiteIDMaker::makeCiteID_volume()
{
    m_volume = bibFields["volume"]->text();
    makeCiteID();
}

void c2bCiteIDMaker::makeCiteID_year_abbreviated()
{
    m_year = bibFields["year"]->text().right(2);
    makeCiteID();
}

void c2bCiteIDMaker::makeCiteID_year_full()
{
    m_year = bibFields["year"]->text();
    makeCiteID();
}

void c2bCiteIDMaker::loadSettings()
{
    m_citeidpattern = c2bSettingsP->value("cb2Bib/CiteIdPattern").toString();
    setPatternConnections();
}

void c2bCiteIDMaker::setPatternConnections()
{
    disconnect(bibFields["author"], SIGNAL(textChanged(const QString&)), this, 0);
    disconnect(bibFields["editor"], SIGNAL(textChanged(const QString&)), this, 0);
    disconnect(bibFields["journal"], SIGNAL(textChanged(const QString&)), this, 0);
    disconnect(bibFields["pages"], SIGNAL(textChanged(const QString&)), this, 0);
    disconnect(bibFields["volume"], SIGNAL(textChanged(const QString&)), this, 0);
    disconnect(bibFields["year"], SIGNAL(textChanged(const QString&)), this, 0);

    // author (track also editor, in case author is empty)
    if (m_citeidpattern.contains("<<author_first>>"))
    {
        connect(bibFields["author"], SIGNAL(textChanged(const QString&)), this, SLOT(makeCiteID_author_first()));
        connect(bibFields["editor"], SIGNAL(textChanged(const QString&)), this, SLOT(makeCiteID_author_first()));
    }
    else if (m_citeidpattern.contains("<<author_all_initials>>"))
    {
        connect(bibFields["author"], SIGNAL(textChanged(const QString&)), this, SLOT(makeCiteID_author_all_initials()));
        connect(bibFields["editor"], SIGNAL(textChanged(const QString&)), this, SLOT(makeCiteID_author_all_initials()));
    }

    // journal
    if (m_citeidpattern.contains("<<journal_initials>>"))
        connect(bibFields["journal"], SIGNAL(textChanged(const QString&)), this, SLOT(makeCiteID_journal_initials()));

    // pages
    if (m_citeidpattern.contains("<<pages_first>>"))
        connect(bibFields["pages"], SIGNAL(textChanged(const QString&)), this, SLOT(makeCiteID_pages_first()));

    // volume
    if (m_citeidpattern.contains("<<volume>>"))
        connect(bibFields["volume"], SIGNAL(textChanged(const QString&)), this, SLOT(makeCiteID_volume()));

    // year
    if (m_citeidpattern.contains("<<year_abbreviated>>"))
        connect(bibFields["year"], SIGNAL(textChanged(const QString&)), this, SLOT(makeCiteID_year_abbreviated()));
    else if (m_citeidpattern.contains("<<year_full>>"))
        connect(bibFields["year"], SIGNAL(textChanged(const QString&)), this, SLOT(makeCiteID_year_full()));
}
