/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "c2bFileDialog.h"
#include "c2bSettings.h"

#include <QDialogButtonBox>
#include <QDirModel>
#include <QPushButton>
#include <QtDebug>


c2bFileDialog::c2bFileDialog(QWidget* parent, const QString& caption, const QString& path, const QString& fn)
        : QFileDialog(parent, caption)
{
    setModal(true);
    settings = c2bSettingsP;
#ifdef Q_WS_X11
    // Distinguish executable files
    fip = new fileIconProvider();
    setIconProvider(fip);
#endif
    _qt_version = QString(qVersion()).remove('.').toInt();
    setDirectory(path);
    selectFile(fn);
}

c2bFileDialog::~c2bFileDialog()
{
#ifdef Q_WS_X11
    delete fip;
#endif
}


QString c2bFileDialog::getFileName(QWidget* parent, const QString& caption, const QString& path, const QString& filter,
                                   QString* selectedFilter)
{
    QString _caption;
    if (caption.isEmpty())
        _caption = tr("Select a filename - cb2Bib");
    else
        _caption = caption + " - cb2Bib";
    c2bFileDialog* dlg = new c2bFileDialog(parent, _caption, _path(path), _filename(path));
    dlg->setConfirmOverwrite(false);
    dlg->setFileMode(AnyFile);
    dlg->setLabelText(QFileDialog::Accept, "Select");
    QString fn = dlg->exec(filter, selectedFilter);
    delete dlg;
    return fn;
}

QString c2bFileDialog::getOpenFileName(QWidget* parent, const QString& caption, const QString& path, const QString& filter,
                                       QString* selectedFilter)
{
    QString _caption;
    if (caption.isEmpty())
        _caption = tr("Select a filename - cb2Bib");
    else
        _caption = caption + " - cb2Bib";
    c2bFileDialog* dlg = new c2bFileDialog(parent, _caption, _path(path), _filename(path));
    dlg->setFileMode(ExistingFile);
    dlg->setLabelText(QFileDialog::Accept, "Select");
    QString fn = dlg->exec(filter, selectedFilter);
    delete dlg;
    return fn;
}

QStringList c2bFileDialog::getOpenFileNames(QWidget* parent, const QString& caption, const QString& path, const QString& filter,
        QString* selectedFilter)
{
    QString _caption;
    if (caption.isEmpty())
        _caption = tr("Select filenames - cb2Bib");
    else
        _caption = caption + " - cb2Bib";
    c2bFileDialog* dlg = new c2bFileDialog(parent, _caption, _path(path), _filename(path));
    dlg->setFileMode(ExistingFiles);
    dlg->setLabelText(QFileDialog::Accept, "Select");
    QString fn = dlg->exec(filter, selectedFilter);
    QStringList fns;
    if (!fn.isEmpty())
        fns = dlg->selectedFiles();
    delete dlg;
    return fns;
}

QString c2bFileDialog::getSaveFileName(QWidget* parent, const QString& caption, const QString& path, const QString& filter,
                                       QString* selectedFilter)
{
    QString _caption;
    if (caption.isEmpty())
        _caption = tr("Save As - cb2Bib");
    else
        _caption = caption + " - cb2Bib";
    c2bFileDialog* dlg = new c2bFileDialog(parent, _caption, _path(path), _filename(path));
    dlg->setFileMode(AnyFile);
    dlg->setAcceptMode(AcceptSave);
    QString fn = dlg->exec(filter, selectedFilter);
    delete dlg;
    return fn;
}

QString c2bFileDialog::getExistingDirectory(QWidget* parent, const QString& caption, const QString& path)
{
    QString _caption;
    if (caption.isEmpty())
        _caption = tr("Select a directory - cb2Bib");
    else
        _caption = caption + " - cb2Bib";
    c2bFileDialog* dlg = new c2bFileDialog(parent, _caption, _path(path), "");
    dlg->setFileMode(DirectoryOnly);
    dlg->setLabelText(QFileDialog::Accept, "Select");
    QString fn = dlg->exec();
    if (!fn.isEmpty())
    {
        fn = QDir::cleanPath(fn);
        fn += QDir::separator();
    }
    delete dlg;
    return fn;
}

QString c2bFileDialog::getSystemFileName(QWidget* parent, const QString& caption, const QString& path, const QString& filter,
        QString* selectedFilter)
{
    QString _caption;
    if (caption.isEmpty())
        _caption = tr("Select a filename - cb2Bib");
    else
        _caption = caption + " - cb2Bib";
    c2bFileDialog* dlg = new c2bFileDialog(parent, _caption, _path(path), _filename(path));
    dlg->setFileMode(AnyFile);
    dlg->setLabelText(QFileDialog::Accept, "Select");
#if QT_VERSION >= 0x040400
    dlg->setFilter(QDir::Drives | QDir::AllDirs | QDir::NoDotAndDotDot | QDir::System);
    dlg->setNameFilterDetailsVisible(false);
#endif
    QString fn = dlg->exec(filter, selectedFilter);
    delete dlg;
    return fn;
}

const QString c2bFileDialog::exec(const QString& _filter, QString* _selectedFilter)
{
    if (!_filter.isEmpty())
        setFilters(_filter.split(";;", QString::SkipEmptyParts));
    if (_selectedFilter)
        selectFilter(*_selectedFilter);
    readSettings();
#if QT_VERSION < 0x040400
    if (_qt_version > 429 && _qt_version < 433)
        enableSelectButton();
#endif
    QString fn;
    if (QFileDialog::exec() == QDialog::Accepted)
    {
        QStringList _selected = selectedFiles();
        if (_selected.count() > 0)
            fn = _selected.first();
        if (_selectedFilter)
            *_selectedFilter = selectedFilter();
    }
    writeSettings();
    return fn;
}

void c2bFileDialog::accept()
{
    // Set suffix for getSaveFileName dialog
    if (acceptMode() == AcceptSave)
    {
        QString suffix = selectedFilter();
        QRegExp sre("\\*\\.(\\w+)");
        if (sre.indexIn(suffix) != -1)
        {
            suffix = sre.cap(1);
            setDefaultSuffix(suffix);
        }
    }
    QFileDialog::accept();
}

void c2bFileDialog::enableSelectButton()
{
    // Bug fix for Qt 4.3.[0-2]
    QStringList sfs = selectedFiles();
    if (sfs.count() == 0)
        return;
    QFileInfo fi(sfs.at(0));
    if (fi.exists())
    {
        QDialogButtonBox* dbb = findChild<QDialogButtonBox*>();
        if (dbb)
        {
            QPushButton* sb = dbb->button(QDialogButtonBox::Open);
            if (sb)
                sb->setEnabled(true);
        }
    }
}

void c2bFileDialog::readSettings()
{
    // Size
    resize(settings->value("c2bFileDialog/size", sizeHint()).toSize());

    // History
    QStringList _history = settings->value("c2bFileDialog/history").toStringList();
    _history.removeAll(directory().absolutePath());
    qSort(_history);
    setHistory(_history);
}

void c2bFileDialog::writeSettings()
{
    // Size
    settings->setValue("c2bFileDialog/size", size());

    // History
    if (result() == QDialog::Accepted)
    {
        // Unique paths, keeping the newest history entries
        QStringList _history = history();
        _history.append(directory().absolutePath());
        _history.removeAll(QDir::homePath());
        _history.removeAll(QDir::rootPath());
        QStringList chronological = settings->value("c2bFileDialog/history").toStringList();
        chronological.append(_history.last());
        // Discard repeated
        _history.clear();
        for (int i = chronological.count() - 1; i > -1; --i)
            if (!_history.contains(chronological.at(i)))
            {
                _history.prepend(chronological.at(i));
                if (_history.count() > 10)
                    break;
            }
        settings->setValue("c2bFileDialog/history", _history);
    }
}

QString c2bFileDialog::_filename(const QString& path)
{
    if (path.isEmpty())
        return path;
    else
        return QFileInfo(path).fileName();
}

QString c2bFileDialog::_path(const QString& path)
{
    if (path.isEmpty())
        return QDir::currentPath();
    else
        return QFileInfo(path).absolutePath();
}


/***************************************************************************
 Class fileIconProvider
 ***************************************************************************/

fileIconProvider::fileIconProvider() : QFileIconProvider()
{
    _exec_icon = QIcon(":/icons/icons/application-x-executable.png");
}

fileIconProvider::~fileIconProvider()
{}


QIcon fileIconProvider::icon(const QFileInfo&  info) const
{
    if (info.isExecutable() && !info.isDir())
        return _exec_icon;
    else
        return QFileIconProvider::icon(info);
}
