/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "c2bNetwork.h"
#include "c2bSettings.h"

#include <QDesktopServices>
#include <QFileInfo>
#include <QProgressDialog>
#include <QTimer>
#include <QUrl>


c2bNetwork::c2bNetwork(QWidget* main, QObject* parent) : QObject(parent)
{
    c2bnetwork_fetching = false;
    c2bnetwork_getfile_error = "";
    c2bnetwork_openfile_error = "";
    loadSettings();
    connect(c2bSettingsP, SIGNAL(newSettings()), this, SLOT(loadSettings()));

    fetcher = new QHttp(this);
    fDestFile = 0;
    progressDialog = new QProgressDialog(main);
    progressDialog->setWindowTitle("cb2Bib");

    connect(fetcher, SIGNAL(requestFinished(int, bool)), this, SLOT(fetcherEnded(int, bool)));
    connect(fetcher, SIGNAL(dataReadProgress(int, int)), this, SLOT(updateDataReadProgress(int, int)));
    connect(progressDialog, SIGNAL(canceled()), this, SLOT(cancelFetcher()));
}

c2bNetwork::~c2bNetwork()
{}


/****************************************************************************

  PUBLIC PART

*****************************************************************************/

bool c2bNetwork::openFile(const QString& fn)
{
    if (FmClient && !FmClientOpenBin.isEmpty())
    {
        // Open NetworkFile through kfmclient_open
        QStringList arglist = FmClientOpenArg.split(" ", QString::SkipEmptyParts);
        arglist.append(fn);
        if (QProcess::startDetached(FmClientOpenBin, arglist))
        {
            c2bnetwork_openfile_error = "";
            return true;
        }
        else
        {
            c2bnetwork_openfile_error = tr("FM Client '%1' could not be launched.").arg(FmClientOpenBin);
            return false;
        }

    }
    else
    {
        // Open NetworkFile through QDesktopServices
        // Urls are assumed percent encoded
        QFileInfo fi = QFileInfo(fn);
        QUrl url;
        if (fi.exists())
            url = QUrl::fromLocalFile(fn);
        else
            url.setEncodedUrl(fn.toUtf8());
        if (QDesktopServices::openUrl(url))
        {
            c2bnetwork_openfile_error = "";
            return true;
        }
        else
        {
            c2bnetwork_openfile_error = tr("Desktop Service openUrl could not open %1.").arg(fn);
            return false;
        }
    }
}

void c2bNetwork::getFile(const QString& orig, const QString& dest, const QString& action, QObject* receiver, const char* callback,
                         const bool overwrite)
{
    if (c2bnetwork_fetching)
    {
        qWarning("[cb2bib] c2bNetwork::getFile: File requested while still fetching previous one. Returned.");
        return;
    }

    c2bnetwork_fetching = true;
    disconnect(this, SIGNAL(getFileDone(bool, const QString&)), 0, 0);
    if (receiver)
        connect(this, SIGNAL(getFileDone(bool, const QString&)), receiver, callback);

    if (overwrite)
        if (QFileInfo(dest).exists())
            QFile::remove(dest);
    if (!getFilePrivate(orig, dest, action))
        emit_getFileDone(false, c2bnetwork_getfile_error);
}

const QString c2bNetwork::encodeUrl(const QString& url)
{
    // Encodes Url to Percent Encoding, as needed by openFile
    // Removes <<post>> tag if present
    // Avoid encoding certains characters that would conflict with QDesktopServices::openUrl
    QString encoded_url = url;
    encoded_url.remove(QRegExp("^<<post>>"));
    encoded_url = QUrl::toPercentEncoding(encoded_url, "+:/?=&\\");
    return encoded_url;
}


/****************************************************************************

  PRIVATE PART

*****************************************************************************/

bool c2bNetwork::getFilePrivate(const QString& orig, const QString& dest, const QString& action)
{
    c2bnetwork_getfile_error = "";
    origFile = orig;
    destFile = dest;
    if (!checkDest())
        return false;

    if (origFile.startsWith("<<post>>"))         // cb2Bib keyword to use post http method
    {
        origFile.remove(QRegExp("^<<post>>"));
        return getFile_c2b(action);
    }

    if (FmClient)
    {
        if (action == "copy" && !FmClientCopyBin.isEmpty())
            return getFile_client(action);
        else if (action == "move" && !FmClientMoveBin.isEmpty())
            return getFile_client(action);
    }
    return getFile_c2b(action);
}

void c2bNetwork::emit_getFileDone(bool status, const QString& error)
{
    c2bnetwork_getfile_status = status;
    c2bnetwork_getfile_error = error;
    // Give some time to cleanup events and to return all c2bNetwork functions
    // before passing the control to the callback routine
    QTimer::singleShot(100, this, SLOT(emit_getFileDone()));
}

void c2bNetwork::emit_getFileDone()
{
    c2bnetwork_fetching = false;
    // Assumed events are clean, all functions returned, then make the callback
    emit getFileDone(c2bnetwork_getfile_status, c2bnetwork_getfile_error);
}

bool c2bNetwork::checkDest()
{
    // Checks whether or not writing to dest is safe
    // Returns false if file exists
    QFile f(destFile);
    if (f.exists())
    {
        c2bnetwork_getfile_error = tr("Destination file '%1' already exists.").arg(destFile);
        return false;
    }
    else
        return true;
}

void c2bNetwork::loadSettings()
{
    c2bSettings* settings = c2bSettingsP;
    FmClient = settings->value("cb2Bib/FmClient").toBool();
    FmClientCopyBin = settings->fileName("cb2Bib/FmClientCopyBin");
    FmClientMoveBin = settings->fileName("cb2Bib/FmClientMoveBin");
    FmClientOpenBin = settings->fileName("cb2Bib/FmClientOpenBin");
    FmClientCopyArg = settings->value("cb2Bib/FmClientCopyArg").toString();
    FmClientMoveArg = settings->value("cb2Bib/FmClientMoveArg").toString();
    FmClientOpenArg = settings->value("cb2Bib/FmClientOpenArg").toString();
}


/****************************************************************************

  PRIVATE PART: FILEMANAGER CLIENT

*****************************************************************************/

bool c2bNetwork::getFile_client(const QString& action)
{
    // Getting NetworkFile through kfmclient
    QString _action = action;
    QUrl u(origFile);            // Move local files only
    QString scheme = u.scheme();
    if (!(scheme == "file" || QFileInfo(origFile).exists()))
        if (_action == "move")
        {
            qWarning("[cb2bib] c2bNetwork::getFile: Source network file will not be deleted.");
            _action = "copy";
        }

    QStringList arglist;
    QString fmclient_bin;
    if (_action == "copy")
    {
        fmclient_bin = FmClientCopyBin;
        arglist = FmClientCopyArg.split(" ", QString::SkipEmptyParts);
    }
    else if (_action == "move")
    {
        fmclient_bin = FmClientMoveBin;
        arglist = FmClientMoveArg.split(" ", QString::SkipEmptyParts);
    }
    arglist.append(origFile);
    arglist.append(destFile);
    clientP = new QProcess(this);
    connect(clientP, SIGNAL(finished(int, QProcess::ExitStatus)), this, SLOT(clientEnded(int, QProcess::ExitStatus)));
    clientP->start(fmclient_bin, arglist);
    if (clientP->waitForStarted())
        return true;
    else
    {
        c2bnetwork_getfile_error = tr("FM Client '%1' could not be launched.").arg(fmclient_bin);
        delete clientP;
        return false;
    }
}

void c2bNetwork::clientEnded(int exitCode, QProcess::ExitStatus exitStatus)
{
    if (exitStatus == QProcess::CrashExit)
        emit emit_getFileDone(false, tr("Client crashed."));
    else
    {
        QFile f(destFile);
        if (f.exists())
            emit emit_getFileDone(true, "");
        else
            emit emit_getFileDone(false, tr("File '%1' has not been written. Exit code '%2'.").arg(origFile).arg(exitCode));
    }
    delete clientP;
}


/****************************************************************************

  PRIVATE PART: C2B FETCHER

*****************************************************************************/

bool c2bNetwork::getFile_c2b(const QString& action)
{
    QUrl u(origFile);
    QString scheme = u.scheme();
    if (scheme == "file" || QFileInfo(origFile).exists())
    {
        // Local File
        QString _origFile;
        if (scheme == "file")
            _origFile = u.toLocalFile();
        else
            _origFile = origFile;
        QFile orig(_origFile);
        bool succeeded = false;
        if (action == "copy")
            succeeded = orig.copy(destFile);
        else if (action == "move")
            succeeded = orig.rename(destFile);
        if (succeeded)
            emit_getFileDone(true, "");
        else
            c2bnetwork_getfile_error = orig.errorString();
        return succeeded;
    }
    else
    {
        // Network File
        if (action == "move")
            qWarning("[cb2bib] c2bNetwork::getFile: Source network file will not be deleted.");

        QString path;
        QString mode;
        QString searchString = u.encodedQuery();
        if (searchString.isEmpty())
        {
            path = u.path();
            mode = "GET";
        }
        else
        {
            path = u.path() + "/";
            mode = "POST";
        }
        QString host = u.host();

        QHttpRequestHeader header(mode, path);
        header.setValue("Host", host);
        header.setValue("User-Agent", "cb2Bib/" + C2B_VERSION + " (Bibliographic Browser Tool)");
        header.setContentType("application/x-www-form-urlencoded");

        progressDialog->setLabelText(tr("Retrieving from %1...").arg(host));
        fetcher->setHost(host, u.port() != -1 ? u.port() : 80);
        if (!u.userName().isEmpty())
            fetcher->setUser(u.userName(), u.password());
        fetcherRequestAborted = false;
        fDestFile = new QFile(destFile);
        fetcherGetId = fetcher->request(header, searchString.toUtf8(), fDestFile);
        return true;
    }
}

void c2bNetwork::fetcherEnded(int requestId, bool error)
{
    if (requestId != fetcherGetId)
        return;
    if (fetcherRequestAborted)
        return;

    progressDialog->reset();
    if (fDestFile)
    {
        fDestFile->close();
        delete fDestFile;          // Delete IO device, not the file
        fDestFile = 0;
    }
    if (error)
        emit_getFileDone(false, fetcher->errorString().toLatin1());
    else
        emit_getFileDone(true, "");
}

void c2bNetwork::cancelFetcher()
{
    fetcherRequestAborted = true;
    fetcher->abort();
    QTimer::singleShot(300, this, SLOT(fetcherCleanUp()));
}

void c2bNetwork::fetcherCleanUp()
{
    fetcherRequestAborted = false;
    if (fDestFile)
    {
        fDestFile->close();
        fDestFile->remove();                    // Delete file
    }
    fetcherEnded(fetcherGetId, true);
}

void c2bNetwork::updateDataReadProgress(int readBytes, int totalBytes)
{
    if (fetcherRequestAborted)
        return;
    if (totalBytes == 0)
        return;
    progressDialog->setMaximum(totalBytes);
    progressDialog->setValue(readBytes);
}
