/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "c2bNetworkQuery.h"
#include "c2b.h"
#include "c2bSettings.h"
#include "c2bUtils.h"

#include <QDir>
#include <QMessageBox>
#include <QTimer>
#include <QUrl>


c2bNetworkQuery::c2bNetworkQuery(QObject* parent) : QObject(parent)
{
    settings = c2bSettingsP;
    network = c2b::network();
    pWidget = c2b::mainWidget();
    settings->setValue("c2bNetworkQuery/isSupervised", true);
    loadSettings();
    connect(settings, SIGNAL(newSettings()), this, SLOT(loadSettings()));

    Queryfile1 = settings->tmp_dir_path + "cb2bib_query_tmp_html1";
    Queryfile2 = settings->tmp_dir_path + "cb2bib_query_tmp_html2";
}

c2bNetworkQuery::~c2bNetworkQuery()
{}


void c2bNetworkQuery::submitQuery(const QString& title, const QString& jn, const QString& vol, const QString& fp,
                                  const QString& doi, const QString& excerpt)
{
    // Submission Initialization
    error = "";
    m_q_numbers.clear();
    m_endoffile = false;
    Title = title;
    JournalName = jn;
    Volume = vol;
    FirstPage = fp;
    DOI = doi.trimmed();
    if (!DOI.isEmpty() && !DOI.contains(QRegExp("^10.[\\d\\.]+/\\S+$")))
    {
        DOI = "";
        qWarning(qPrintable(tr("[cb2bib] Warning: DOI skipped: '%1' is not a valid DOI.").arg(doi)));
    }
    Excerpt = excerpt;
    submitQuery1();
}

void c2bNetworkQuery::submitQuery1()
{
    // Submission, first step, setting journal codes

    if (!setQueryParameters())
    {
        error = tr("No data for query.");
        emit queryEnded(false, targetQ, Queryfile1);
        return;
    }
    if (m_endoffile)
    {
        error = tr("Performed %1 queries: No reference found.").arg(m_q_numbers.count());
        emit queryEnded(false, targetQ, Queryfile1);
        return;
    }

    c2bUtils::debug(tr("Query Number = %1").arg(m_q_numbers.count()));
    c2bUtils::debug(tr("targetQ[%1]").arg(targetQ));
    c2bUtils::debug(tr("captionQ[%1]").arg(captionQ));
    c2bUtils::debug(tr("referenceurl_prefix[%1]").arg(referenceurl_prefix));
    c2bUtils::debug(tr("referenceurl_sufix[%1]").arg(referenceurl_sufix));
    c2bUtils::debug(tr("pdfurl_prefix[%1]").arg(pdfurl_prefix));
    c2bUtils::debug(tr("pdfurl_sufix[%1]").arg(pdfurl_sufix));
    c2bUtils::debug(tr("action[%1]").arg(action));
    c2bUtils::debug(tr("POST1[%1]").arg(targetQ));

    if (action == "browse_query")
    {
        if (network->openFile(network->encodeUrl(targetQ)))
        {
            error = "Browsing query.";
            emit queryEnded(true, "", "");
        }
        else
        {
            error = network->openfileErrorString();
            emit queryEnded(false, "", "");
        }
        return;
    }
    c2b::showMessage(tr("Query: %1.").arg(targetQ));
    network->getFile(targetQ, Queryfile1, "copy", this, SLOT(submitQuery2(bool)), !KeepTmpNQFiles);
}

void c2bNetworkQuery::submitQuery2(bool stat)
{
    // Submission, second part: extracting reference location

    if (!stat)
    {
        error = network->getfileErrorString();
        emit queryEnded(false, targetQ, Queryfile1);
        return;
    }

    QString lines = c2bUtils::fileToString(Queryfile1, !KeepTmpNQFiles);
    QRegExp rx(captionQ);
    rx.setMinimal(true);
    if (!rx.isValid())
        qWarning(qPrintable(tr("[cb2bib] Warning: RegExp '%1' is not valid.").arg(captionQ)));
    int ncap = rx.indexIn(lines);
    if (ncap > -1)
        lines = c2bUtils::fromHtmlString(rx.cap(1));
    else
    {
        QTimer::singleShot(10, this, SLOT(submitQuery1()));
        return;
    }
    targetBib = referenceurl_prefix + lines + referenceurl_sufix;
    if (pdfurl_prefix.isEmpty() && pdfurl_sufix.isEmpty())
        targetPDF = "";
    else if (AutomaticPdfDownload)
        targetPDF = pdfurl_prefix + lines + pdfurl_sufix;

    c2bUtils::debug(tr("CAPTURED[%1]").arg(lines));
    c2bUtils::debug(tr("POST2[%1]").arg(targetBib));
    c2bUtils::debug(tr("POST3[%1]").arg(targetPDF));

    if (action == "browse_referenceurl")
    {
        if (network->openFile(network->encodeUrl(targetBib)))
        {
            error = "Browsing reference.";
            emit queryEnded(true, "", "");
        }
        else
        {
            error = network->openfileErrorString();
            emit queryEnded(false, "", "");
        }
        return;
    }
    c2b::showMessage(tr("Retrieving: %1.").arg(targetBib));
    network->getFile(targetBib, Queryfile2, "copy", this, SLOT(queryDone(bool)), !KeepTmpNQFiles);
}

void c2bNetworkQuery::queryDone(bool stat)
{
    // Submission Done

    if (!stat)
    {
        error = network->getfileErrorString();
        emit queryEnded(false, targetBib, Queryfile2);
        return;
    }

    QString lines = c2bUtils::fileToString(Queryfile2, !KeepTmpNQFiles);
    if (targetBib.contains("MEDLINE"))
    {
        QRegExp med_rx;
        med_rx.setMinimal(true);
        med_rx.setPattern("^.+\"pre_replace\">(.+)</table><.+$");
        lines.replace(med_rx, "\\1");
        lines.replace("</tr>", "_NEW_LINE_");
        med_rx.setPattern("<.+>");
        lines.replace(med_rx, " ");
        lines = c2bUtils::fromHtmlString(lines);     // Convert to unicode the text-related HTML tags in Pubmed, keep lines
        lines.replace("_NEW_LINE_", "\n");
    }
    targetBib = lines.trimmed();
    emit queryEnded(true, targetPDF, targetBib);
}

const QString c2bNetworkQuery::errorString()
{
    return error;
}

void c2bNetworkQuery::loadSettings()
{
    AutomaticPdfDownload = settings->value("cb2Bib/AutomaticPdfDownload").toBool();
    KeepTmpNQFiles = settings->value("cb2Bib/KeepTmpNQFiles").toBool();
    NetworkFile = settings->fileName("cb2Bib/NetworkFile");
}

bool c2bNetworkQuery::setQueryParameters()
{
    if (NetworkFile.isEmpty())
    {
        QMessageBox::information(pWidget, tr("Information - cb2Bib"),
                                 tr("No Network Query file has been specified.\n\n"
                                    "Note: Network Query files are specified through the cb2Bib Configure dialog."),
                                 QMessageBox::Ok);
        return false;
    }
    QFile file(NetworkFile);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
    {
        QMessageBox::warning(pWidget, tr("Warning - cb2Bib"),
                             tr("Unable to open the Network Query file %1 for reading.\nError: '%2'.\n\n"
                                "Note: Network Query files are specified through the cb2Bib Configure dialog.")
                             .arg(NetworkFile).arg(file.errorString()),
                             QMessageBox::Ok);
        return false;
    }

    QString line;
    QTextStream stream(&file);
    stream.setCodec("UTF-8");
    stream.setAutoDetectUnicode(true);
    QRegExp Journal("journal=" + JournalName + "\\|");
    QRegExp AnyJournal("journal=\\s*$");
    uint readQueryParams = 0;
    m_endoffile = false;
    while (!stream.atEnd())
    {
        line = stream.readLine();

        // Skip comments and blanks
        if (!(line.isEmpty() || line.contains(QRegExp("^#"))))
        {
            if (line.contains(Journal))
                JournalCode = line.split("|").at(1);
            else if (line.contains(AnyJournal))
                JournalCode = JournalName;

            // Get appropiate parameters for Journal or AnyJournal
            if (line.contains(Journal) || line.contains(AnyJournal))
            {

                // Skip if already performed
                if (!m_q_numbers.contains(++readQueryParams))
                {
                    while (line.contains(QRegExp("^journal=")))
                        line = stream.readLine();
                    targetQ = line.replace(QRegExp("^query="), "");
                    line = stream.readLine();
                    captionQ = line.replace(QRegExp("^capture_from_query="), "");
                    line = stream.readLine();
                    referenceurl_prefix = line.replace(QRegExp("^referenceurl_prefix="), "");
                    line = stream.readLine();
                    referenceurl_sufix = line.replace(QRegExp("^referenceurl_sufix="), "");
                    line = stream.readLine();
                    pdfurl_prefix = line.replace(QRegExp("^pdfurl_prefix="), "");
                    line = stream.readLine();
                    pdfurl_sufix = line.replace(QRegExp("^pdfurl_sufix="), "");
                    line = stream.readLine();
                    action = line.replace(QRegExp("^action="), "");

                    // Setting Query Parameters
                    updateQueryPlaceholders();

                    // Finally, check for unresolved cb2Bib tags
                    if (areQueryParametersValid())
                    {
                        m_q_numbers.append(readQueryParams);
                        return true;
                    }
                }
            }
        }
    }
    file.close();
    m_endoffile = true;
    return (!m_q_numbers.isEmpty());
}

void c2bNetworkQuery::updateQueryPlaceholders()
{
    if (!Title.isEmpty())
    {
        targetQ.replace("<<title>>", Title);
        captionQ.replace("<<title>>", Title);
        referenceurl_prefix.replace("<<title>>", Title);
        referenceurl_sufix.replace("<<title>>", Title);
        pdfurl_prefix.replace("<<title>>", Title);
        pdfurl_sufix.replace("<<title>>", Title);
    }
    if (!JournalCode.isEmpty())
    {
        targetQ.replace("<<journal>>", JournalCode);
        captionQ.replace("<<journal>>", JournalCode);
        referenceurl_prefix.replace("<<journal>>", JournalCode);
        referenceurl_sufix.replace("<<journal>>", JournalCode);
        pdfurl_prefix.replace("<<journal>>", JournalCode);
        pdfurl_sufix.replace("<<journal>>", JournalCode);
    }
    if (!FirstPage.isEmpty())
    {
        targetQ.replace("<<pages>>", FirstPage);
        captionQ.replace("<<pages>>", FirstPage);
        referenceurl_prefix.replace("<<pages>>", FirstPage);
        referenceurl_sufix.replace("<<pages>>", FirstPage);
        pdfurl_prefix.replace("<<pages>>", FirstPage);
        pdfurl_sufix.replace("<<pages>>", FirstPage);
    }
    if (!Volume.isEmpty())
    {
        targetQ.replace("<<volume>>", Volume);
        captionQ.replace("<<volume>>", Volume);
        referenceurl_prefix.replace("<<volume>>", Volume);
        referenceurl_sufix.replace("<<volume>>", Volume);
        pdfurl_prefix.replace("<<volume>>", Volume);
        pdfurl_sufix.replace("<<volume>>", Volume);
    }
    if (!DOI.isEmpty())
    {
        targetQ.replace("<<doi>>", DOI);
        captionQ.replace("<<doi>>", DOI);
        referenceurl_prefix.replace("<<doi>>", DOI);
        referenceurl_sufix.replace("<<doi>>", DOI);
        pdfurl_prefix.replace("<<doi>>", DOI);
        pdfurl_sufix.replace("<<doi>>", DOI);
    }
    if (!Excerpt.isEmpty())
    {
        targetQ.replace("<<excerpt>>", Excerpt);
        captionQ.replace("<<excerpt>>", Excerpt);
        referenceurl_prefix.replace("<<excerpt>>", Excerpt);
        referenceurl_sufix.replace("<<excerpt>>", Excerpt);
        pdfurl_prefix.replace("<<excerpt>>", Excerpt);
        pdfurl_sufix.replace("<<excerpt>>", Excerpt);
    }
}

bool c2bNetworkQuery::areQueryParametersValid()
{
    if (!action.isEmpty())
        if (!settings->value("c2bNetworkQuery/isSupervised").toBool())
            return false;
    QString allParams = targetQ + captionQ + referenceurl_prefix + referenceurl_sufix + pdfurl_prefix + pdfurl_sufix;
    return !(allParams.contains(QRegExp("(<<title>>|<<journal>>|<<pages>>|<<volume>>|<<doi>>|<<excerpt>>)")));
}
