/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "c2bNetworkQueryInfo.h"

#include "c2b.h"
#include "c2bBibParser.h"
#include "c2bSettings.h"

#include <QAction>
#include <QFileSystemWatcher>
#include <QTextStream>
#include <QUrl>


c2bNetworkQueryInfo::c2bNetworkQueryInfo(c2bBibParser* bp, QObject* parent) :
        QObject(parent), bibFields(bp->bibFields), bibFieldList(bp->bibFieldList)
{
    connect(c2bSettingsP, SIGNAL(newSettings()), this, SLOT(loadSettings()));
    fsw = new QFileSystemWatcher(this);
    connect(fsw, SIGNAL(fileChanged(const QString&)), this, SLOT(parseBookmarks()));
    loadSettings();
}

c2bNetworkQueryInfo::~c2bNetworkQueryInfo()
{}


void c2bNetworkQueryInfo::loadSettings()
{
    QString nfn = c2bSettingsP->fileName("cb2Bib/NetworkFile");
    if (nfn == NetworkFile)
        return;
    if (QFileInfo(nfn).exists())
    {
        if (!NetworkFile.isEmpty())
            fsw->removePath(NetworkFile);
        NetworkFile = nfn;
        fsw->addPath(NetworkFile);
    }
    parseBookmarks();
}

void c2bNetworkQueryInfo::parseBookmarks()
{
    BookmarkList.clear();
    if (NetworkFile.isEmpty())
        return;
    QFile file(NetworkFile);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
        return;
    QRegExp _bookmark("^bookmark=");
    QString line;
    QTextStream stream(&file);
    stream.setCodec("UTF-8");
    stream.setAutoDetectUnicode(true);
    while (!stream.atEnd())
    {
        line = stream.readLine();
        if (line.contains(_bookmark))
        {
            line.remove(_bookmark);
            if (line.count("|") == 1)
                BookmarkList.append(line);
            else
                qDebug(tr("[cb2bib] parseBookmarks: Syntax error at '%1'").arg(line).toLatin1());
        }
    }
    file.close();
    while (!BookmarkActionList.isEmpty())
        delete BookmarkActionList.takeFirst();
    QRegExp rx = QRegExp("<<.+>>");
    for (int i = 0; i < BookmarkList.count(); ++i)
        if (BookmarkList.at(i).contains(rx))
        {
            QStringList bm_parts = BookmarkList.at(i).split("|");
            QAction* act = new QAction(bm_parts.at(0), this);
            act->setData(bm_parts.at(1));
            act->setStatusTip(QString("%1  [%2]").arg(bm_parts.at(0)).arg(bm_parts.at(1)));
            connect(act, SIGNAL(triggered()), this, SLOT(actionTriggered()));
            BookmarkActionList.append(act);
        }
}

QList<QAction*> c2bNetworkQueryInfo::bookmarkActions(const QString& selection)
{
    _current_selection = selection;
    for (int i = 0; i < BookmarkActionList.count(); ++i)
        BookmarkActionList[i]->setVisible(false);
    QRegExp rx = QRegExp("<<.+>>");
    for (int i = 0; i < BookmarkActionList.count(); ++i)
    {
        QString bmd = BookmarkActionList.at(i)->data().toString();
        for (int j = 0; j < bibFieldList.count(); ++j)
        {
            QString tag = QString("<<%1>>").arg(bibFieldList.at(j));
            if (bmd.contains(tag))
                if (!bibFields[bibFieldList.at(j)]->text().isEmpty())
                    bmd.remove(tag);
        }
        if (bmd.contains("<<selection>>"))
            if (!_current_selection.isEmpty())
                bmd.remove("<<selection>>");
        BookmarkActionList[i]->setVisible(!bmd.contains(rx));
    }
    return BookmarkActionList;
}

void c2bNetworkQueryInfo::actionTriggered()
{
    QAction* act = static_cast<QAction*>(sender());
    if (act)
    {
        QString link = act->data().toString();
        c2b::openFile(encodeLink(link));
    }
}

const QString c2bNetworkQueryInfo::encodeLink(const QString& ln) const
{
    QString eln = ln;
    // See comment in c2bNetworkQueryInfo::bookmarksToHtml
    eln.replace("[[[", "<<");
    eln.replace("]]]", ">>");
    QRegExp rx = QRegExp("<<.+>>");
    if (eln.contains(rx))
    {
        for (int j = 0; j < bibFieldList.count(); ++j)
        {
            QString tag = QString("<<%1>>").arg(bibFieldList.at(j));
            if (eln.contains(tag))
            {
                QString fld;
                if (bibFieldList.at(j) == "pages")
                    fld = bibFields[bibFieldList.at(j)]->text().replace(QRegExp("-.+$"), "");
                else if (bibFieldList.at(j) == "author")
                {
                    fld = bibFields[bibFieldList.at(j)]->text();
                    fld.replace(" and ", " ");
                }
                else if (bibFieldList.at(j) == "title")
                {
                    fld = bibFields[bibFieldList.at(j)]->text();
                    fld = fld.remove("{");
                    fld = fld.remove("}");
                }
                else
                    fld = bibFields[bibFieldList.at(j)]->text();
                fld = QUrl::toPercentEncoding(fld);
                eln.replace(tag, fld);
                if (!eln.contains(rx))
                    return eln;
            }
        }
        eln.replace("<<selection>>", _current_selection);
    }
    return eln;
}

const QString c2bNetworkQueryInfo::bookmarksToHtml() const
{
    if (BookmarkList.count() == 0)
        return "";
    QString html = "<p align=\"center\">Bookmarks:<br /><p align=\"center\"><em>";
    for (int i = 0; i < BookmarkList.count(); ++i)
    {
        QStringList bm_parts = BookmarkList.at(i).split("|");
        QString url = bm_parts.at(1);
        // Tags <<.>> conflict with the Qt > 4.4.x HTML parser, even being them inside a
        // literal, quoted string. The substitution below works as a workaround to prevent
        // cb2Bib tags from clashing with the parser.
        // See c2bNetworkQueryInfo::encodeLink. A similar issue was resolved in
        // fromQtXmlString/toQtXmlString.
        url.replace("<<", "[[[");
        url.replace(">>", "]]]");
        url = QString("<a href=\"%1\">%2</a>").arg(url).arg(bm_parts.at(0));
        html += QString("%1 <br />").arg(url);
    }
    html += "</em></p><br />";
    return html;
}
