/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "c2bPostprocess.h"
#include "c2b.h"
#include "c2bSettings.h"

#include <QDir>
#include <QMessageBox>
#include <QProcess>
#include <QPushButton>


c2bPostprocess::c2bPostprocess(const QString& fn, QWidget* parent) : QDialog(parent)
{
    ui.setupUi(this);
    setWindowTitle(tr("BibTeX Postprocessing Log - cb2Bib"));
    setAttribute(Qt::WA_DeleteOnClose);
    buttonOpenFile = new QPushButton(tr("&Open File"));
    ui.buttonBox->addButton(buttonOpenFile, QDialogButtonBox::ActionRole);
    buttonOpenFile->setAutoDefault(true);
    buttonOpenFile->setDefault(true);
    buttonOpenFile->setEnabled(false);
    ui.buttonBox->button(QDialogButtonBox::Close)->setEnabled(false);
    connect(ui.buttonBox, SIGNAL(helpRequested()), this, SLOT(help()));
    connect(buttonOpenFile, SIGNAL(clicked()), this, SLOT(openFile()));
    c2bSettings* settings = c2bSettingsP;
    resize(settings->value("c2bLogWidget/size", size()).toSize());
    PPBIBTEX_BIN = QDir::toNativeSeparators(settings->fileName("cb2Bib/PostprocessBibTeXBin"));
    PPBIBTEX_ARG = settings->value("cb2Bib/PostprocessBibTeXArg").toString();
    PPBIBTEX_EXT = settings->value("cb2Bib/PostprocessBibTeXExt").toString();

    QFile f(fn);
    if (!f.open(QIODevice::ReadOnly | QIODevice::Text))
    {
        QMessageBox::warning(parent, tr("Warning - cb2Bib"),
                             tr("Unable to open the file %1 for reading.\nError: '%2'.").
                             arg(fn).arg(f.errorString()), QMessageBox::Ok);
        ui.buttonBox->button(QDialogButtonBox::Close)->setEnabled(true);
        ui.buttonBox->button(QDialogButtonBox::Close)->setFocus();
        ui.Log->append(tr("BibTeX Post Processing: Erroneous termination.\n"));
    }
    else
    {
        f.close();
        bib = fn;
        QFileInfo fi(fn);
        out = QDir::toNativeSeparators(fi.absolutePath() + '/' + fi.baseName() + '.' + PPBIBTEX_EXT);
        QString cargs = PPBIBTEX_ARG;
        cargs.replace("%finput", bib);
        cargs.replace("%foutput", out);
        ui.Log->append(tr("BibTeX Post Processing Command:\n%1 %2\n").arg(PPBIBTEX_BIN).arg(cargs));
        PProcessFile();
    }
}

c2bPostprocess::~c2bPostprocess()
{
    c2bSettingsP->setValue("c2bLogWidget/size", size());
}


void c2bPostprocess::PProcessFile()
{
    // Postprocessing BibTeX
    QStringList arglist = PPBIBTEX_ARG.split(" ", QString::SkipEmptyParts);
    int n = arglist.count();
    QStringList args;
    for (int i = 0; i < n; i++)
    {
        if (arglist.at(i) == "%finput")
            args.append(bib);
        else if (arglist.at(i) == "%foutput")
            args.append(out);
        else
            args.append(arglist.at(i));
    }
    ppBibTeX = new QProcess(this);
    connect(ppBibTeX, SIGNAL(finished(int, QProcess::ExitStatus)), this, SLOT(conversionEnded()));
    ppBibTeX->start(PPBIBTEX_BIN, args);
    if (!ppBibTeX->waitForStarted())
    {
        ui.Log->append(tr("Failed to call %1.").arg(PPBIBTEX_BIN));
        ui.buttonBox->button(QDialogButtonBox::Close)->setEnabled(true);
        ui.buttonBox->button(QDialogButtonBox::Close)->setFocus();
        delete ppBibTeX;
    }
}

void c2bPostprocess::conversionEnded()
{
    addToLog();
    buttonOpenFile->setEnabled(true);
    ui.buttonBox->button(QDialogButtonBox::Close)->setEnabled(true);
    buttonOpenFile->setFocus();
    ui.Log->append(tr("\n\nBibTeX Post Processing Command Ended."));
    delete ppBibTeX;
}

void c2bPostprocess::addToLog()
{
    QString log = QString::fromUtf8(ppBibTeX->readAllStandardOutput().trimmed());
    if (!log.isEmpty())
        ui.Log->append(log);
    log = QString::fromUtf8(ppBibTeX->readAllStandardError().trimmed());
    if (!log.isEmpty())
        ui.Log->append(log);
}

void c2bPostprocess::openFile()
{
    c2b::openFile(out);
    close();
}

void c2bPostprocess::help()
{
    c2b::displayHelp("http://www.molspaces.com/d_cb2bib-overview.php#keyboard");
}
