/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 *
 *   The LyX pipe procedure in citeEntry has been adapted from Tellico
 *   (Tellico (C) 2003-2005 by Robby Stephenson)
 ***************************************************************************/
#include "c2bReferenceList.h"

#include "c2b.h"
#include "c2bApproximatePattern.h"
#include "c2bBibParser.h"
#include "c2bBibSearcher.h"
#include "c2bFileDialog.h"
#include "c2bRLWebSearchSettings.h"
#include "c2bSettings.h"
#include "c2bTextBrowser.h"
#include "c2bTextEdit.h"
#include "c2bUtils.h"

#include <QByteArray>
#include <QClipboard>
#include <QDir>
#include <QHeaderView>
#include <QMenu>
#include <QMessageBox>
#include <QScrollBar>
#include <QStandardItemModel>
#include <QTextCursor>
#include <QTimer>
#include <QUrl>

// fifo to lyx
#include <fcntl.h>
#include <unistd.h>


c2bReferenceList::c2bReferenceList(c2bTextEdit* ed, c2bTextBrowser* br, QWidget* parent) : QWidget(parent)
{
    ui.setupUi(this);
    editorPtr = ed;
    browserPtr = br;
    settings = c2bSettingsP;
    settings->setDefaultValue("c2bReferenceList/SearchEngine", "http://www.google.com/search?ie=UTF-8&q=");
    settings->setDefaultValue("c2bReferenceList/SearchEngineAddQuotes", false);
    setupFilterBar();

    m_key_index = 0;
    m_id_index = 1;
    m_year_index = 2;
    m_journal_index = 3;
    m_author_index = 4;
    m_title_index = 5;
    QStringList header;
    header.append(tr("Key"));
    header.append(tr("ID"));
    header.append(tr("Year"));
    header.append(tr("Journal"));
    header.append(tr("Authors"));
    header.append(tr("Title"));
    ui.listView->sortItems(m_key_index, Qt::AscendingOrder);
    ui.listView->setHeaderLabels(header);
    ui.listView->hideColumn(m_id_index);
    ui.listView->setContextMenuPolicy(Qt::ActionsContextMenu);

    settings->setDefaultValue("c2bReferenceList/LyXPipe", QDir::cleanPath(QDir::homePath() + QDir::separator() + ".lyx/lyxpipe.in"));
    lyxpipe = settings->value("c2bReferenceList/LyXPipe").toString();
    citeEntryAction = new QAction(this);
    citeEntryAction->setShortcut(tr("Alt+X"));
    citeEntryAction->setText(tr("Cite Selected Entries -- LyX pipe"));
    connect(citeEntryAction, SIGNAL(triggered()), this, SLOT(citeEntry()));
    ui.listView->addAction(citeEntryAction);
    citeEntry2CbAction = new QAction(this);
    citeEntry2CbAction->setShortcut(tr("Alt+C"));
    citeEntry2CbAction->setText(tr("Cite Selected Entries to Clipboard"));
    connect(citeEntry2CbAction, SIGNAL(triggered()), this, SLOT(citeEntry2Cb()));
    ui.listView->addAction(citeEntry2CbAction);

    c2bUtils::addSeparator(ui.listView);
    openFileAction = new QAction(this);
    openFileAction->setText(tr("Open Document File"));
    openFileAction->setStatusTip(tr("Open Document File"));
    connect(openFileAction, SIGNAL(triggered()), this, SLOT(openFile()));
    ui.listView->addAction(openFileAction);

    openUrlAction = new QAction(this);
    openUrlAction->setText(tr("Open Document Url"));
    openUrlAction->setStatusTip(tr("Open Document Url"));
    connect(openUrlAction, SIGNAL(triggered()), this, SLOT(openUrl()));
    ui.listView->addAction(openUrlAction);

    openDOIAction = new QAction(this);
    openDOIAction->setText(tr("Browse Reference by DOI"));
    openDOIAction->setStatusTip(tr("Browse Reference by DOI"));
    connect(openDOIAction, SIGNAL(triggered()), this, SLOT(openDOI()));
    ui.listView->addAction(openDOIAction);

    c2bUtils::addSeparator(ui.listView);
    openAuthorsWebSearchAction = new QAction(this);
    openAuthorsWebSearchAction->setText(tr("Web Search by Authors"));
    openAuthorsWebSearchAction->setStatusTip(tr("Web Search by Authors"));
    connect(openAuthorsWebSearchAction, SIGNAL(triggered()), this, SLOT(openAuthorWebSearch()));
    ui.listView->addAction(openAuthorsWebSearchAction);
    openTitleWebSearchAction = new QAction(this);
    openTitleWebSearchAction->setText(tr("Web Search by Title"));
    openTitleWebSearchAction->setStatusTip(tr("Web Search by Title"));
    connect(openTitleWebSearchAction, SIGNAL(triggered()), this, SLOT(openTitleWebSearch()));
    ui.listView->addAction(openTitleWebSearchAction);

    c2bUtils::addSeparator(ui.listView);
    webSearchSettingsAction = new QAction(this);
    webSearchSettingsAction->setText(tr("Web Search Settings"));
    webSearchSettingsAction->setStatusTip(tr("Web Search Settings"));
    connect(webSearchSettingsAction, SIGNAL(triggered()), this, SLOT(webSearchSettings()));
    ui.listView->addAction(webSearchSettingsAction);

    c2bUtils::addSeparator(ui.listView);
    clearSelectionsAction = new QAction(this);
    clearSelectionsAction->setText(tr("Clear Entry Selection"));
    clearSelectionsAction->setStatusTip(tr("Clear Entry Selection"));
    connect(clearSelectionsAction, SIGNAL(triggered()), this, SLOT(clearSelections()));
    ui.listView->addAction(clearSelectionsAction);

    c2bUtils::addSeparator(ui.listView);
    refreshListAction = new QAction(this);
    refreshListAction->setText(tr("Refresh List and Browser View"));
    refreshListAction->setStatusTip(tr("Refresh List and Browser View"));
    connect(refreshListAction, SIGNAL(triggered()), this, SLOT(refreshList()));
    ui.listView->addAction(refreshListAction);
    connect(ui.listView, SIGNAL(itemActivated(QTreeWidgetItem*, int)), this, SLOT(goToReference(QTreeWidgetItem*)));

    // Actions dependent on selection and availability
    itemSelectionChanged();
    connect(ui.listView, SIGNAL(itemSelectionChanged()), this, SLOT(itemSelectionChanged()));

    // Browser functionality
    m_references_html = c2bUtils::fileToString(":/htm/htm/references.html");
    m_references_html.replace("GET_CB2BIB_VERSION_NUMBER", C2B_VERSION);
    m_bib_item_html = c2bUtils::fileToString(":/htm/htm/bib_item.html");
    QString references_css = c2bUtils::fileToString(settings->fileName("cb2Bib/BrowserCssFile"));
    if (references_css.isEmpty())
        references_css = c2bUtils::fileToString(":/htm/htm/references.css");
    browserPtr->document()->setDefaultStyleSheet(references_css);

    openTextLocalSearchAction = new QAction(this);
    openTextLocalSearchAction->setText(tr("Local Search for selected text"));
    openTextLocalSearchAction->setStatusTip(tr("Local Search for selected text"));
    connect(openTextLocalSearchAction, SIGNAL(triggered()), this, SLOT(openTextLocalSearch()));
    openTextLocalSearchAction->setEnabled(false);
    connect(browserPtr, SIGNAL(copyAvailable(bool)), openTextLocalSearchAction, SLOT(setEnabled(bool)));
    openTextWebSearchAction = new QAction(this);
    openTextWebSearchAction->setText(tr("Web Search for selected text"));
    openTextWebSearchAction->setStatusTip(tr("Web Search for selected text"));
    connect(openTextWebSearchAction, SIGNAL(triggered()), this, SLOT(openTextWebSearch()));
    openTextWebSearchAction->setEnabled(false);
    connect(browserPtr, SIGNAL(copyAvailable(bool)), openTextWebSearchAction, SLOT(setEnabled(bool)));

    QList<QAction*> al = ui.listView->actions();
    for (int i = 0; i < 3; ++i)
        browserPtr->addAction(al.at(i));
    browserPtr->addAction(openTextLocalSearchAction);
    browserPtr->addAction(openTextWebSearchAction);
    c2bUtils::addSeparator(browserPtr);
    for (int i = 7; i < al.count(); ++i)
        browserPtr->addAction(al.at(i));
    browserPtr->setContextMenuPolicy(Qt::ActionsContextMenu);

    connect(browserPtr, SIGNAL(anchorClicked(const QUrl&)), this, SLOT(openLink(const QUrl&)));
    connect(browserPtr, SIGNAL(highlighted(const QString&)), parent->parent(), SLOT(showMessage(const QString&)));
}

c2bReferenceList::~c2bReferenceList()
{}


void c2bReferenceList::loadList(const QString& txt, c2bBibSearcher* bibSearcher)
{
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    c2bBibParser* bp = c2b::bibParser();
    QString references_html = m_references_html;
    QString references;

    QStringList fields;
    fields.append("abstract");
    fields.append("author");
    fields.append("booktitle");
    fields.append("doi");
    fields.append("file");
    fields.append("journal");
    fields.append("pages");
    fields.append("title");
    fields.append("url");
    fields.append("volume");
    fields.append("year");
    bibReference ref;
    bp->initReferenceParsing(editorPtr->editFileName(), fields, &ref);
    int ref_counter = 0;

    while (bp->referencesIn(txt, &ref))
    {
        ref_counter++;
        QString key = ref.keyName;
        QString abstract = ref.value("abstract");
        QString author = ref.value("author");
        QString doi = ref.value("doi");
        QString file = ref.value("file");
        QString journal = ref.value("journal");
        QString pages = ref.value("pages");
        QString url = ref.value("url");
        QString title = ref.value("title");
        QString volume = ref.value("volume");
        QString year = ref.value("year");
        QString author_long;
        QString ID;
        QString link;

        if (!author.isEmpty())
        {
            author_long = author;
            author_long.remove(QRegExp("[^\\w\\s-]"));
            author_long.replace(" and ", ", ");
            author_long = author_long.simplified();
            author.remove(QRegExp("\\b\\w\\b"));
            author.remove(QRegExp("[^\\w\\s]"));
            author.replace(" and ", ", ");
            author = author.simplified();
        }
        if (title.isEmpty())
            title = ref.value("booktitle");
        c2bUtils::cleanTitle(title);

        if (!doi.isEmpty())
            if (!doi.startsWith("http://"))
                doi = "http://dx.doi.org/" + QUrl::toPercentEncoding(doi);
        if (!file.isEmpty())
            link = file;
        else if (!url.isEmpty())
            link = url;
        else if (!doi.isEmpty())
            link = doi;
        ID.setNum(ref_counter);

        doiList.append(doi);
        fileList.append(file);
        urlList.append(url);
        positionList.append(ref.positionValue);
        QStringList dataList;
        dataList.append(key);
        dataList.append(ID);
        dataList.append(year);
        dataList.append(journal);
        dataList.append(author);
        dataList.append(title);
        QTreeWidgetItem* qi = new QTreeWidgetItem(dataList);
        qi->setStatusTip(m_journal_index, bp->fullJournal(journal));
        qi->setStatusTip(m_author_index, author_long);
        qi->setStatusTip(m_title_index, title);
        ui.listView->insertTopLevelItem(0, qi);

        if (!author_long.isEmpty())
            author_long += ".";
        title.remove(QRegExp("\\.$"));
        if (!pages.isEmpty())
        {
            if (!volume.isEmpty())
                pages = ", " + pages;
            else
                pages = " " + pages;
        }
        if (!year.isEmpty())
            year = "(" + year + ").";
        QString bib_item_html = m_bib_item_html;
        bib_item_html.replace("GET_REFERENCE_ANCHOR_ID", ID);
        bib_item_html.replace("GET_REFERENCE_ANCHOR_NAME", ID);
        bib_item_html.replace("GET_REFERENCE_AUTHOR", author_long);
        bib_item_html.replace("GET_REFERENCE_LINK", link);
        bib_item_html.replace("GET_REFERENCE_PAGES", pages);
        bib_item_html.replace("GET_REFERENCE_SOURCE", journal);
        bib_item_html.replace("GET_REFERENCE_TITLE", c2bUtils::toHtmlString(title));
        bib_item_html.replace("GET_REFERENCE_VOLUME", volume);
        bib_item_html.replace("GET_REFERENCE_YEAR", year);
        if (bibSearcher)
            bib_item_html.replace("GET_REFERENCE_ABSTRACT", bibSearcher->highlight(abstract) +
                                  bibSearcher->hitMap.value(ref.rawReference));
        else
            bib_item_html.replace("GET_REFERENCE_ABSTRACT", c2bUtils::toHtmlString(abstract));
        references += bib_item_html;
    }

    ui.listView->sortItems(ui.listView->sortColumn(), ui.listView->header()->sortIndicatorOrder());
    ui.listView->resizeColumnToContents(m_year_index);
    references_html.replace("GET_REFERENCES", references);
    browserPtr->updateHtml(references_html);
    QApplication::restoreOverrideCursor();
    c2b::showMessage(tr("Parsed %1 references.").arg(ref_counter));
}

void c2bReferenceList::citeEntry()
{
    // This procedure was adapted from Tellico
    // Tellico (C) 2003-2005 by Robby Stephenson
    QByteArray pipe = QFile::encodeName(lyxpipe);
    QString errorStr = tr("Unable to write to the server pipe at '%1'.").arg(QString(pipe));

    if (!QFile::exists(pipe))
    {
        QMessageBox::warning(this, tr("Warning - cb2Bib"), errorStr, QMessageBox::Ok);
        return;
    }

    int pipeFd = ::open(pipe, O_WRONLY);
    QFile file(QString::fromUtf8(pipe));
    if (file.open(pipeFd, QIODevice::WriteOnly))
    {
        QString refs;
        QTreeWidgetItemIterator it(ui.listView, QTreeWidgetItemIterator::Selected | QTreeWidgetItemIterator::NotHidden);
        while (*it)
        {
            if (!(*it)->text(m_key_index).isEmpty())
                refs += (*it)->text(m_key_index) + " ";
            ++it;
        }
        refs = refs.trimmed();
        // pybliographer uses comma-space, and pyblink expects the space there
        refs.replace(" ", ", ");
        QTextStream st(&file);
        st << QString::fromLatin1("LYXCMD:cb2bib:citation-insert:%1\n").arg(refs).toLatin1();
        file.flush();
        file.close();
    }
    else
        QMessageBox::warning(this, tr("Warning - cb2Bib"), errorStr, QMessageBox::Ok);
    ::close(pipeFd);
}

void c2bReferenceList::citeEntry2Cb()
{
    QString refs;
    QTreeWidgetItemIterator it(ui.listView, QTreeWidgetItemIterator::Selected | QTreeWidgetItemIterator::NotHidden);
    while (*it)
    {
        if (!(*it)->text(m_key_index).isEmpty())
            refs += (*it)->text(m_key_index) + " ";
        ++it;
    }
    refs = refs.trimmed();
    refs.replace(" ", ", ");
    refs = "\\cite{" + refs + "}";
    QClipboard *cb = QApplication::clipboard();
    cb->setText(refs, QClipboard::Clipboard);
}

void c2bReferenceList::openDOI()
{
    bool zdoi = false;
    QTreeWidgetItemIterator it(ui.listView, QTreeWidgetItemIterator::Selected | QTreeWidgetItemIterator::NotHidden);
    while (*it)
    {
        QString doi = doiList.at((*it)->text(m_id_index).toInt() - 1);
        if (!doi.isEmpty())
        {
            zdoi = true;
            c2b::openFile(doi);
        }
        ++it;
    }
    if (!zdoi)
        c2b::showMessage(tr("No DOI entry among selected references."));
}

void c2bReferenceList::openFile()
{
    bool zfile = false;
    QTreeWidgetItemIterator it(ui.listView, QTreeWidgetItemIterator::Selected | QTreeWidgetItemIterator::NotHidden);
    while (*it)
    {
        QString file = fileList.at((*it)->text(m_id_index).toInt() - 1);
        if (!file.isEmpty())
        {
            zfile = true;
            c2b::openFile(file);
        }
        ++it;
    }
    if (!zfile)
        c2b::showMessage(tr("No file entry among selected references."));
}

void c2bReferenceList::openUrl()
{
    bool zurl = false;
    QTreeWidgetItemIterator it(ui.listView, QTreeWidgetItemIterator::Selected | QTreeWidgetItemIterator::NotHidden);
    while (*it)
    {
        QString url = urlList.at((*it)->text(m_id_index).toInt() - 1);
        if (!url.isEmpty())
        {
            zurl = true;
            c2b::openFile(url);
        }
        ++it;
    }
    if (!zurl)
        c2b::showMessage(tr("No Url entry among selected references."));
}

void c2bReferenceList::openLink(const QUrl& ln)
{
    QString ln_str = ln.toString();
    if (ln_str.isEmpty())
        c2b::showMessage(tr("No available link for this reference."));
    else if (ln_str.startsWith("Select reference "))
        scrollToReference(ln_str.remove(QRegExp("\\D")));
    else if (ln_str.startsWith("Edit reference"))
        emit openFile(ln_str.replace(QRegExp("^.+\\[(.+)\\]$"), "\\1"));
    else
        c2b::openFile(ln_str);
}

void c2bReferenceList::openAuthorWebSearch()
{
    QString searchEngine = settings->value("c2bReferenceList/SearchEngine").toString();
    bool zwebs = false;
    QTreeWidgetItemIterator it(ui.listView, QTreeWidgetItemIterator::Selected | QTreeWidgetItemIterator::NotHidden);
    while (*it)
    {
        if (!(*it)->text(m_author_index).isEmpty())
        {
            zwebs = true;
            QString webs = (*it)->text(m_author_index);
            webs.remove(",");
            webs = webs.simplified();
            webs = searchEngine + QUrl::toPercentEncoding(webs);
            c2b::openFile(webs);
        }
        ++it;
    }
    if (!zwebs)
        c2b::showMessage(tr("No author entry among selected references."));
}

void c2bReferenceList::openTitleWebSearch()
{
    QString searchEngine = settings->value("c2bReferenceList/SearchEngine").toString();
    bool addQuotes = settings->value("c2bReferenceList/SearchEngineAddQuotes").toBool();
    bool zwebs = false;
    QTreeWidgetItemIterator it(ui.listView, QTreeWidgetItemIterator::Selected | QTreeWidgetItemIterator::NotHidden);
    while (*it)
    {
        if (!(*it)->text(m_title_index).isEmpty())
        {
            zwebs = true;
            QString webs = (*it)->text(m_title_index);
            if (addQuotes)
                webs = "\"" + webs + "\"";
            webs = searchEngine + QUrl::toPercentEncoding(webs);
            c2b::openFile(webs);
        }
        ++it;
    }
    if (!zwebs)
        c2b::showMessage(tr("No title entry among selected references."));
}

void c2bReferenceList::openTextWebSearch()
{
    QString searchEngine = settings->value("c2bReferenceList/SearchEngine").toString();
    QString webs = browserPtr->textCursor().selectedText();
    if (settings->value("c2bReferenceList/SearchEngineAddQuotes").toBool())
        webs = "\"" + webs + "\"";
    webs = searchEngine + QUrl::toPercentEncoding(webs);
    c2b::openFile(webs);
}

void c2bReferenceList::webSearchSettings()
{
    c2bRLWebSearchSettings searchSettings(this);
    searchSettings.exec();
}

void c2bReferenceList::openTextLocalSearch()
{
    QString text = browserPtr->textCursor().selectedText();
    QString fn = editorPtr->editFileName();
    c2b::doSearchInFiles(text, fn);
}

void c2bReferenceList::refreshList()
{
    doiList.clear();
    fileList.clear();
    urlList.clear();
    positionList.clear();
    ui.listView->clear();
    ui.comboFilter->clearEditText();
    loadList(editorPtr->toPlainText());
}

void c2bReferenceList::goToReference(const QString& id)
{
    QTextCursor tc = editorPtr->textCursor();
    tc.setPosition(positionList.at(id.toInt() - 1));
    editorPtr->setTextCursor(tc);
    editorPtr->verticalScrollBar()->setValue(editorPtr->verticalScrollBar()->maximum());
    editorPtr->ensureCursorVisible();

    browserPtr->scrollToAnchor(id);
    int scrollPosition = browserPtr->verticalScrollBar()->value();
    browserPtr->verticalScrollBar()->setValue(scrollPosition - 10);
}

void c2bReferenceList::goToReference(const int position)
{
    // Maps position to internal reference id. It is needed by the search browser, when a
    // citation is edited in the original file. If this file is changed multiple times,
    // positions in the search browser get outdated, so the mapping position-id is inexact.
    // Thus, search for the closet reference.
    int ip = 0;
    int pdiff_min = 1000000;
    for (int i = 0; i < positionList.count(); ++i)
    {
        int p = positionList.at(i);
        if (p == position)
        {
            ip = i;
            break;
        }
        int pdiff = abs(p - position);
        if (pdiff < pdiff_min)
        {
            pdiff_min = pdiff;
            ip = i;
        }
    }
    QString id;
    goToReference(id.setNum(ip + 1));
}

void c2bReferenceList::goToReference(QTreeWidgetItem* ref)
{
    if (ref == 0)
        return;
    QString id = ref->text(m_id_index);
    goToReference(id);
}

void c2bReferenceList::scrollToReference(const QString& id)
{
    if (id.isEmpty())
        return;
    QList<QTreeWidgetItem*> items = ui.listView->findItems(id, Qt::MatchExactly, 1);
    QTreeWidgetItem* item = items.at(0);
    ui.listView->setFocus();
    item->setSelected(!item->isSelected());
    ui.listView->scrollToItem(item);

    QTextCursor tc = editorPtr->textCursor();
    tc.setPosition(positionList.at(id.toInt() - 1));
    editorPtr->setTextCursor(tc);
    editorPtr->verticalScrollBar()->setValue(editorPtr->verticalScrollBar()->maximum());
    editorPtr->ensureCursorVisible();
}

void c2bReferenceList::clearSelections()
{
    disconnect(ui.listView, SIGNAL(itemSelectionChanged()), this, SLOT(itemSelectionChanged()));
    QTreeWidgetItemIterator it(ui.listView, QTreeWidgetItemIterator::Selected);
    while (*it)
    {
        (*it)->setSelected(false);
        ++it;
    }
    itemSelectionChanged();
    connect(ui.listView, SIGNAL(itemSelectionChanged()), this, SLOT(itemSelectionChanged()));
}

void c2bReferenceList::itemSelectionChanged()
{
    int entries = 0;
    int doi = 0;
    int file = 0;
    int url = 0;
    int author = 0;
    int title = 0;
    QTreeWidgetItemIterator it(ui.listView, QTreeWidgetItemIterator::Selected | QTreeWidgetItemIterator::NotHidden);
    while (*it)
    {
        if (!doiList.at((*it)->text(m_id_index).toInt() - 1).isEmpty())
            ++doi;
        if (!fileList.at((*it)->text(m_id_index).toInt() - 1).isEmpty())
            ++file;
        if (!urlList.at((*it)->text(m_id_index).toInt() - 1).isEmpty())
            ++url;
        if (!(*it)->text(m_author_index).isEmpty())
            ++author;
        if (!(*it)->text(m_title_index).isEmpty())
            ++title;
        ++entries;
        ++it;
    }
    citeEntryAction->setEnabled(entries > 0);
    citeEntryAction->setStatusTip(tr("Cite Selected Entries -- LyX pipe. %1 Entries selected.").arg(entries));
    citeEntry2CbAction->setEnabled(entries > 0);
    citeEntry2CbAction->setStatusTip(tr("Cite Selected Entries to Clipboard. %1 Entries selected.").arg(entries));
    openDOIAction->setEnabled(doi > 0);
    openDOIAction->setStatusTip(tr("Browse Reference by DOI. %1 DOIs in selected entries.").arg(doi));
    openFileAction->setEnabled(file > 0);
    openFileAction->setStatusTip(tr("Open Document File. %1 Files in selected entries.").arg(file));
    openUrlAction->setEnabled(url > 0);
    openUrlAction->setStatusTip(tr("Open Document Url. %1 Urls in selected entries.").arg(url));
    openAuthorsWebSearchAction->setEnabled(author > 0);
    openAuthorsWebSearchAction->setStatusTip(tr("Web Search by Authors. %1 Entries selected.").arg(author));
    openTitleWebSearchAction->setEnabled(title > 0);
    openTitleWebSearchAction->setStatusTip(tr("Web Search by Title. %1 Entries selected.").arg(title));
    emit hasSelectedItems(entries > 0);
}

QStringList c2bReferenceList::selectedKeys() const
{
    QStringList keys;
    QTreeWidgetItemIterator it(ui.listView, QTreeWidgetItemIterator::Selected | QTreeWidgetItemIterator::NotHidden);
    while (*it)
    {
        if (!(*it)->text(m_key_index).isEmpty())
            keys.append((*it)->text(m_key_index));
        ++it;
    }
    return keys;
}

void c2bReferenceList::setLyxPipe()
{
    QString new_pn = c2bFileDialog::getSystemFileName(this, "Select LyX pipe", lyxpipe, "All (*)");
    if (!new_pn.isEmpty())
    {
        lyxpipe = new_pn;
        settings->setValue("c2bReferenceList/LyXPipe", lyxpipe);
    }
}


/***************************************************************************
    Filter Bar
 ***************************************************************************/

void c2bReferenceList::setupFilterBar()
{
    m_filter_mode = settings->value("c2bReferenceList/FilterMode", "Approximate string").toString();
    QActionGroup* filterMode = new QActionGroup(this);
    filterMode->setExclusive(true);
    QAction* act = filterMode->addAction("Approximate string");
    act->setCheckable(true);
    if (m_filter_mode == "Approximate string") act->setChecked(true);
    act = filterMode->addAction("Fixed string");
    act->setCheckable(true);
    if (m_filter_mode == "Fixed string") act->setChecked(true);
    act = filterMode->addAction("Fixed string: All Words");
    act->setCheckable(true);
    if (m_filter_mode == "Fixed string: All Words") act->setChecked(true);
    act = filterMode->addAction("Fixed string: Any Word");
    act->setCheckable(true);
    if (m_filter_mode == "Fixed string: Any Word") act->setChecked(true);
    act = filterMode->addAction("Regular expression");
    act->setCheckable(true);
    if (m_filter_mode == "Regular expression") act->setChecked(true);
    act = filterMode->addAction("Wildcard");
    act->setCheckable(true);
    if (m_filter_mode == "Wildcard") act->setChecked(true);
    QMenu* filterModeM = new QMenu(this);
    filterModeM->addActions(filterMode->actions());
    ui.filterB->setMenu(filterModeM);
    connect(filterMode, SIGNAL(triggered(QAction*)), this, SLOT(setFilterMode(QAction*)));
    connect(ui.comboFilter, SIGNAL(editTextChanged(const QString&)), this, SLOT(blockFilterSignals()));
}

void c2bReferenceList::setFilterMode(QAction* mode)
{
    m_filter_mode = mode->text();
    settings->setValue("c2bReferenceList/FilterMode", m_filter_mode);
    filterList();
}

void c2bReferenceList::blockFilterSignals()
{
    ui.comboFilter->blockSignals(true);
    QTimer::singleShot(500, this, SLOT(filterList()));
}

void c2bReferenceList::filterList()
{
    QString text = ui.comboFilter->currentText();
    if (text.isEmpty())
    {
        QTreeWidgetItemIterator it(ui.listView);
        while (*it)
        {
            (*it)->setHidden(false);
            ++it;
        }
    }
    else
    {
        c2bSearchPattern pattern(text, m_filter_mode);
        QTreeWidgetItemIterator it(ui.listView);
        while (*it)
        {
            QTreeWidgetItem *item = *it;
            if (pattern.matches(item->text(m_author_index)) || pattern.matches(item->text(m_title_index)))
                item->setHidden(false);
            else
                item->setHidden(true);
            ++it;
        }
        QTimer::singleShot(5000, this, SLOT(keepFilterPattern()));
    }
    itemSelectionChanged();
    ui.comboFilter->blockSignals(false);
}

void c2bReferenceList::keepFilterPattern()
{
    QString txt = ui.comboFilter->currentText();
    if (!txt.isEmpty())
        if (ui.comboFilter->findText(txt) < 0)
            ui.comboFilter->addItem(txt);
}
