/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "c2bSearchInFiles.h"
#include "c2b.h"
#include "c2bBibSearcher.h"

#include <QFileInfo>
#include <QTime>


c2bSearchInFiles::c2bSearchInFiles(const QString& file, QWidget* parent) : QDialog(parent)
{
    ui.setupUi(this);
    // Set blank title to columns 0 and 4. Qt 4.4.0 needs it, otherwise it takes column number as default title.
    // Recheck whenever ui files are generated for 4.4 instead of current 4.2.
    ui.patternList->headerItem()->setText(0, QApplication::translate("c2bSearchInFiles", "", 0, QApplication::UnicodeUTF8));
    ui.patternList->headerItem()->setText(4, QApplication::translate("c2bSearchInFiles", "", 0, QApplication::UnicodeUTF8));

    for (int i = 0; i < ITEMS_IN_SEARCH_PATTERN; ++i)
        ui.patternList->resizeColumnToContents(i);
    ui.buttonBox->button(QDialogButtonBox::Abort)->setEnabled(false);
    searchB = new QPushButton(tr("Search"));
    ui.buttonBox->addButton(searchB, QDialogButtonBox::ActionRole);
    searchB->setAutoDefault(true);
    searchB->setDefault(true);
    searchB->setEnabled(false);

    m_bibtex_file.clear();
    m_bibtex_dir.clear();
    setBibTeXFile(file);

    connect(searchB, SIGNAL(clicked()), this, SLOT(searchB_clicked()));
    connect(ui.allBibFiles, SIGNAL(clicked()), this, SLOT(setBibTeXFile()));
    connect(ui.buttonBox, SIGNAL(helpRequested()), this, SLOT(help()));
    connect(ui.pattern, SIGNAL(patternAvailable(bool)), this, SLOT(updateForm()));
    connect(ui.patternList, SIGNAL(itemActivated(QTreeWidgetItem*, int)), this, SLOT(on_editPatternB_clicked()));
    connect(ui.patternList, SIGNAL(itemClicked(QTreeWidgetItem*, int)), this, SLOT(updateForm()));
}

c2bSearchInFiles::~c2bSearchInFiles()
{}


void c2bSearchInFiles::updateForm()
{
    ui.addPatternB->setEnabled(ui.pattern->isAvailable());
    ui.Boolean->setEnabled((ui.patternList->topLevelItemCount() > 1) ||
                           (ui.pattern->isAvailable() && ui.patternList->topLevelItemCount() > 0));
    searchB->setEnabled(ui.pattern->isAvailable() || ui.patternList->topLevelItemCount() > 0);
    if (ui.patternList->currentItem())
    {
        ui.deletePatternB->setEnabled(true);
        ui.editPatternB->setEnabled(true);
    }
    else
    {
        ui.deletePatternB->setEnabled(false);
        ui.editPatternB->setEnabled(false);
    }
    ui.searchLog->setText(searchLogText);
}

void c2bSearchInFiles::on_addPatternB_clicked()
{
    new QTreeWidgetItem(ui.patternList, ui.pattern->pattern());
    for (int i = 0; i < ITEMS_IN_SEARCH_PATTERN; ++i)
        ui.patternList->resizeColumnToContents(i);
    ui.pattern->clear();
    updateForm();
}

void c2bSearchInFiles::on_deletePatternB_clicked()
{
    QTreeWidgetItem* item = ui.patternList->currentItem();
    if (item)
        delete item;
    item = ui.patternList->currentItem();
    if (item)
        ui.patternList->currentItem()->setSelected(true);
    updateForm();
}

void c2bSearchInFiles::on_editPatternB_clicked()
{
    QDialog editor(this);
    ui_editor.setupUi(&editor);
    QTreeWidgetItem* item = ui.patternList->currentItem();
    ui_editor.pattern->setPattern(item);
    connect(ui_editor.pattern, SIGNAL(patternAvailable(bool)),
            ui_editor.buttonBox->button(QDialogButtonBox::Ok), SLOT(setEnabled(bool)));
    if (editor.exec() == QDialog::Accepted)
    {
        QStringList pattern = ui_editor.pattern->pattern();
        for (int i = 0; i < ITEMS_IN_SEARCH_PATTERN; ++i)
            item->setText(i, pattern.at(i));
        for (int i = 0; i < ITEMS_IN_SEARCH_PATTERN; ++i)
            ui.patternList->resizeColumnToContents(i);
    }
}

void c2bSearchInFiles::searchB_clicked()
{
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    searchB->setEnabled(false);
    ui.buttonBox->button(QDialogButtonBox::Abort)->setEnabled(true);

    ui.searchLog->setText(searchLogText);
    c2bBibSearcher searcher;
    connect(ui.buttonBox->button(QDialogButtonBox::Abort), SIGNAL(clicked()), &searcher, SLOT(abort()));
    searcher.setScope(m_bibtex_file, m_bibtex_dir, ui.allBibFiles->isChecked(), ui.documentFiles->isChecked());
    searcher.setSimplifySource(ui.simplifySource->isChecked());
    searcher.setBoolean(ui.Boolean->currentText() == "All");
    if (ui.addPatternB->isEnabled())
    {
        QStringList p = ui.pattern->pattern();
        searcher.addPattern(p.at(0) == "NOT", p.at(1) == "Sensitive", p.at(2), p.at(3), p.at(4), p.at(5));
    }
    QTreeWidgetItemIterator it(ui.patternList);
    while (*it)
    {
        QStringList p;
        for (int i = 0; i < ITEMS_IN_SEARCH_PATTERN; ++i)
            p.append((*it)->text(i));
        searcher.addPattern(p.at(0) == "NOT", p.at(1) == "Sensitive", p.at(2), p.at(3), p.at(4), p.at(5));
        ++it;
    }
    if (searcher.patternCount() == 0)
        return;

    QTime t;
    t.start();
    searcher.exec();

    if (searcher.hitCount() > 0 || searcher.errorCount() > 0)
    {
        if (searcher.hitCount() > 0)
            ui.searchLog->setText(tr("Found %1 hits from %2 references (%3s)")
                                  .arg(searcher.hitCount()).arg(searcher.referenceCount()).arg(t.elapsed() / 1000.0));
        else
            ui.searchLog->setText(searchLogText);
        QApplication::processEvents();
        emit searchDone(m_bibtex_dir, &searcher);
    }
    else if (searcher.errorCount() < 0)
        ui.searchLog->setText(tr("Search aborted"));
    else
        ui.searchLog->setText(tr("No reference found (%1s)").arg(t.elapsed() / 1000.0));

    ui.buttonBox->button(QDialogButtonBox::Abort)->setEnabled(false);
    searchB->setEnabled(true);
    QApplication::restoreOverrideCursor();
}

void c2bSearchInFiles::setBibTeXFile(const QString& file)
{
    if (!file.isEmpty())
    {
        QFileInfo fi(file);
        m_bibtex_file = fi.absoluteFilePath();
        m_bibtex_dir = fi.absolutePath();
    }
    if (ui.allBibFiles->isChecked())
        searchLogText = tr("Search in files at %1").arg(m_bibtex_dir);
    else
        searchLogText = tr("Search in file %1").arg(m_bibtex_file);
    ui.searchLog->setText(searchLogText);
}

void c2bSearchInFiles::setCurrentText(const QString& text)
{
    if (text.isEmpty())
        return;
    ui.pattern->setText(text);
    updateForm();
}

void c2bSearchInFiles::show()
{
    if (isMinimized())
        showNormal();
    raise();
    activateWindow();
    ui.pattern->setFocus();
    QDialog::show();
}

void c2bSearchInFiles::help()
{
    c2b::displayHelp("http://www.molspaces.com/d_cb2bib-c2bsearch.php");
}
