/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "c2bSettings.h"
#include "c2bUtils.h"

#include <QApplication>
#include <QClipboard>
#include <QDir>
#include <QFSFileEngine>
#include <QIcon>
#include <QMessageBox>
#include <QTimer>


/**
    Global settings for the cb2Bib
*/
c2bSettings::c2bSettings(const QString& organization, const QString& application, QObject* parent)
        : QSettings(organization, application, parent)
{
    is_usbConfMode = false;
    init();
}

c2bSettings::c2bSettings(const QString& fn, QObject* parent) : QSettings(fn, QSettings::IniFormat, parent)
{
    is_usbConfMode = fn == (qApp->applicationDirPath() + QDir::separator() + "cb2bib.conf");
    init();
}

c2bSettings::~c2bSettings()
{}


void c2bSettings::init()
{
    Q_ASSERT_X(!c2b_global_settings, "c2bSettings", "c2bSettings already instantiated");
    c2bSettings::c2b_global_settings = this;
    if (is_usbConfMode)
    {
        qAppDir = QDir(qApp->applicationDirPath());
        QFSFileEngine::setCurrentPath(qApp->applicationDirPath());
        tmp_dir_path = qApp->applicationDirPath() + QDir::separator();
    }
    else
        tmp_dir_path = QDir::homePath() + QDir::separator();
    setDefaults();
    // Set relative to BibTeX file in USB mode, overriding defaults.
    if (is_usbConfMode)
        setValue("cb2Bib/RelativePdfDirectory", true);
    readCommandLine();
    connect(qApp, SIGNAL(lastWindowClosed()), this, SLOT(guiExits()));
}

void c2bSettings::setDefaultValue(const QString &key, const QVariant &defaulValue)
{
    // Makes sure a key is initialized
    setValue(key, value(key, defaulValue));
}

void c2bSettings::notify()
{
    update();
    emit newSettings();
}

void c2bSettings::update()
{
    // Values for very general GUI settings
    c2bMonoFont = value("cb2Bib/MonoFont").value<QFont>();
    colors.loadColors(this);
}

void c2bSettings::setDefaults()
{
    QHash<QString, QVariant> defaults;
    QString dsep = QDir::separator();

    // Files
    defaults["cb2Bib/BibTeXFile"] = QFileInfo("references.bib").absoluteFilePath();
    defaults["cb2Bib/CacheDirectory"] = QDir::cleanPath(QDir::homePath() + dsep + "PdfDocuments" + dsep + "c2b.tmp");
    defaults["cb2Bib/PdfDirectory"] = QDir::cleanPath(QDir::homePath() + dsep + "PdfDocuments");

#ifdef Q_WS_X11
    defaults["c2bBibSearcher/Pdf2TextBin"] = QDir::cleanPath(C2B_BIBSEARCHER_PDF2TXT_BIN);
    defaults["c2bMetaPreparser/ExifToolBin"] = QDir::cleanPath(C2B_METAPREPARSER_EXIFTOOL_BIN);
    defaults["c2bPdfImport/LastFile"] = QDir::cleanPath(C2B_DATA_DIR + dsep + "testPDFImport/nomeaning1.pdf");
    defaults["c2bPdfImport/Pdf2TextBin"] = QDir::cleanPath(C2B_PDFIMPORT_PDF2TXT_BIN);
    defaults["cb2Bib/BrowserCssFile"] = QDir::cleanPath(C2B_DATA_DIR + dsep + "data/references.css");
    defaults["cb2Bib/JournalFile"] = QDir::cleanPath(C2B_DATA_DIR + dsep + "data/abbreviations.txt");
    defaults["cb2Bib/NetworkFile"] = QDir::cleanPath(C2B_DATA_DIR + dsep + "data/netqinf.txt");
    defaults["cb2Bib/PostprocessBibTeXBin"] = QDir::cleanPath(C2B_DATA_DIR + dsep + C2B_POSTPROCESS_BIBTEX_BIN);
    defaults["cb2Bib/PreparserBin"] = QDir::cleanPath(C2B_DATA_DIR + dsep + "c2btools/ris2bib");
    defaults["cb2Bib/RegularExpressionFile"] = QDir::cleanPath(C2B_DATA_DIR + dsep + "data/regexps.txt");
#endif
#ifdef Q_WS_MACX
    QString C2B_DATA_DIR_MAC = C2B_DATA_DIR;
    if (QDir(C2B_DATA_DIR + dsep + "data" + dsep).exists())
        C2B_DATA_DIR_MAC = C2B_DATA_DIR + dsep;
    else if (QDir(qApp->applicationDirPath() + dsep + "Contents/Resources/data" + dsep).exists())
        C2B_DATA_DIR_MAC = qApp->applicationDirPath() + dsep + "Contents/Resources" + dsep;
    defaults["c2bBibSearcher/Pdf2TextBin"] = QDir::cleanPath(C2B_BIBSEARCHER_PDF2TXT_BIN);
    defaults["c2bMetaPreparser/ExifToolBin"] = QDir::cleanPath(C2B_METAPREPARSER_EXIFTOOL_BIN);
    defaults["c2bPdfImport/LastFile"] = QDir::cleanPath(C2B_DATA_DIR_MAC + dsep + "testPDFImport/nomeaning1.pdf");
    defaults["c2bPdfImport/Pdf2TextBin"] = QDir::cleanPath(C2B_PDFIMPORT_PDF2TXT_BIN);
    defaults["cb2Bib/BrowserCssFile"] = QDir::cleanPath(C2B_DATA_DIR_MAC + dsep + "data/references.css");
    defaults["cb2Bib/JournalFile"] = QDir::cleanPath(C2B_DATA_DIR_MAC + dsep + "data/abbreviations.txt");
    defaults["cb2Bib/NetworkFile"] = QDir::cleanPath(C2B_DATA_DIR_MAC + dsep + "data/netqinf.txt");
    defaults["cb2Bib/PostprocessBibTeXBin"] = QDir::cleanPath(C2B_DATA_DIR_MAC + dsep + C2B_POSTPROCESS_BIBTEX_BIN);
    defaults["cb2Bib/PreparserBin"] = QDir::cleanPath(C2B_DATA_DIR_MAC + dsep + "c2btools/ris2bib");
    defaults["cb2Bib/RegularExpressionFile"] = QDir::cleanPath(C2B_DATA_DIR_MAC + dsep + "data/regexps.txt");
#endif
#ifdef Q_WS_WIN
    QString C2B_APP_DIR = qApp->applicationDirPath() + dsep;
    defaults["c2bBibSearcher/Pdf2TextBin"] = C2B_APP_DIR + "externtools\\xpdf\\pdftotext.exe";
    defaults["c2bPdfImport/LastFile"] = C2B_APP_DIR + "testPDFImport\\nomeaning1.pdf";
    defaults["c2bPdfImport/Pdf2TextBin"] = C2B_APP_DIR + "externtools\\xpdf\\pdftotext.exe";
    defaults["cb2Bib/BrowserCssFile"] = C2B_APP_DIR + "data\\references.css";
    defaults["cb2Bib/JournalFile"] = C2B_APP_DIR + "data\\abbreviations.txt";
    defaults["cb2Bib/NetworkFile"] = C2B_APP_DIR + "data\\netqinf.txt";
    defaults["cb2Bib/PostprocessBibTeXBin"] = C2B_APP_DIR + "c2btools\\bib2end.bat";
    defaults["cb2Bib/PreparserBin"] = C2B_APP_DIR + "c2btools\\ris2bib.bat";
    defaults["cb2Bib/RegularExpressionFile"] = C2B_APP_DIR + "data\\regexps.txt";
#endif

    // External file manager
    defaults["cb2Bib/FmClient"] = false;
    defaults["cb2Bib/FmClientCopyArg"] = C2B_FM_CLIENT_COPY_ARG;
    defaults["cb2Bib/FmClientCopyBin"] = QDir::cleanPath(C2B_FM_CLIENT_COPY_BIN);
    defaults["cb2Bib/FmClientMoveArg"] = C2B_FM_CLIENT_MOVE_ARG;
    defaults["cb2Bib/FmClientMoveBin"] = QDir::cleanPath(C2B_FM_CLIENT_MOVE_BIN);
    defaults["cb2Bib/FmClientOpenArg"] = C2B_FM_CLIENT_OPEN_ARG;
    defaults["cb2Bib/FmClientOpenBin"] = QDir::cleanPath(C2B_FM_CLIENT_OPEN_BIN);

    // Postprocess BibTeX
    defaults["cb2Bib/PostprocessBibTeXArg"] = C2B_POSTPROCESS_BIBTEX_ARG;
    defaults["cb2Bib/PostprocessBibTeXExt"] = C2B_POSTPROCESS_BIBTEX_EXT;

    // Misc
    defaults["cb2Bib/AddMetadata"] = true;
    defaults["cb2Bib/AutomaticPdfDownload"] = false;
    defaults["cb2Bib/AutomaticQuery"] = false;
    defaults["cb2Bib/CiteIdPattern"] = C2B_CITE_ID_PATTERN;
    defaults["cb2Bib/ConvertReferenceToLaTeX"] = true;
    defaults["cb2Bib/CopyMovePdf"] = "copy";
    defaults["cb2Bib/DoHeuristicGuess"] = true;
    defaults["cb2Bib/FindReplaceInput"] = true;
    defaults["cb2Bib/FindReplaceInputList"] = C2B_FIND_REPLACE_LIST.split("^e");
    defaults["cb2Bib/InsertMetadata"] = false;
    defaults["cb2Bib/KeepTmpNQFiles"] = false;
    defaults["cb2Bib/MonoFont"] = QFont("Courier");
    defaults["cb2Bib/MovePdf"] = false;
    defaults["cb2Bib/PageNumberSeparator"] = " - ";
    defaults["cb2Bib/PostprocessMonth"] = false;
    defaults["cb2Bib/PreAppendMetadata"] = "prepend";
    defaults["cb2Bib/PreparserAutomatic"] = false;
    defaults["cb2Bib/PreparserShowLog"] = true;
    defaults["cb2Bib/RelativePdfDirectory"] = false;
    defaults["cb2Bib/SetJournalsToFullname"] = false;
    defaults["cb2Bib/UseDoubleBraces"] = true;
    defaults["cb2Bib/UseFullNames"] = false;

    // Pdf2Text arguments
    defaults["c2bBibSearcher/Pdf2TextArg"] = "-raw";
    defaults["c2bPdfImport/Pdf2TextArg"] = "-f 1 -l 1";

    // Construct current settings
    QStringList keys = defaults.keys();
    for (int i = 0; i < keys.size(); ++i)
        setValue(keys.at(i), value(keys.at(i), defaults.value(keys.at(i))));
    update();
}

QString c2bSettings::fileName(const QString& key)
{
    if (is_usbConfMode)
        return qAppDir.relativeFilePath(value(key).toString());
    else
        return value(key).toString();
}

void c2bSettings::setFileName(const QString& key, const QString& fn)
{
    if (is_usbConfMode)
        setValue(key, qAppDir.relativeFilePath(fn));
    else
        setValue(key, fn);
}

c2bSettings* c2bSettings::c2b_global_settings = 0;


/***************************************************************************
 *  Command line processing
 ***************************************************************************/

/** \page commandline The cb2Bib Command Line

\verbatim

 Usage: cb2bib [tmp_imported_reference_file] [options]
 Options:
        --bibedit [filename1.bib [filename2.bib ...  ]]  Open BibTeX editor
        --conf [filename.conf]                           Use configuration file
 Examples:
        cb2bib                                           Launches cb2Bib
        cb2bib tmp_reference                             Imports reference
        cb2bib tmp_reference --conf cb2bib.conf          Imports reference, uses cb2bib.conf configuration
        cb2bib --bibedit filename.bib                    Edits BibTeX filename.bib
        cb2bib --conf                                    Uses /cb2bib/full/path/cb2bib.conf configuration
 Notes:
  -To import references from a browser just select either cb2bib or dl_cb2bib. The browser will provide
   the temporary reference filename.
  -The file cb2bib.conf must be readable and writable. If it does not exists, the cb2Bib will write one
   based on predefined defaults.
  -If launching the cb2Bib from a removable media use the command 'cb2bib --conf'. Settings will be found
   independently of the mounting address that the host computer will provide.

\endverbatim

*/

bool c2bSettings::confMode()
{
    if (qApp->argc() > 1)
        for (int i = 1; i < qApp->argc(); i++)
            if (QString(qApp->argv()[i]) == "--conf")
                return true;
    return false;
}

const QString c2bSettings::confFileName()
{
    QString fileName;
    for (int i = 1; i < qApp->argc() - 1; ++i)
        if (QString(qApp->argv()[i]) == "--conf")
        {
            fileName = QString(qApp->argv()[i+1]);
            break;
        }
    if (fileName.isEmpty())
        fileName = qApp->applicationDirPath() + QDir::separator() + "cb2bib.conf";
    QFileInfo fi(fileName);
    if (fi.isDir())
    {
        fprintf(stdout, QObject::tr("[cb2Bib] Error: Could not open %1 configuration file for reading/writing.").arg(fileName).toLatin1());
        fprintf(stdout, QObject::tr("[cb2Bib] Error: Filename is a directory.").toLatin1());
        exit(1);
    }
    QFile file(fileName);
    if (!file.open(QIODevice::ReadWrite | QIODevice::Text))
    {
        fprintf(stdout, QObject::tr("[cb2Bib] Error: Could not open %1 configuration file for reading/writing.").arg(fileName).toLatin1());
        fprintf(stdout, QObject::tr("[cb2Bib] Error: %1.").arg(file.errorString()).toLatin1());
        exit(1);
    }
    return fileName;
}

void c2bSettings::readCommandLine()
{
    // Check for errors
    QStringList arguments = qApp->arguments();
    arguments.removeAll("");
    for (int i = 1; i < arguments.count(); ++i)
        if (arguments.at(i).startsWith("-"))
        {
            if (arguments.at(i) == "--bibedit" || arguments.at(i) == "--conf")
                continue;
            else
                printUsage();
        }

    // Check for --bibedit
    autostart_bibtex_editor = false;
    bibTeXFilelist.clear();
    for (int i = 1; i < arguments.count(); ++i)
        if (arguments.at(i) == "--bibedit")
        {
            autostart_bibtex_editor = true;
            for (int j = i + 1; j < arguments.count(); ++j)
                if (arguments.at(j).startsWith("-"))
                    break;
                else
                {
                    QFileInfo f(arguments.at(j));
                    bibTeXFilelist.append(f.absoluteFilePath());
                }
            break;
        }

    // Check for tmp_imported_reference_file
    init_gui = true;
    if (arguments.count() > 1)
    {
        if (autostart_bibtex_editor && !arguments.at(1).startsWith("-"))
            printUsage();
        if (!arguments.at(1).startsWith("-"))
        {
            QFileInfo f(arguments.at(1));
            tmp_imported_reference_file = f.absoluteFilePath();
            // Process tmp_imported_reference_file and update init_gui
            processReferenceFile();
        }
    }

    // Done
    autostart_bibtex_editor = bibTeXFilelist.count() > 0;
    if (autostart_bibtex_editor)
        setValue("cb2Bib/BibTeXFile", QFileInfo(bibTeXFilelist.first()).absoluteFilePath());
    if (init_gui)
    {
        setValue("cb2Bib/isRunning", true);
        sync();
    }
}

void c2bSettings::printUsage()
{
    fprintf(stdout, "Usage: cb2bib [tmp_imported_reference_file] [options]\n");
    fprintf(stdout, "Options:\n");
    fprintf(stdout, "       --bibedit [filename1.bib [filename2.bib ...  ]]  Open BibTeX editor\n");
    fprintf(stdout, "       --conf [filename.conf]                           Use configuration file\n");
    fprintf(stdout, "Examples:\n");
    fprintf(stdout, "       cb2bib                                           Launches cb2Bib\n");
    fprintf(stdout, "       cb2bib tmp_reference                             Imports reference\n");
    fprintf(stdout, "       cb2bib tmp_reference --conf cb2bib.conf          Imports reference, uses cb2bib.conf configuration\n");
    fprintf(stdout, "       cb2bib --bibedit filename.bib                    Edits BibTeX filename.bib\n");
    fprintf(stdout, "       cb2bib --conf                                    Uses /cb2bib/full/path/cb2bib.conf configuration\n");
    fprintf(stdout, "Notes:\n");
    fprintf(stdout, " -To import references from a browser just select either cb2bib or dl_cb2bib. The browser will provide\n");
    fprintf(stdout, "  the temporary reference filename.\n");
    fprintf(stdout, " -The file cb2bib.conf must be readable and writable. If it does not exists, the cb2Bib will write one\n");
    fprintf(stdout, "  based on predefined defaults.\n");
    fprintf(stdout, " -If launching the cb2Bib from a removable media use the command 'cb2bib --conf'. Settings will be found\n");
    fprintf(stdout, "  independently of the mounting address that the host computer will provide.\n");
    exit(1);
}


/***************************************************************************
 *  Downloaded reference extraction
 ***************************************************************************/

void c2bSettings::processReferenceFile()
{
    // The "Download reference to reference manager" functionality to cb2Bib is implemented as follows:
    // - Put downloaded reference into the clipboard
    // - If no cb2Bib instance is running, launch cb2Bib GUI, otherwise exit
    // This is simple, enough, and consistent with a 'clipboard to BibTeX' program.
    // The 'single instance' implementation is naive, but it is sufficient for cb2Bib, and does not prevent
    // from having multiple cb2Bib instances in its usual deployment, if so desired.
    // Aware of QClipboard X11 note: "You should not store or retrieve the clipboard contents in response
    // to timer or non-user-input events". Functions processReferenceFile() and processprocessReferenceFileEvent()
    // do not follow it. It seems to work smooth, for all platforms, and, overall, it is a feasible solution
    // in this particular case.

    QFile file(tmp_imported_reference_file);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
    {
        init_gui = false;
        QMessageBox::warning(0, tr("Warning - cb2Bib"), tr("Unable to open the reference file %1 for reading.\nError: '%2'.").
                             arg(tmp_imported_reference_file).arg(file.errorString()), QMessageBox::Ok);
        QTimer::singleShot(500, qApp, SLOT(quit()));
    }
    else
    {
        if (value("cb2Bib/isRunning", false).toBool())
        {
            init_gui = false;
            progBar.setRange(0, 100);
            progBar.setWindowTitle(QObject::tr("Importing..."));
            progBar.setWindowIcon(QIcon(QString::fromUtf8(":/icons/icons/cb2bib.png")));
            progBar.show();
            progBar.setValue(50);
        }
        QTimer::singleShot(500, this, SLOT(processReferenceFileEvent()));
    }
}

void c2bSettings::processReferenceFileEvent()
{
    const QString reference = c2bUtils::fileToString(tmp_imported_reference_file).trimmed();
    QClipboard* cb = qApp->clipboard();
    cb->setText(reference, QClipboard::Clipboard);
    if (!init_gui)
    {
        progBar.setValue(100);
        QTimer::singleShot(500, qApp, SLOT(quit()));
    }
}

void c2bSettings::guiExits()
{
    setValue("cb2Bib/isRunning", false);
    sync();
}
