/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "c2bTextEdit.h"

#include "c2bSettings.h"
#include "c2bUtils.h"

#include <QKeyEvent>
#include <QList>
#include <QMenu>
#include <QScrollBar>
#include <QTextBlock>
#include <QTextCursor>


c2bTextEdit::c2bTextEdit(QWidget* parent) : QTextEdit(parent)
{
    setFrameShape(QTextEdit::StyledPanel);
    setFrameShadow(QTextEdit::Raised);
    setAcceptRichText(false);

    c2bUtils::addSeparator(this);
    wordWrapAction = new QAction(this);
    addAction(wordWrapAction);
    wordWrapAction->setText(tr("Wrap Text"));
    wordWrapAction->setStatusTip(tr("Wrap Text"));
    connect(wordWrapAction, SIGNAL(triggered()), this, SLOT(wordWrap()));

    wordUnWrapAction = new QAction(this);
    addAction(wordUnWrapAction);
    wordUnWrapAction->setText(tr("Unwrap Text"));
    wordUnWrapAction->setStatusTip(tr("Unwrap Text"));
    connect(wordUnWrapAction, SIGNAL(triggered()), this, SLOT(wordUnWrap()));

    if (c2bSettingsP->value("c2bTextEdit/WrapMode", false).toBool())
        wordWrap();
    else
        wordUnWrap();
}

c2bTextEdit::~c2bTextEdit()
{
    c2bSettingsP->setValue("c2bTextEdit/WrapMode", lineWrapMode() == QTextEdit::WidgetWidth);
}


void c2bTextEdit::setBibTeXActions()
{
    c2bUtils::addSeparator(this);
    pasteCurrentBibTeXAction = new QAction(this);
    addAction(pasteCurrentBibTeXAction);
    pasteCurrentBibTeXAction->setShortcut(tr("Shift+Ins"));
    pasteCurrentBibTeXAction->setText(tr("Paste Current BibTeX"));
    pasteCurrentBibTeXAction->setStatusTip(tr("Paste Current BibTeX"));

    c2bUtils::addSeparator(this);
    convertSelection2LaTeXAction = new QAction(this);
    addAction(convertSelection2LaTeXAction);
    convertSelection2LaTeXAction->setText(tr("Selection to LaTeX"));
    convertSelection2LaTeXAction->setStatusTip(tr("Selection to LaTeX"));
    connect(convertSelection2LaTeXAction, SIGNAL(triggered()), this, SLOT(convert2LaTeX()));
    convertSelection2LaTeXAction->setEnabled(false);
    connect(this, SIGNAL(copyAvailable(bool)), convertSelection2LaTeXAction, SLOT(setEnabled(bool)));
    convertSelection2UnicodeAction = new QAction(this);
    addAction(convertSelection2UnicodeAction);
    convertSelection2UnicodeAction->setText(tr("Selection to Unicode"));
    convertSelection2UnicodeAction->setStatusTip(tr("Selection to Unicode"));
    connect(convertSelection2UnicodeAction, SIGNAL(triggered()), this, SLOT(convert2Unicode()));
    convertSelection2UnicodeAction->setEnabled(false);
    connect(this, SIGNAL(copyAvailable(bool)), convertSelection2UnicodeAction, SLOT(setEnabled(bool)));

    c2bUtils::addSeparator(this);
    journalsToFullAction = new QAction(this);
    addAction(journalsToFullAction);
    journalsToFullAction->setText(tr("Set Journals to Full Name"));
    journalsToFullAction->setStatusTip(tr("Set Journals to Full Name"));
    journalsToAbbreviatedAction = new QAction(this);
    addAction(journalsToAbbreviatedAction);
    journalsToAbbreviatedAction->setText(tr("Set Journals to Abbreviated Name"));
    journalsToAbbreviatedAction->setStatusTip(tr("Set Journals to Abbreviated Name"));

    c2bUtils::addSeparator(this);
    updateMetadataAction = new QAction(this);
    addAction(updateMetadataAction);
    updateMetadataAction->setText(tr("Update Documents Metadata"));
    updateMetadataAction->setStatusTip(tr("Update Documents Metadata"));
}

void c2bTextEdit::contextMenuEvent(QContextMenuEvent * e)
{
    QMenu *menu = createStandardContextMenu();
    QList<QAction*> acts = actions();
    for (int i = 0; i < acts.count(); ++i)
        menu->addAction(acts.at(i));
    menu->exec(e->globalPos());
    delete menu;
}

void c2bTextEdit::keyPressEvent(QKeyEvent * e)
{
    QTextEdit::keyPressEvent(e);
    if (e->key() == Qt::Key_Insert)
    {
        setOverwriteMode(!overwriteMode());
        emit overwriteModeChanged();
    }
}

void c2bTextEdit::wordWrap()
{
    setLineWrapMode(QTextEdit::WidgetWidth);
    wordWrapAction->setEnabled(false);
    wordUnWrapAction->setEnabled(true);
}

void c2bTextEdit::wordUnWrap()
{
    setLineWrapMode(QTextEdit::NoWrap);
    wordWrapAction->setEnabled(true);
    wordUnWrapAction->setEnabled(false);
}

void c2bTextEdit::updatePlainText(const QString& newText)
{
    int scrollPosition = verticalScrollBar()->value();
    int cursorBlock = textCursor().blockNumber();
    int cursorColumn = textCursor().columnNumber();
    setPlainText(newText);

    QTextBlock p = document()->begin();
    int i = 0;
    while (p.isValid())
    {
        if (cursorBlock == i)
            break;
        i++;
        p = p.next();
    }
    QTextCursor cursor = textCursor();
    cursor.setPosition(p.position() + cursorColumn, QTextCursor::MoveAnchor);
    setTextCursor(cursor);
    verticalScrollBar()->setValue(scrollPosition);
}

void c2bTextEdit::convert2LaTeX()
{
    QString txt = textCursor().selectedText();
    // Better do the long way: Convert anything to c2b, and then
    // translate to TeX
    c2bUtils::fullBibToC2b(txt);
    insertPlainText(c2bUtils::c2bToBib(txt));
}

void c2bTextEdit::convert2Unicode()
{
    QString txt = textCursor().selectedText();
    // Strictly, this should be bibToC2b; but convert also
    // possible PNAS like, and, for consistency, HTML sub
    // and superscripts to TeX
    insertPlainText(c2bUtils::fullBibToC2b(txt));
}
