/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "c2bUpdateMetadata.h"

#include "c2b.h"
#include "c2bSettings.h"

#include <QPushButton>
#include <QTimer>


c2bUpdateMetadata::c2bUpdateMetadata(QWidget* parent) : QDialog(parent)
{
    ui.setupUi(this);
    setWindowTitle(tr("Documents Updating Log - cb2Bib"));
    ui.Log->setFont(c2bSettingsP->c2bMonoFont);
    ui.Log->setLineWrapMode(QTextEdit::NoWrap);
    buttonAbort = new QPushButton(tr("Abort"));
    ui.buttonBox->addButton(buttonAbort, QDialogButtonBox::ActionRole);
    connect(buttonAbort, SIGNAL(clicked()), this, SLOT(aborted()));
    connect(ui.buttonBox, SIGNAL(helpRequested()), this, SLOT(help()));
    resize(c2bSettingsP->value("c2bLogWidget/size", size()).toSize());
}

c2bUpdateMetadata::~c2bUpdateMetadata()
{
    c2bSettingsP->setValue("c2bLogWidget/size", size());
}


void c2bUpdateMetadata::update(const QString& fn, const QString& bibtex)
{
    ui.Log->clear();
    _fn = fn;
    _bibtex = bibtex;
    QTimer::singleShot(500, this, SLOT(update()));
    QDialog::exec();
}

void c2bUpdateMetadata::update()
{
    ui.Log->append(tr("[cb2bib] Updating documents metadata...\n"));
    QString exiftool_bin = c2bSettingsP->fileName("c2bMetaPreparser/ExifToolBin");
    if (exiftool_bin.isEmpty())
    {
        ui.Log->append(tr("[cb2bib] ExifTool location has not been specified."));
        return ;
    }
    if (!QFileInfo(exiftool_bin).exists())
    {
        ui.Log->append(tr("[cb2bib] ExifTool file %1 does not exist.").arg(exiftool_bin));
        return;
    }
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    ui.buttonBox->button(QDialogButtonBox::Close)->setEnabled(false);
    buttonAbort->setEnabled(true);
    buttonAbort->setFocus();
    _aborted = false;
    ui.Log->append(tr("[cb2bib] Processing file %1...\n").arg(_fn));

    int doc_counter = 0;
    int error_counter = 0;
    int updated_counter = 0;
    bp = c2b::bibParser();
    c2bMetaPreparser* mp = c2b::metaPreparser();
    bibReference ref;
    bp->initReferenceParsing(_fn, bp->bibFieldList, &ref);
    while (bp->referencesIn(_bibtex, &ref) && !_aborted)
    {
        QCoreApplication::processEvents();
        QString file = ref.value("file");
        if (file.isEmpty())
        {
            ui.Log->append(tr("[cb2bib] %1: No file in reference.").arg(ref.keyName));
            continue;
        }
        QFileInfo fi(file);
        if (!fi.exists())
        {
            ui.Log->append(tr("[cb2bib] %1: Warning: File %2 does not exist.").arg(ref.keyName).arg(file));
            continue;
        }
        ++doc_counter;
        bibReference mref;
        if (mp->metadata(file, &mref))
            if (!needsUpdating(ref, mref))
                continue;
        if (!fi.isWritable())
        {
            ++error_counter;
            ui.Log->append(tr("[cb2bib] %1: Error: File %2 is not writable.").arg(ref.keyName).arg(file));
            continue;
        }
        QString error_str;
        if (mp->insertMetadata(ref, file, &error_str))
        {
            if (mp->metadata(file, &mref))
                if (!needsUpdating(ref, mref))
                {
                    ++updated_counter;
                    ui.Log->append(tr("[cb2bib] %1: File %2 successfully updated.").arg(ref.keyName).arg(file));
                    continue;
                }
            ++error_counter;
            ui.Log->append(tr("[cb2bib] %1: Warning: File %2 was not properly updated.").arg(ref.keyName).arg(file));
            writeDifferences(ref, mref);
        }
        else
        {
            ++error_counter;
            ui.Log->append(tr("[cb2bib] %1: Error: File %2 could not be updated.").arg(ref.keyName).arg(file));
            ui.Log->append(tr("[exiftool] '%1'.").arg(error_str));
        }
        QCoreApplication::processEvents();
    }

    QApplication::restoreOverrideCursor();
    ui.buttonBox->button(QDialogButtonBox::Close)->setEnabled(true);
    ui.buttonBox->button(QDialogButtonBox::Close)->setFocus();
    buttonAbort->setEnabled(false);

    ui.Log->append(tr("\n[cb2bib] Checked %1 documents.").arg(doc_counter));
    if (doc_counter > 0 && updated_counter == 0 && error_counter == 0)
        ui.Log->append(tr("[cb2bib] Documents Metadata was up to date."));
    else
        ui.Log->append(tr("[cb2bib] Updated %1 documents.").arg(updated_counter));
    if (error_counter > 0)
        ui.Log->append(tr("[cb2bib] Found %1 errors.").arg(error_counter));
    c2b::showMessage(tr("Updated %1 documents.").arg(updated_counter));
}

bool c2bUpdateMetadata::needsUpdating(const bibReference& ref, const bibReference& mref)
{
    if (ref.typeName != mref.typeName)
        return true;
    const QStringList bibFieldList = bp->bibFieldList;
    for (int i = 0; i < bibFieldList.size(); ++i)
    {
        QString key = bibFieldList.at(i);
        if (key == "file")
            continue;
        else if (key == "id")
            continue;
        QString value = ref.value(key);
        c2bUtils::fullBibToC2b(value);
        if (key == "title")
            c2bUtils::cleanTitle(value);
        else if (key == "booktitle")
            c2bUtils::cleanTitle(value);
        if (value != mref.value(key))
            return true;
    }
    return false;
}

void c2bUpdateMetadata::writeDifferences(const bibReference& ref, const bibReference& mref)
{
    QString diff("  [Key] '%1'\n  [In Reference] '%2'\n  [In Document]  '%3'");
    if (ref.typeName != mref.typeName)
        ui.Log->append(diff.arg("type").arg(ref.typeName).arg(mref.typeName));
    const QStringList bibFieldList = bp->bibFieldList;
    for (int i = 0; i < bibFieldList.size(); ++i)
    {
        QString key = bibFieldList.at(i);
        if (key == "file")
            continue;
        else if (key == "id")
            continue;
        QString value = ref.value(key);
        c2bUtils::fullBibToC2b(value);
        if (key == "title")
            c2bUtils::cleanTitle(value);
        else if (key == "booktitle")
            c2bUtils::cleanTitle(value);
        if (value != mref.value(key))
            ui.Log->append(diff.arg(key).arg(value).arg(mref.value(key)));
    }
}

void c2bUpdateMetadata::aborted()
{
    _aborted = true;
    buttonAbort->setEnabled(false);
    ui.Log->append(tr("\n\n[cb2bib] Aborted.\n"));
}

void c2bUpdateMetadata::help()
{
    c2b::displayHelp("http://www.molspaces.com/d_cb2bib-c2beditor.php#update_metadata");
}
