/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#ifndef C2BUTILS_H
#define C2BUTILS_H

#include <QAction>
#include <QDir>
#include <QFile>
#include <QFileInfo>
#include <QGraphicsTextItem>
#include <QRegExp>
#include <QString>
#include <QtDebug>


/**
    General cb2Bib Utils

    @author Pere Constans
*/

namespace c2bUtils
{

extern QString c2bToAscii(QString str, bool keepWords = false);
extern QString& bibToC2b(QString& str);
extern QString& c2bToBib(QString& str);

const QString PNAS_greek_letters = "[\\{\\[\\(](alpha|beta|gamma|delta|varepsilon|"
                                   "zeta|eta|theta|iota|kappa|lambda|mu|nu|xi|"
                                   "omicron|pi|rho|sigmaf|varsigma|sigma|tau|"
                                   "upsilon|phi|chi|psi|omega|vartheta|varphi|"
                                   "varpi|digamma|varkappa|varrho|epsilon)[\\}\\]\\)]";
static const QRegExp isBibToC2b("[\\{\\[\\$\\\\<>]");


inline QString& fullBibToC2b(QString& str)
{
    // Escape TeX and other special characters to Unicode
    if (!str.contains(isBibToC2b))
        return str;
    // PNAS Style
    str.replace(QRegExp(PNAS_greek_letters, Qt::CaseInsensitive), "$\\\\1$");
    str.replace("{micro}", "$\\mu$");
    str.replace("{middle dot}", "$\\cdot$");
    // Sub and superscripts
    str.replace(QRegExp("<sub>([^<]*)</sub>"), "$_{\\1}$");
    str.replace(QRegExp("<sup>([^<]*)</sup>"), "$^{\\1}$");
    str.replace(QRegExp("\\[sub ([^\\]]*)\\]"), "$_{\\1}$");
    str.replace(QRegExp("\\[sup ([^\\]]*)\\]"), "$^{\\1}$");
    str.replace(QRegExp("\\}\\$\\$([_\\^])\\{"), "}\\1{"); // merge if both
    // TeX to Unicode
    bibToC2b(str);
    return str;
}

inline QString& cleanEquations(QString& str)
{
    // Simplify equations
    if (!str.contains('$'))
        return str;
    int n = str.length();
    int p = 0;
    int i = 0;
    while (i < n)
    {
        if (str.at(i) == '$')
        {
            bool has_eq = false;
            for (int j = i + 1; j < qMin(i + 70, n); ++j) // Set a max equation length
                if (str.at(j) == '$')
                {
                    has_eq = true;
                    QChar ck;
                    for (int k = i + 1; k < j; ++k)
                    {
                        ck = str.at(k);
                        if (ck != '_' && ck != '^' && ck != '\\' && ck != '{' && ck != '}')
                            str[p++] = str.at(k);
                    }
                    i = j;
                    break;
                }
            if (!has_eq)
                str[p++] = str.at(i);
        }
        else
            str[p++] = str.at(i);
        ++i;
    }
    str.chop(n - p);
    return str;
}

inline QString& cleanTitle(QString& title)
{
    // Remove TeX braces and equation $
    cleanEquations(title);
    title.remove("{");
    title.remove("}");
    title = title.trimmed();
    return title;
}

inline QString toHtmlString(QString str)
{
    // Encode some symbols to HTML for proper browser display
    str.replace("<", "&#060;");
    str.replace(">", "&#062;");
    str.replace("%", "&#037;");
    str.replace(QRegExp("_\\{([^\\}]*)\\}"), "<sub>\\1</sub>");
    str.replace(QRegExp("\\^\\{([^\\}]*)\\}"), "<sup>\\1</sup>");
    cleanEquations(str);
    return str;
}

inline QString fromHtmlString(const QString& str)
{
    QGraphicsTextItem converter;
    converter.setHtml(str);
    QString converted = converter.toPlainText();
    return converted;
}

inline QString firstPage(QString pages)
{
    pages.simplified();
    pages.replace(QRegExp("[\\s-].*$"), "");
    return (pages);
}

inline bool isUpperCaseString(const QString& str)
{
    if (str.isEmpty())
        return false;
    for (int i = 0; i < str.length(); i++)
        if (str.at(i).isLetter())
            if (str.at(i).category() == QChar::Letter_Lowercase)
                return false;
    return true;
}

inline QString setCapitalization(const QString& str)
{
    QString cap_string = str;
    cap_string = cap_string.toLower();
    bool do_upper = true;
    for (int i = 0; i < cap_string.length(); i++)
        if (cap_string.at(i).isLetter())
        {
            if (do_upper)
                cap_string[i] = cap_string.at(i).toUpper();
            do_upper = false;
        }
        else if (cap_string.at(i) == '.' || cap_string.at(i) == ':')
            do_upper = true;
    return (cap_string);
}

/**
    Gives a document absolute, full path-name
*/
inline QString documentAbsoluteName(const bool bibtexrelative, const QString& bibtexfn, const QString& docdir, const QString& docfn)
{
    QString name;
    if (bibtexrelative)
    {
        // bibtexfn might be relative in USB mode, calling absolutePath will restore its fullname.
        QDir ddir(QFileInfo(bibtexfn).absolutePath());
        // When bibtexrelative, should only be relative to bibtexfn, (safely) disregard if it is absolute.
        if (QDir(docdir).isRelative())
            name = ddir.absolutePath() + QDir::separator() + docdir + QDir::separator() + QFileInfo(docfn).fileName();
        else
            name = ddir.absolutePath() + QDir::separator() + QFileInfo(docfn).fileName();
    }
    else
        // On USB mode, docdir will be relative to the application directory. Calling absolutePath will restore its
        // fullname, since cb2Bib does not change (and should not change for this to work) the application directory.
        name = QDir(docdir).absolutePath() + QDir::separator() + QFileInfo(docfn).fileName();
    return QDir::toNativeSeparators(QDir::cleanPath(name));
}

/**
    Gives a document filename, as it will be written to BibTeX file tag
*/
inline QString documentFileName(const bool bibtexrelative, const QString& bibtexfn, const QString& docdir, const QString& docfn)
{
    QString name;
    if (bibtexrelative)
    {
        QDir ddir(QFileInfo(bibtexfn).absolutePath());
        if (QDir(docdir).isRelative())
            name = ddir.absolutePath() + QDir::separator() + docdir + QDir::separator() + QFileInfo(docfn).fileName();
        else
            name = ddir.absolutePath() + QDir::separator() + QFileInfo(docfn).fileName();
        name = ddir.relativeFilePath(name);
    }
    else
        name = QDir(docdir).absolutePath() + QDir::separator() + QFileInfo(docfn).fileName();
    return QDir::toNativeSeparators(QDir::cleanPath(name));
}

/**
    Gives a document path as it will be written to BibTeX file tag
*/
inline QString documentDirName(const bool bibtexrelative, const QString& bibtexfn, const QString& docdir)
{
    QString name;
    if (bibtexrelative)
    {
        QDir ddir(QFileInfo(bibtexfn).absolutePath());
        if (QDir(docdir).isRelative())
            name = ddir.absolutePath() + QDir::separator() + docdir;
        else
            name = docdir + QDir::separator();
        name = ddir.relativeFilePath(name);
    }
    else
        name = QDir(docdir).absolutePath();
    name = QDir::toNativeSeparators(QDir::cleanPath(name));
    if (!name.isEmpty())
        name += QDir::separator();
    return name;
}

inline QString fileToString(const QString& fn, bool delete_on_close = false)
{
    if (fn.isEmpty())
        return QString();
    QFile f(fn);
    if (f.open(QIODevice::ReadOnly | QIODevice::Text))
    {
        QTextStream stream(&f);
        stream.setCodec("UTF-8");
        stream.setAutoDetectUnicode(true);
        QString contents = stream.readAll();
        if (delete_on_close)
            f.remove();
        else
            f.close();
        return (contents);
    }
    else
        return QString();
}

inline bool stringToFile(const QString& str, const QString& fn)
{
    if (fn.isEmpty())
        return false;
    QFile f(fn);
    if (f.open(QIODevice::WriteOnly | QIODevice::Text))
    {
        QTextStream stream(&f);
        stream.setCodec("UTF-8");
        stream.setAutoDetectUnicode(true);
        stream << str;
        f.close();
        return true;
    }
    else
        return false;
}

inline int nearInteger(double a)
{
    int ia = (int)a;
    a = a - (double)ia;
    if (a >= 0.5)
        ia++;
    else if (a <= -0.5)
        ia--;
    return ia;
}

inline void addSeparator(QWidget* w)
{
    // Adds separator to widget w
    QAction* action = new QAction(w);
    action->setSeparator(true);
    w->addAction(action);
}

inline void debug(const QString&
#ifdef CB2BIB_DEBUG
                  debug_ms
#endif
                 )
{
#ifdef CB2BIB_DEBUG
    qDebug() << "[cb2bib] " << debug_ms;
#endif
}

} // namespace c2bUtils

#endif
