/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "cb2Bib.h"

#include "c2b.h"
#include "c2bBibHighlighter.h"
#include "c2bBibParser.h"
#include "c2bClipEdit.h"
#include "c2bClipboard.h"
#include "c2bEditor.h"
#include "c2bFileDialog.h"
#include "c2bNetwork.h"
#include "c2bNetworkQuery.h"
#include "c2bPdfImport.h"
#include "c2bPostprocess.h"
#include "c2bSaveRegExp.h"
#include "c2bSearchInFiles.h"
#include "c2bSettings.h"
#include "c2bTextBrowser.h"

#include <QtGui>


/** \page overview Overview

    The cb2Bib is a tool for rapidly extracting unformatted, or unstandardized
    bibliographic references from email alerts, journal Web pages, and PDF
    files.

    <b>Current version: cb2Bib GET_VERSION</b>. (See \ref changelog for a
    detailed list of changes and acknowledgments, and \ref relnotes for
    additional notes and information.)

<p>&nbsp;</p>

\section contents Contents:
- \ref descrip
- \ref usage
- \ref detail
- \ref deposform
- \ref requirements
- \ref credits

<p>&nbsp;</p>
\section descrip Description


    The cb2Bib reads the clipboard text contents and process it against a set
    of predefined patterns. If this automatic detection is successful, cb2Bib
    formats the clipboard data according to the structured BibTeX reference
    standard.

    Otherwise, if no predefined format pattern is found or if detection proves
    to be difficult, manual data extraction is greatly simplified by cb2Bib. In
    most cases, such manual data extraction will provide with a new,
    personalized pattern to be included within the predefined pattern set for
    future automatic extractions.

    Once the bibliographic reference is correctly extracted, it is added to a
    specified BibTeX database file. Optionally, document files are renamed to
    its citeID and moved to a desired directory as a personal article library,
    and their metadata is updated with the bibliographic reference. See \ref
    c2bconf_documents section.


    <p>&nbsp;</p>
\section usage Using cb2Bib


\subsection procedure Procedure

    - <b>Select the reference to import from the email or web browser</b> \n On
    Unix machines, cb2Bib automatically detects mouse selections and clipboard
    changes. On Windows machines, copy or Ctrl-C is necessary to activate
    cb2Bib automatic processing.

    - <b>cb2Bib automatic processing</b> \n Once text is selected cb2Bib
    initiates the automatic reference extraction. It uses the predefined
    patterns from file <tt>regexp.txt</tt> to attempt automatic extraction. See
    \ref c2bconf_files section for setting the user predefined pattern matching
    expression file. After a successful detection bibliographic fields appear
    on the cb2Bib item line edits. Manual editing is possible at this stage.

    - <b>cb2Bib manual processing</b> \n If no predefined format pattern is
    found or if detection proves to be difficult, a manual data extraction must
    be performed. Select individual reference items from the cb2Bib clipboard
    area. A popup menu will appear after selection is made. Choose the
    corresponding bibliographic field. See \ref entrytype. Selection is
    postprocessed and added to the cb2Bib item line edit. cb2Bib field tags
    will show on the cb2Bib clipboard area. Once the manual processing is done,
    cb2Bib clipboard area will contain the matching pattern. The pattern can be
    further edited and stored to the <tt>regexp.txt</tt> file using <b>Insert
    Regular Expression</b>, Alt+I. See the \ref bibproc and \ref regexpedit
    sections.

    - <b>Download reference to cb2Bib</b> \n The cb2Bib has the built-in
    functionality to interact with publishers "Download reference to Citation
    Manager" service. Choose BibTeX format, or any other format that you can
    translate using <b>External Clipboard Preparsing Command</b>. See \ref
    keyboard, <b>Alt C</b>. Click "Download" from your browser. When asked
    "Open with..." select cb2Bib. The cb2Bib will be launched if no running
    instance is found. If already running, it will place the downloaded
    reference to the clipboard, and it will start processing. Make sure your
    running instance is aware of clipboard changes. See \ref c2bbuttons. For
    convenience, the shell script <tt>dl_cb2bib</tt>, and the desktop config
    file <tt>dl_cb2bib.desktop</tt> are also provided.

    - <b>Adding documents</b> \n PDF and other documents can be added to the
    BibTeX reference by dragging the file icon and dropping it into the
    cb2Bib's panel. Optionally, document files, are renamed to its citeID and
    moved to a desired directory as a personal article library (See \ref
    c2bconf_documents section). Linked to a reference documents correspond to
    the BibTeX tag <tt>file</tt>. Usual reference manager software will permit
    to retrieve and visualize these files. Download, copy and/or moving is
    scheduled and performed once the reference is accepted, e.g., once it is
    saved by pressing <b>Save Reference</b> button.

    - <b>Multiple retrieving from PDF files</b> \n Multiple PDF or convertible
    to text files can be sequentially processed by dragging a set of files into
    cb2Bib's PDFImport dialog. By starting the processing button, files are
    sequentially converted to text and send to cb2Bib clipboard panel for
    reference extraction. See \ref c2bpdfimport for details.

    - <b>Journal-Volume-Page Queries</b> \n Takes input Journal, Volume, and
    first page from the corresponding edit lines and attempts to complete the
    reference. Additionally, queries consider Title, DOI, and an
    <it>excerpt</it>, which is a simplified clipboard panel contents. See \ref
    c2bconf_network section, the distribution file <tt>netqinf.txt</tt>, and
    \ref relnotes035 for customization and details.

    - <b>BibTeX Editor</b> \n cb2Bib includes a practical text editor suitable
    for corrections and additions. cb2Bib capabilities are readily available
    within the editor. E.g., the reference is first sent to cb2Bib by selecting
    it, and later retrieved from cb2Bib to the editor using 'right click' +
    'Paste Current BibTeX'. Interconversions Unicode <-> LaTeX, long <->
    abbreviated journal name, and adding/renaming PDF files are easily
    available. BibTeX Editor is also accessible through a shell command line.
    See \ref commandline and \ref c2beditor.


\subsection c2bbuttons Buttons Functionality

    - \b About \n About cb2Bib, bookmarks, and online help

    - \b Configure \n Configure cb2Bib. See \ref c2bconf section.

    - <b>Search references</b> \n Opens the cb2Bib's search dialog. The search
    is performed either on the current BibTeX file, or for all BibTeX files on
    the current directory. Optionally, the search is extended to reference's
    files. Hits are displayed on an editor window. See \ref c2bsearch. See also
    \ref c2bconf_utilities section to configure the external to text converter.

    - \b PDFImport \n Launches cb2Bib's PDFImport window. Files dragged into
    PDFImport window are sequentially translated to text and sent to cb2Bib
    clipboard panel. The cb2Bib automatic and manual capabilities are then
    easily available to extract and supervise reference extractions. See \ref
    c2bpdfimport.

    - \b Exit \n Exits the cb2Bib

    - <b>Dis/Connect Clipboard</b> \n Toggles automatic cb2Bib and desktop
    clipboard connection. While the automatic cb2Bib-clipboard connection
    permits reducing keystrokes, the disconnected mode is needed in cases where
    multiple mouse selections or copies are required to complete a reference
    extraction. See also \ref relnotes041 and \ref relnotes021 if you
    experience problems with this feature.

    - <b>Network Reference Query</b> \n Starts Network Query. It usually takes
    input Journal, Volume, and first page from the corresponding edit lines and
    attempts to complete the reference. See \ref c2bconf_network network
    section to customize querying. See the distribution file
    <tt>netqinf.txt</tt> and also \ref relnotes035 for the details.

    - <b>View BibTeX Reference</b> \n View current reference as will be output
    to the BibTeX file. Eventual manual changes should be done on the item line
    edit.

    - <b>Save Reference</b> \n Inserts the current bibliographic reference to
    the output BibTeX file. This action decides whether or not a reference is
    accepted. Scheduled actions such as PDF downloading, copying or renaming
    will be performed at this time.

    - <b>Open BibTeX File</b> \n Opens the current BibTeX output file. Right
    click within the BibTeX Editor window for its particular functionality. See
    also \ref c2beditor.


\subsection keyboard Additional, Keyboard Functionality

    - <b>Alt B</b> \n Edits the Bookmarks and Network Query Info file
    <tt>netqinf.txt</tt>.

    - <b>Alt C</b> \n Preparses cb2Bib's clipboard through a user specified
    external script or tool. Preparsing is necessary to catch formatted
    references that can not be easily extracted using recognition patterns, or
    that are written in ambiguous formats. Many available scripts or specific
    user-written tools can be incorporated to cb2Bib through this external
    preparsing capability. In addition, simple, one-line scripts can be used
    within PDFImport to provide, for instance, the journal name when missing
    from the PDF first page. The cb2Bib distribution contains the sample
    scripts <tt>isi2bib</tt> and <tt>ris2bib</tt> that convert ISI and RIS
    formatted strings to BibTeX. See \ref c2bconf_clipboard for details.

    - <b>Alt D</b> \n Deletes temporary BibTeX output file. This permits using
    cb2Bib output files as temporary media to transfer references to a
    preferred reference manager and preferred format. <b>Caution</b>: This
    feature is not intended for the users that actually store their references
    in one or several BibTeX files. Remember to import references prior to
    delete cb2Bib output file.

    - <b>Alt E</b> \n Edits the regular expression file. It permits an easy
    access and modification of stored extraction patterns. New patterns are
    conveniently added to to the regular expression file by using the <b>RegExp
    Editor</b> button functionality.

    - <b>Alt F</b> \n Launches a file dialog for selecting the source file name
    for the BibTeX entry <tt>file</tt>. Selected files are displayed either, as
    the actual source filename, or, as the target filename, depending on the
    file copy/rename/move settings. See \ref c2bconf_documents. Alternatively to
    <b>Alt F</b>, documents can be easily linked to a reference by dragging the
    document file and dropping it to the cb2Bib panel.

    - <b>Alt I</b> \n Edits and optionally inserts the current regular
    expression pattern. See the \ref bibproc and \ref regexpedit sections.

    - <b>Alt J</b> \n Edits the Journal Abbreviations file.

    - <b>Alt O</b> \n Opens the currently linked document for browsing.
    Documents can be easily linked to a reference by dragging the document file
    and dropping it to the cb2Bib panel, or with <b>Alt F</b>. Linked documents
    correspond to the BibTeX tag <tt>file</tt>.

    - <b>Alt P</b> \n Postprocess BibTeX output file. It launches a user
    specified script or program to postprocess the current BibTeX file. The
    cb2Bib distribution contains two sample scripts. One, <tt>bib2pdf</tt> is a
    shell script for running <tt>latex</tt> and <tt>bibtex</tt>; this permits
    to check the BibTeX file for possible errors, and to easily produce a
    suitable output for printing. The other one, <tt>bib2end.bat</tt> is a
    batch script for running <tt>bib2xml</tt> and <tt>xml2end</tt>, which
    converts references into Endnote format. See \ref c2bconf_bibtex for
    details.

    - <b>Alt R</b> \n Restarts the cb2Bib automatic engine. Takes input data
    not from the system clipboard but from the cb2Bib clipboard panel. This
    permits editting the input stream from poorly translated PDF captions,
    correcting for author superscripts, or helps in debugging regular
    expressions.

    - <b>Esc</b> \n Quits cb2Bib popup menu. The cb2Bib menu pops up each time
    a selection in made in the clipboard panel. This saves keystrokes in a
    normal bibliographic extraction. Press <b>Esc</b> or <b>Right Click</b>
    mouse button if you need to gain access to the editor cut/copy/paste
    functionality instead.


    <p>&nbsp;</p>
\section detail The cb2Bib detailed procedure
    This section describes the details of the internal cb2Bib recognition scheme.

    -# \ref bibproc
    -# \ref authorproc
    -# \ref journalproc

    <p>&nbsp;</p>
\section deposform Regular Expression Submission Form
    cb2Bib usefullness increases when having a set of reliable regular
    expressions. It can therefore be interesting to share one's favorite
    regexps among cb2Bib users. If you have a working -which does not mean
    perfect- regexp that could benefit other users, please take a moment and
    fill out the \ref submission. These regexp will be later included into the
    cb2Bib distribution, as received, without any additional editing.
    Interested users could then copy/paste needed cb2Bib regexps into their own
    regexp file. In this way, no much of anybody's time and effort should be
    needed.

    <p>&nbsp;</p>
\section requirements cb2Bib Requirements

\subsection requirements_comp Compilation
    To compile cb2Bib, the following libraries must be present and accessible:

   - Qt 4.3.0 or higher from
    \htmlonly
    <a href="http://trolltech.com/products/qt" target="_blank">Trolltech</a>.
    \endhtmlonly
    On a Linux platform with Qt preinstalled, make sure that the <tt>devel</tt>
    packages and Qt tools are also present.

    - X11 header files if compiling on Unix platforms. Concretely, headers
    <tt>X11/Xlib.h</tt> and <tt>X11/Xatom.h</tt> are needed, unless cb2Bib is
    configured as <tt>./configure --disable_cbpoll</tt>. On a native MacOSX and
    Windows <tt>cbpoll</tt> is already disabled by default.

    - The header files <tt>fcntl.h</tt> and <tt>unistd.h</tt> from
    <tt>glibc-devel</tt> package are also required. Otherwise compilation will
    fail with <tt>referencelist.cpp:227: `::close' undeclared</tt>.

\subsection requirements_run Deployment
    Although not needed for running cb2Bib, the following tools extend cb2Bib
    applicability:

    - <tt>ExifTool</tt>, version 7.31 or later, available at
    \htmlonly
    <a href="http://www.sno.phy.queensu.ca/~phil/exiftool/" target="_blank">http://www.sno.phy.queensu.ca/~phil/exiftool/</a>.
    \endhtmlonly

    - <tt>pdftotext</tt>, found packaged as <tt>xpdf</tt>, and downloadable from
    \htmlonly
    <a href="http://www.foolabs.com/xpdf/download.html" target="_blank">http://www.foolabs.com/xpdf</a>.
    \endhtmlonly

    - The <tt>bib2xml</tt> and <tt>xml2end</tt>
    \htmlonly
    <a href="http://www.scripps.edu/~cdputnam/software/bibutils/bibutils.html" target="_blank">BibUtils</a>,
    \endhtmlonly
    to test the postprocess script <tt>bib2end.bat</tt> on Windows platforms.

    - ... and LaTeX and friends, to check for BibTeX file correctness and to
    get a nice printing through the shell script <tt>bib2pdf</tt>.


    <p>&nbsp;</p>
\section credits Credits and License

    The cb2Bib icons are taken from the <em>Oxygen</em>, <em>Crystal SVG</em>,
    and <em>Noia</em> icon sets, to be found at the
    \htmlonly
    <a href="http://www.kde.org" target="_blank">KDE Desktop Environment</a>.
    \endhtmlonly
    Several people has contributed with suggestions, bug reports or patches.
    For a detailed list of acknowledgments see the \ref changelog.

    The cb2Bib program is licensed under the terms of the
    \htmlonly
    <a href="http://www.gnu.org/licenses/gpl.html" target="_blank">GNU General Public License</a>
    \endhtmlonly
    version 3.


    <p>&nbsp;</p>

    <em>The <b>cb2Bib</b>, Pere Constans, Copyright &copy; 2004-2008. All rights reserved.\n
    First realeased, version 0.1.0 on 2004-06-29. \n
    GET_UPDATEDON</em>
*/

/** \page relnotes Release Notes

\section relnotes100 Release Note cb2Bib 1.0.0

    Approximately four years ago the first cb2Bib was released. It included the
    possibility of easily linking a document to its bibliographic reference, in
    a handy way, by dragging the file to the main (at that time, single) panel.
    Now, in version 1.0.0, when a file is dropped, the cb2Bib scans the
    document for metadata packets, and checks, in a rather experimental way,
    whether or not they contain relevant bibliographic information.

    Publishers metadata might or might not be accurate. Some, for instance,
    assign the DOI to the key Title. The cb2Bib extracts possibly relevant
    key-value pairs and adds them to clipboard panel. Whenever key-value pairs
    are found accurate, just pressing Alt+G imports them to the line edits. If
    keys with the prefix <tt>bibtex</tt> are found, then, most probable the
    data was written by JabRef or the cb2Bib itself, and then, the values are
    automatically imported.

    The preparsed metadata that is added to the clipboard panel begins with
    <tt>[Bibliographic Metadata</tt> and ends with <tt>/Bibliographic
    Metadata]</tt>. Therefore, if you are using PDFImport together with a set
    of regular expressions, such that they contain the begin (^) or end ($)
    anchors, you can safely replace them by the above tags. In this manner,
    existing regular expressions remain useful with this minor change. And,
    with the advantage that, if recognition fails for a given document,
    metadata might give the hardest fields to extract from a PDF article,
    which are author and title.

    See also \ref metadata.


    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes084 Release Note cb2Bib 0.8.4

    The previous cb2Bib release added the command line option <tt>--conf
    [full_path]cb2bib.conf</tt> to specify the settings location. This feature
    was intended, mainly, as a clean way to run the program on a host computer
    from a removable drive. The work done focused on arranging the command line
    and settings related code. It was left for a later release to solve some
    requirements regarding the managing of file pathnames and temporary files.

    This release addresses these two points. Now, when the cb2Bib is launched
    as <tt>cb2bib --conf</tt> --without a configuration filename-- it treats
    filenames as being relative to the cb2Bib actual location. Temporary files,
    if needed, will be placed at this location as well. Therefore, no data is
    being written on the host, and the cb2Bib works independently of the actual
    address that the host assigns to the removable drive.

    The Windows' un/installer cleans/sets configuration data on the registry.
    Being aware of this particular, it might be better not to install the
    program directly to the USB drive. Just copy the cb2Bib base directory from
    a home/own computer to the removable drive, and then run it on the host
    computer as <tt>cb2bib --conf</tt>.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes083 Release Note cb2Bib 0.8.3

    The cb2Bib accepts several arguments on its command line to access specific
    functionality. So far, the command <tt>cb2bib tmp_ref</tt> permits
    importing references from the browser, whenever a <em>download to reference
    manager</em> choice is available. In addition, the command <tt>cb2bib
    --bibedit ref.bib</tt> directly launches the BibTeX editor for file
    browsing and editing.

    This release adds the command line option <tt>--conf
    [full_path]cb2bib.conf</tt> to specifically set a file where all internal
    settings are being retrieved and stored. This has two interesting
    applications. On one hand, it easily permits switching from several sets of
    extraction rules, since the files <tt>abbreviations.txt</tt>,
    <tt>regexps.txt</tt>, and <tt>netqinf.txt</tt> are all stored in the
    cb2Bib's settings. And, on the other hand, it allows installing the program
    on a USB flash drive, and cleanly running it on any (e. g., library)
    computer. Settings can be stored and kept on the external device, and
    therefore, no data will be written on the registry or settings directory of
    the host computer.

    So far, however, this feature should be regarded as experimental. The Qt
    library to which the cb2Bib is linked does read/write access to system
    settings in a few places (concretely, in file and color dialogs). On Unix
    and Mac OS systems this access can be modified by setting the environment
    variable DAG_CONFIG_HOME. No such workaround is presently available in
    Windows.

    See \ref commandline for a detailed syntax description.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes081 Release Note cb2Bib 0.8.1

    Several changes in this release affect installation and deployment. First,
    the cb2Bib internals for settings management has been reorganized. Version
    0.8.1 will not read previous settings, as user colors, file locations, etc.
    On Unix, settings are stored at <tt>~/.config/MOLspaces/cb2Bib.conf</tt>.
    This file can be removed, or renamed. On Windows, it is recommended to
    uninstall previous versions before upgrading.

    Second, cb2Bib tags are not shown by default. Instead, it is shown plain,
    raw clipboard data, as it is easier to identify with the original source.
    To write a regular expression, right click, on the menu, check 'View Tagged
    Clipboard Data', and perform the extraction from this view.

    And finally, the cb2Bib adds the tag <<excerpt>> for network queries. It
    takes a simplified version of the clipboard contents and sends it to, e.g.
    Google Scholar. From there, one can easily import BibTeX references related
    to that contents. Therefore one should unchecked in most cases the 'Perform
    Network Queries after automatic reference extractions' box.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes072 Release Note cb2Bib 0.7.2

    The cb2Bib reads the clipboard contents, processes it, and places it to the
    main cb2Bib's panel. If clipboard contents can be recognized as a
    reference, it writes the corresponding BibTeX entry. If not, the user can
    interact from the cb2Bib panel and complete or correct the reference.
    Additionally, this process permits to write down a regular expression
    matching the reference's pattern.

    To ease pattern writing, cb2Bib preprocesses the raw input data. This can
    consider format conversion by external tools and general substitutions, in
    addition to including some special tags. The resulting preprocessed data is
    usually less readable. A particularly illustrating case is when input data
    comes from a PDF article.

    The cb2Bib now optionally presents input data, as raw, unprocessed data.
    This preserves the block text format of the source, and thus identifying
    the relevant bibliographic fields by visual inspection is more
    straightforward. In this raw mode view panel, interaction works in a
    similar manner. Except that, no conversions or substitutions are seen
    there, and that no regular expression tags are written.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes070 Release Note cb2Bib 0.7.0

    This realease moves forward cb2Bib base requirement to Qt 4.2.0.
    Compilation errors related to rehighlight() library calls, kindly reported
    by Bongard, Seemann, and Luisser, should not appear anymore. File/URL
    opening is carried now by this library, in a desktop integrated manner.
    Additionally, Gnome users will enjoy better integration, as Cleanlooks
    widget style is available.

    All known regressions in 0.6.9x series have been fixed. Also, a few minor
    improvements have been included. In particular, file selection dialogs
    display navigation history, and BibTeX output file can be conveniently
    selected from the list of '*.bib' files at the current directory. Such a
    feature will be specially useful to users that sort references in thematic
    files located at a given directory.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes0691 Release Note cb2Bib 0.6.91

    This release fixes a regression in the cb2Bib network capabilities.
    Network, and hence querying was erratic, both for the internal HTTP
    routines and for external clients. In addition to this fix, the
    <tt>netqinf.txt</tt> has been updated. PubMed is working again. Queries are
    also extended to include DOI's. A possible applicability will be for
    indexing a set of PDF articles with PDFImport. If the article contains its
    DOI number, and 'Perform Network Queries after automatic reference
    extractions' is checked, chances are that automatic extractions will work
    smooth.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes0690 Release Note cb2Bib 0.6.90

    The cb2Bib has been ported from Qt3 to Qt4, a migration in its underlying
    system library. Qt experienced many changes and improvements in this major
    release upgrade. Relevant to cb2Bib, these changes will provide a better
    file management, word completion, faster searches, and better desktop
    integration.

    Upgrading to Qt4 it is not a "plug and recompile" game. Thorough
    refactoring and rewriting was required. The resulting cb2Bib code is
    cleaner and more suitable to further development. As one might expect,
    major upgrades introduce new bugs that must be fixed. The cb2Bib 0.6.90 is
    actually a preview version. It has approximately the same functionality
    than its predecessor. So, no additions were considered at this point. Its
    use, bug reporting, and feedback are encouraged. This will help to get
    sooner a stable cb2Bib 0.7.

    To compile it, type <tt>./configure</tt> as usual. The <tt>configure</tt>
    script calls the <tt>qmake</tt> tool to generate an appropriate
    <tt>Makefile</tt>. To make sure the right, Qt4 <tt>qmake</tt> is invocated,
    you can setup <tt>QTDIR</tt> environment variable prior to
    <tt>./configure</tt>. The <tt>configure</tt>'s call statement will then be
    <tt>'$QTDIR/bin/qmake'</tt>. E. g., type <tt>'setenv QTDIR /usr'</tt> if
    <tt>qmake</tt> happens to be at the directory<tt>/usr/bin</tt>.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes060 Release Note cb2Bib 0.6.0

    The cb2Bib uses the internal tags <tt><<NewLine_n>></tt> and
    <tt><<Tab_n>></tt> to ease the creation of regular expressions for
    reference extraction. New line and tabular codes from the input stream are
    substituted by these numbered tags. Numbering new lines and tabulars gives
    an extra safety when writing down a regular expression. E. g., suppose
    field title is 'anything' between '<tt><<NewLine1>></tt> and
    <tt><<NewLine2>></tt>'. We can then easily write 'anything' as '.+' without
    the risk of overextending the caption to several '\\n' codes. On the other
    hand, one still can use <tt><<NewLine\\d>></tt> if not interested in a
    specific numbering. All these internal tags are later removed, once cb2Bib
    postprocesses the entry fields.

    The cb2Bib identified so far new lines by checking for '\\n' codes. I was
    unaware that this was a platform dependent, as well as a not completely
    accurate way of detecting new lines. McKay Euan reported that
    <tt><<NewLine_n>></tt> tags were not appearing as expected in the MacOSX
    version. I later learn that MacOSX uses '\\r' codes, and that Windows uses
    '\\r\\n', instead of '\\n' for new line encoding.

    This realease addresses this issue. It is supposed now that the cb2Bib
    regular expressions will be more transferable among the different
    platforms. Extraction from plain text sources is expected to be completely
    platform independent. Extraction from web pages will still remain browser
    dependent. In fact, each browser adds its peculiar interpretation of a
    given HTML source. For example, in Wiley webpages we see the sectioning
    header 'Abstract' in its source and in several browsers, but we see, and
    get, 'ABSTRACT' if using Konqueror.

    What we pay for this more uniform approach is, however, a <b>break in
    compatibility</b> with previous versions of cb2Bib. Unix/Linux users should
    not expect many differences, though. Only one from the nine regular
    expressions in the examples needed to be modified, and the two contributed
    regular expressions work perfectly without any change. Windows users will
    not see a duplication of <tt><<NewLine_n>></tt> tags. To update previous
    expressions it should be enough just shifting the <tt><<NewLine_n>></tt>
    numbering. And, of course, any working regular expression that does not
    uses <tt><<NewLine_n>></tt> tags will still be working in this new version.

    Finally, just to mention that I do not have a MacOSX to test any of the
    cb2Bib realeases in this particular platform. I am therefore assuming that
    these changes will fix the problem at hand. If otherwise, please, let me
    know. Also, let me know if release 0.6.0 'break' your own expressions. I
    consider this realease a sort of experimental or beta version, and the
    previous version 0.5.3, will still be available during this testing period.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes050 Release Note cb2Bib 0.5.0

    Two issues had appeared regarding cb2Bib installation and deployment on
    MacOSX platforms.

    First, if you encounter a 'nothing to install'-error during installation on
    MacOSX 10.4.x using the cb2bib binary installer available at
    \htmlonly
    <a href="http://naranja.umh.es/~atg/" target="_blank">http://naranja.umh.es/~atg/</a>,
    \endhtmlonly
    please delete the cb2bib-receipts from <tt>/Library/Receipts</tt> and then
    rerun the installer. See also M. Bongard's clarifying note 'MACOSX 10.4.X
    "NOTHING TO INSTALL"-ERROR' for details.

    Second, and also extensible to other cb2Bib platform versions, if PDFImport
    issues the error message 'Failed to call <i>some_format_to_text</i>' tool,
    make sure such a tool is installed and available. Go to
    Configure->PDFImport, click at the 'Select External Convert Tool' button,
    and navigate to set its full path. Since version 0.5.0 the default full
    path for the MacOSX is already set, and pointing to
    <tt>/usr/local/bin/pdftotext</tt>.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes041 Release Note cb2Bib 0.4.1

    Qt/KDE applications emit notifications whenever they change the clipboard
    contents. The cb2Bib uses these notifications to automatically start its
    'clipboard to BibTeX' processing. Other applications, however, does not
    notify about them. Since version 0.2.1, see \ref relnotes021, cb2Bib
    started checking the clipboard periodically. This checking was later
    disabled as a default, needing a few lines of code to be uncomented to
    activate it. Without such a checking, the cb2Bib appears unresponsive when
    selecting/copying from e.g., acroread or Mozilla. This release includes the
    class <tt>clipboardpoll</tt> written by L. Lunak for the KDE's Klipper.
    Checking is performed in a very optimized way. This checking is enabled by
    default. If you experience problems with this feature, or if the required
    X11 headers aren't available, consider disabling it by typing
    <tt>./configure --disable_cbpoll</tt> prior to compilation. This will
    disable checking completely. If the naive, old checking is preferred,
    uncomment the four usual lines, <tt>./configure --disable_cbpoll</tt>, and
    compile.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes035 Release Note cb2Bib 0.3.5

    Releases 0.3.3 and 0.3.4 brought querying functionality to cb2Bib. In
    essence, cb2Bib was rearranged to accommodate copying and opening of
    network files. Queries were then implemented as user customizable HTML
    posts to journal databases. In addition, these arrangements permitted
    defining convenience, dynamic bookmarks that were placed at the cb2Bib's
    'About' panel.

    cb2Bib contains three viewing panels: 'About', 'Clipboard' and 'View
    BibTeX', being the 'Clipboard' panel the main working area. To keep cb2Bib
    simple, only two buttons, 'About' and 'View BibTeX', are set to navigate
    through the panels. The 'About' and 'View BibTeX' buttons are toggle
    buttons for momentarily displaying their corresponding panels. Guidance was
    so far provided by enabling/disabling the buttons.

    After the bookmark introduction, the 'About' panel has greatly increased
    its usefullness. Button functionality has been slightly redesigned now to
    avoid as many keystrokes and mouse clicks as possible. The buttons remain
    switchable, but they no longer disable the other buttons. User is guided by
    icon changes instead. Hopefully these changes will not be confusing or
    counterintuitive.

    Bookmarks and querying functionality are customizable through the
    <tt>netqinf.txt</tt> file, which is editable by pressing the <tt>Alt+B</tt>
    keys. Supported queries are of the form 'Journal-Volume-First Page'. cb2Bib
    parses <tt>netqinf.txt</tt> each time a query is performed. It looks for
    <tt>journal=<i>Full_Name|[code]</i></tt> to obtain the required information
    for a specific journal. Empty, '<tt>journal=</tt>' entries have a meaning
    of 'any journal'. New in this realease, cb2Bib will test all possible
    queries for a given journal instead of giving up at the first <tt><i>No
    article found</i></tt> message. The query process stops at the first
    successfull hit or, otherwise, once <tt>netqinf.txt</tt> is parsed
    completely (in an equivalent way as the automatic pattern recognition
    works). This permits querying multiple -and incomplete- journal databases.

    Users should order the <tt>netqinf.txt</tt> file in a way it is more
    convenient. E.g., put PubMed in front of JACS if desired an automatic
    extraction. Or JACS in front of PubMed and extract from the journal web
    page, if author accented characters are wanted.

    So far, this querying functionality is still tagged as
    <em>experimental</em>. Either the querying itself or its syntax seem quite
    successful. However, downloading of PDF files, on windows OS + T1 network,
    <b>was found to freeze</b> once progress reaches the 30-50%. Any feedback
    on this issue will be greatly appreciated. Also, information on
    <tt>kfmclient</tt> equivalent tools for non KDE desktops would be worth to
    be included in the cb2Bib documentation.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes030 Release Note cb2Bib 0.3.0

    cb2Bib considers the whole set of authors as an author-string pattern. This
    string is later postprocessed, without requirements on the actual number of
    authors it may contain, or on how the names are written. Once considered
    author-string patterns, the extraction of bibliographic references by means
    of regular expressions becomes relatively simple.


    There are situations, however, where several author-strings are required.
    The following box shows one of these cases. Authors are grouped according
    to their affiliations. Selecting from 'F. N. First' to 'F. N. Fifth' would
    include 'First Affiliation' within the author string. Cleaning up whatever
    wording 'First Affiliation' may contain is a rather ill-posed problem.
    Instead, cb2Bib includes an <tt>Add Authors</tt> option. The way of
    operation is then to select 'F. N. First, F. N. Second, F. N. Third' and
    chose <tt>Authors</tt> and right after, select 'F. N. Fourth and F. N.
    Fifth' and chose <tt>Add Authors</tt>.

    \htmlonly
    <? require("cb2bib/p_cb2bib_example_multiple_author_sets.html"); ?>
    \endhtmlonly

    At this point in the manual extraction, the user was faced with a red
    <tt><<moreauthors>></tt> tag in the cb2Bib clipboard panel. The
    <tt><<moreauthors>></tt> tag was intended to warn the user about the fact
    that cb2Bib would not be able to consider the resulting extraction pattern
    as a valid, general regular expression. Usual regular expressions are built
    up from an a priori known level of nesting. In these cases, however, the
    level of nesting is variable. It depends on the number of different
    affiliations occurring in a particular reference.

    So far the <tt><<moreauthors>></tt> tag has become a true FAQ about cb2Bib
    and a source of many confusions. There is no real need, however, for such
    an user warning. The <tt><<moreauthors>></tt> has therefore been removed
    and cb2Bib has taken an step further, to its 0.3.0 version.

    The cb2Bib 0.3.0 manual extraction works as usual. By clicking
    <tt>Authors</tt> the Authors edit line is reseted and selection contents
    moved there. Alternatively, if <tt>Add Authors</tt> is clicked, selection
    contents is added to the author field. On this version, however, both
    operations are tagged as <tt><<author>></tt> (singular form, as it is the
    BibTeX keyword for Authors). The generated extraction pattern can now
    contain any number of <tt><<author>></tt> fields.

    In automatic mode, cb2Bib now adds all <tt>author</tt> captions to Authors.
    In this way, cb2Bib can treat interlaced author-affiliation cases.
    Obviously, users needing such extractions will have to write particular
    regular expressions for cases with one set of authors, for two sets, and so
    on. Eventhough it is not rare a work having a hundred of authors, it would
    be quite umprobable that they were working on so many different
    institutions. Therefore, few regular expressions should actually be
    required in practice. Although not elegant, this breaks what was a cb2Bib
    limitation and broadens its use when extracting from PDF sources. Remember
    here to sort these regular expressions in decreasing order, since at
    present, cb2Bib stops at the first hit. Also, consider <tt>Any Pattern</tt>
    to get ride of the actual affiliation contents, as you might not want to
    extract authors addresses.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes027 Release Note cb2Bib 0.2.7

    The cb2Bib 0.2.7 realease introduces multiple retrieving from PDF files.
    PDF documents are becoming more and more widely used, not only to transfer
    and printing articles, but also are substituting the personal paper files
    and classifiers for the electronic equivalents.

    cb2Bib is intended to help updating personal databases of papers. It is a
    tool focused on what is left behind in database retrieving. Cases such as
    email alerts, or inter colleague references and PDF sharing are example
    situations. Though in an electronic format, sources are not standardized or
    not globally used as to permit using habitual import filters in reference
    managers. cb2Bib is designed to consider a direct user intervention, either
    by creating its own useful filters or by a simple copy-paste assistance
    when handtyping.

    Hopefully someday cb2Bib will be able to take that old directory, with
    perhaps a few hundreds of papers, to automatically index the references and
    rename the files by author, in a consistent manner. The required mechanism
    is already there, in this version. But I guess that this new feature will
    manifest some present limitations in cb2Bib. For instance, most printed and
    PDF papers interlace author names and affiliations. cb2Bib doesn't has the
    means to automatically discern an author name from a department or street
    name. So far one needs to manually use the 'Add to Authors' feature to deal
    with these situations. Also, the managing of regular expressions needs
    developing, specially thinking in the spread variety of design patterns in
    publications.

    In summary, this current version is already useful in classifying and
    extracting the reference of that couple of papers that someone send right
    before submitting a work. A complete unsupervised extraction is still far
    away, however.

    <p>&nbsp;</p>
*/

/** \page relnotes Release Notes

\section relnotes021 Release Note cb2Bib 0.2.1

    The cb2Bib mechanism 'select-and-catch' failed in some cases. Acrobat and
    Mozilla selections were not always notified to the cb2Bib. Indeed, this
    'window manager - application' connection seems to be broken on a KDE 3.3.0
    + Qt 3.3.3 system.

    The cb2Bib 0.2.1 continues to listen to system clipboard change
    notifications, whenever they are received and whenever cb2Bib is on
    connected mode. Additionally, the cb2Bib 0.2.1 periodically checks for
    changes in the system clipboard. Checks are performed every second,
    approximately. This permits cb2Bib to work as usual, although one could
    experience 1-2 seconds delays in systems where the automatic notification
    is broken.

    If the 'select-and-catch' functionality appears 'sticky', possibly
    happening while using non KDE applications from where text is selected,
    check the source file <tt>c2bclipboard</tt><tt>.cpp</tt>, look for
    <tt>'Setting timer'</tt>, and set variable <tt>interval</tt> to 1000. This
    is the interval of time in ms that cb2Bib will use to check for clipboard
    changes.

    <p>&nbsp;</p>
*/


cb2Bib::cb2Bib(QWidget* parent) : QMainWindow(parent)
{
    settings = c2bSettingsP;
    ui.setupUi(this);

    // Creating cb2Bib global resources
    c2b* c2b_resources = new c2b(this, this);
    connect(c2b_resources, SIGNAL(statusMessage(const QString&)), this, SLOT(showMessage(const QString&)));
    connect(c2b_resources, SIGNAL(searchInFiles(const QString&, const QString&)), this,
            SLOT(doSearchInFiles(const QString&, const QString&)));

    // Actions and shortcuts
    c2bUtils::addSeparator(this);
    addAction(ui.clearFieldsAction);
    addAction(ui.guessFieldsAction);
    addAction(ui.startEngineAction);
    c2bUtils::addSeparator(this);
    addAction(ui.preprocessClipboardAction);
    addAction(ui.postprocessBibTeXAction);
    c2bUtils::addSeparator(this);
    addAction(ui.editBookmarksAction);
    addAction(ui.editAbbreviationsAction);
    addAction(ui.editRegExpsAction);
    c2bUtils::addSeparator(this);
    addAction(ui.insertRegExpAction);
    addAction(ui.viewTaggedClipEditAction);
    setAcceptDrops(true);

    c2bUtils::addSeparator(ui.file);
    ui.file->addAction(ui.selectFileAction);
    ui.file->addAction(ui.openFileDocumentAction);
    ui.openFileDocumentAction->setEnabled(false);

    QShortcut* sc = new QShortcut(QKeySequence(Qt::ALT + Qt::Key_D), this);
    connect(sc, SIGNAL(activated()), this, SLOT(deleteTmpBib()));

    ClipEdit = new c2bClipEdit(this);
    ClipEdit->setObjectName("ClipEdit");
    ClipEdit->addActions(actions());
    ui.stackedWidget->addWidget(ClipEdit);
    ClipEdit->viewport()->setAcceptDrops(false);

    AboutBrowser = new c2bTextBrowser(this);
    AboutBrowser->setObjectName("AboutBrowser");
    AboutBrowser->addActions(actions());
    ui.stackedWidget->addWidget(AboutBrowser);
    BibBrowser = new c2bTextBrowser(this);
    bibS = new c2bBibHighlighter(BibBrowser);
    BibBrowser->setObjectName("BibBrowser");
    BibBrowser->addActions(actions());
    ui.stackedWidget->addWidget(BibBrowser);
    connect(AboutBrowser, SIGNAL(anchorClicked(const QUrl&)), this, SLOT(linkClicked(const QUrl&)));
    connect(AboutBrowser, SIGNAL(highlighted(const QString&)), this, SLOT(showLink(const QString &)));

    // Connect Actions
    connect(ui.clearFieldsAction, SIGNAL(triggered()), this, SLOT(c2bInit()));
    connect(ui.editAbbreviationsAction, SIGNAL(triggered()), this, SLOT(abbreviationsEditor()));
    connect(ui.editBookmarksAction, SIGNAL(triggered()), this, SLOT(bookmarksEditor()));
    connect(ui.editRegExpsAction, SIGNAL(triggered()), this, SLOT(regExpsEditor()));
    connect(ui.guessFieldsAction, SIGNAL(triggered()), this, SLOT(guessFields()));
    connect(ui.insertRegExpAction, SIGNAL(triggered()), this, SLOT(insertRegExp()));
    connect(ui.openFileDocumentAction, SIGNAL(triggered()), this, SLOT(openFileDocument()));
    connect(ui.postprocessBibTeXAction, SIGNAL(triggered()), this, SLOT(postprocessBibTeX()));
    connect(ui.preprocessClipboardAction, SIGNAL(triggered()), this, SLOT(preparseClipboard()));
    connect(ui.selectFileAction, SIGNAL(triggered()), this, SLOT(selectFile()));
    connect(ui.startEngineAction, SIGNAL(triggered()), this, SLOT(restartEngine()));
    connect(ui.viewTaggedClipEditAction, SIGNAL(triggered(bool)), this, SLOT(setTaggedClipEditMode(bool)));
    setBibMenu();
    pdfImport = 0;
    searchInFiles = 0;
    setTaggedClipEditMode(false);

    // cb2Bib Settings
    loadSettings();
    connect(settings, SIGNAL(newSettings()), this, SLOT(loadSettings()));
    resize(settings->value("cb2Bib/size", size()).toSize());
    ui.bibtexfile->completer()->setCompletionMode(QCompleter::PopupCompletion);
    updateBibfile(settings->fileName("cb2Bib/BibTeXFile"));

    // Setting Bibliographic Parser
    bp = c2b::bibParser();
    setBibWidgetMapping();
    bp->connectBibWidgets(bibFields, ui.typecombo, ClipEdit);
    connect(bp, SIGNAL(bibModified()), this, SLOT(bibModified()));
    m_modified = false;
    ui.addBibB->setEnabled(m_modified);
    connect(bp, SIGNAL(preparserDataAvailable(const QString&)), this, SLOT(preParserEnded(const QString&)));

    // Setting cb2bib network queries
    netQ = new c2bNetworkQuery(this);
    connect(netQ, SIGNAL(queryEnded(bool, const QString&, const QString&)),
            this, SLOT(netQueryEnded(bool, const QString&, const QString&)));

    // Connecting cb2Bib clipboard and start
    m_connected = false;
    cb = new c2bClipboard(this);
    connect(cb, SIGNAL(cbDataChanged(const QString&)), this, SLOT(dataChanged(const QString&)));
    on_connectB_clicked();            // Set button and update m_connected
    cb->setConnected(m_connected);
    raw_input_data = "";
    dataChanged(cb->text());
    if (settings->autostart_bibtex_editor)
        initAsBibTeXEditor();
}

cb2Bib::~cb2Bib()
{
    if (searchInFiles)
        delete searchInFiles;
}


/****************************************************************************

  GUI FUNCTIONALITY

*****************************************************************************/

void cb2Bib::on_viewAboutB_clicked()
{
    if (ui.stackedWidget->currentWidget()->objectName() == "AboutBrowser")
    {
        ui.stackedWidget->setCurrentWidget(ClipEdit);
        ClipEdit->setFocus();
    }
    else
    {
        QColor linkColor(palette().buttonText().color());
        QString sheet = QString::fromLatin1("a { text-decoration: underline; color: %1 }").arg(linkColor.name());
        AboutBrowser->document()->setDefaultStyleSheet(sheet);
        QString AboutS =
            tr(
                "<br /><br /><p align=\"center\"><b>cb2Bib %1</b></p>"
                "<p align=\"center\">"
                "A tool for rapidly extracting unformatted bibliographic references from email "
                "alerts, journal web pages, and PDF files.<br />"
                "<p align=\"center\"><img src=\":/icons/icons/cb2bib.png\"></p>"
                "<p align=\"center\">"
                "<a href=\"http://www.molspaces.com/cb2bib/index.html\">"
                "cb2Bib Help and User's Guide</a></p><br />"
                "<p align=\"center\">"
                "Additional Functionality:</p>"
                "<p align=\"center\">"
                "<table border=\"1\" align=\"center\" cellspacing=\"0\" cellpadding=\"4\">"
                "<thead><tr>"
                "<th>Keys</th><th>Description</th></tr>"
                "</thead><tbody><tr>"
                "<td>Alt B</td><td>Edit Bookmarks</td></tr><tr>"
                "<td>Alt C</td><td>Preparses cb2Bib's clipboard</td></tr><tr>"
                "<td>Alt D</td><td>Deletes temporary BibTeX output file</td></tr><tr>"
                "<td>Alt E</td><td>Edit Regular Expressions</td></tr><tr>"
                "<td>Alt F</td><td>Select File source filename</td></tr><tr>"
                "<td>Alt I</td><td>Insert Regular Expression</td></tr><tr>"
                "<td>Alt J</td><td>Edit Journal Abbreviations</td></tr><tr>"
                "<td>Alt O</td><td>Open File Document</td></tr><tr>"
                "<td>Alt P</td><td>Postprocess BibTeX</td></tr><tr>"
                "<td>Alt R</td><td>Restarts automatic recognition</td></tr><tr>"
                "<td>Esc</td><td>Quits cb2Bib popup menu</td></tr><tr>"
                "<td>Right Click</td><td>Editor popup menu"
                "</td></tr></tbody>"
                "</table></p><br /> %2 <br />"
                "<p align=\"center\">Copyright (C) 2004-2008 by Pere Constans<br />"
                "<a href=\"http://www.molspaces.com/cb2bib/index.html\">"
                "http://www.molspaces.com/cb2bib</a></p>&nbsp;"
                "<p align=\"center\"><i>The cb2Bib is licensed under the terms of the "
                "<a href=\"http://www.gnu.org/licenses/gpl.html\">"
                "GNU General Public License</a> version 3"
                "</i><br /></p>&nbsp;")
            .arg(C2B_VERSION).arg(c2b::netqinf()->bookmarksToHtml());
        int scrollPosition = AboutBrowser->verticalScrollBar()->value();
        AboutBrowser->setHtml(AboutS);
        AboutBrowser->verticalScrollBar()->setValue(scrollPosition);
        ui.stackedWidget->setCurrentWidget(AboutBrowser);
        AboutBrowser->setFocus();
    }
    updateViewButtons();
}

void cb2Bib::on_configureB_clicked()
{
    c2b::Configure();
}

void cb2Bib::on_searchInFilesB_clicked()
{
    doSearchInFiles();
}

void cb2Bib::on_PdfImportB_clicked()
{
    if (!pdfImport)
    {
        pdfImport = new c2bPdfImport;
        pdfImport->setAttribute(Qt::WA_DeleteOnClose);
        connect(this, SIGNAL(addedBibToFile(const QString&)), pdfImport, SLOT(processNext()));
        connect(this, SIGNAL(endedProcessing(bool)), pdfImport, SLOT(referenceExtacted(bool)));
        connect(pdfImport, SIGNAL(textProcessed(const QString&)), this, SLOT(forceDataChange(const QString&)));
        connect(pdfImport, SIGNAL(fileProcessed(const QString&)), this, SLOT(fileDropped(const QString&)));
        connect(pdfImport, SIGNAL(saveReferenceRequest()), this, SLOT(on_addBibB_clicked()));
        connect(pdfImport, SIGNAL(setClipboardDisabled(bool)), ui.connectB, SLOT(setDisabled(bool)));
        showMessage(tr("PDF Import launched."));
    }
    pdfImport->show();
}

void cb2Bib::on_exitB_clicked()
{
    qApp->closeAllWindows();
}

void cb2Bib::on_connectB_clicked()
{
    if (m_connected)
    {
        ui.connectB->setIcon(QIcon(C2B_ICON_DIR + C2B_ICON_DISCONNECT_B));
        ui.connectB->setStatusTip(tr("Connect clipboard"));
        m_connected = false;
    }
    else
    {
        ui.connectB->setIcon(QIcon(C2B_ICON_DIR + C2B_ICON_CONNECT_B));
        ui.connectB->setStatusTip(tr("Disconnect clipboard"));
        m_connected = true;
    }
}

void cb2Bib::on_makeNetworkQueryB_clicked()
{
    makeNetQuery();
}

void cb2Bib::on_viewBibB_clicked()
{
    if (ui.stackedWidget->currentWidget()->objectName() == "BibBrowser")
    {
        ui.stackedWidget->setCurrentWidget(ClipEdit);
        ClipEdit->setFocus();
    }
    else
    {
        BibBrowser->setPlainText(bp->makeBib());
        ui.stackedWidget->setCurrentWidget(BibBrowser);
        BibBrowser->setFocus();
    }
    updateViewButtons();
}

void cb2Bib::on_addBibB_clicked()
{
    // Prepare Save Reference to file
    ui.addBibB->setEnabled(false);
    if (m_dropped_article_file.isEmpty() || ui.file->text().isEmpty() || !settings->value("cb2Bib/MovePdf").toBool())
    {
        m_target_article_file.clear();
        addToFile(true);
    }
    else
    {
        QString CopyMovePdf = settings->value("cb2Bib/CopyMovePdf").toString();
        m_target_article_file = c2bUtils::documentAbsoluteName(settings->value("cb2Bib/RelativePdfDirectory").toBool(),
                                ui.bibtexfile->currentText(), settings->value("cb2Bib/PdfDirectory").toString(), ui.file->text());
        QString doc_dir_name = QFileInfo(m_target_article_file).absolutePath();
        QDir doc_dir(doc_dir_name);
        if (!doc_dir.exists())
            if (!doc_dir.mkpath(doc_dir_name))
            {
                QMessageBox::information(this, tr("Information - cb2Bib"),
                                         tr("Unable to create directory %1.").arg(doc_dir_name), QMessageBox::Ok);
                ui.addBibB->setEnabled(m_modified);
                return;
            }
        c2b::network()->getFile(m_dropped_article_file, m_target_article_file, CopyMovePdf, this, SLOT(addToFile(bool)));
    }
}

void cb2Bib::on_editBibTeXB_clicked()
{
    bibEditor(ui.bibtexfile->currentText());
}

void cb2Bib::on_file_dirs_clicked()
{
    QString fn = c2bFileDialog::getFileName(this, "Select a BibTeX filename", ui.bibtexfile->currentText(),
                                            "BibTeX (*.bib);;All (*)");
    if (!fn.isEmpty())
        updateBibfile(fn);
}

void cb2Bib::loadSettings()
{
    ClipEdit->setFont(settings->c2bMonoFont);
    BibBrowser->setFont(settings->c2bMonoFont);
    AboutBrowser->setFont(settings->c2bMonoFont);
}

void cb2Bib::updateBibfile(const QString& fn)
{
    ui.bibtexfile->clear();
    QFileInfo fi = QFileInfo(fn);
    QDir m_bibdir = fi.absoluteDir();
    QFileInfoList flist = m_bibdir.entryInfoList(QStringList() << "*.bib");
    for (int i = 0; i < flist.count(); ++i)
        ui.bibtexfile->addItem(flist.at(i).absoluteFilePath());
    int index = ui.bibtexfile->findText(fn);
    if (index == -1)
        ui.bibtexfile->setEditText(fn);
    else
        ui.bibtexfile->setCurrentIndex(index);
}

void cb2Bib::updateViewButtons()
{
    QString viewing = ui.stackedWidget->currentWidget()->objectName();
    if (viewing == "BibBrowser")
    {
        ui.viewBibB->setIcon(QIcon(C2B_ICON_DIR + C2B_ICON_VIEWC2B_B));
        ui.viewAboutB->setIcon(QIcon(C2B_ICON_DIR + C2B_ICON_ABOUT_B));
        ui.viewBibB->setStatusTip(tr("View cb2Bib clipboard - Alt+V"));
        ui.viewAboutB->setStatusTip(tr("About cb2bib and cb2bib Bookmarks - Ctrl+Tab"));
    }
    else if (viewing == "AboutBrowser")
    {
        ui.viewBibB->setIcon(QIcon(C2B_ICON_DIR + C2B_ICON_VIEWBIB_B));
        ui.viewAboutB->setIcon(QIcon(C2B_ICON_DIR + C2B_ICON_ABOUT_B_BACK));
        ui.viewBibB->setStatusTip(tr("View BibTeX Reference - Ctrl+V"));
        ui.viewAboutB->setStatusTip(tr("Back to cb2Bib clipboard - Ctrl+Tab"));
    }
    else
    {
        ui.viewBibB->setIcon(QIcon(C2B_ICON_DIR + C2B_ICON_VIEWBIB_B));
        ui.viewAboutB->setIcon(QIcon(C2B_ICON_DIR + C2B_ICON_ABOUT_B));
        ui.viewBibB->setStatusTip(tr("View BibTeX Reference - Alt+V"));
        ui.viewAboutB->setStatusTip(tr("About cb2bib and cb2bib Bookmarks - Ctrl+Tab"));
    }
}

void cb2Bib::showLink(const QString& ln)
{
    statusBar()->showMessage(c2b::encodeLink(ln), C2B_MESSAGE_TIME);
}

void cb2Bib::showMessage(const QString& ms)
{
    statusBar()->showMessage(ms, C2B_MESSAGE_TIME);
}

void cb2Bib::dragEnterEvent(QDragEnterEvent* event)
{
    if (event->mimeData()->hasUrls())
        event->acceptProposedAction();
}

void cb2Bib::dropEvent(QDropEvent* event)
{
    QList<QUrl> fns = event->mimeData()->urls();
    if (fns.count() > 0)
    {
        if (fns.at(0).scheme() == "file")
        {
            showNormal();
            raise();
            activateWindow();
            fileDroppedByUser(fns.at(0).toLocalFile().trimmed());
        }
        else
            fileDroppedByUser(fns.at(0).toString().trimmed());
    }
    event->acceptProposedAction();
}

void cb2Bib::fileDroppedByUser(const QString& fn)
{
    if (fn.isEmpty())
        return;
    bool has_bibtex = false;
    if (settings->value("cb2Bib/AddMetadata").toBool())
    {
        QString metadata = c2b::fileMetadata(fn);
        if (!metadata.isEmpty())
        {
            QString text = ClipEdit->toPlainText();
            if (settings->value("cb2Bib/PreAppendMetadata").toString() == "prepend")
                text = metadata + text;
            else
                text = text + "\n" + metadata;
            has_bibtex = bp->hasBibTeX(metadata);
            if (has_bibtex)
                forceDataChange(text);
            else
                ClipEdit->setText(text);
        }
    }
    fileDropped(fn);
    if (!has_bibtex)
        showMessage(tr("Scheduled inclusion of file '%1'.").arg(fn));
    if (QFileInfo(fn).exists())
        settings->setFileName("cb2Bib/LastDroppedFile", fn);
}

void cb2Bib::fileDropped(const QString& fn)
{
    if (fn.isEmpty())
        return;
    m_dropped_article_file = fn.trimmed();
    ui.openFileDocumentAction->setEnabled(true);
    ui.openFileDocumentAction->setText(tr("Open '%1'").arg(m_dropped_article_file));
    setArticleFileName();
    if (settings->value("cb2Bib/MovePdf").toBool())
        ui.file->setStatusTip(tr("From file '%1'.").arg(m_dropped_article_file));
}

void cb2Bib::closeEvent(QCloseEvent* ce)
{
    if (pdfImport)
        pdfImport->close();
    if (searchInFiles)
        searchInFiles->close();
    settings->setValue("cb2Bib/size", size());
    settings->setFileName("cb2Bib/BibTeXFile", QFileInfo(ui.bibtexfile->currentText()).absoluteFilePath());
    ce->accept();
}

void cb2Bib::setTaggedClipEditMode(bool tagged)
{
    taggedViewMode = tagged;
    ui.insertRegExpAction->setEnabled(tagged);
    ClipEdit->setTaggedViewMode(tagged);
    if (taggedViewMode)
        bib_menu_any_pattern_action->setText(tr("Any Pattern"));
    else
        bib_menu_any_pattern_action->setText(tr("Delete"));
    ui.stackedWidget->setCurrentWidget(ClipEdit);
    ClipEdit->setFocus();
    updateViewButtons();
}

void cb2Bib::doSearchInFiles(const QString& string, const QString& fn)
{
    if (!searchInFiles)
    {
        searchInFiles = new c2bSearchInFiles(ui.bibtexfile->currentText());
        connect(ui.bibtexfile, SIGNAL(editTextChanged(const QString&)), searchInFiles, SLOT(setBibTeXFile(const QString&)));
        connect(searchInFiles, SIGNAL(searchDone(const QString&, c2bBibSearcher*)),
                this, SLOT(bibEditor(const QString&, c2bBibSearcher*)));
        showMessage(tr("BibTeX Search launched."));
    }
    if (fn.isEmpty())
        searchInFiles->setBibTeXFile(ui.bibtexfile->currentText());
    else
        searchInFiles->setBibTeXFile(fn);
    searchInFiles->setCurrentText(string);
    searchInFiles->show();
}


/****************************************************************************

  CB2BIB FUNCTIONALITY

*****************************************************************************/

void cb2Bib::setBibWidgetMapping()
{
    bibFields.insert("abstract", ui.abstract);
    bibFields.insert("address", ui.address);
    bibFields.insert("annote", ui.annote);
    bibFields.insert("author", ui.author);
    bibFields.insert("booktitle", ui.booktitle);
    bibFields.insert("chapter", ui.chapter);
    bibFields.insert("doi", ui.doi);
    bibFields.insert("edition", ui.edition);
    bibFields.insert("editor", ui.editor);
    bibFields.insert("eprint", ui.eprint);
    bibFields.insert("file", ui.file);
    bibFields.insert("id", ui.id);
    bibFields.insert("institution", ui.institution);
    bibFields.insert("isbn", ui.isbn);
    bibFields.insert("issn", ui.issn);
    bibFields.insert("journal", ui.journal);
    bibFields.insert("keywords", ui.keywords);
    bibFields.insert("month", ui.month);
    bibFields.insert("note", ui.note);
    bibFields.insert("number", ui.number);
    bibFields.insert("organization", ui.organization);
    bibFields.insert("pages", ui.pages);
    bibFields.insert("publisher", ui.publisher);
    bibFields.insert("school", ui.school);
    bibFields.insert("series", ui.series);
    bibFields.insert("title", ui.title);
    bibFields.insert("url", ui.url);
    bibFields.insert("volume", ui.volume);
    bibFields.insert("year", ui.year);
}

void cb2Bib::setBibMenu()
{
    bib_menu = new QMenu(this);

    QAction* a = bib_menu->addAction("Title");
    a->setData("title");
    a = bib_menu->addAction("Authors");
    a->setData("author");
    a = bib_menu->addAction("Add author", this, SLOT(addAuthors()));
    a->setIcon(QIcon(C2B_ICON_DIR + "add.png"));
    a = bib_menu->addAction("Journal");
    a->setData("journal");

    a = bib_menu->addAction("Pages");
    bib_menu->insertSeparator(a);
    a->setData("pages");
    a = bib_menu->addAction("Volume");
    a->setData("volume");
    a = bib_menu->addAction("Number");
    a->setData("number");
    a = bib_menu->addAction("Year");
    a->setData("year");

    a = bib_menu->addAction("Abstract");
    bib_menu->insertSeparator(a);
    a->setData("abstract");
    a = bib_menu->addAction("Keywords");
    a->setData("keywords");
    a = bib_menu->addAction("File");
    a->setData("file");

    QMenu* submenu = new QMenu("Other fields...", this);
    a = submenu->addAction("Address");
    a->setData("address");
    a = submenu->addAction("Annote");
    a->setData("annote");
    a = submenu->addAction("Booktitle");
    a->setData("booktitle");
    a = submenu->addAction("Chapter");
    a->setData("chapter");
    a = submenu->addAction("DOI");
    a->setData("doi");
    a = submenu->addAction("Edition");
    a->setData("edition");
    a = submenu->addAction("Editor");
    a->setData("editor");
    a = submenu->addAction("Add editor", this, SLOT(addEditor()));
    a->setIcon(QIcon(C2B_ICON_DIR + "add.png"));
    a = submenu->addAction("Eprint");
    a->setData("eprint");
    a = submenu->addAction("Institution");
    a->setData("institution");
    a = submenu->addAction("ISBN");
    a->setData("isbn");
    a = submenu->addAction("ISSN");
    a->setData("issn");
    a = submenu->addAction("Month");
    a->setData("month");
    a = submenu->addAction("Note");
    a->setData("note");
    a = submenu->addAction("Organization");
    a->setData("organization");
    a = submenu->addAction("Publisher");
    a->setData("publisher");
    a = submenu->addAction("School");
    a->setData("school");
    a = submenu->addAction("Series");
    a->setData("series");
    a = submenu->addAction("URL");
    a->setData("url");

    bib_menu->insertSeparator(a);
    bib_menu->addMenu(submenu);
    bib_menu->insertSeparator(a);
    bib_menu_any_pattern_action = bib_menu->addAction("", this, SLOT(anyPattern()));

    connect(ClipEdit, SIGNAL(userEndedSelection()), this, SLOT(readField()));
}

void cb2Bib::readField()
{
    QAction* action = bib_menu->exec(QCursor::pos());
    if (!action)
        return;
    QString field = action->data().toString();
    if (field.isEmpty())
        return;
    bibFields[field]->setText(bp->parse(field, ClipEdit->textCursor().selectedText()));
    ClipEdit->insertTag("<<" + field + ">>");
}

void cb2Bib::addAuthors()
{
    ui.author->setText(bp->parse("addauthors", ClipEdit->textCursor().selectedText()));
    ClipEdit->insertTag("<<author>>");
    if (taggedViewMode)
        showMessage(tr("Info: Creating a pattern with multiple author sets."));
}

void cb2Bib::addEditor()
{
    ui.editor->setText(bp->parse("addeditor", ClipEdit->textCursor().selectedText()));
    ClipEdit->insertTag("<<editor>>");
    if (taggedViewMode)
        showMessage(tr("Info: Creating a pattern with multiple ui.editor sets."));
}

void cb2Bib::anyPattern()
{
    if (taggedViewMode)
        ClipEdit->insertTag("<AnyPattern>");
    else
        ClipEdit->insertPlainText("");
}

void cb2Bib::preparseClipboard()
{
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    bp->preparse(raw_input_data);
}

void cb2Bib::preParserEnded(const QString& text)
{
    if (!text.isEmpty())
    {
        c2bInit();
        bp->parse(text);           // Field Initialization & Auto Detecttion
        processingEnded();
    }
    QApplication::restoreOverrideCursor();
}

void cb2Bib::restartEngine()
{
    QString text = ClipEdit->toPlainText();
    if (!text.isEmpty())
    {
        c2bInit();
        bp->parse(text);           // Field Initialization & Auto Detecttion
        processingEnded();
    }
}

void cb2Bib::guessFields()
{
    QString text = ClipEdit->toPlainText();
    if (!text.isEmpty())
    {
        QString keep_dropped_article_file = m_dropped_article_file;
        c2bInit();
        bp->guessFields(text);
        fileDropped(keep_dropped_article_file);
        processingEnded();
    }
}

void cb2Bib::dataChanged(const QString& text)
{
    if (m_connected && ui.connectB->isEnabled() && !ClipEdit->textCursor().hasSelection())
        forceDataChange(text);
}

void cb2Bib::forceDataChange(const QString& text)
{
    if (!text.isEmpty())
    {
        raw_input_data = text.trimmed();
        if (settings->value("cb2Bib/PreparserAutomatic").toBool())
            bp->preparse(text);
        else
        {
            c2bInit();
            bp->parse(text);      // Field Initialization & Auto Detecttion
            processingEnded();
        }
    }
}

void cb2Bib::c2bInit()
{
    setWindowTitle(tr("(none) - cb2Bib"));
    bp->clearFields();
    m_dropped_article_file.clear();
    ui.openFileDocumentAction->setEnabled(false);
    ui.openFileDocumentAction->setText(tr("Open File Document"));
    ui.file->setStatusTip("");
    disconnect(ui.id, SIGNAL(textChanged(const QString&)), this, SLOT(setArticleFileName()));
    m_modified = false;
    ui.addBibB->setEnabled(m_modified);
}

void cb2Bib::deleteTmpBib()
{
    QMessageBox msgBox(this);
    QPushButton* deleteButton = msgBox.addButton(tr("&Delete"), QMessageBox::DestructiveRole);
    msgBox.setDefaultButton(msgBox.addButton(QMessageBox::Cancel));
    msgBox.setWindowTitle(tr("Confirmation - cb2Bib"));
    msgBox.setText(tr("Delete temporary BibTeX file %1 ?").arg(ui.bibtexfile->currentText()));
    msgBox.setIcon(QMessageBox::Question);
    msgBox.exec();
    if (msgBox.clickedButton() == deleteButton)
    {
        QFile BibFile(ui.bibtexfile->currentText());
        if (BibFile.remove())
            showMessage(tr("File %1 deleted.").arg(ui.bibtexfile->currentText()));
        else
            QMessageBox::warning(this, tr("Warning - cb2Bib"), tr("Unable to delete the file %1.\nError: '%2'").
                                 arg(ui.bibtexfile->currentText()).arg(BibFile.errorString()), QMessageBox::Ok);
    }
}

void cb2Bib::insertRegExp()
{
    //  Extracting Reference Type
    QString reft = ui.typecombo->currentText();

    //  Extracting Field Order
    QString order = ClipEdit->toPlainText();
    QRegExp rx("<<(" + bp->bibFieldList.join("|") + ")>>");
    QStringList list;
    int pos = 0;
    while (pos >= 0)
    {
        pos = rx.indexIn(order, pos);
        if (pos > -1)
        {
            list += rx.cap(1);
            pos  += rx.matchedLength();
            c2bUtils::debug(rx.cap(1));
        }
    }
    order = list.join(" ");
    c2bUtils::debug(order);

    //  Parsing Regular Expression
    QString str = "^" + QRegExp::escape(ClipEdit->toPlainText()) + "$";

    str.replace(QRegExp("<<abstract>>"), "(.+)");
    str.replace(QRegExp("<<address>>"), "(.+)");
    str.replace(QRegExp("<<annote>>"), "(.+)");
    str.replace(QRegExp("<AnyPattern>"), ".+");
    str.replace(QRegExp("<<author>>"), "(.+)");
    str.replace(QRegExp("<<booktitle>>"), "(.+)");
    str.replace(QRegExp("<<chapter>>"), "(\\d+)");
    str.replace(QRegExp("<<doi>>"), "(10.[\\d\\.]+/\\S+)");
    str.replace(QRegExp("<<edition>>"), "(.+)");
    str.replace(QRegExp("<<editor>>"), "(.+)");
    str.replace(QRegExp("<<eprint>>"), "(.+)");
    str.replace(QRegExp("<<file>>"), "(\\S+)");
    str.replace(QRegExp("<<institution>>"), "(.+)");
    str.replace(QRegExp("<<isbn>>"), "(.+)");
    str.replace(QRegExp("<<issn>>"), "(.+)");
    str.replace(QRegExp("<<journal>>"), "(.+)");
    str.replace(QRegExp("<<keywords>>"), "(.+)");
    str.replace(QRegExp("<<month>>"), "(.+)");
    str.replace(QRegExp("<<note>>"), "(.+)");
    str.replace(QRegExp("<<number>>"), "(\\d+)");
    str.replace(QRegExp("<<organization>>"), "(.+)");
    str.replace(QRegExp("<<pages>>"), "([\\d\\-" + QString(QChar(8211)) + "\\s]+)");
    str.replace(QRegExp("<<publisher>>"), "(.+)");
    str.replace(QRegExp("<<school>>"), "(.+)");
    str.replace(QRegExp("<<series>>"), "(.+)");
    str.replace(QRegExp("<<title>>"), "(.+)");
    str.replace(QRegExp("<<url>>"), "(\\S+)");
    str.replace(QRegExp("<<volume>>"), "(\\d+)");
    str.replace(QRegExp("<<year>>"), "(\\d\\d\\d\\d)");
    c2bUtils::debug(str);

    //  User Editting
    str = reft + "\n" + order + "\n" + str;
    c2bSaveRegExp Sdiag(str, this);
    connect(&Sdiag, SIGNAL(savePatternInfo(const QString&, const QString&)),
            this, SLOT(savePattern(const QString&, const QString&)));
    if (Sdiag.exec() == QDialog::Rejected)
        showMessage(tr("Cancelled."));
}

void cb2Bib::savePattern(const QString& rx, const QString& rxname)
{
    QString regular_expression_f = settings->fileName("cb2Bib/RegularExpressionFile");
    QFile file(regular_expression_f);
    bool fopen;
    if (file.exists())
        fopen = file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Append);
    else
        fopen = file.open(QIODevice::WriteOnly | QIODevice::Text);

    if (fopen)
    {
        QTextStream stream(&file);
        stream.setCodec("UTF-8");
        stream.setAutoDetectUnicode(true);
        stream << ("# cb2Bib " + C2B_VERSION + "  Pattern:") << endl
        << rxname << endl << rx << endl << endl;
        file.close();
        showMessage(tr("Pattern '%1' added to %2.").arg(rxname).arg(regular_expression_f));
    }
    else
        QMessageBox::warning(this, tr("Warning - cb2Bib"),
                             tr("Unable to open the file %1 for writing.\nError: '%2'.").
                             arg(regular_expression_f).arg(file.errorString()), QMessageBox::Ok);
    c2bUtils::debug("signal:");
    c2bUtils::debug(rx);
    c2bUtils::debug(rxname);
}

void cb2Bib::setArticleFileName()
{
    disconnect(ui.id, SIGNAL(textChanged(const QString&)), this, SLOT(setArticleFileName()));
    if (m_dropped_article_file.isEmpty())
        return;
    if (!settings->value("cb2Bib/MovePdf").toBool())
    {
        QUrl u(m_dropped_article_file);
        QString scheme = u.scheme();
        if (scheme == "file")
            ui.file->setText(bp->parse("file", u.toLocalFile()));
        else
            ui.file->setText(bp->parse("file", m_dropped_article_file));
        return;
    }
    QString fileName = ui.id->text();
    if (fileName.isEmpty())
        fileName = "no_cite_id";
    QFileInfo fi(m_dropped_article_file);
    QString pdfExtension = "." + fi.suffix().toLower();

    // Possible article file extensions
    if (!pdfExtension.contains(QRegExp("^\\.\\w{2,4}$")))
        pdfExtension = ".pdf";  // Default
    if (pdfExtension == ".gz")
        if (fi.completeSuffix().toLower() == ".ps.gz")
            pdfExtension = ".ps.gz";  // Composite extension
    fileName = c2bUtils::documentFileName(settings->value("cb2Bib/RelativePdfDirectory").toBool(), ui.bibtexfile->currentText(),
                                          settings->value("cb2Bib/PdfDirectory").toString(), fileName + pdfExtension);
    ui.file->setText(bp->parse("file", fileName));
    connect(ui.id, SIGNAL(textChanged(const QString&)), this, SLOT(setArticleFileName()));
}

void cb2Bib::addToFile(bool ready)
{
    if (!ready)
    {
        QMessageBox::information(this, tr("Information - cb2Bib"), tr("Not ready to add reference.\n%2").
                                 arg(c2b::network()->getfileErrorString()), QMessageBox::Ok);
        ui.addBibB->setEnabled(m_modified);
        return;
    }
    QFile file(ui.bibtexfile->currentText());
    bool fopen;
    if (file.exists())
        fopen = file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Append);
    else
        fopen = file.open(QIODevice::WriteOnly | QIODevice::Text);

    if (fopen)
    {
        QTextStream stream(&file);
        stream.setCodec("UTF-8");
        stream.setAutoDetectUnicode(true);
        stream << bp->makeBib();
        file.close();
        c2b::showMessage(tr("Bib Item %1 added to %2.").arg(ui.id->text()).arg(ui.bibtexfile->currentText()));
        if (settings->value("cb2Bib/InsertMetadata").toBool())
            if (!m_target_article_file.isEmpty())
                if (!c2b::metaPreparser()->insertMetadata(bp->currentReference(), m_target_article_file))
                    c2b::showMessage(tr("Unable to insert metadata to file %1.").arg(m_target_article_file));
        m_modified = false;
        ui.addBibB->setEnabled(m_modified);
        emit addedBibToFile(ui.bibtexfile->currentText());
    }
    else
        QMessageBox::warning(this, tr("Warning - cb2Bib"), tr("Unable to open file %1 for writing.\nError: '%2'.").
                             arg(ui.bibtexfile->currentText()).arg(file.errorString()), QMessageBox::Ok);
}

void cb2Bib::postprocessBibTeX()
{
    c2bPostprocess* ppBib = new c2bPostprocess(ui.bibtexfile->currentText(), this);
    showMessage(tr("BibTeX Postprocessing launched."));
    if (ppBib->exec() != QDialog::Accepted)
        showMessage(tr("BibTeX Postprocessing exited."));
}

void cb2Bib::bookmarksEditor()
{
    if (openDocuments.contains("bookmark_network_editor"))
    {
        c2bEditor* ed = openDocuments.value("bookmark_network_editor");
        if (ed->isMinimized())
            ed->showNormal();
        ed->raise();
        ed->activateWindow();
        return;
    }
    c2bEditor* ed = new c2bEditor(settings->fileName("cb2Bib/NetworkFile"), "NetQInfo", this);
    ed->setObjectName("bookmark_network_editor");
    openDocuments.insert("bookmark_network_editor", ed);
    ui.editBookmarksAction->setEnabled(false);
    connect(ed, SIGNAL(restartQuery()), this, SLOT(makeNetQuery()));
    connect(ed, SIGNAL(destroyed(QObject*)), this, SLOT(updateOpenDocuments(QObject*)));
    connect(ed, SIGNAL(openFile(const QString&)), this, SLOT(bibEditor(const QString&)));
    ed->show();
    showMessage(tr("Bookmark File Editor launched."));
}

void cb2Bib::abbreviationsEditor()
{
    c2bEditor* ed = new c2bEditor(settings->fileName("cb2Bib/JournalFile"), "JAbbrev", this);
    ed->setObjectName("journal_abbreviations_editor");
    openDocuments.insert("journal_abbreviations_editor", ed);
    ui.editAbbreviationsAction->setEnabled(false);
    connect(ed, SIGNAL(destroyed(QObject*)), this, SLOT(updateOpenDocuments(QObject*)));
    connect(ed, SIGNAL(openFile(const QString&)), this, SLOT(bibEditor(const QString&)));
    connect(ed, SIGNAL(openBookmarks()), this, SLOT(bookmarksEditor()));
    ed->show();
    showMessage(tr("Journal Abbreviations Editor launched."));
}

void cb2Bib::regExpsEditor()
{
    c2bEditor* ed = new c2bEditor(settings->fileName("cb2Bib/RegularExpressionFile"), "RegExp", this);
    ed->setObjectName("regular_expression_editor");
    openDocuments.insert("regular_expression_editor", ed);
    ui.editRegExpsAction->setEnabled(false);
    connect(ed, SIGNAL(restartEngine()), this, SLOT(restartEngine()));
    connect(ed, SIGNAL(destroyed(QObject*)), this, SLOT(updateOpenDocuments(QObject*)));
    connect(ed, SIGNAL(openFile(const QString&)), this, SLOT(bibEditor(const QString&)));
    connect(ed, SIGNAL(openBookmarks()), this, SLOT(bookmarksEditor()));
    ed->show();
    showMessage(tr("RegExp File Editor launched."));
}

void cb2Bib::initAsBibTeXEditor()
{
    for (int i = 0; i < settings->bibTeXFilelist.count(); ++i)
        bibEditor(settings->bibTeXFilelist.at(i));
}

void cb2Bib::bibEditor(const QString& bibtexfile, c2bBibSearcher* searcher)
{
    if (bibtexfile.isEmpty())
        return;
    if (openDocuments.contains(bibtexfile))
    {
        c2bEditor* ed = openDocuments.value(bibtexfile);
        if (ed->isMinimized())
            ed->showNormal();
        ed->raise();
        ed->activateWindow();
        return;
    }
    c2bEditor* ed = new c2bEditor(bibtexfile, searcher, this);
    ed->setObjectName(ed->editFileName());
    openDocuments.insert(ed->editFileName(), ed);
    connect(ed, SIGNAL(restartQuery()), this, SLOT(makeNetQuery()));
    connect(ed, SIGNAL(destroyed(QObject*)), this, SLOT(updateOpenDocuments(QObject*)));
    connect(ed, SIGNAL(openFile(const QString&)), this, SLOT(bibEditor(const QString&)));
    connect(ed, SIGNAL(openBookmarks()), this, SLOT(bookmarksEditor()));
    connect(ed, SIGNAL(editFileRenamed(const QString&, const QString&)), this,
            SLOT(updateOpenDocuments(const QString&, const QString&)));
    ed->show();
    showMessage(tr("BibTeX Editor launched."));
}

void cb2Bib::updateOpenDocuments(QObject* obj)
{
    openDocuments.remove(obj->objectName());

    if (!openDocuments.contains("bookmark_network_editor"))
        ui.editBookmarksAction->setEnabled(true);
    if (!openDocuments.contains("journal_abbreviations_editor"))
        ui.editAbbreviationsAction->setEnabled(true);
    if (!openDocuments.contains("regular_expression_editor"))
        ui.editRegExpsAction->setEnabled(true);

    if (settings->autostart_bibtex_editor && openDocuments.count() == 0)
        close();
}

void cb2Bib::updateOpenDocuments(const QString& old_name, const QString& new_name)
{
    c2bEditor* ed = openDocuments.take(old_name);
    if (ed)
    {
        if (ed->objectName() == "bookmark_network_editor" ||
                ed->objectName() == "journal_abbreviations_editor" ||
                ed->objectName() == "regular_expression_editor")
            return;
        ed->setObjectName(new_name);
        openDocuments.insert(new_name, ed);
        updateBibfile(new_name);
    }
}

void cb2Bib::bibModified()
{
    m_modified = true;
    ui.addBibB->setEnabled(m_modified);
}


/****************************************************************************

  NETWORK FUNCTIONALITY

*****************************************************************************/

void cb2Bib::makeNetQuery()
{
    if (!ui.makeNetworkQueryB->isEnabled())
    {
        showMessage(tr("Currently processing previous query. Resubmit later."));
        return;
    }
    QString Qtitle = bibFields["title"]->text();
    QString Qjournal = bp->fullJournal(bibFields["journal"]->text());
    QString Qvolume = bibFields["volume"]->text();
    QString Qfpage = bibFields["pages"]->text();
    Qfpage = c2bUtils::firstPage(Qfpage);
    QString Qdoi = bibFields["doi"]->text();
    QStringList hints;
    hints.append(Qdoi);
    QString Qexcerpt = bp->excerpt(ClipEdit->toPlainText(), hints);
    ui.makeNetworkQueryB->setEnabled(false);
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    netQ->submitQuery(Qtitle, Qjournal, Qvolume, Qfpage, Qdoi, Qexcerpt);
}

void cb2Bib::netQueryEnded(bool stat, const QString& pdffile, const QString& reference)
{
    if (!stat)
    {
        emit endedProcessing(false);
        c2b::showMessage(tr("Query submission error. %1").arg(netQ->errorString()));
    }
    else if (pdffile.isEmpty() && reference.isEmpty())
    {
        emit endedProcessing(false);
        c2b::showMessage(tr("Query submitted. %1").arg(netQ->errorString()));
    }
    else
    {
        QString previous_article_file = m_dropped_article_file;  // Keep dropped file in case this is auto recognized
        forceDataChange(reference);
        if (bp->isAutoRecognized() && !pdffile.isEmpty())
            fileDropped(pdffile);
        else if (bp->isAutoRecognized())
            fileDropped(previous_article_file);
    }
    QApplication::restoreOverrideCursor();
    ui.makeNetworkQueryB->setEnabled(true);
}

void cb2Bib::processingEnded()
{
    if (settings->value("cb2Bib/AutomaticQuery").toBool()
            && ui.makeNetworkQueryB->isEnabled()
            && !bp->c2bAutoRecogStr.contains("BibTeX"))
    {
        bp->c2bAutoRecogStr = tr("Starting network query, using '%1'").arg(bp->c2bAutoRecogStr);
        makeNetQuery();
        return;
    }
    if (pdfImport)
        fileDropped(pdfImport->processedFile);
    emit endedProcessing(bp->isAutoRecognized());
}

void cb2Bib::linkClicked(const QUrl& ln)
{
    c2b::openFile(c2b::encodeLink(ln.toString()));
}

void cb2Bib::openFileDocument()
{
    if (!m_dropped_article_file.isEmpty())
        c2b::openFile(m_dropped_article_file);
}

void cb2Bib::selectFile()
{
    QString fn = c2bFileDialog::getOpenFileName(this, "", settings->fileName("cb2Bib/LastDroppedFile"),
                 "Portable Document Format (*.pdf);;DjVu (*.djvu);;Post Script (*.ps);;TeX (*.tex);;All (*)");
    if (!fn.isEmpty())
        fileDroppedByUser(fn);
}
