/***************************************************************************
 *   Copyright (C) 2004-2008 by Pere Constans
 *   constans@molspaces.com
 *   cb2Bib version 1.0.4. Licensed under the GNU GPL version 3.
 *   See the LICENSE file that comes with this distribution.
 ***************************************************************************/
#include "journalDB.h"

#include <QFile>
#include <QObject>
#include <QtDebug>


journalDB::journalDB(const QString &dbfile)
{
    if (dbfile.isEmpty())
    {
        qDebug(QObject::tr("[cb2Bib] No journal file especified.").toLatin1());
        return;
    }
    nitems = 0;
    QFile file(dbfile);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
    {
        qDebug(QObject::tr("[cb2Bib] Could not open %1 journal file for reading.").arg(dbfile).toLatin1());
        return;
    }
    QTextStream stream(&file);
    stream.setCodec("UTF-8");
    stream.setAutoDetectUnicode(true);
    QString line;

    while (!stream.atEnd())
    {
        line = stream.readLine();
        if (!(line.isEmpty() || line.contains(QRegExp("^#"))))
        {
            QStringList spLine = line.split("|");
            nitems++;
            JCode += spLine.at(0);
            QString dum = spLine.at(1);
            JAbbrev += dum;
            dum.replace(QRegExp("\\W"), " ");     //  Keeps word structure
            JAbbrev_simp_w += dum.simplified();
            dum.replace(QRegExp("\\W"), "");      //  Removes whitespaces also
            JAbbrev_simp += dum;
            dum = spLine.at(2);
            JExtended += dum;
            dum.replace(QRegExp("\\W"), "");      //  Removes whitespaces also
            JExtended_simp += dum;
        }
    }
    file.close();
}


journalDB::~journalDB()
{
    JCode.clear();
    JAbbrev.clear();
    JExtended.clear();
    JAbbrev_simp.clear();
    JExtended_simp.clear();
}

/** \page journalproc Processing of journal names

     cb2Bib processes journal names according to its editable database,
     stored at <tt>abbreviations.txt</tt>. See \ref c2bconf_files and
     \ref c2bconf_bibtex.

     Journal names processing is performed in the following situations
     whenever a string is recognized as 'journal'. Additionally,
     it is also processed by pressing <b>Intro Key</b> at the journal
     edit line.

     <p>&nbsp;</p>

            - Retrieves Journal name, <b>abbreviated form</b>, if found.

            - If Journal name is not found in the database, returns
            input Journal name.

            - Search is case unsensitive.

            - Warning: Journal codes can be duplicated. If duplicated,
            returns input Journal name.

*/
QString journalDB::retrieve(QString JQuery)
{
    QString query = JQuery;
    query.replace(QRegExp("\\W"), "");      //  Removes whitespaces also
    QRegExp JQrx = QRegExp(QString("^%1$").arg(query), Qt::CaseInsensitive);
    int jfound = 0;
    int foundat = -1;

    for (int i = 0; i < nitems; i++)
    {
        if (JCode.at(i).contains(JQrx))
        {
            jfound++;
            foundat = i;
        }
        if (JAbbrev_simp.at(i).contains(JQrx))
            if (++jfound == 1)
                return(JAbbrev.at(i));
        if (JExtended_simp.at(i).contains(JQrx))
            if (++jfound == 1)
                return(JAbbrev.at(i));
    }
    if (jfound == 1 && foundat != -1)
        return(JAbbrev.at(foundat));
    else
        return(JQuery);
}

/** \page journalproc Processing of journal names
    <p>&nbsp;</p>

    - Retrieves Journal name, <b>full form</b>, if found.

    - If Journal name is not found in the database, returns
    input Journal name.

    - Search is case unsensitive.

    - Warning: Journal codes can be duplicated. If duplicated,
    returns input Journal name.

*/
QString journalDB::retrieveFull(const QString& JQuery)
{
    QString query = JQuery;
    query.replace(QRegExp("\\W"), "");      //  Removes whitespaces also
    QRegExp JQrx = QRegExp(QString("^%1$").arg(query), Qt::CaseInsensitive);
    int jfound = 0;
    int foundat = -1;

    for (int i = 0; i < nitems; i++)
    {
        if (JCode.at(i).contains(JQrx))
        {
            jfound++;
            foundat = i;
        }
        if (JAbbrev_simp.at(i).contains(JQrx))
            if (++jfound == 1)
                return(JExtended.at(i));
        if (JExtended_simp.at(i).contains(JQrx))
            if (++jfound == 1)
                return(JExtended.at(i));
    }
    if (jfound == 1 && foundat != -1)
        return(JExtended.at(foundat));
    else
        return(JQuery);
}
