//
// C++ Implementation: graph3d
//
// Description:
//
//
// Author: Mike Arrison <arrison@graphcalc.com>, (C) 2003
// Major Contributor: Charlie Hall
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include <iostream>
#include "expression.h"
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qprogressdialog.h>
#include <qspinbox.h>
#include "graph3d.h"
using namespace std;

graph3d::graph3d( QWidget* parent, const char* name )
 : QGLWidget( parent, name )
{
	OptionsWindow = new graph3doptions;
	xRot = 0.0;
	yRot = 0.0;
	zRot = 0.0;			//If these guys ever change... change them in zoomStandard() too
	xTran = 0.0;
	yTran = 0.0;
	zTran = -60.0;
	xCurrent = 0;
	yCurrent = 0;
	xPressed = 0;
	yPressed = 0;
	mouse_mode = false;  //0 for Rotation, 1 for translation
	modifier_shift = false;
	modifier_control = false;

	resetInputs();
	refresh_needed = true;

	m_GLlist[0] = 1;
	m_GLlist[1] = 2;
	m_GLlist[2] = 3;
	m_GLlist[3] = 4;
	m_GLlist[4] = 5;
	m_GLlist[5] = 6;

}


graph3d::~graph3d()
{
	makeCurrent();
}

void graph3d::initializeGL()
{
	//glViewport(0,0,w,h);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	glDrawBuffer(GL_BACK);
	glEnable(GL_DEPTH_TEST);

	glShadeModel(GL_SMOOTH);
	glClearDepth(1.0f);

	//glEnable(GL_LINE_SMOOTH);
	//glEnable(GL_BLEND);
	////glBlendFunc(GL_DST_COLOR,GL_ZERO);
	//glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
	//glHint(GL_LINE_SMOOTH_HINT,GL_NICEST);
	//cout << "initializeGL:" << flush;
}

void graph3d::resetInputs()
{
	expression x_min(OptionsWindow->input_x_minimum->text());
	expression x_max(OptionsWindow->input_x_maximum->text());
	expression y_min(OptionsWindow->input_y_minimum->text());
	expression y_max(OptionsWindow->input_y_maximum->text());
	m_x_min = x_min.evaluate_num(0);
	m_x_max = x_max.evaluate_num(0);
	m_y_min = y_min.evaluate_num(0);
	m_y_max = y_max.evaluate_num(0);
	m_x_interval = OptionsWindow->input_x_intervals->value();//(int)x_int.evaluate_num(0);
	m_y_interval = OptionsWindow->input_y_intervals->value();//(int)y_int.evaluate_num(0);
}

void graph3d::reGraph()
{
	resetInputs();
	refresh_needed=true;
	updateGL();
}

void graph3d::paintGL()
{
	glClear(GL_COLOR_BUFFER_BIT);
	glClear(GL_DEPTH_BUFFER_BIT);

	glClearColor(255,
				 255,
				 255,
				 1.0); // sets the background color

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
	glPushMatrix();

	glLoadIdentity();
	glTranslatef( xTran, yTran, zTran);
	glRotatef( xRot, 1.0, 0.0, 0.0 );
	glRotatef( yRot, 0.0, 1.0, 0.0 );
	glRotatef( zRot, 0.0, 0.0, 1.0 );

	GLuint axislist=7;
	GLuint ticklist=8;
	GLuint labelslist=9;

	if (refresh_needed) {
		//cout << "refreshing:" << flush;
		/* Bad Progress Bar code
		int max_cycles = 0, current_cycle = 0;
		for (int graph_num = 0; graph_num < 6; ++graph_num) {
			if (OptionsWindow->graphchecked[graph_num]->isChecked()) {
				max_cycles += m_x_interval;
			}
		}
		QProgressDialog progress ("Updating Graph...", "Cancel", max_cycles, this, "progress", TRUE);
		*/
		glNewList(axislist,GL_COMPILE);
			glLineWidth(GLfloat(OptionsWindow->input_axes_width->value()));
			glBegin(GL_LINES);
				glColor3f(1.0, 0.0, 0.0);			// Un-hardcode these next 9 lines
				glVertex3f(-10.0, 0.0, 0.0);	// Colors and position should be configurable
				glVertex3f(10.0, 0.0, 0.0);		// Z values should be based on min and max function results
				glColor3f(0.0, 1.0, 0.0);			// So this gllist should be after the equations
				glVertex3f(0.0, -10.0, 0.0);
				glVertex3f(0.0, 10.0, 0.0);
				glColor3f(0.0, 0.0, 1.0);
				glVertex3f(0.0, 0.0, -10.0);
				glVertex3f(0.0, 0.0, 10.0);
			glEnd();
		glEndList();
		refresh_needed = false;
		double x_step((m_x_max-m_x_min)/m_x_interval), y_step((m_y_max-m_y_min)/m_y_interval);
		int x_count, y_count;
		double x, y;
		for (int graph_num = 0; graph_num < 6; ++graph_num) {
			if (OptionsWindow->graphchecked[graph_num]->isChecked()) {
				//cout << "recomputing" <<graph_num<<":"<< flush;
				expression graph3d_expression(OptionsWindow->graphequations[graph_num]->text());
				glNewList(m_GLlist[graph_num],GL_COMPILE);
				glLineWidth(GLfloat(OptionsWindow->input_wire_width->value()));
				//glLineWidth(2);
				for (x = m_x_min, x_count = 0; x_count <= m_x_interval; ++x_count, x+=x_step){
					glBegin(GL_LINE_STRIP);
					glColor3f(0,0,0);
					for (y = m_y_min, y_count = 0; y_count <= m_y_interval; ++y_count, y+=y_step){
						glVertex3d( x, y, graph3d_expression.evaluate_num(x,y));
					}
					glEnd();
				}
				for (y = m_y_min, y_count = 0; y_count <= m_y_interval; ++y_count, y+=y_step){
					glBegin(GL_LINE_STRIP);
					glColor3f(0,0,0);
					for (x = m_x_min, x_count = 0; x_count <= m_x_interval; ++x_count, x+=x_step){
						glVertex3d( x, y, graph3d_expression.evaluate_num(x,y));
					}
					glEnd();
				}
				glEndList();
			} //end of IF CHECKED
		} // end of FOR EACH FUNCTION loop
	} // end of IF REFRESH_NEEDED
	if (OptionsWindow->input_axes->isChecked())
	{
		glCallList(axislist);
		//cout << "axes:" << flush;
	}
	if (OptionsWindow->input_ticks->isChecked())
	{
		glCallList(ticklist);
		//cout << "ticks:" << flush;
	}
	if (OptionsWindow->input_axes_labels->isChecked())
	{
		glCallList(labelslist);
		//cout << "axeslabels:" << flush;
	}
	for (int graph_num = 0; graph_num < 6; ++graph_num) {
		if (OptionsWindow->graphchecked[graph_num]->isChecked()) {
			glCallList(m_GLlist[graph_num]);
			//cout << "graph" <<m_GLlist[graph_num] <<":"<< flush;
		}
	}
	//cout << "Function computed:" << flush;
}

void graph3d::resizeGL( int width, int height )
{
	GLfloat w = 0.0;
	GLfloat h = 0.0;
	if(width > height)
	{
		w = (GLfloat)width / (GLfloat)height;
		h = 1.0;
	}
	else
	{
		h = (GLfloat)height / (GLfloat)width;
		w = 1.0;
	}
	glViewport(0, 0, (GLint)width, (GLint)height);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glFrustum(-w, w, -h, h, 5.0, 1500.0);
	glMatrixMode(GL_MODELVIEW);
}

void graph3d::setXRotation( int degrees )
{
	xRot = (GLfloat)(degrees % 360);
	//refresh_needed = true;
}

void graph3d::setYRotation( int degrees )
{
	yRot = (GLfloat)(degrees % 360);
	//refresh_needed = true;
}

void graph3d::setZRotation( int degrees )
{
	zRot = (GLfloat)(degrees % 360);
	//refresh_needed = true;
}

void graph3d::mousePressEvent(QMouseEvent* me)
{
	xPressed = me->x();
	yPressed = me->y();
	if (me->button() == LeftButton) { mouse_mode = false; } //Translation
	else if (me->button() == MidButton) { mouse_mode = true; } //Rotation
	else if (me->button() == RightButton) {
		//popup options
	}
}

void graph3d::mouseMoveEvent(QMouseEvent* me)
{
	xCurrent = me->x();
	yCurrent = me->y();

	if(mouse_mode) { //Translation
		xTran += (GLfloat)(xCurrent - xPressed)/4.0;
		yTran -= (GLfloat)(yCurrent - yPressed)/4.0;
		updateGL();
	} else { //Rotation
		if(modifier_control){
			if (modifier_shift) {
				setZRotation((int)(zRot +((xCurrent-xPressed)*0.1)));
			} else {
				setYRotation((int)(yRot +((xCurrent-xPressed)*0.1)));
				setXRotation((int)(xRot +((yCurrent-yPressed)*0.1)));
			}
		} else {
			if (modifier_shift) {
				setZRotation((int)zRot + xCurrent-xPressed);
			} else {
				setYRotation((int)yRot + xCurrent-xPressed);
				setXRotation((int)xRot + yCurrent-yPressed);
			}
		}
		updateGL();
	}

	xPressed = xCurrent;
	yPressed = yCurrent;
}

void graph3d::zoomIn()
{
	zTran += 10;
	updateGL();
}

void graph3d::zoomOut()
{
  zTran -= 10;
	updateGL();
}

void graph3d::zoomPrevious()
{
	updateGL();
}

void graph3d::zoomStandard()
{
	xRot = 0.0;
	yRot = 0.0;
	zRot = 0.0;			//If these guys ever change... change them in the constructor too
	xTran = 0.0;
	yTran = 0.0;
	zTran = -60.0;
	glRotatef( xRot, 1.0, 0.0, 0.0 );
	glRotatef( yRot, 0.0, 1.0, 0.0 );
	glRotatef( zRot, 0.0, 0.0, 1.0 );
	updateGL();
}

void graph3d::wheelEvent(QWheelEvent* we)
{
	zTran += we->delta()/100;
	updateGL();
}

void graph3d::keyPressEvent(QKeyEvent* ke)
{
	switch(ke->key())
	{
	case Qt::Key_Up:
		zTran += 5.0;
		updateGL();
		break;
	case Qt::Key_Down:
		zTran -= 5.0;
		updateGL();
		break;
	case Qt::Key_Left:
		break;
	case Qt::Key_Right:
		break;
	case Qt::Key_Shift:
		modifier_shift = true;
	case Qt::Key_Control:
		modifier_control = true;
	default:
		ke->ignore();
		break;
	}
}

void graph3d::keyReleaseEvent(QKeyEvent* ke)
{
	switch(ke->key())
	{
	case Qt::Key_Shift:
		modifier_shift = false;
	case Qt::Key_Control:
		modifier_control = false;
	default:
		ke->ignore();
		break;
	}
}
