/***************************************************************************
 *   Copyright (C) 2005 by Emiliano Gulmini   *
 *   emi_barbarossa@yahoo.it   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
//QT Headers
#include <qstring.h>
#include <qstringlist.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qdir.h>
#include <qregexp.h>

//KDE Headers
#include <kprocess.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>

//Local Headers
#include "pythoncodegenerator.h"
#include "bccodegenerator.h"
#include "codegenerator.h"
#include "configuration.h"
#include "kalcoolusrc.h"
#include "service.h"

QString CodeGenerator::Mode = "BC_MODE";

using namespace Global;

CodeGenerator::CodeGenerator(const QString& text,
                             const LanguagesMap& languages,
                             const ConstantsMap& constants,
                             Configuration* configuration)
{
  m_displayText = text;
  m_languagesMap = languages;
  m_constantsMap = constants;
  m_bcScriptFilePath = locateLocal("data", "kalcoolus/program_bc");
  m_pythonScriptFilePath = locateLocal("data", "kalcoolus/program_python.py");
  m_configuration = configuration;
  m_numberOfBCObjects = 0;
  m_bcCodeGenerator = new BCCodeGenerator(languages["bc"], constants);
  m_pythonCodeGenerator = new PythonCodeGenerator(languages["python"], constants);
}

CodeGenerator::~CodeGenerator()
{
  if(m_bcCodeGenerator) delete m_bcCodeGenerator;
  if(m_pythonCodeGenerator) delete m_pythonCodeGenerator;
}

QString CodeGenerator::executeProgram(bool b)
{
  QString result;

  if(b) executeProgramPython(result);
  else executeProgramBC(result);

  return result;
}

/*******************************************************************/
/************************ PRIVATE SLOTS ****************************/
/*******************************************************************/
/*!
    \fn CodeGenerator::slotGetScriptError(KProcess* proc, char* s, int i)
 */
void CodeGenerator::slotGetScriptError(KProcess* proc, char* s, int i)
{
  if(Mode == "PYTHON_MODE")
    getScriptErrorPython(proc,s,i);

  if(Mode == "BC_MODE")
    getScriptErrorBC(proc,s,i);
}

/*!
    \fn CodeGenerator::slotGetScriptOutput(KProcess* proc, char* s, int i)
 */
void CodeGenerator::slotGetScriptOutput(KProcess* proc, char* s, int i)
{
  if(Mode == "PYTHON_MODE")
    getScriptOutputPython(proc,s,i);

  if(Mode == "BC_MODE")
    getScriptOutputBC(proc,s,i);
}

/*!
    \fn CodeGenerator::slotProcessExited(KProcess* proc)
 */
void CodeGenerator::slotProcessExited(KProcess* proc)
{
  Q_UNUSED(proc);
}

/*******************************************************************/
/************************ PRIVATE METHODS **************************/
/*******************************************************************/

/*!
    \fn CodeGenerator::generateProgramBC()
 */
void CodeGenerator::generateProgramBC()
{
  // Parsing the input display content 
  m_displayText.remove(" ");
  QStringList instructionList = QStringList::split("\n",m_displayText);
  uint j = 0,
       c = instructionList.count();
  while(j < c)
    {
      QString instruction = instructionList[j];
      m_bcCodeGenerator->convertToBC(instruction);
      instruction.remove("#");
      if(not instruction.isEmpty()) m_program += instruction + "\n";
      j++;
    }

  m_program = Resource::gpl +
              "scale=" + m_configuration->scale() + ";\n" +
              "obase=" + m_configuration->obase() + ";\n" +
              "ibase=" + m_configuration->ibase() + ";\n" + builtinBCFunctions() + "## MAIN ##\n" + m_program + "\nhalt;";
}

/*!
    \fn CodeGenerator::executeProgramBC(QString& result)
 */
void CodeGenerator::executeProgramBC(QString& result)
{
  // Marking bc mode
  Mode = "BC_MODE";
  // Creating BC program
  generateProgramBC();
  // Creating the command line to be executed 
  QFile generatedBCSourceFile(m_bcScriptFilePath);
  if(not generatedBCSourceFile.open(IO_WriteOnly | IO_Truncate)) KMessageBox::error(0,"Cannot execute bc instructions. It may be that your permissions are wrong");
  else
    {
      KProcess* proc = new KProcess();

      //proc->setEnvironment("BC_LINE_LENGTH",m_configuration->lineLength());
      proc->setWorkingDirectory(QDir::currentDirPath());
      proc->setUseShell(true);

      QTextStream stream(&generatedBCSourceFile);
      stream << m_program + "\n";
      generatedBCSourceFile.close();

      m_program = QString::null;

      *proc << m_configuration->bcPath();
      *proc << KProcess::quote("-l") << KProcess::quote("-q") << KProcess::quote(m_bcScriptFilePath);

      connectKProcess(proc);

      /* Through slotGetScriptOutput, m_resultBuffer contains
       * the result of the KProcess call 
       */
      if(not proc->start(KProcess::Block, KProcess::All))
        {
          KMessageBox::error(0,"Error: cannot executes bc code");
          m_resultBuffer = "Error: cannot executes bc code";
        }

      if(proc) delete proc;

      m_bcCodeGenerator->formatResultString(result, m_resultBuffer);
    }
}

/*!
    \fn CodeGenerator::getScriptErrorBC(KProcess* proc, char* s, int i)
 */
void CodeGenerator::getScriptErrorBC(KProcess* proc, char* s, int i)
{
  Q_UNUSED(proc);

  QCString temp(s,i+1);

  if(temp.isEmpty() or temp == "\n") return;

  m_resultBuffer += temp + "\n";
}

/*!
    \fn CodeGenerator::getScriptOutBC(KProcess* proc, char* s, int i)
 */
void CodeGenerator::getScriptOutputBC(KProcess* proc, char* s, int i)
{
  Q_UNUSED(proc);
  QCString temp(s,i+1);

  if(temp.isEmpty() or temp == "\n") return;

  m_resultBuffer += temp+"\n";
}

/*!
    \fn CodeGenerator::builtinBCFunctions()
 */
QString CodeGenerator::builtinBCFunctions()
{
  // Getting predefined functions from kalcoolusrc.h
  QString defaultBCFunctions = Resource::BCDefaultFunctions;
  // Getting user-defined functions 
  FunctionsCodeMap::Iterator it;
  for(it = m_languagesMap["bc"].begin(); it not_eq m_languagesMap["bc"].end(); ++it)
    defaultBCFunctions += it.data()+ "\n";

  return defaultBCFunctions;
}

/*!
    \fn CodeGenerator::generateProgramPython()
 */
void CodeGenerator::generateProgramPython()
{
  // Parsing the input display content
  m_displayText.remove(" ");
  QStringList instructionList = QStringList::split("\n",m_displayText);
  QString dec = m_configuration->pythonDecimalDigits();
  QString printFun = "def print_result (x): \n"
                     "\tresult=complex(x)\n"
                     "\tif (result-result.real) == 0j:\n"
                     "\t\tprint ('%"+dec+"."+dec+"f'%(result.real))\n"
                     "\telse:\n"
                     "\t\tif (((result)-result.real)*-1j).real < 0:\n"
                     "\t\t\tprint ('%"+dec+"."+dec+"f-i%"+dec+"."+dec+"f'%(result.real,abs(result-result.real)))\n"
                     "\t\telse:\n"
                     "\t\t\tprint ('%"+dec+"."+dec+"f+i%"+dec+"."+dec+"f'%(result.real,abs(result-result.real)))\n\n";

  m_program += printFun + "## MAIN ##\n";

  uint j = 0,
       c = instructionList.count();
  while(j < c)
    {
      QString instruction = instructionList[j];
      m_pythonCodeGenerator->convertToPython(instruction);

      instruction = "print_result(" + instruction + ")\n";

      if(not instruction.isEmpty()) m_program += instruction;
      j++;
    }
  QString importedFunctions = Resource::gpl +
                              "import math\n"
                              "import cmath\n"
                              "import random\n";
  importedFunctions += builtinPythonFunctions();
  /* Number of generated code lines. If we get an error this varriable is
   * used to calculate the relative input editor line number that caused 
   * the error.
   */
  m_pythonProgramLinesNumber = importedFunctions.contains("\n") + printFun.contains("\n") + 1;
  m_program = importedFunctions + m_program;
}

/*!
    \fn CodeGenerator::executeProgramPython(QString& result)
 */
void CodeGenerator::executeProgramPython(QString& result)
{
  Mode = "PYTHON_MODE";
  // Creating Python program
  generateProgramPython();
  // Creating the command line to be executed 
  QFile generatedPythonSourceFile(m_pythonScriptFilePath);
  if(not generatedPythonSourceFile.open(IO_WriteOnly | IO_Truncate)) KMessageBox::error(0,"Cannot execute python instructions, control your permissions");
  else
    {
      KProcess* proc = new KProcess();
      QString pyPath = m_configuration->pythonModulesPath();
      if(not pyPath.isEmpty())
        proc->setEnvironment("PYTHONPATH",pyPath);
      proc->setWorkingDirectory(QDir::currentDirPath());
      proc->setUseShell(true);

      QTextStream stream(&generatedPythonSourceFile);
                  stream << m_program + "\n";
      generatedPythonSourceFile.close();

      QString script = "python " + m_pythonScriptFilePath;
      *(proc) << script;

      connectKProcess(proc);

      /* Through slotGetScriptOutput, m_resultBuffer contains
       * the result of the KProcess call 
       */
      if(not proc->start(KProcess::Block, KProcess::All))
        {
          KMessageBox::error(0,"Error: cannot executes Python code");
          m_resultBuffer = "Error: cannot executes Python code";
        }

      if(proc) delete proc;
      m_pythonCodeGenerator->formatResultString(result, m_resultBuffer, m_pythonProgramLinesNumber);
    }
}

/*!
    \fn CodeGenerator::getScriptErrorPython(KProcess* proc, char* s, int i)
 */
void CodeGenerator::getScriptErrorPython(KProcess* proc, char* s, int i)
{
  Q_UNUSED(proc);

  QCString temp(s,i+1);
  // if line is empty then exit
  if(temp.isEmpty() or temp == "\n") return;
  // removing spourious new-lines
  m_resultBuffer.remove("\n");

  m_resultBuffer += temp;
}

/*!
    \fn CodeGenerator::getScriptOutputPython(KProcess* proc, char* s, int i)
 */
void CodeGenerator::getScriptOutputPython(KProcess* proc, char* s, int i)
{
  Q_UNUSED(proc);
  QCString temp(s,i+1);
  if(temp.isEmpty() or temp == "\n") return;

  m_resultBuffer += temp;
}

/*!
    \fn CodeGenerator::builtinPythonFunctions()
 */
QString CodeGenerator::builtinPythonFunctions()
{
  // Getting predefined functions from kalcoolusrc.h
  QString defaultPythonFunctions = Resource::PythonDefaultFunctions;
  // Getting user-defined functions
  FunctionsCodeMap::Iterator it;
  for(it = m_languagesMap["python"].begin(); it not_eq m_languagesMap["python"].end(); ++it)
    defaultPythonFunctions += it.data()+ "\n";

  return defaultPythonFunctions;
}

/*!
   \fn CodeGenerator::connectKProcess(KProcess* proc);
*/
void CodeGenerator::connectKProcess(KProcess* proc)
{
  connect(proc, SIGNAL(receivedStdout(KProcess*,char*,int)), this, SLOT(slotGetScriptOutput(KProcess*,char*,int)));
  connect(proc, SIGNAL(receivedStderr(KProcess*,char*,int)), this, SLOT(slotGetScriptError(KProcess*,char*,int)));
  connect(proc, SIGNAL(processExited(KProcess*)), this, SLOT(slotProcessExited(KProcess*)));
}
#include "codegenerator.moc"

