/***************************************************************************
 *   Copyright (C) 2005 by Emiliano Gulmini   *
 *   emi_barbarossa@yahoo.it   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
//QT Headers
#include <qfile.h>
#include <qtextstream.h>
#include <qstring.h>
#include <qdom.h>

//KDE Headers
#include <kmessagebox.h>
#include <klistview.h>
#include <kconfig.h>
#include <kpushbutton.h>
#include <klineedit.h>
#include <kcombobox.h>
#include <kstandarddirs.h>
#include <ktextedit.h>
//Local Headers
#include "service.h"
#include "constantsdb.h"

ConstantsDB::ConstantsDB(KConfig* config, KListView* list, KLineEdit* name, KLineEdit* value, KPushButton* add, KPushButton* del, KComboBox* cBox, KTextEdit* display) : QObject()
{
  m_config = config;
  m_cbxConstantsBox = cBox;
  m_constantsListview = list;
  m_leConstName = name;
  m_leConstValue = value;
  m_bConstAdd = add;
  m_bConstClear = del;
  m_inputDisplay = display;

  m_constantsDatabasePath = locateLocal("data", "kalcoolus/constantsDB.xml");

  connect(m_bConstAdd, SIGNAL(clicked()), this, SLOT(slotAddConstant()));
  connect(m_bConstClear, SIGNAL(clicked()), this, SLOT(slotClearConstant()));
  connect(m_constantsListview, SIGNAL(clicked(QListViewItem*)), this, SLOT(slotConstSelected(QListViewItem*)));
  connect(m_cbxConstantsBox, SIGNAL(activated(const QString &)), this, SLOT(slotConstantsBox(const QString &)));
}

ConstantsDB::~ConstantsDB()
{
  m_config = 0;
  m_constantsListview = 0;
  m_cbxConstantsBox = 0;
  m_leConstName = 0;
  m_leConstValue = 0;
  m_bConstAdd = 0;
  m_bConstClear = 0;
  m_inputDisplay = 0;
}

/*!
    \fn ConstantsDB::slotAddConstant()
 */
void ConstantsDB::slotAddConstant()
{
  QString name = m_leConstName->text(),
          value = m_leConstValue->text();
  if(not name.isEmpty() and not value.isEmpty())
    {
      QListViewItem* item = new KListViewItem(m_constantsListview);
      item->setText(0, name);
      item->setText(1, value);
      m_constantsMap[name] = value;
      m_leConstName->clear();
      m_leConstValue->clear();
    }
  loadConstantsIntoBox();
}

/*!
    \fn ConstantsDB::slotClearConstant()
 */
void ConstantsDB::slotClearConstant()
{
  int i;
  for(i = 0; i < m_cbxConstantsBox->count(); i++)
    if(m_cbxConstantsBox->text(i) == m_leConstName->text())
      {
        m_cbxConstantsBox->removeItem(i);
        break;
      }

  m_leConstName->clear();
  m_leConstValue->clear();
}

/*!
    \fn ConstantsDB::slotConstSelected(QListViewItem*)
 */
void ConstantsDB::slotConstSelected(QListViewItem* item)
{
  /* When a constant is selected in the constants page it is removed from
     memory until we have finished to reedit its value
  */
  if(item)
    {
      m_leConstName->setText(item->text(0));
      m_leConstValue->setText(item->text(1));
      m_constantsMap.remove(item->text(0));
      delete item;
    }
}

/*!
    \fn ConstantsDB::slotConstantsBox(const QString &)
 */
void ConstantsDB::slotConstantsBox(const QString & s)
{
  Service::insertStringIntoEditor(m_inputDisplay, s, 0, s.length());
}

/*!
    \fn ConstantsDB::loadDatabase()
 */
void ConstantsDB::loadDatabase()
{
  QDomDocument dbDocument;
    QFile constantsDataBaseFile(m_constantsDatabasePath);

    if(not constantsDataBaseFile.open(IO_ReadOnly)) return;

    if(not dbDocument.setContent(&constantsDataBaseFile))
      {
        constantsDataBaseFile.close();
        return;
      }
    constantsDataBaseFile.close();

    // print out the element names of all elements that are direct children
    // of the outermost element.
    QDomElement docElem = dbDocument.documentElement();

    QDomNode n = docElem.firstChild();
    while(not n.isNull())
      {
        QDomElement e = n.toElement(); // try to convert the node to an element.
        if(not e.isNull())
          {
            QListViewItem* tempItem = new QListViewItem(m_constantsListview);
            tempItem->setText(0, e.tagName());
            tempItem->setText(1, e.attribute("value"));
          }
        n = n.nextSibling();
      }
  loadConstantsIntoBox();
}

/*!
    \fn ConstantsDB::writeToDatabase()()
 */
void ConstantsDB::writeToDatabase()
{
  QDomDocument dbDocument;
  dbDocument.appendChild(dbDocument.createElement("constants"));

  QListViewItem* tempItem = m_constantsListview->firstChild();
  while(tempItem)
    {
      QDomElement tempElement = dbDocument.createElement(tempItem->text(0));

      tempElement.setAttribute("value", tempItem->text(1));

      dbDocument.firstChild().appendChild(tempElement);
      tempItem = tempItem->nextSibling();
    }

  QFile constantsDataBaseFile(m_constantsDatabasePath);
  if(not constantsDataBaseFile.open(IO_WriteOnly))
    KMessageBox::error(0,"Cannot store constants into database");
  else
    {
      QTextStream constantsDataBaseStream(&constantsDataBaseFile);
      constantsDataBaseStream << "<?xml version=\"1.0\" ?>\n";
      constantsDataBaseStream << dbDocument.toString(2);
      constantsDataBaseFile.close();
    }
}

/*!
    \fn ConstantsDB::loadConstantsIntoBox()
 */
void ConstantsDB::loadConstantsIntoBox()
{
  // Adding constants to the user-defined list 
  m_cbxConstantsBox->clear();
  QListViewItem* tempItem = m_constantsListview->firstChild();
  while(tempItem)
    {
      m_cbxConstantsBox->insertItem(tempItem->text(0));
      m_constantsMap[tempItem->text(0)] = tempItem->text(1);
      tempItem = tempItem->nextSibling();
    }
}

#include "constantsdb.moc"


