/***************************************************************************
 *   Copyright (C) 2005 by Emiliano Gulmini   *
 *   emi_barbarossa@yahoo.it   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

//QT Headers
#include <qobject.h>
#include <qdom.h>
#include <qmap.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qregexp.h>

//KDE Headers
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <kcombobox.h>
#include <kpushbutton.h>
#include <klineedit.h>
#include <ktextedit.h>

//Local Headers
#include "kalcoolusrc.h"
#include "functionsdb.h"
#include "service.h"
#include "bccodegenerator.h"
#include "pythoncodegenerator.h"

using namespace Global;

FunctionsDB::FunctionsDB(KConfig* config, KPushButton* complex, KTextEdit* d, KLineEdit* name, KLineEdit* arguments, KTextEdit* body, KComboBox* list, KComboBox* fBox, KPushButton* add, KPushButton* del) : QObject()
{
  m_config = config;
  m_bComplex = complex;
  m_inputDisplay = d;
  m_functionsDatabasePath = locateLocal("data","kalcoolus/functionsDB.xml");

  m_leUserFunctionName = name;
  m_leUserFunctionArguments = arguments;
  m_editorUserFunctionBody = body;
  m_cbxUserFunctionsList = list;
  m_cbxFunctionsBox = fBox;
  m_bAddUserFunction = add;
  m_bDeleteUserFunction = del;

  connect(m_bAddUserFunction, SIGNAL(clicked()), this, SLOT(slotAddUserFunction()));
  connect(m_bDeleteUserFunction, SIGNAL(clicked()), this, SLOT(slotDeleteUserFunction()));
  connect(m_cbxUserFunctionsList, SIGNAL(activated(int)), this, SLOT(slotEditUserFunction(int)));
  connect(m_cbxFunctionsBox, SIGNAL(activated(const QString &)), this, SLOT(slotSelectUserFunction(const QString&)));
}

FunctionsDB::~FunctionsDB()
{
  m_leUserFunctionName = 0;
  m_leUserFunctionArguments = 0;
  m_editorUserFunctionBody = 0;
  m_cbxUserFunctionsList = 0;
  m_cbxFunctionsBox = 0;
  m_bAddUserFunction = 0;
  m_bDeleteUserFunction = 0;
  m_config = 0;
  m_bComplex = 0;
  m_inputDisplay = 0;
  m_bc = 0;
  m_python = 0;
}

/*!
    \fn FunctionsDB::loadDatabase()
 */
void FunctionsDB::loadDatabase()
{
  QDomDocument dbDocument;
  QFile functionsDataBaseFile(m_functionsDatabasePath);
  if(not functionsDataBaseFile.open(IO_ReadOnly)) return ;
  if(not dbDocument.setContent(&functionsDataBaseFile))
    {
      functionsDataBaseFile.close();
      return ;
    }
  functionsDataBaseFile.close();

  getCode(dbDocument);

  loadFunctionsIntoBoxes();
}

/*!
    \fn FunctionsDB::loadFunctionsIntoBoxes()
 */
void FunctionsDB::loadFunctionsIntoBoxes()
{
  // Adding user-defined function name to the list 
  m_cbxUserFunctionsList->clear();
  m_cbxFunctionsBox->clear();

  FunctionsCodeMap fcMap;
  QRegExp r;
  if(m_bComplex->isOn())
    {
      r = QRegExp(Resource::pythonFunctionNamePattern);
      fcMap = m_languagesMap["python"];
    }
  else
    {
      r = QRegExp(Resource::bcFunctionNamePattern);
      fcMap = m_languagesMap["bc"];
    }
  FunctionsCodeMap::Iterator it;
  for(it = fcMap.begin(); it not_eq fcMap.end(); ++it)
    {
      r.search((*it).data());
      QString functionName = r.cap(1);
      m_cbxUserFunctionsList->insertItem(functionName);
      m_cbxFunctionsBox->insertItem(functionName);
    }
}

/*!
    \fn FunctionsDB::getCode(const QDomDocument& db)
 */
void FunctionsDB::getCode(const QDomDocument& db)
{
  // print out the element names of all elements that are direct children
  // of the outermost element.
  QDomElement docElem = db.documentElement();

  QDomNode n = docElem.firstChild();
  while(not n.isNull())
    {
      QDomElement e = n.toElement(); // try to convert the node to an element.
      if(not e.isNull())
        {
          if(e.attribute("language") == "python")
            {
              QString userFunctionName = e.attribute("name"),
                      userFunctionArguments = e.attribute("arguments"),
                      userFunctionBody = e.text();
                      m_languagesMap["python"][userFunctionName] = QString("def %1 (%2):\n%3\n\n").arg(userFunctionName).
                               arg(userFunctionArguments).
                               arg(userFunctionBody);
            }
          if(e.attribute("language") == "bc")
            {
              QString userFunctionName = e.attribute("name"),
                      userFunctionArguments = e.attribute("arguments"),
                      userFunctionBody = e.text();
                      m_languagesMap["bc"][userFunctionName] = QString("define %1 (%2) {\n%3\n}\n").arg(userFunctionName).
                                   arg(userFunctionArguments).
                                   arg(userFunctionBody);
            }
        }
      n = n.nextSibling();
    }
}

/*!
    \fn FunctionsDB::writeToDatabase()
 */
void FunctionsDB::writeToDatabase()
{
  /** Opens an xml file to use as database */
  QDomDocument dbDocument;
  dbDocument.appendChild(dbDocument.createElement("functions"));
  /** adds to database BC and Python user defined functions */
  putBCIntoDatabase(dbDocument);
  putPythonIntoDatabase(dbDocument);
  /** writes database on disk */
  QFile functionsDataBaseFile(m_functionsDatabasePath);
  if(not functionsDataBaseFile.open(IO_WriteOnly))
    KMessageBox::error(0,"Cannot store functions into database");
  else
    {
      QTextStream functionsDataBaseStream(&functionsDataBaseFile);
      functionsDataBaseStream << "<?xml version=\"1.0\" ?>\n";
      functionsDataBaseStream << dbDocument.toString(2);
      functionsDataBaseFile.close();
    }
}

/*!
    \fn FunctionsDB::createBCDatabase(QDomDocument& db)
 */
void FunctionsDB::putBCIntoDatabase(QDomDocument& db)
{
  // This function is used by FunctionsDB::writeToDatabase
  FunctionsCodeMap::ConstIterator it;
  for(it = m_languagesMap["bc"].constBegin(); it not_eq m_languagesMap["bc"].constEnd(); ++it)
    {
      QString functionCode = (*it).data();
      QRegExp r = QRegExp(Resource::bcFunctionNamePattern);
              r.search(functionCode);
      QString userFunctionName = r.cap(1);
              r = QRegExp(Resource::bcFunctionArgumentsPattern);
              r.search(functionCode);
      QString userFunctionArguments = r.cap(1);
              r = QRegExp(Resource::bcFunctionBodyPattern);
              r.search(functionCode);
      QString userFunctionBody = r.cap(1);

      QDomElement tempElement = db.createElement("define");
      tempElement.setAttribute("name",userFunctionName);
      tempElement.setAttribute("arguments",userFunctionArguments);
      tempElement.setAttribute("language","bc");
      QDomCDATASection cdata = db.createCDATASection(userFunctionBody);
      tempElement.appendChild(cdata);
      db.firstChild().appendChild(tempElement);
    }
}

/*!
    \fn FunctionsDB::createPythonDatabase(QDomDocument& db)
 */
void FunctionsDB::putPythonIntoDatabase(QDomDocument& db)
{
  // This function is used by FunctionsDB::writeToDatabase
  FunctionsCodeMap::ConstIterator it;
  for(it = m_languagesMap["python"].begin(); it not_eq m_languagesMap["python"].end(); ++it)
    {
      QString functionCode = (*it).data();
      QRegExp r = QRegExp(Resource::pythonFunctionNamePattern);
              r.search(functionCode);
      QString userFunctionName = r.cap(1);
              r = QRegExp(Resource::pythonFunctionArgumentsPattern);
              r.search(functionCode);
      QString userFunctionArguments = r.cap(1);
              r = QRegExp(Resource::pythonFunctionBodyPattern);
              r.search(functionCode);
      QString userFunctionBody = r.cap(1);

      QDomElement tempElement = db.createElement("define");
      tempElement.setAttribute("name",userFunctionName);
      tempElement.setAttribute("arguments",userFunctionArguments);
      tempElement.setAttribute("language","python");
      QDomCDATASection cdata = db.createCDATASection(userFunctionBody);
      tempElement.appendChild(cdata);
      db.firstChild().appendChild(tempElement);
    }
}

/*!
    \fn FunctionsDB::slotAddUserFuntion()
 */
void FunctionsDB::slotAddUserFunction()
{
  QString userFunctionName = m_leUserFunctionName->text(),
          userFunctionArguments = m_leUserFunctionArguments->text(),
          userFunctionBody = m_editorUserFunctionBody->text();

  userFunctionName.remove(" ");
  userFunctionArguments.remove(" ");

  if(userFunctionName.isEmpty() or (userFunctionName.contains(QRegExp("^[A-Z]\\w*")) not_eq 0))
    {
      KMessageBox::error(0,"Function name must be not empty and must begin with a lower case letter");
      return;
    }

  if(m_bComplex->isOn())
    {
      QStringList instructionList = QStringList::split("\n",userFunctionBody);
      uint j = 0,
           count = instructionList.count();
      QString body;
      while(j < count)
        {
          QString instruction = instructionList[j];
          m_python->convertToPython(instruction);
          if(instruction[0] == ' ')
            body += instruction + "\n";
          else
            body += " " + instruction + "\n";
          j++;
        }
      QString userFunctionCode = QString("def %1 (%2):\n%3\n\n").
                                     arg(userFunctionName).
                                     arg(userFunctionArguments).
                                     arg(body);
      userFunctionCode.stripWhiteSpace();
      m_languagesMap["python"][userFunctionName] = userFunctionCode;
    }
  else
    {
      QStringList instructionList = QStringList::split("\n",userFunctionBody);
      uint j = 0,
           count = instructionList.count();
      QString body;
      while(j < count)
        {
          QString instruction = instructionList[j];
          m_bc->convertToBC(instruction);
          if(instruction[0] == ' ') 
            body += instruction + "\n";
          else
            body += " " + instruction + "\n";
          j++;
        }
      QString userFunctionCode = QString("define %1 (%2) {\n%3\n}\n").
                                     arg(userFunctionName).
                                     arg(userFunctionArguments).
                                     arg(body);
      userFunctionCode.stripWhiteSpace();
      m_languagesMap["bc"][userFunctionName] = userFunctionCode;
    }

  if(m_cbxUserFunctionsList->currentText() not_eq userFunctionName)
    m_cbxUserFunctionsList->insertItem(userFunctionName);

  m_cbxUserFunctionsList->setCurrentText(userFunctionName);

  if(m_cbxFunctionsBox->currentText() not_eq userFunctionName)
    m_cbxFunctionsBox->insertItem(userFunctionName);

  m_cbxFunctionsBox->setCurrentText(userFunctionName);

  m_leUserFunctionName->clear();
  m_leUserFunctionArguments->clear();
  m_editorUserFunctionBody->clear();
}

/*!
    \fn FunctionsDB::slotDeleteUserFuntion()
 */
void FunctionsDB::slotDeleteUserFunction()
{
  QString currentFunctionName = m_cbxUserFunctionsList->currentText();
  if(m_bComplex->isOn())
    m_languagesMap["python"].remove(currentFunctionName);
  else
    m_languagesMap["bc"].remove(currentFunctionName);

  m_leUserFunctionName->clear();
  m_leUserFunctionArguments->clear();
  m_editorUserFunctionBody->clear();

  m_cbxUserFunctionsList->removeItem(m_cbxUserFunctionsList->currentItem());
  int i = 0,
      count = m_cbxFunctionsBox->count();
  while(i < count)
    {
      if(currentFunctionName == m_cbxFunctionsBox->text(i))
        {
          m_cbxFunctionsBox->removeItem(i);
          break;
        }
      else i++;
    }
}

/*!
    \fn FunctionsDB::slotEditUserFuntion(int index)
 */
void FunctionsDB::slotEditUserFunction(int index)
{
  Q_UNUSED(index);
  if(m_bComplex->isOn())
    {
      QString currentText = m_languagesMap["python"][m_cbxUserFunctionsList->currentText()];
      QRegExp r = QRegExp(Resource::pythonFunctionNamePattern);
              r.search(currentText);
              m_leUserFunctionName->setText(r.cap(1));

              r = QRegExp(Resource::pythonFunctionArgumentsPattern);
              r.search(currentText);
              m_leUserFunctionArguments->setText(r.cap(1));

              r = QRegExp(Resource::pythonFunctionBodyPattern);
              r.search(currentText);
      QString functionBody = r.cap(1);
              m_python->convertToKalcoolus(functionBody);

      int k = functionBody.length()-1;
              while(QString("\n ").contains(functionBody[k]) not_eq 0)
                if(k >= 0) functionBody.truncate(k--);

      m_editorUserFunctionBody->setText(functionBody);
      if(functionBody[0] == '\t') functionBody.remove(0,1);
    }
  else
    {
      QString currentText = m_languagesMap["bc"][m_cbxUserFunctionsList->currentText()];
      QRegExp r = QRegExp(Resource::bcFunctionNamePattern);
              r.search(currentText);
              m_leUserFunctionName->setText(r.cap(1));

              r = QRegExp(Resource::bcFunctionArgumentsPattern);
              r.search(currentText);
              m_leUserFunctionArguments->setText(r.cap(1));

              r = QRegExp(Resource::bcFunctionBodyPattern);
              r.search(currentText);
      QString functionBody = r.cap(1);
              m_bc->convertToKalcoolus(functionBody);
              m_editorUserFunctionBody->setText(functionBody.stripWhiteSpace());
    }
}

/*!
    \fn FunctionsDB::slotSelectUserFunction(const QString& s)
 */
void FunctionsDB::slotSelectUserFunction(const QString& s)
{
  QString currentItem = s,
          language,
          patternName,
          patternArgs,
          patternBody;
  if(m_bComplex->isOn())
    {
      language = "python";
      patternName = Resource::pythonFunctionNamePattern;
      patternArgs = Resource::pythonFunctionArgumentsPattern;
      patternBody = Resource::pythonFunctionBodyPattern;
    }
  else
    {
      language = "bc";
      patternName = Resource::bcFunctionNamePattern;
      patternArgs = Resource::bcFunctionArgumentsPattern;
      patternBody = Resource::bcFunctionBodyPattern;
    }
  QRegExp r = QRegExp(patternName);
              r.search(m_languagesMap[language][currentItem]);
  QString function = r.cap(1);

          r = QRegExp(patternArgs);
          r.search(m_languagesMap[language][currentItem]);
  QString argList = r.cap(1);

  if(argList.isEmpty()) function += "(#)";
  else
    {
      int i = argList.contains(",");
      if(i not_eq 0) function += "(" + QString().fill(',',i)+")";
      else function +="()";
    }

  Service::insertStringIntoEditor(m_inputDisplay,function, 0, function.length() - argList.contains(",") - 1);
}

void FunctionsDB::getCodeGenerators(BCCodeGenerator* bc, PythonCodeGenerator* python)
{
  m_bc = bc;
  m_python = python;
}

#include "functionsdb.moc"
