/***************************************************************************
 *   Copyright (C) 2005 by Emiliano Gulmini                                *
 *   emi_barbarossa@yahoo.it                                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "kalcoolus.h"

//QT Headers
#include <qmap.h>
#include <qspinbox.h>
#include <qwidgetstack.h>
#include <qiconview.h>
#include <qpopupmenu.h>
#include <qtabwidget.h>

//KDE Headers
#include <klocale.h>
#include <kprocess.h>
#include <kmessagebox.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kstandarddirs.h>
#include <khelpmenu.h>
#include <kaboutdata.h>
#include <kurlrequester.h>

//Local and Other Headers
//#include "matrix_editor.h"
#include "matrix.h"
#include "bccodegenerator.h"
#include "pythoncodegenerator.h"
#include "keyboard.h"
#include "constantsdb.h"
#include "functionsdb.h"
#include "codegenerator.h"
#include "configuration.h"
#include "utilities.h"
#include "kalcoolusrc.h"
#include "kalcoolusimplementif.h"
#include "service.h"

using namespace Global;

Kalcoolus::Kalcoolus() : Kbc( 0, "Kalcoolus" )
{
  // Initializing config file 
  m_config = new KConfig("kalcoolusrc");

  // Initializing display 
  m_inputDisplay->setText("\n");
  m_inputDisplay->setCursorPosition(0,0);

  // Constants database object
  m_constantsDB = new ConstantsDB(m_config,
                                  m_constantsListview,
                                  m_leConstName, m_leConstValue,
                                  m_bConstAdd, m_bConstClear,
                                  m_cbxConstantsBox,
                                  m_inputDisplay);
  m_constantsDB->loadDatabase();

  // Functions database object
  m_functionsDB = new FunctionsDB(m_config,
                                  m_bComplex,
                                  m_inputDisplay,
                                  m_leUserFunctionName,
                                  m_leUserFunctionArguments,
                                  m_editorUserFunctionBody,
                                  m_cbxUserFunctionsList,
                                  m_cbxFunctionsBox,
                                  m_bAddUserFunction,
                                  m_bDeleteUserFunction);
  m_functionsDB->loadDatabase();

  m_bcCodeGenerator = new BCCodeGenerator(m_functionsDB->functionsDB()["bc"], m_constantsDB->constantsDB());
  m_pythonCodeGenerator = new PythonCodeGenerator(m_functionsDB->functionsDB()["python"], m_constantsDB->constantsDB());
  //FIXME: bad method!
  m_functionsDB->getCodeGenerators(m_bcCodeGenerator, m_pythonCodeGenerator);

  //Grouping buttons to one object
  QMap<QString,  KPushButton*> keyboardMap;
  keyboardMap["0"] = m_b0;
  keyboardMap["1"] = m_b1;
  keyboardMap["2"] = m_b2;
  keyboardMap["3"] = m_b3;
  keyboardMap["4"] = m_b4;
  keyboardMap["5"] = m_b5;
  keyboardMap["6"] = m_b6;
  keyboardMap["7"] = m_b7;
  keyboardMap["8"] = m_b8;
  keyboardMap["9"] = m_b9;
  keyboardMap["A"] = m_bA;
  keyboardMap["B"] = m_bB;
  keyboardMap["C"] = m_bC;
  keyboardMap["D"] = m_bD;
  keyboardMap["E"] = m_bE;
  keyboardMap["F"] = m_bF;
  keyboardMap["Im"] = m_bImaginary;
  keyboardMap["+"] = m_bPlus;
  keyboardMap["*"] = m_bMult;
  keyboardMap["/"] = m_bDiv;
  keyboardMap["-"] = m_bMinus;
  keyboardMap["="] = m_bEqual;
  keyboardMap["=="] = m_bIdentical;
  keyboardMap["!="] = m_bDifferent;
  keyboardMap[">"] = m_bGreat;
  keyboardMap[">="] = m_bGreatEqual;
  keyboardMap["<"] = m_bLess;
  keyboardMap["<="] = m_bLessEqual;
  keyboardMap["complex"] = m_bComplex;
  keyboardMap["last"] = m_bLastResult;
  keyboardMap["newline"] = m_bNewLine;
  keyboardMap["."] = m_bDot;
  keyboardMap["("] = m_bLeftRound;
  keyboardMap[")"] = m_bRightRound;
  keyboardMap["sin"] = m_bSin;
  keyboardMap["cos"] = m_bCos;
  keyboardMap["tan"] = m_bTan;
  keyboardMap["atan"] = m_bAtan;
  keyboardMap["log"] = m_bLog;
  keyboardMap["ln"] = m_bLn;
  keyboardMap["exp"] = m_bExp;
  keyboardMap["sqr"] = m_bSqr;
  keyboardMap["sqrt"] = m_bSqrt;
  keyboardMap["pow"] = m_bPower;
  keyboardMap["fact"] = m_bFact;
  keyboardMap["npr"] = m_bSimpleDisposition;
  keyboardMap["nrr"] = m_bRepeatedDisposition;
  keyboardMap["binom"] = m_bBinomial;
  keyboardMap["ncr"] = m_bCombination;
  keyboardMap["percent"] = m_bPercentage;
  keyboardMap["real"] = m_bRealPart;
  keyboardMap["imag"] = m_bImaginaryPart;
  keyboardMap["conj"] = m_bConjugate;
  keyboardMap["arg"] = m_bArgument;
  keyboardMap["mod"] = m_bModulus;
  keyboardMap["warn"] = m_bWarning;
  keyboardMap["bye"] = m_bBye;
  keyboardMap["save"] = m_bSave;

  //Keyboard
  m_keyboard = new Keyboard(m_config,
                            m_inputDisplay,
                            m_resultDisplay,
                            keyboardMap);

  //Configuration
  m_configuration = new Configuration(m_config,
                                      keyboardMap,
                                      m_inputDisplay,
                                      m_resultDisplay,
                                      m_editorUserFunctionBody,
                                      m_spbxScale,
                                      m_spbxIBase,
                                      m_spbxOBase,
                                      m_kurlBCPath,
                                      m_spbxPythonDecimalDigits,
                                      m_kurlPythonModulesPath,
                                      m_fontRequester,
                                      m_colbxNumbers,
                                      m_colbxOperations,
                                      m_colbxFunctions,
                                      m_colbxKeywords,
                                      m_colbxSpecials,
                                      m_kurlLogFilePath,
                                      m_cbxEnableLogFile);
  m_configuration->loadBCSettings();
  m_configuration->loadPythonSettings();
  m_configuration->loadFontSettings();
  m_configuration->loadOptionsSettings();

  //Preparing utilities
  m_utilities = new Utilities(m_configuration,
                              m_spbxDegrees,
                              m_spbxMinutes,
                              m_spbxSeconds,
                              m_leRadians,
                              m_bAngleReset,
                              m_bCopyRadians,
                              m_bCopyDegrees,
                              m_iconWorkspace,
                              m_bNewMatrix,
//                              m_bCancelMatrix,
                              m_inputDisplay);

  // Mapping buttons into symbols
  mapButtons();

  // Connecting signals 
  ConnectKeyboardPage();
  ConnectConstantsPage();
  ConnectFunctionsPage();
  ConnectUtilitiesPage();
  ConnectConfigurationPage();

  // Creating help menu
  createHelp();

  // Initializing DCOP interface
  m_interface = new KalcoolusImplementIf(m_inputDisplay, m_resultDisplay, m_bEqual);

  m_menu = 0;
}

Kalcoolus::~Kalcoolus()
{
  if(m_keyboard) delete m_keyboard;
  if(m_configuration) delete m_configuration;
  if(m_functionsDB) delete m_functionsDB;
  if(m_constantsDB) delete m_constantsDB;
  if(m_config) delete m_config;
  if(m_utilities) delete m_utilities;
  if(m_interface) delete m_interface;
  if(m_bcCodeGenerator) delete m_bcCodeGenerator;
  if(m_pythonCodeGenerator) delete m_pythonCodeGenerator;
  if(m_menu) delete m_menu;
}
/*******************************************************************/
/************************ PRIVATE SLOTS ****************************/
/*******************************************************************/

// Keyboard Page Slots
/*!
    \fn Kalcoolus::slotSave()
 */
void Kalcoolus::slotSave()
{
  // Saving BC settings 
  m_configuration->saveBCSettings();
  // Saving Python settings 
  m_configuration->savePythonSettings();
  // Saving font settings
  m_configuration->saveFontSettings();
  // Saving option settings
  m_configuration->saveOptionsSettings();
  // Saving constants/functions code to databases 
  m_constantsDB->writeToDatabase();
  m_functionsDB->writeToDatabase();
}

/*!
    \fn Kalcoolus::slotBye()
 */
void Kalcoolus::slotBye()
{
  accept();
}

/*!
    \fn Kalcoolus::slotEqualClicked()
 */
void Kalcoolus::slotEqualClicked()
{
  if(m_inputDisplay->text() == "\n") return;
  // Cleaning display 
  m_resultDisplay->clear();
  // Creating the internal bc/python script
  CodeGenerator codeGenerator(m_inputDisplay->text(),
                              m_functionsDB->functionsDB(),
                              m_constantsDB->constantsDB(),
                              m_configuration);

  QString result = codeGenerator.executeProgram(m_bComplex->isOn());

  QStringList resultList = QStringList::split("\n",result);

  m_last = resultList[resultList.count()-1];
  m_resultDisplay->setText(result);

  // Enabling log file features
  if(m_configuration->isEnabledLogFile())
    {
      QString logInfo = CodeGenerator::Mode + " - " + QDate::currentDate(Qt::LocalTime).toString(Qt::LocalDate)+"\n";
      QFile logFile(m_configuration->logFilePath());
      if(logFile.open( IO_WriteOnly | IO_Append ))
        {
          QTextStream stream( &logFile );
          stream << logInfo << m_inputDisplay->text();
        }
      logFile.close();
    }
}

/*!
    \fn Kalcoolus::slotLastResultButton()
 */
void Kalcoolus::slotLastResultButton()
{
  Service::insertStringIntoEditor(m_inputDisplay,m_last, 0, m_last.length());
  m_inputBuffer = m_inputDisplay->text();
}

/*!
    \fn Kalcoolus::slotEraseClicked()
 */
void Kalcoolus::slotEraseClicked()
{
  // Cleaning all
  m_inputDisplay->clear();
  m_inputDisplay->setText("\n");
  m_resultDisplay->clear();
  m_inputBuffer = m_inputDisplay->text();
  m_inputDisplay->setCursorPosition(0,0);
  m_resultDisplay->setCursorPosition(0,0);
}

/*!
    \fn Kalcoolus::slotCancelClicked()
 */
void Kalcoolus::slotCancelClicked()
{
  // We want to cancel a character from the cursor position
  int para,
      index;
  m_inputDisplay->getCursorPosition(&para,&index);
  // Splitting text into single lines
  QStringList inputDisplayList = QStringList::split("\n", m_inputDisplay->text(), true);
  /* Deleting characters.
     Updating cursor position */
  index --;
  if(index < 0)
    {
      para --;
      if(para < 0)
        {
          para = 0;
          index = 0;
        }
      else
        index = inputDisplayList[para].length()-1;
    }
  if(index == 0 and inputDisplayList[para][index+1] == '\n')
    inputDisplayList[para].remove(index, 2);
  else
    inputDisplayList[para].remove(index, 1);
  // Reconstructing text
  m_inputDisplay->setText(inputDisplayList.join("\n"));
  m_inputDisplay->setCursorPosition(para, index);

  m_inputBuffer = m_inputDisplay->text();
}

/*!
    \fn Kalcoolus::slotCopyrights()
 */
void Kalcoolus::slotCopyrights()
{
  KProcess* proc = new KProcess();

  proc->setUseShell(true);
  QString mode = "bc";
  if(not m_bComplex->isOn())
    {
      *proc << m_configuration->bcPath();
      *proc <<  KProcess::quote("-v");
    }
  else
    {
      mode = "python";
      *proc << "python -c \"copyright()\"";
    }

  connect(proc, SIGNAL(receivedStdout(KProcess*,char*,int)), this, SLOT(slotGetScriptOutput(KProcess*,char*,int)));
  connect(proc, SIGNAL(receivedStderr(KProcess*,char*,int)), this, SLOT(slotGetScriptError(KProcess*,char*,int)));
  connect(proc, SIGNAL(processExited(KProcess*)), this, SLOT(slotProcessExited(KProcess*)));

  if(not proc->start(KProcess::Block, KProcess::All))
    KMessageBox::error(0,"Error: cannot retrive "+mode+" copyrights notice");

  if(proc) delete proc;
}

/*!
    \fn Kalcoolus::slotComplexMode()
 */
void Kalcoolus::slotComplexMode()
{
  /** Enabling/disabling capabilities when we switch between
   *  normal and complex mode */
  m_functionsDB->loadFunctionsIntoBoxes();
  bool complexMode = m_bComplex->isOn();

  m_bImaginary->setEnabled(complexMode);
  m_bLess->setEnabled(not complexMode);
  m_bLessEqual->setEnabled(not complexMode);
  m_bGreat->setEnabled(not complexMode);
  m_bGreatEqual->setEnabled(not complexMode);
  m_bPercentage->setEnabled(not complexMode);
  m_bRealPart->setEnabled(complexMode);
  m_bImaginaryPart->setEnabled(complexMode);
  m_bConjugate->setEnabled(complexMode);
  m_bArgument->setEnabled(complexMode);
  m_bModulus->setEnabled(complexMode);
  m_configView->findItem("Setting python",0)->setSelectable(complexMode);
  m_configView->findItem("Setting bc",0)->setSelectable(not complexMode);

  if(complexMode)
    {
      m_configTool->raiseWidget(2);
      m_configView->findItem("Setting python",0)->setSelected(true);
      m_grpboxExpressions->setTitle("Expressions : complex mode");
      m_grpboxResults->setTitle("Results : complex mode");
      m_bA->setEnabled(false);
      m_bB->setEnabled(false);
      m_bC->setEnabled(false);
      m_bD->setEnabled(false);
      m_bE->setEnabled(false);
      m_bF->setEnabled(false);
    }
  else
    {
      m_configTool->raiseWidget(0);
      m_configView->findItem("Setting bc",0)->setSelected(true);
      m_grpboxExpressions->setTitle("Expressions : normal mode");
      m_grpboxResults->setTitle("Results : normal mode");
      m_configuration->enableButtons();
    }
}

// Constants Page Slots
// Functions Page Slots


// Utilities Page Slots
/*!
    \fn Kalcoolus::slotChooseUtilitiesItem(QIconViewItem*)
 */
void Kalcoolus::slotChooseUtilitiesItem(QIconViewItem* item)
{
  // Enabling choosed configuration item 
  if(not item) return;

  QString itemCaption = item->text();
  bool complexMode = m_bComplex->isOn();

  if(itemCaption == "Angle converter" and not complexMode)     m_utilitiesStack->raiseWidget(0);
  else
  if(itemCaption == "Matrices") m_utilitiesStack->raiseWidget(1);
  /*else
  if(itemCaption == "Setting python" and complexMode) m_configTool->raiseWidget(2);
  else
  if(itemCaption == "Kalcoolus options") m_configTool->raiseWidget(3);*/
}

/*!
    \fn Kalcoolus::slotUtilitiesPageSelected(QWidget* p)
 */
void Kalcoolus::slotUtilitiesPageSelected(QWidget* p)
{
  Q_UNUSED(p);
  m_utilitiesStack->raiseWidget(0);
}

/*!
    \fn Kalcoolus::slotApplyFunctionToMatrix()
 */
/*void Kalcoolus::slotApplyFunctionToMatrix()
{
  
  
}*/

/*void Kalcoolus::slotNewMatrixButton()
{
  m_utilities->matrixShow();
}

void Kalcoolus::slotAddToWorkspace(const Matrix& m)
{
  QIconViewItem* item = m_icoWorkspace->firstItem();

  while(item) if(item->text() == m.name) return;
              else item = item->nextItem();
  QString iconText = m.name+"("+QString::number(m.rows)+"x"+QString::number(m.cols)+")";
  (void) new QIconViewItem(m_icoWorkspace, iconText);
}*/

// Configuration Page Slots
/*!
    \fn Kalcoolus::slotChooseConfigurationItem(QIconViewItem*)
 */
void Kalcoolus::slotChooseConfigurationItem(QIconViewItem* item)
{
  // Enabling choosed configuration item 
  if(not item) return;

  QString itemCaption = item->text();
  bool complexMode = m_bComplex->isOn();
  QMap<QString, int> rw;

  rw["Fonts"] = 1;
  rw["Options"] = 3;
  if(complexMode)
    {
      rw["Setting python"] = 2;
      rw["Setting bc"] = -1;
      m_configTool->raiseWidget(rw[item->text()]);
    }
  else
    {
      rw["Setting python"] = -1;
      rw["Setting bc"] = 0;
      m_configTool->raiseWidget(rw[item->text()]);
    }
}
/*!
    \fn Kalcoolus::slotConfigurationPageSelected(QWidget*)
 */
void Kalcoolus::slotConfigurationPageSelected(QWidget* p)
{
  Q_UNUSED(p);

  QMap<QString, int> rw;

  rw["Fonts"] = 1;
  rw["Options"] = 3;

  QString iconText;
  if(not m_bComplex->isOn())
    {
      if(m_configView->currentItem() == 0)
        {
          iconText = "Setting bc";
          m_configTool->raiseWidget(0);
        }
      else
        iconText = m_configView->currentItem()->text();

      rw["Setting bc"] = 0;
    }
  else
    {
      if(m_configView->currentItem() == 0)
        {
          iconText = "Setting python";
          m_configTool->raiseWidget(2);
        }
      else
        iconText = m_configView->currentItem()->text();
      rw["Setting python"] = 2;
    }
  m_configTool->raiseWidget(rw[iconText]);
  m_configView->findItem(iconText,0)->setSelected(true);
}

// Other Slots
/*!
    \fn Kalcoolus::slotHelp()
 */
void Kalcoolus::slotHelp()
{
  kapp->invokeHelp("","","");
}

/*******************************************************************/
/************************ PRIVATE METHODS **************************/
/*******************************************************************/

/*!
    \fn Kalcoolus::Connect()
 */
void Kalcoolus::ConnectKeyboardPage()
{
  // Connecting Keyboard Page signals 
  connect(m_bDelete, SIGNAL(clicked()), this, SLOT(slotEraseClicked()));
  connect(m_bCancel, SIGNAL(clicked()), this, SLOT(slotCancelClicked()));
  connect(m_bLastResult, SIGNAL(clicked()), this, SLOT(slotLastResultButton()));
  connect(m_bEqual, SIGNAL(clicked()), this, SLOT(slotEqualClicked()));
  connect(m_bWarning, SIGNAL(clicked()), this, SLOT(slotCopyrights()));
  connect(m_bBye, SIGNAL(clicked()), this, SLOT(slotBye()));
  connect(m_bSave, SIGNAL(clicked()), this, SLOT(slotSave()));
  connect(m_bComplex, SIGNAL(clicked()), this, SLOT(slotComplexMode()));
}

/*!
    \fn Kalcoolus::ConnectConstantsPage()
 */
void Kalcoolus::ConnectConstantsPage()
{
  // Connecting Constants Page signals 
}

/*!
    \fn Kalcoolus::ConnectFunctionsPage
 */
void Kalcoolus::ConnectFunctionsPage()
{
  // Connecting Functions Page signals 
}

/*!
    \fn Kalcoolus::ConnectUtilitiesPage
 */
void Kalcoolus::ConnectUtilitiesPage()
{
  connect(m_utilitiesView, SIGNAL(clicked(QIconViewItem*)), this, SLOT(slotChooseUtilitiesItem(QIconViewItem*)));
  connect(m_tabMenu, SIGNAL(currentChanged(QWidget*)), this, SLOT(slotUtilitiesPageSelected(QWidget*)));
 // connect(m_utilities, SIGNAL(addToWorkspace(const Matrix&)), this, SLOT(slotAddToWorkspace(const Matrix&)));
 // connect(m_bNewMatrix, SIGNAL(clicked()), this, SLOT(slotNewMatrixButton()));
 // connect(m_icoWorkspace, SIGNAL(contextMenuRequested(QIconViewItem*, const QPoint&)), this, SLOT(slotPopupMenu(QIconViewItem*, const QPoint&)));
}

/*void Kalcoolus::slotPopupMenu(QIconViewItem* item, const QPoint& p)
{
  //m_menu->reparent(0,p);
  if(item)
    {
      QPopupMenu menu(m_icoWorkspace);
      menu.insertItem(item->text());
      menu.insertSeparator();
      menu.insertItem("edit");
      //m_menu->insertItem("delete",this,SLOT(slotDelWorkspace));
      menu.exec(p);
    }
}*/

/*!
    \fn Kalcoolus::ConnectConfigurationPage
 */
void Kalcoolus::ConnectConfigurationPage()
{
  // Connecting Configuration Page signals 
  connect(m_configView, SIGNAL(clicked(QIconViewItem*)), this, SLOT(slotChooseConfigurationItem(QIconViewItem*)));
  connect(m_tabMenu, SIGNAL(currentChanged(QWidget*)), this, SLOT(slotConfigurationPageSelected(QWidget*)));
}

/*!
    \fn Kalcoolus::mapButtons()
 */
void Kalcoolus::mapButtons()
{
  /** This object maps the names of the GUI buttons
   *  to GUI symbols */
  m_buttonsMap["m_bWarning"] = "copyrights(#)";
}

/*!
    \fn Kalcoolus::createHelp()
 */
void Kalcoolus::createHelp()
{
  // Connecting help 
  KAboutData* aboutData = new KAboutData("Kalcoolus",
                                         I18N_NOOP("Kalcoolus"),
                                         "0.2 alfa 1",
                                         I18N_NOOP("This is a KDE calculator built on top of GNU bc 1.06, with support for python 2.3.4"),
                                         KAboutData::License_GPL_V2,
                                         "(c) 2005, Emiliano Gulmini",
                                         I18N_NOOP("\n"),
                                         "http://web.tiscali.it/gulbros","submit@bugs.kde.org");

  aboutData->addAuthor("Emiliano Gulmini",
                       I18N_NOOP("Original author and current maintainer"),
                       "emi_barbarossa@yahoo.it",
                       "http://web.tiscali.it/gulbros");

  aboutData->setOtherText("Articolo 11 -  L'Italia ripudia la guerra come\nstrumento di offesa alla liberta' degli altri popoli e come mezzo di\nrisoluzione delle controversie internazionali;\nconsente, in condizioni di parita' con gli altri Stati,\nalle limitazioni di sovranita' necessarie ad\nun ordinamento che assicuri la pace e la giustizia fra le Nazioni;\npromuove e favorisce le organizzazioni internazionali rivolte a tale scopo."); 

  connect(m_bHelp, SIGNAL(clicked()), this, SLOT(slotHelp()));

  KHelpMenu *helpMenu = new KHelpMenu(this, aboutData, true);

  m_bHelp->setPopup((QPopupMenu*)helpMenu->menu());
}

void Kalcoolus::slotGetScriptError(KProcess* proc, char* s, int i)
{
  Q_UNUSED(proc);
  KMessageBox::information(0, i18n(QCString(s,i+1)), "Copyright notice");
}

void Kalcoolus::slotGetScriptOutput(KProcess* proc, char* s, int i)
{
  Q_UNUSED(proc);
  KMessageBox::information(0, i18n(QCString(s,i+1)), "Copyright notice");
}

void Kalcoolus::slotProcessExited(KProcess* proc)
{
  Q_UNUSED(proc);
}

#include "kalcoolus.moc"
