/***************************************************************************
 *   Copyright (C) 2005 by Emiliano Gulmini   *
 *   emi_barbarossa@yahoo.it   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
// Qt headers
#include <qregexp.h>
#include <qstring.h>
// Local headers
#include "service.h"
#include "pythoncodegenerator.h"

PythonCodeGenerator::PythonCodeGenerator(const Global::FunctionsCodeMap& fm, const Global::ConstantsMap& cm,QObject *parent, const char *name)
 : QObject(parent, name)
{
  m_fcm = fm;
  m_cm = cm;
}


PythonCodeGenerator::~PythonCodeGenerator()
{
}

/*!
    \fn PythonCodeGenerator::formatResultString(QString&, QString& resultBuffer, int internalScriptNumberOfLines)
 */
void PythonCodeGenerator::formatResultString(QString& result, QString& resultBuffer, int internalScriptNumberOfLines)
{
  // catching error output strings and formatting them
  QRegExp r("(line\\s)(\\d+)");
          r.search(resultBuffer);
  QString bugMessageLineNumber = r.cap(1) + " ";
  int n = r.cap(2).toInt();
  /* When python finds an error it returns the line number of the bug,
   * but we must subtract the number of lines generated automatically by
   * kalcoolus to obtain the real position of this bug*/ 

  n -= internalScriptNumberOfLines;
  // Creating error output text 
  bugMessageLineNumber += QString::number(n);
  QString errorDescription = resultBuffer.section("Error:",1,1);

  if(errorDescription.isEmpty()) 
    resultBuffer += "\n";
  else 
    resultBuffer = bugMessageLineNumber + " : " + errorDescription + "\n";

  resultBuffer.remove("(").remove(")");
  while (resultBuffer.contains(QRegExp("[0-9]j")) not_eq 0)
    resultBuffer.replace("j","i");

  int pos = resultBuffer.find("i ");
      while(pos >= 0)
        {
          resultBuffer.replace(pos,2,"i");
          pos = resultBuffer.find("i ",pos);
        }
      result = resultBuffer;
      resultBuffer = QString::null;
}

/*!
    \fn PythonCodeGenerator::convertToPython(QString& expression)
 */
void PythonCodeGenerator::convertToPython(QString& expression)
{
  /*Now we mask the name of the functions and constants, so the parser
    can convert the 'i' of kalcoolus in the 'j' of python otherwise it could
    generate something like 's1jn(5)' from 'sin(5)'.
    We also convert constants to their value.
    In the future I will write a REAL PREPROCESSOR!!!
  */

  // Masking function names with the place holders $1$, $2$, $3$ etc.
  int i = 1;
  Global::FunctionsCodeMap::Iterator itf;
  QMap<QString, QString> placeHoldersMap;
  for(itf = m_fcm.begin(); itf not_eq m_fcm.end(); ++itf)
    {
      QString funcName = itf.key(),
              placeHolderString = QString("$%1$").arg(i);
      Service::replaceAllSubstrings(expression,
                                    funcName.length(),
                                    funcName + "(",
                                    placeHolderString);
      placeHoldersMap[placeHolderString] = funcName;
      i++;
    }

  // Masking builtin-function names with the strings $1$, $2$, $3$ etc.
  QStringList list = QStringList::split(",","sin,cos,tan,atan,log,ln,exp,sqr,sqrt,pow,fact,percent,re,im,conj,arg,mod");
  QStringList::Iterator itl;
  for(itl = list.begin(); itl not_eq list.end(); ++itl)
    {
      QString funcName = (*itl),
              placeHolderString = QString("$%1$").arg(i);
      Service::replaceAllSubstrings(expression,
                                    funcName.length(),
                                    funcName+"(",
                                    placeHolderString);
      placeHoldersMap[placeHolderString] = funcName;
      i++;
    }

  // Masking builtin-keywords with the strings $1$, $2$, $3$ etc.
  list = QStringList::split(",","and,else,import,raise,assert,except,in,return,break,exec,is,try,class,finally,lambda,while,continue,for,not,yield,def,from,or,del,global,pass,elif,if,print");
  for(itl = list.begin(); itl not_eq list.end(); ++itl)
    {
      QString keyword = (*itl),
              placeHolderString = QString("$%1$").arg(i);
      Service::replaceAllSubstrings(expression,
                                    keyword.length(),
                                    keyword,
                                    placeHolderString);
      placeHoldersMap[placeHolderString] = keyword;
      i++;
    }

  // Replacing constants names with their numerical value
  QString expressionBeforeReplacing = QString::null;
  while(expressionBeforeReplacing not_eq expression)
    {
      expressionBeforeReplacing = expression;
      Global::ConstantsMap::Iterator itc;
      for(itc = m_cm.begin(); itc not_eq m_cm.end(); ++itc)
        {
          QString key = itc.key(),
                  data = itc.data();
          while(m_cm.contains(data)) data = m_cm[data];
          Service::replaceAllSubstrings(expression, key.length(), key, data);
        }
      //if(expressionBeforeReplacing == expression) break;
    }

  // Masking substrings like 'ia' or 'iw' or 'iZ' to prevent conversion into '1ja' or '1jw' or '1jZ'
  QString keywordToMask = "i";
  QString letters = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
  int j = 0,
      l = letters.length();
  while(j < l)
    {
      QString placeHolderString = QString("$%1$").arg(i);
      placeHoldersMap[placeHolderString] = keywordToMask+letters[j];
      Service::replaceAllSubstrings(expression,
                                    2,
                                    keywordToMask+letters[j],
                                    placeHolderString);
      i++;
      j++;
    }

  /* Now we can safely search for expressions like 4+i or i*i that are not legal in python
   * and replace them with the correct form 
   */
  QRegExp r("[0-9](i)");
  int pos = r.search(expression, 0);
  while(pos >= 0)
    {
      expression.replace(r.pos(1),1,"j");
      pos = r.search(expression, r.pos(1));
    }
  r = QRegExp("(i[0-9]+)");
  pos = r.search(expression, 0);
  while(pos >= 0)
    {
      QString cap = r.cap(1).remove(0,1);
              cap += "j";
      expression.replace(pos,cap.length(),cap);
      pos = r.search(expression, r.pos(1));
    }
  // Previous loop could generate code like 5j.03 that is incorrect. We fix it
  pos = expression.find("j.");
  if(pos >= 0)
    {
      expression.remove(pos,1);
      expression += "j";
    }
  r = QRegExp("[\\*\\+\\-/=><\\s\\(\\)\\[\\]](i)");
  pos = r.search(expression, 0);
  while(pos >= 0)
    {
      expression.replace(r.pos(1),1,"1j");
      pos = r.search(expression, r.pos(1)+1);
    }
  r = QRegExp("(^i[^a-z]+)");
  pos = r.search(expression, 0);
  while(pos >= 0)
    {
      expression.replace(r.pos(1),1,"1j");
      pos = r.search(expression, r.pos(1)+1);
    }

  // the "(#)" stands for "(no-arguments-here)"
  expression.replace("(#)","()");
  // Now we must restore the original functions names
  QMap<QString,QString>::Iterator itk;
  for(itk = placeHoldersMap.begin(); itk not_eq placeHoldersMap.end(); ++itk)
    {
      QString placeHolderString = itk.key();
      int pos = expression.find(placeHolderString, 0);
      while (pos not_eq -1)
        {
           expression.replace(pos, placeHolderString.length(), itk.data());
           pos = expression.find(placeHolderString, pos);
        }
    }
}

/*!
    \fn PythonCodeGenerator::convertToKalcoolus(QString& expression)
 */
void PythonCodeGenerator::convertToKalcoolus(QString& expression)
{
  /* We search for expressions like 4+1j or 1j*1j that are not kalcoolus notations
   * and replace them with the correct form 
   */
  uint i = 0;
  while (i < expression.length())
    {
     if((i == 0) and (expression[i] not_eq ' '))
       {
         expression = " "+expression;
         i += 2;
       }
     if(expression[i] == '\n')
        {
          if(((i+1) < expression.length()) and (expression[i+1] not_eq ' '))
            {
              expression.replace(i,1,"\n ");
              i += 2;
            }
        }
      i++;
    }
  expression.replace("0j" , "0i")
            .replace("1j" , "i")
            .replace("2j" , "2i")
            .replace("3j" , "3i")
            .replace("4j" , "4i")
            .replace("5j" , "5i")
            .replace("6j" , "6i")
            .replace("7j" , "7i")
            .replace("8j" , "8i")
            .replace("9j" , "9i");
}

#include "pythoncodegenerator.moc"
