#include "acomplex.h"
#include "atomutil.h"

CComplex CCNull( 0.0, 0.0 );
//------------------------------------------------------------------------------------------------
//
//	abs(z)
//
//		This function returns the magnitude of z.  It uses the 
//      algorithm presented in Numerical Recipes to avoid overflow
//      and underflow of the result
//
//------------------------------------------------------------------------------------------------
double  abs   (const CComplex& z)
{
	double x, y;		// absolute value of real and imaginary parts of z
	double temp;		// temporary value
	
	x = fabs(z.real());
	y = fabs(z.imag());
	
	if (x == 0.0)
		return y;
	else if (y == 0.0)
		return x;
	else if (x > y) {
		temp = y/x;
		return x*sqrt(1.0+temp*temp);
	}
	else {
		temp = x/y;
		return y*sqrt(1.0+temp*temp);
	}
}
//------------------------------------------------------------------------------------------------
//
//	operator+=
//
//		This is the += operator with complex argument
//
//------------------------------------------------------------------------------------------------
CComplex& CComplex::operator+=(const CComplex& z)
{
	mReal += z.mReal;
	mImag += z.mImag;
	return *this;
}
//------------------------------------------------------------------------------------------------
//
//	operator+=
//
//		This is the += operator with real argument
//
//------------------------------------------------------------------------------------------------
inline CComplex& CComplex::operator+=(const double&  x)
{
	mReal += x;
	return *this;
}
//------------------------------------------------------------------------------------------------
//
//	operator-=
//
//		This is the -= operator with complex argument
//
//------------------------------------------------------------------------------------------------
CComplex& CComplex::operator-=(const CComplex& z)
{
	mReal -= z.mReal;
	mImag -= z.mImag;
	return *this;
}
//------------------------------------------------------------------------------------------------
//
//	operator-=
//
//		This is the -= operator with real argument
//
//------------------------------------------------------------------------------------------------
CComplex& CComplex::operator-=(const double&  x)
{
	mReal -= x;
	return *this;
}
//------------------------------------------------------------------------------------------------
//
//	operator*=
//
//		This is the *= operator with complex argument
//
//------------------------------------------------------------------------------------------------
CComplex& CComplex::operator*=(const CComplex& z)
{
	double re = mReal;
	double im = mImag;
	mReal = re*z.mReal - im*z.mImag;
	mImag = im*z.mReal + re*z.mImag;
	return *this;
}
//------------------------------------------------------------------------------------------------
//
//	operator*=
//
//		This is the *= operator with real argument
//
//------------------------------------------------------------------------------------------------
CComplex& CComplex::operator*=(const double&  x)
{
	mReal *= x;
	mImag *= x;
	return *this;
}
//------------------------------------------------------------------------------------------------
//
//	operator/=
//
//		This is the /= operator with complex argument.  Uses the 
//      algorithm in Numerical Recipes to avoid overflow and underflow
//
//------------------------------------------------------------------------------------------------
inline CComplex& CComplex::operator/=(const CComplex& z)
{
	double re = mReal;
	double im = mImag;
	double ratio, den;	// ratio and denominator used in calculations
	
	if (fabs(z.mReal) >= fabs(z.mImag)) {
		ratio = z.mImag / z.mReal;
		den = z.mReal + ratio*z.mImag;
		mReal = (re+ratio*im) / den;
		mImag = (im-ratio*re) / den;
	} else {
		ratio = z.mReal / z.mImag;
		den = z.mImag + ratio*z.mReal;
		mReal = (re*ratio+im) / den;
		mImag = (im*ratio-re) / den;
	}
	return *this;
}
//------------------------------------------------------------------------------------------------
//
//	operator /=
//
//		this is the /= operator with real argument
//
//------------------------------------------------------------------------------------------------
CComplex& CComplex::operator/=(const double&  x)
{
	mReal /= x;
	mImag /= x;
	return *this;
}
//------------------------------------------------------------------------------------------------
//
//	operator !
//
//------------------------------------------------------------------------------------------------
CComplex CComplex::operator!( void )
{
   CComplex c;
   
   c.mReal = ( mReal == 0.0 ? 1.0 : 0.0 );
   c.mImag = ( mImag == 0.0 ? 1.0 : 0.0 );
   
   return c;
}
//------------------------------------------------------------------------------------------------
//  A Sizeof operator
//------------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------------
CComplex::operator size_t( void ) const
{
   return 2 * sizeof( double );
}
//------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------
//  A Binary Operators
//------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------
//
//	operator+
//
//		This is the + operator with 2 complex arguments
//
//------------------------------------------------------------------------------------------------
CComplex operator+(const CComplex& z1, const CComplex& z2)
{
	CComplex r = z1;
	return r += z2;
}
//------------------------------------------------------------------------------------------------
//
//	operator+
//
//		This is the + operator with (real, complex) arguments
//
//------------------------------------------------------------------------------------------------
CComplex operator+(const double&  x,  const CComplex& z)
{
	CComplex r = z;
	return r += x;
}
//------------------------------------------------------------------------------------------------
//
//	operator+
//
//		This is the + operator with (complex, real) arguments
//
//------------------------------------------------------------------------------------------------
CComplex operator+(const CComplex& z,  const double&  x)
{
	CComplex r = z;
	return r += x;
}
//------------------------------------------------------------------------------------------------
//
//	operator-
//
//		This is the - operator with complex arguments
//
//------------------------------------------------------------------------------------------------
CComplex operator-(const CComplex& z1, const CComplex& z2)
{
	CComplex r = z1;
	return r -= z2;
}
//------------------------------------------------------------------------------------------------
//
//	operator-
//
//		This is the - operator with (real, complex) arguments
//
//------------------------------------------------------------------------------------------------
CComplex operator-(const double&  x,  const CComplex& z)
{
	CComplex r(x, 0.0);
	return r -= z;
}
//------------------------------------------------------------------------------------------------
//
//	operator-
//
//		This is the - operator with (complex, real) arguments
//
//------------------------------------------------------------------------------------------------
CComplex operator-(const CComplex& z,  const double&  x)
{
	CComplex r = z;
	return r -= x;
}
//------------------------------------------------------------------------------------------------
//
//	operator*
//
//		This is the * operator with complex arguments
//
//------------------------------------------------------------------------------------------------
CComplex operator*(const CComplex& z1, const CComplex& z2)
{
	CComplex r = z1;
	return r *= z2;
}
//------------------------------------------------------------------------------------------------
//
//	operator*
//
//		This is the * operator with (real, complex) arguments
//
//------------------------------------------------------------------------------------------------
CComplex operator*(const double&  x,  const CComplex& z)
{
	CComplex r = z;
	return r *= x;
}
//------------------------------------------------------------------------------------------------
//
//	operator*
//
//		This is the * operator with (complex, real) arguments
//
//------------------------------------------------------------------------------------------------
CComplex operator*(const CComplex& z,  const double&  x)
{
	CComplex r = z;
	return r *= x;
}
//------------------------------------------------------------------------------------------------
//
//	operator/
//
//		This is the / operator with complex arguments
//
//------------------------------------------------------------------------------------------------
CComplex operator/(const CComplex& z1, const CComplex& z2)
{
	CComplex r = z1;
	return r /= z2;
}
//------------------------------------------------------------------------------------------------
//
//	operator/
//
//		This is the / operator with (real, complex) arguments
//
//------------------------------------------------------------------------------------------------
CComplex operator/(const double&  x,  const CComplex& z)
{
	CComplex r(x, 0.0);
	return r /= z;
}
//------------------------------------------------------------------------------------------------
//
//	operator/
//
//		This is the / operator with (complex, real) arguments
//
//------------------------------------------------------------------------------------------------
CComplex operator/(const CComplex& z,  const double&  x)
{
	CComplex r = z;
	return r /= x;
}
//------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------
//  A Unary Operators
//------------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------------
//
//	operator+
//
//		This is the unary + operator
//
//------------------------------------------------------------------------------------------------
CComplex operator+(const CComplex& z)
{
	CComplex r = z;
	return r;
}
//------------------------------------------------------------------------------------------------
//
//	operator-
//
//		This is the unary - operator
//
//------------------------------------------------------------------------------------------------
CComplex operator-(const CComplex& z)
{
	CComplex r = z;
	return r *= -1.0;
}
//------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------
//  A Comparison Functions
//------------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------------
//
//	operator==
//
//		This is the == operator for complex arguments.  Relies on 
//      default promotion of double to CComplex for real RHS arguments.
//
//------------------------------------------------------------------------------------------------
bool operator==(const CComplex& z1, const CComplex& z2)
{
	return (z1.real() == z2.real()) && (z1.imag() == z2.imag());
}
//------------------------------------------------------------------------------------------------
//
//	operator!=
//
//		This is the != operator for complex arguments.  Relies on 
//      default promotion of double to CComplex for real RHS arguments.
//
//------------------------------------------------------------------------------------------------
bool operator!=(const CComplex& z1, const CComplex& z2)
{
	return (z1.real() != z2.real()) || (z1.imag() != z2.imag());
}
//------------------------------------------------------------------------------------------------
//
//	operator<
//
//		This is the < operator for complex arguments.  Relies on 
//      default promotion of double to CComplex for real RHS arguments.
//      Uses abs(z1) < abs(z2) for comparison.
//
//------------------------------------------------------------------------------------------------
bool operator<(const CComplex& z1, const CComplex& z2)
{
	return (abs(z1) < abs(z2));
}
//------------------------------------------------------------------------------------------------
//
//	operator<=
//
//		This is the <= operator for complex arguments.  Relies on 
//      default promotion of double to CComplex for real RHS arguments.
//      Uses abs(z1) <= abs(z2) for comparison.
//
//------------------------------------------------------------------------------------------------
bool operator<=(const CComplex& z1, const CComplex& z2)
{
	return (abs(z1) <= abs(z2));
}
//------------------------------------------------------------------------------------------------
//
//	operator>
//
//		This is the > operator for complex arguments.  Relies on 
//      default promotion of double to CComplex for real RHS arguments.
//      Uses abs(z1) > abs(z2) for comparison.
//
//------------------------------------------------------------------------------------------------
bool operator>(const CComplex& z1, const CComplex& z2)
{
	return (abs(z1) > abs(z2));
}
//------------------------------------------------------------------------------------------------
//
//	operator>=
//
//		This is the >= operator for complex arguments.  Relies on 
//      default promotion of double to CComplex for real RHS arguments.
//      Uses abs(z1) >= abs(z2) for comparison.
//
//------------------------------------------------------------------------------------------------
bool operator>=(const CComplex& z1, const CComplex& z2)
{
	return (abs(z1) >= abs(z2));
}
//------------------------------------------------------------------------------------------------
//
//	operator==
//
//		This is the == operator for complex arguments.  Relies on 
//      default promotion of double to CComplex for real RHS arguments.
//
//------------------------------------------------------------------------------------------------
bool operator==(const double& x, const CComplex& z)
{
	return (x == z.real()) && (z.imag() == 0.0);
}
//------------------------------------------------------------------------------------------------
//
//	operator!=
//
//		This is the != operator for complex arguments.  Relies on 
//      default promotion of double to CComplex for real RHS arguments.
//
//------------------------------------------------------------------------------------------------
bool operator!=(const double& x, const CComplex& z)
{
	return (x != z.real()) || (z.imag() != 0.0);
}
//------------------------------------------------------------------------------------------------
//
//	operator<
//
//		This is the < operator for complex arguments.  Relies on 
//      default promotion of double to CComplex for real RHS arguments.
//
//------------------------------------------------------------------------------------------------
bool operator<(const double& x, const CComplex& z)


{
	return (x < z.real()) && (z.imag() == 0.0);
}
//------------------------------------------------------------------------------------------------
//
//	operator<=
//
//		This is the <= operator for complex arguments.  Relies on 
//      default promotion of double to CComplex for real RHS arguments.
//
//------------------------------------------------------------------------------------------------
bool operator<=(const double& x, const CComplex& z)
{
	return (x <= z.real()) && (z.imag() == 0.0);
}
//------------------------------------------------------------------------------------------------
//
//	operator>
//
//		This is the > operator for complex arguments.  Relies on 
//      default promotion of double to CComplex for real RHS arguments.
//
//------------------------------------------------------------------------------------------------
bool operator>(const double& x, const CComplex& z)
{
	return (x > z.real()) && (z.imag() == 0.0);
}
//------------------------------------------------------------------------------------------------
//
//	operator>=
//
//		This is the >= operator for complex arguments.  Relies on 
//      default promotion of double to CComplex for real RHS arguments.
//
//------------------------------------------------------------------------------------------------
bool operator>=(const double& x, const CComplex& z)
{
	return (x >= z.real()) && (z.imag() == 0.0);
}
//------------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------------
//
//	ToString(z)
//
//------------------------------------------------------------------------------------------------
AnsiString CComplex::ToString( void ) const
{
   AnsiString res;
   
   if( mImag )
   {
      res = AnsiString( "(" ) + 
            DoubleToStr( mReal ) + 
            AnsiString( "," ) +
            DoubleToStr( mImag ) +
            AnsiString( "i)" );
   }
   else
   {
      res = DoubleToStr( mReal );
   }
   return res;
}
//------------------------------------------------------------------------------------------------
//  An I/O Operators
//------------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------------
int CComplex::Save( FILE* f )
{
   int written = 0;
   if( f )
   {
      written += fwrite( &mReal, 1, sizeof( double ), f );
      written += fwrite( &mImag, 1, sizeof( double ), f );
   }
   return written;
}
//------------------------------------------------------------------------------------------------
int CComplex::Load( FILE* f )
{
   int readed = 0;
   if( f )
   {
      readed += fread( &mReal, 1, sizeof( double ), f );
      readed += fread( &mImag, 1, sizeof( double ), f );
   }
   return readed;
}
//------------------------------------------------------------------------------------------------
//  A Polar Operators
//------------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------------
//
//	polar(r,theta)
//
//		This function returns a complex number from its polar 
//      representation.  theta should be in radians
//
//------------------------------------------------------------------------------------------------
CComplex polar (const double& r, const double& theta)
{
	double re, im;		// real and imaginary parts of new complex number
	re = r*cos(theta);
	im = r*sin(theta);
	CComplex z(re, im);
	return z;
}
//------------------------------------------------------------------------------------------------
//
//	conj(z)
//
//		This function returns the complex conjugate of z
//
//------------------------------------------------------------------------------------------------
CComplex conj  (const CComplex& z)
{
	CComplex zz(z.real(), -z.imag());
	return zz;
}
//------------------------------------------------------------------------------------------------
//
//	angle(z)
//
//		This function returns the argument (angle) of z
//
//------------------------------------------------------------------------------------------------
double  angle (const CComplex& z)
{
	return atan2(z.imag(), z.real());
}
//------------------------------------------------------------------------------------------------
//
//	arg(z)
//

//		This function returns the argument (angle) of z
//
//------------------------------------------------------------------------------------------------
double  arg (const CComplex& z)
{
	return atan2(z.imag(), z.real());
}
//------------------------------------------------------------------------------------------------
//
//	real(z)
//
//		This function returns the real part of z
//
//------------------------------------------------------------------------------------------------
double  real  (const CComplex& z)
{
	return z.real();
}
//------------------------------------------------------------------------------------------------
//
//	imag(z)
//
//		This function returns the imaginary part of z
//
//------------------------------------------------------------------------------------------------
double  imag  (const CComplex& z)
{
	return z.imag();
}
//------------------------------------------------------------------------------------------------
//
//	sign(z)
//
//		This function returns the "sign" of z defined by 
//      sign(z) = z/abs(z) (as defined in MATLAB)
//
//------------------------------------------------------------------------------------------------
CComplex sign  (const CComplex& z)
{
	CComplex ans = z;
	ans /= abs(z);
	return ans;
}
//------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------
//  A Math Functions
//------------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------------
//
//	sin(z)
//
//		This function returns the sine of z
//
//------------------------------------------------------------------------------------------------
CComplex sin   (const CComplex& z)
{
    CComplex i(0.0, 1.0);
    
	return (exp(i*z) - exp(-i*z)) / (2.0*i);
}
//------------------------------------------------------------------------------------------------
//
//	sinh(z)
//
//		This function returns the hyperbolic sine of z
//
//------------------------------------------------------------------------------------------------
CComplex sinh  (const CComplex& z)
{
	return (exp(z) - exp(-z)) / 2.0;
}
//------------------------------------------------------------------------------------------------
//
//	asin(z)
//
//		This function returns the inverse sine of z
//
//------------------------------------------------------------------------------------------------
CComplex asin  (const CComplex& z)
{
	double x, y;		// real and imaginary parts of z
	double u, v;
	double a, b;
	
	x = z.real();
	y = z.imag();
	
	u = 0.5*sqrt((x+1.0)*(x+1.0) + y*y);
	v = 0.5*sqrt((x-1.0)*(x-1.0) + y*y);
	
	a = u + v;
	b = u - v;
	
	if (y > 0.0)
		return CComplex(asin(b), log(a+sqrt(a*a-1.0)));
	else
		return CComplex(asin(b), -log(a+sqrt(a*a-1.0)));
}
//------------------------------------------------------------------------------------------------
//
//	asinh(z)
//
//		This function returns the inverse hyperbolic sine of z
//
//------------------------------------------------------------------------------------------------
CComplex asinh (const CComplex& z)
{
    CComplex i(0.0, 1.0);
    
	return -i*asin(i*z);
}
//------------------------------------------------------------------------------------------------
//
//	cos(z)
//
//		This function returns the cosine of z
//
//------------------------------------------------------------------------------------------------
CComplex cos   (const CComplex& z)
{
    CComplex i(0.0, 1.0);
    
	return (exp(i*z) + exp(-i*z)) / 2.0;
}
//------------------------------------------------------------------------------------------------
//
//	cosh(z)
//
//		This function returns the hyperbolic cosine of z
//
//------------------------------------------------------------------------------------------------
CComplex cosh  (const CComplex& z)
{
	return (exp(z) + exp(-z)) / 2.0;
}
//------------------------------------------------------------------------------------------------
//
//	acos(z)
//
//		This function returns the inverse cosine of z
//
//------------------------------------------------------------------------------------------------
CComplex acos  (const CComplex& z)
{
	double x, y;		// real and imaginary parts of z
	double u, v;
	double a, b;
	
	x = z.real();
	y = z.imag();
	
	u = 0.5*sqrt((x+1.0)*(x+1.0) + y*y);
	v = 0.5*sqrt((x-1.0)*(x-1.0) + y*y);
	
	a = u + v;
	b = u - v;
	
	if (y > 0.0)
		return CComplex(acos(b), -log(a+sqrt(a*a-1.0)));
	else
		return CComplex(acos(b), log(a+sqrt(a*a-1.0)));
}
//------------------------------------------------------------------------------------------------
//
//	acosh(z)
//
//		This function returns the inverse hyperbolic cosine of z
//
//------------------------------------------------------------------------------------------------

CComplex acosh (const CComplex& z)
{
    CComplex i(0.0, 1.0);
    
	if (z.real()*z.imag() > 0.0)
		return i*acos(z);
	else
		return -i*acos(z);
}
//------------------------------------------------------------------------------------------------
//
//	tan(z)
//
//		This function returns the tangent of z
//
//------------------------------------------------------------------------------------------------
CComplex tan   (const CComplex& z)
{
	return sin(z)/cos(z);
}
//------------------------------------------------------------------------------------------------
//
//	tanh(z)
//
//		This function returns the hyperbolic tangent of z
//
//------------------------------------------------------------------------------------------------
CComplex tanh  (const CComplex& z)
{
	return sinh(z)/cosh(z);
}
//------------------------------------------------------------------------------------------------
//
//	atan(z)
//
//		This function returns the inverse tangent of z
//
//------------------------------------------------------------------------------------------------
CComplex atan  (const CComplex& z)
{
	double x, y;		// real and imaginary parts of z
	double u, v, w;
	
	x = z.real();
	y = z.imag();
	
	u = 1.0 - x*x - y*y;
	v = x*x + (y+1.0)*(y+1.0);
	w = x*x + (y-1.0)*(y-1.0);
	
	if (abs(z) <= 1.0)
		return CComplex(0.5*atan(2.0*x/u), 0.25*log(v/w));
	else
		return CComplex(-atan(2.0*x/u), 0.25*log(v/w));
}
//------------------------------------------------------------------------------------------------
//
//	atanh(z)
//
//		This function returns the inverse hyperbolic tangent of z
//
//------------------------------------------------------------------------------------------------
CComplex atanh (const CComplex& z)
{
    CComplex i(0.0, 1.0);
    
	return -i*atan(i*z);
}
//------------------------------------------------------------------------------------------------
//
//	sec(z)
//
//		This function returns the secant of z
//
//------------------------------------------------------------------------------------------------
CComplex sec   (const CComplex& z)
{
	return 1.0/cos(z);
}
//------------------------------------------------------------------------------------------------
//
//	sech(z)
//
//		This function returns the hyperbolic secant of z
//
//------------------------------------------------------------------------------------------------
CComplex sech  (const CComplex& z)
{
	return 1.0/cosh(z);
}
//------------------------------------------------------------------------------------------------
//
//	asec(z)
//
//		This function returns the inverse secant of z
//
//------------------------------------------------------------------------------------------------
CComplex asec  (const CComplex& z)
{
	return acos(1.0/z);
}
//------------------------------------------------------------------------------------------------
//
//	asech(z)
//
//		This function returns inverse hyperbolic secant of z
//
//------------------------------------------------------------------------------------------------
CComplex asech (const CComplex& z)
{
    CComplex i(0.0, 1.0);
    
	if (z.real()*z.imag() > 0.0)
		return -i*asec(z);
	else
		return i*asec(z);
}
//------------------------------------------------------------------------------------------------
//
//	csc(z)
//
//		This function returns the cosecant of z
//
//------------------------------------------------------------------------------------------------
CComplex csc   (const CComplex& z)
{
	return 1.0/sin(z);
}
//------------------------------------------------------------------------------------------------
//
//	csch(z)
//
//		This function returns the hyperbolic cosecant of z
//
//------------------------------------------------------------------------------------------------
CComplex csch  (const CComplex& z)
{
	return 1.0/sinh(z);
}
//------------------------------------------------------------------------------------------------
//
//	acsc(z)
//
//		This function returns the inverse cosecant of z
//
//------------------------------------------------------------------------------------------------
CComplex acsc  (const CComplex& z)
{
	return asin(1.0/z);
}
//------------------------------------------------------------------------------------------------
//
//	acsch(z)
//
//		This function returns the inverse hyperbolic cosecant of z
//
//------------------------------------------------------------------------------------------------
CComplex acsch (const CComplex& z)
{
    CComplex i(0.0, 1.0);
    
	return i*acsc(i*z);
}
//------------------------------------------------------------------------------------------------
//
//	cot(z)
//
//		This function returns the cotangent of z
//
//------------------------------------------------------------------------------------------------
CComplex cot   (const CComplex& z)
{
	return cos(z)/sin(z);
}
//------------------------------------------------------------------------------------------------
//
//	coth
//
//		This function returns the hyperbolic cotangent of z
//
//------------------------------------------------------------------------------------------------
CComplex coth  (const CComplex& z)
{
	return cosh(z)/sinh(z);
}
//------------------------------------------------------------------------------------------------
//
//	acot(z)
//
//		This function returns the inverse cotangent of z
//
//------------------------------------------------------------------------------------------------
CComplex acot  (const CComplex& z)
{
	return atan(1.0/z);
}
//------------------------------------------------------------------------------------------------
//
//	acoth(z)
//
//		This function returns the inverse hyperbolic cotangent of z
//
//------------------------------------------------------------------------------------------------
CComplex acoth (const CComplex& z)
{
    CComplex i(0.0, 1.0);
    
	return i*acot(i*z);
}
//------------------------------------------------------------------------------------------------
//
//	exp(z)
//
//		This function returns the exponent of z
//
//------------------------------------------------------------------------------------------------
CComplex exp   (const CComplex& z)
{
	double y;		// imaginary part of z
	double ex;		// value of e^x (x == real part of z)
	ex = exp(z.real());
	y = z.imag();
	return CComplex(ex*cos(y), ex*sin(y));
}
//------------------------------------------------------------------------------------------------
//
//	log(z)
//
//		This functin returns the natural logarithm of z
//
//------------------------------------------------------------------------------------------------
CComplex log   (const CComplex& z)
{
	double r, theta;		// polar representation
	r = abs(z);
	theta = angle(z);
	return CComplex(log(r), theta);
}
//------------------------------------------------------------------------------------------------
CComplex ln    (const CComplex& z)
{
   return log( z );
}
//------------------------------------------------------------------------------------------------
//
//	log10(z)
//
//		This function returns the base-10 log of z
//
//------------------------------------------------------------------------------------------------
CComplex log10 (const CComplex& z)
{
	return log(z)/log(10.0);
}
//------------------------------------------------------------------------------------------------
//
//	pow(z,x)
//
//		This function returns the power z of x.  
//
//------------------------------------------------------------------------------------------------
CComplex pow( const CComplex& v, double& expon )
{
   CComplex r;

   if( v.iszero() && expon > 0.0 )
   {
      //r.__re = 0.0;
      //r.__im = 0.0;
      r = CComplex( 0.0, 0.0 );
   }
   else if( expon == 1.0 )
   {
      //r.__re = 1.0;
      //r.__im = 0.0;
      r = CComplex( 1.0, 0.0 );
   }
   else
   {
      r = polar( pow( abs( v ), expon ), expon * arg( v ) );
   }
   return r;
}
//---------------------------------------------------------------------------
CComplex pow( double& v, const CComplex& expon )
{
   CComplex r;

   if( v == 0.0 && expon.real() > 0.0 )
   {
      r = CComplex( 0.0, 0.0 );
      //r.__re = 0.0;
      //r.__im = 0.0;
   }
   else
   {
      double lnx = log( fabs( v ) );
      if( v > 0.0 )
      {
         r = exp( expon * lnx );
      }
      else
      {
         r = exp( expon * CComplex( lnx, M_PI ) );
      }
   }
   return r;
}
//---------------------------------------------------------------------------
CComplex pow( const CComplex& v, const CComplex& expon )
{
   CComplex r;

   if( v.iszero() && expon.real() > 0.0 )
   {
      //r.__re = 0.0;
      //r.__im = 0.0;
      r = CComplex( 0.0, 0.0 );
   }
   else if( expon.real() == 0.0 )
   {
      //r.__re = 1.0;
      //r.__im = 0.0;
      r = CComplex( 1.0, 0.0 );
   }
   else
   {
      r = exp( expon * ln( v ) );
   }
   return r;
}
//------------------------------------------------------------------------------------------------
//
//	sqrt(z)
//
//		This function returns the square-root of z.  It uses the 
//      algorithm presented in Numerical Recipes to avoid overflow
//      and underflow of the result
//
//------------------------------------------------------------------------------------------------
CComplex sqrt  (const CComplex& z)
{
	CComplex c;
	double x, y, w, r;
	if ((z.real() == 0.0) && (z.imag() == 0)) {
		return c;		// default constructor makes c = 0
	}
	else {
		x = fabs(z.real());
		y = fabs(z.imag());
		if (x >= y) {
			r = y/x;
			w = sqrt(x)*sqrt(0.5*(1.0+sqrt(1.0+r*r)));
		}
		else {
			r = x/y;
			w = sqrt(y)*sqrt(0.5*(r+sqrt(1.0+r*r)));
		}
		if (z.real() >= 0.0) {
			c = CComplex(w, z.imag()/(2.0*w));
		}
		else {
			// reuse x and y here to save variables
			y = (z.imag() >= 0.0) ? w : -w;
			x = z.imag()/(2.0*x);
			c = CComplex(x,y);
		}
		return c;
	}
}
//------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------
//  A Rounding Functions
//------------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------------
//
//	fix(z)
//
//		This function rounds the elements of z toward zeros
//
//------------------------------------------------------------------------------------------------
CComplex fix   (const CComplex& z)
{
	double x, y;
	if (z.real() >= 0.0)
		x = floor(z.real());
	else
		x = ceil(z.real());
	if (z.imag() >= 0.0)
		y = floor(z.imag());
	else
		y = ceil(z.imag());
	return CComplex(x,y);
}
//------------------------------------------------------------------------------------------------
//
//	floor(z)
//
//		This function rounds the elements of z to -infinity
//
//------------------------------------------------------------------------------------------------
CComplex floor (const CComplex& z)
{
	return CComplex(floor(z.real()), floor(z.imag()));
}
//------------------------------------------------------------------------------------------------
//
//	ceil(x)
//
//		This function rounds the elements of z to +infinity
//
//------------------------------------------------------------------------------------------------
CComplex ceil  (const CComplex& z)
{
	return CComplex(ceil(z.real()), ceil(z.imag()));
}
//------------------------------------------------------------------------------------------------
//
//	round(z)
//
//		This function rounds the elements of z to the nearest integer
//
//------------------------------------------------------------------------------------------------
CComplex round (const CComplex& z)
{
	double x, y;
	if (z.real() >= 0.0)
		x = floor(z.real() + 0.5);
	else
		x = ceil(z.real() - 0.5);
	if (z.imag() >= 0.0)
		y = floor(z.imag() + 0.5);
	else
		y = ceil(z.imag() - 0.5);
	return CComplex(x,y);
}
//------------------------------------------------------------------------------------------------
CComplex round( const CComplex& z, int prec )
{
	double x, y;
	x = dround( z.real(), prec );
	y = dround( z.imag(), prec );
	return CComplex( x, y );
}
//------------------------------------------------------------------------------------------------
AnsiString ComplexToStr( const CComplex& z )
{
   return z.ToString();
}
//------------------------------------------------------------------------------------------------
AnsiString ComplexToStrF( const CComplex& z, int prec )
{
   AnsiString p;
   
   if( prec < 0 )
   {
      prec = 6;
   }
   if( z.imag() != 0.0 )
   {
      p = AnsiString( "(" ) + DoubleToStr( z.real(), prec ) +
          AnsiString( "," ) + DoubleToStr( z.imag(), prec ) +
          AnsiString( "i)" );
   }
   else
   {
      p = DoubleToStr( z.real(), prec );
   }
   return p;
}
//------------------------------------------------------------------------------------------------
AnsiString ComplexToStrE( const CComplex& z, int prec )
{
   AnsiString p;

   if( prec < 0 )
   {
      prec = 6;
   }
   if( z.imag() != 0.0 )
   {
      p = AnsiString( "(" ) + DoubleToStrE( z.real(), prec ) +
          AnsiString( "," ) + DoubleToStrE( z.imag(), prec ) +
          AnsiString( "i)" );
   }
   else
   {
      p = DoubleToStrE( z.real(), prec );
   }
   return p;
}
//------------------------------------------------------------------------------------------------
CComplex StrToComplex( const char* Value )
{
   //bool lError = false;
   int len;
   int sep;
   char buf[ 200 ];
   char cre[ 100 ];
   char cim[ 100 ];
   double re = 0.0;
   double im = 0.0;

   strtrim( buf, Value );

   len = strlen( buf ); //buf.Length();
   if( len && isdoublestr( buf ) )
   {
      re = StrToDouble( buf );
      im = 0.0;
   }
   else if( len >= 5 )
   {
      //lprintf( "StrToComplex(): Buf = [%s]\n", buf );

      sep = strpos( buf, ',' );
      //lprintf( "Sep pos( , ) = %d\n", sep );
      if( sep == -1 )
      {
         sep = strpos( buf, ';' );
         //lprintf( "Sep pos( ; ) = %d\n", sep );
      }

      if( buf[ 0 ] == '(' && buf[ len - 1 ] == ')' && sep > 0 && sep < len - 1 )
      {
         strncpy( cre, buf + 1, sep - 1 );
         cre[ sep - 1 ] = '\0';
         if( isdoublestr( cre ) )
         {
            re = StrToDouble( cre );
         }
         else if( isexponentstr( cre ) )
         {
            re = StrEToDouble( cre );
         }
         else
         {
            re = 0.0;
         }

         if( buf[ len - 2 ] == 'i' )
         {
            strncpy( cim, buf + sep + 1, len - sep - 3 );
            cim[ len - sep - 3 ] = '\0';
         }
         else
         {
            strncpy( cim, buf + sep + 1, len - sep - 2 );
            cim[ len - sep - 2 ] = '\0';
         }

         if( isdoublestr( cim ) )
         {
            im = StrToDouble( cim );
         }
         else if( isexponentstr( cim ) )
         {
            im = StrEToDouble( cim );
         }
         else
         {
            im = 0.0;
         }
      }
      else if( sep == -1 && isdoublestr( buf ) )
      {
         re = StrToDouble( buf );
         im = 0.0;
      }
      else if( sep == -1 && isexponentstr( buf ) )
      {
         re = StrEToDouble( buf );
         im = 0.0;
      }
   }
   else
   {
      //lError = true;
   }
   return CComplex( re, im );
}
//------------------------------------------------------------------------------------------------

