#include <stdio.h>
#include <string.h>

#ifdef __WIN32__
#include <windows.h>
#endif

#include <astring.h>
#include <atomutil.h>

#include "atom.h"

//-----------------------------------------------------------------------------
const char* __AtomTypes[] = {
                      "UNKNOWN", // 0x00
                      "INT", // 0x01
                      "BOOL", // 0x02
                      "DOUBLE", // 0x03
                      "EXPONENT", // 0x04
                      "MEASURE", // 0x05
                      "COMPLEX", // 0x06
                      "VECTOR", // 0x07
                      "MATRIX", // 0x08
                      "", // 0x09
                      "", // 0x0A
                      "", // 0x0B
                      "", // 0x0C
                      "", // 0x0D
                      "", // 0x0E
                      "", // 0x0F
                      "STRING", // 0x10
                      "EXPRSS", // 0x11
                      "", // 0x12
                      "", // 0x13
                      "", // 0x14
                      "", // 0x15
                      "", // 0x16
                      "", // 0x17
                      "", // 0x18
                      "", // 0x19
                      "", // 0x1A
                      "", // 0x1B
                      "", // 0x1C
                      "", // 0x1D
                      "", // 0x1E
                      "", // 0x1F
                      "SYMBOL", // 0x20
                      "", // 0x21
                      "?"
                      };
//-----------------------------------------------------------------------------
double cot( double x )
{
   return 1.0 / tan( x );
}
//-----------------------------------------------------------------------------
double coth( const double x )
{
	return cosh( x ) / sinh( x );
}
//------------------------------------------------------------------------------------------------
double acot( const double x )
{
	return atan( 1.0 / x );
}
//------------------------------------------------------------------------------------------------
double sec( const double x )
{
	return 1.0 / cos( x );
}
//------------------------------------------------------------------------------------------------
double sech( const double x )
{
	return 1.0 / cosh( x );
}
//------------------------------------------------------------------------------------------------
double asec( const double x )
{
	return acos( 1.0 / x );
}
//------------------------------------------------------------------------------------------------
double csc( const double x )
{
	return 1.0 / sin( x );
}
//------------------------------------------------------------------------------------------------
double csch( const double x )
{
	return 1.0 / sinh( x );
}
//------------------------------------------------------------------------------------------------
double acsc( const double x )
{
	return asin( 1.0 / x );
}
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
void double2str( double x, char* buf, int prec = -1 )
{
   int j;
   bool lExit = false;
   
   if( !buf )
   {
      return;
   }
   if( prec < 0 )
   {
      prec = 6;
   }
   
   sprintf( buf, "%50.*f", prec, x );
   for( j = strlen( buf ) - 1; j >= 0 && !lExit; j-- )
   {
      if( buf[ j ] == '0' || buf[ j ] == '\0' )
      {
         buf[ j ] = '\0';
      }
      else if( buf[ j ] == '.' )
      {
         buf[ j ] = '\0';
      }
      else
      {
         lExit = true;
      }
   }
}
//-----------------------------------------------------------------------------
//
//
//
//-----------------------------------------------------------------------------
atom nullatom;
atom NULLA;
atom ZeroAtom = atom( 0.0 );
atom invalidatom( atom( false ), false );
//-----------------------------------------------------------------------------
atom::atom( void )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   ////lprintf( "Atom Void()\n" );
   //_type = aUnknown;
   //_format = aifUnknown;
   //_value = NULL;
   //_name = NULL;
   //strbuf = NULL;
   //_valid = false;
}
//-----------------------------------------------------------------------------
atom::atom( int avalue, unsigned char aformat )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   //lprintf( "Atom Int( %d )\n", avalue );
   _type = aInteger;
   _format = aformat;
   _value = new int( avalue );
   _name = NULL;
   strbuf = NULL;
   _valid = true;
}
//-----------------------------------------------------------------------------
atom::atom( bool avalue )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   //lprintf( "Atom Bool( %s )\n", avalue ? "True" : "False" );
   _type = aBoolean;
   _format = aifDec;
   _value = new bool( avalue );
   _name = NULL;
   strbuf = NULL;
   _valid = true;
}
//-----------------------------------------------------------------------------
atom::atom( double avalue, unsigned char aformat )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   //lprintf( "Atom Double( %f )\n", avalue );
   _type = aDouble;
   _format = aformat;
   _value = new double( avalue );
   _name = NULL;
   strbuf = NULL;
   _valid = true;
}
//-----------------------------------------------------------------------------
atom::atom( double are, double aim, unsigned char aformat )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   _type = aComplex;
   _format = aformat;
   _value = new CComplex( are, aim );
   _name = NULL;
   strbuf = NULL;
   _valid = true;
}
//-----------------------------------------------------------------------------
atom::atom( CComplex c, unsigned char aformat )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   _type = aComplex;
   _format = aformat;
   _value = new CComplex( c.real(), c.imag() );
   _name = NULL;
   strbuf = NULL;
   _valid = true;
}
//-----------------------------------------------------------------------------
atom::atom( CVector v )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   _type = aVector;
   _format = aifDec;
   _value = new CVector( v );
   _name = NULL;
   strbuf = NULL;
   _valid = true;
}
//-----------------------------------------------------------------------------
atom::atom( CMatrix m )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   _type = aMatrix;
   _format = aifDec;
   _value = new CMatrix( m );
   _name = NULL;
   strbuf = NULL;
   _valid = true;
}
//-----------------------------------------------------------------------------
atom::atom( CMeasure m, unsigned char aformat )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   _type = aMeasure;
   _format = aformat;
   _value = new CMeasure( m );
   _name = NULL;
   strbuf = NULL;
   _valid = true;
}
//-----------------------------------------------------------------------------
atom::atom( const char* str )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   _format = aifDec;
   if( str )
   {
      if( isstringstr( str ) )
      {
         _type = aString;
      }
      else if( isexprstr( str ) )
      {
         _type = aExpression;
      }
      else if( issymbolstr( str ) )
      {
         _type = aSymbol;
      }
      else
      {
         _type = aUnknown;
      }
   }
   if( _type != aUnknown )
   {
      char* ptr;
      int len = strlen( str );
      if( _type == aString || _type == aExpression )
      {
         _value = new char[ len - 1 ];
         ptr = (char*)( _value );
         strncpy( ptr, str + 1, len - 2 );
         ptr[ len - 2 ] = '\0';
      }
      else if( _type == aSymbol )
      {
         _value = new char[ len + 1 ];
         ptr = (char*)( _value );
         strncpy( ptr, str, len );
         ptr[ len ] = '\0';
         //setname( str );
      }
      _valid = true;
   }
   else
   {
      _value = NULL;
      _name = NULL;
      _valid = false;
   }
   strbuf = NULL;
}
//-----------------------------------------------------------------------------
atom::atom( const atom& a )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   //lprintf( "Atom Atom&( %s, %s )\n", a.typestr(), a.getstring() );
   /*
   _type = a.type();
   _valid = a.valid();
   setname( a._name );
   switch( _type )
   {
      case aInteger:
      _value = new int( *(int*)( a._value ) );
      break;

      case aBoolean:
      _value = new bool( *(bool*)( a._value ) );
      break;

      case aDouble:
      ////lprintf( "Getting double value...\n" );
      _value = new double( *(double*)( a._value ) );
      break;

      case aComplex:
      _value = new CComplex( ((CComplex*)(a._value))->real(), ((CComplex*)(a._value))->imag() );
      break;

      case aVector:
      _value = new CVector( *((CVector*)(a._value)) );
      break;

      case aMatrix:
      _value = new CMatrix( *((CMatrix*)(a._value)) );
      break;

      case aMeasure:
      _value = new CMeasure( *((CMeasure*)(a._value)) );
      break;

      case aString:
      case aExpression:
      _value = new char[ strlen( (char*)( a._value ) ) + 1 ];
      memcpy( _value, a._value, strlen( (char*)( a._value ) ) + 1 );
      break;

      default:
      _value = NULL;
      break;
   }
   */
   *this = a;
}
//-----------------------------------------------------------------------------
atom::atom( const atom& a, const char* name )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   *this = a;
   setname( name );
}
//-----------------------------------------------------------------------------
atom::atom( const atom& a, bool valid )
   :  strbuf(NULL),
     _type( aUnknown ),
     _format(aifUnknown),
     _value(NULL),
     _name(NULL),
     _valid(false)
{
   *this = a;
   _valid = valid;
}
//-----------------------------------------------------------------------------
atom::~atom()
{
   clear();
}
//-----------------------------------------------------------------------------
//bool atom::valid( void ) const
//{
//   return _valid;
//}
//-----------------------------------------------------------------------------
const atom& atom::operator=( const atom& a )
{
   clear();
   
   _type = a._type;
   _format = a._format;
   _valid = a._valid;
   switch( _type )
   {
      case aInteger:
      _value = new int( *(int*)( a._value ) );
      break;

      case aBoolean:
      _value = new bool( *(bool*)( a._value ) );
      break;
      
      case aDouble:
      _value = new double( *(double*)( a._value ) );
      break;
      
      case aComplex:
      if( ((CComplex*)(a._value))->imag() )
      {
         _value = new CComplex( ((CComplex*)(a._value))->real(), ((CComplex*)(a._value))->imag() );
      }
      else
      {
         _type = aDouble;
         _value = new double( ((CComplex*)(a._value))->real() );
      }
      break;
      
      case aVector:
      _value = new CVector( *((CVector*)(a._value)) );
      break;
      
      case aMatrix:
      _value = new CMatrix( *((CMatrix*)(a._value)) );
      break;
      
      case aMeasure:
      _value = new CMeasure( *((CMeasure*)(a._value)) );
      break;
      
      case aString:
      case aExpression:
      _value = new char[ strlen( (char*)( a._value ) ) + 1 ];
      memcpy( _value, a._value, strlen( (char*)( a._value ) ) + 1 );
      break;

      case aSymbol:
      _value = new char[ strlen( (char*)( a._value ) ) + 1 ];
      memcpy( _value, a._value, strlen( (char*)( a._value ) ) + 1 );
      //setname( (char*)( _value ) );
      break;
      
      default:
      _value = NULL;
      break;
   }
   
   if( a._name )
   {
      setname( a._name );
   }
   return *this;
}
//-----------------------------------------------------------------------------
atom atom::operator!( void )
{
   bool b = false;
   atom x;
   if( _type == aBoolean )
   {
      b = *(bool*)(_value);
   }
   else if( isnumeric() )
   {
      b = ( (CComplex)( *this ) == CComplex( 0.0, 0.0 ) ); //false;
   }
   x = atom( !b );
   return x;
}
//-----------------------------------------------------------------------------
bool atom::iscompatible( atom& a )
{
   bool res;
   
   if( isnumeric() && a.isnumeric() )
   {
      res = true;
   }
   else
   {
      res = ( _type == a.type() );
   }
   return res;
}
//-----------------------------------------------------------------------------
void atom::clear( void )
{
   if( strbuf )
   {
      delete [] strbuf;
      strbuf = NULL;
		#ifdef __DEBUG__
      fprintf( stderr, "Deleted strbuf...\n" );
		#endif
   }
   else
   {
   }

   if( _name )
   {
      delete [] _name;
      _name = NULL;
		#ifdef __DEBUG__
      fprintf( stderr, "Delete name...\n" );
		#endif
   }
   
   //fprintf( stderr, "Will try to delete value...\n" ); 
   if( _value )
   {
		#ifdef __DEBUG__
      fprintf( stderr, "Will delete the value of type %d...\n", _type );
		#endif

      switch( _type )
      {
      case aInteger:
      delete (int*)( _value );
      break;
      
      case aBoolean:
      delete (bool*)( _value );
      break;

      case aDouble:
      delete (double*)( _value );
		#ifdef __DEBUG__
      fprintf( stderr, "Deleted double value...\n ");
		#endif
      break;

      case aComplex:
      delete (CComplex*)( _value );
      #ifdef __DEBUG__
      fprintf( stderr, "Deleted CComplex value...\n ");
      #endif
      break;

      case aVector:
      delete (CVector*)( _value );
      break;

      case aMatrix:
      delete (CMatrix*)( _value );
      break;

      case aMeasure:
      delete (CMeasure*)( _value );
      break;

      case aString:
      case aExpression:
      case aSymbol:
      delete [] (char*)( _value );
      break;

      default:
      break;
      }
      _value = NULL;
   }
   _type = aUnknown;

	#ifdef __DEBUG__
   fprintf( stderr, "Clear() function has finished...\n" );
	#endif
}
//-----------------------------------------------------------------------------
void atom::setname( const char* buf )
{
   int len;
   if( _name )
   {
      delete [] _name;
   }
   _name = NULL;
   if( buf )
   {
      len = strlen( buf );
      if( len )
      {
         _name = new char[ len + 1 ];
         strcpy( _name, buf );
      }
   }
}
//-----------------------------------------------------------------------------
void atom::setformat( unsigned char aformat )
{
   if( _type == aInteger )
   {
      _format = aformat;
   }
}
//-----------------------------------------------------------------------------
const char* atom::name( void )
{
   return _name;
}
//-----------------------------------------------------------------------------
void atom::release( void )
{
   if( strbuf )
   {
      delete [] strbuf;
      strbuf = NULL;
   }
}
//-----------------------------------------------------------------------------
bool atom::save( FILE* f )
{
   bool res = false;

   if( f )
   {
      //double re, im;
      int written = 0;
      //int j, x, y;
      int namelen = _name ? strlen( _name ) + 1 : 0;
      int size = 3 * sizeof( int ) + atomsize() + sizeof( bool );
      if( namelen )
      {
         size += namelen;
      }

      written += fwrite( &size, 1, sizeof( int ), f );
      written += fwrite( &namelen, 1, sizeof( int ), f );
      written += fwrite( &_type, 1, sizeof( int ), f );
      if( namelen )
      {
         written += fwrite( _name, 1, namelen, f );
      }
      written += fwrite( &_valid, 1, sizeof( bool ), f );
      switch( _type )
      {
         case aBoolean:
         written += fwrite( _value, 1, sizeof( bool ), f );
         break;
         
         case aInteger:
         written += fwrite( _value, 1, sizeof( int ), f );
         written += fwrite( &_format, 1, sizeof( unsigned char ), f );
         break;
         
         case aDouble:
         written += fwrite( _value, 1, sizeof( double ), f );
         written += fwrite( &_format, 1, sizeof( unsigned char ), f );
         break;
         
         case aComplex:
         written += ((CComplex*)(_value))->Save( f );
         written += fwrite( &_format, 1, sizeof( unsigned char ), f );
         break;
         
         case aVector:
         written += ((CVector*)(_value))->Save( f );
         break;
         
         case aMatrix:
         written += ((CMatrix*)(_value))->Save( f );
         break;

         case aMeasure:
         written += ((CMeasure*)(_value))->Save( f );
         written += fwrite( &_format, 1, sizeof( unsigned char ), f );
         break;
                  
         case aString:
         written += fwrite( _value, 1, strlen( (char*)(_value) ) + 1, f );
         break;

         case aExpression:
         written += fwrite( _value, 1, strlen( (char*)(_value) ) + 1, f );
         break;
         
         case aSymbol:
         written += fwrite( _value, 1, strlen( (char*)(_value) ) + 1, f );
         break;

         default:
         break;
      }
      if( written == size )
      {
         res = true;
      }
      else
      {
      }
   }
   return res;
}
//-----------------------------------------------------------------------------
bool atom::load( FILE* f )
{
   bool res = false;
   
   clear();
   
   if( f )
   {
      // double re, im;
      int readed = 0;
      int x;
      int namelen; // = _name ? strlen( _name ) + 1 : 0;
      int size; // = 3 * sizeof( int ) + atomsize() + sizeof( bool );
      
      readed += fread( &size, 1, sizeof( int ), f );

      readed += fread( &namelen, 1, sizeof( int ), f );
      readed += fread( &_type, 1, sizeof( int ), f );
      if( namelen )
      {
         readed += fread( _name, 1, namelen, f );
      }
      readed += fread( &_valid, 1, sizeof( bool ), f );
      switch( _type )
      {
         case aBoolean:
         _value = new bool;
         readed += fread( _value, 1, sizeof( bool ), f );
         break;
         
         case aInteger:
         _value = new int;
         readed += fread( _value, 1, sizeof( int ), f );
         readed += fread( &_format, 1, sizeof( unsigned char ), f );
         break;
         
         case aDouble:
         _value = new double;
         readed += fread( _value, 1, sizeof( double ), f );
         readed += fread( &_format, 1, sizeof( unsigned char ), f );
         break;
         
         case aComplex:
         _value = new CComplex( 0.0, 0.0 );
         readed += ((CComplex*)(_value))->Load( f );
         readed += fread( &_format, 1, sizeof( unsigned char ), f );
         break;
         
         case aVector:
         _value = new CVector();
         readed += ((CVector*)(_value))->Load( f );
         break;
         
         case aMatrix:
         _value = new CMatrix();
         readed += ((CMatrix*)(_value))->Load( f );
         break;

         case aMeasure:
         _value = new CMeasure();
         readed += ((CMeasure*)(_value))->Load( f );
         readed += fread( &_format, 1, sizeof( unsigned char ), f );
         break;
                  
         case aString:
         x = size - ( 3 * sizeof( int ) + namelen + sizeof( bool ) );
         _value = new char[ x ];
         readed += fread( _value, 1, x, f );
         break;

         case aExpression:
         x = size - ( 3 * sizeof( int ) + namelen + sizeof( bool ) );
         _value = new char[ x ];
         readed += fread( _value, 1, x, f );
         break;

         case aSymbol:
         x = size - ( 3 * sizeof( int ) + namelen + sizeof( bool ) );
         _value = new char[ x ];
         readed += fread( _value, 1, x, f );
         break;

         default:
         break;
      }
      if( readed == size )
      {
         res = true;
      }
      else
      {
      }
   }
   return res;
}
//-----------------------------------------------------------------------------
int atom::atomsize( void ) const
{
   int i = 0;
   switch( _type )
   {
      case aInteger:
      i = sizeof( int ) + sizeof( unsigned char );
      break;
      case aBoolean:
      i = sizeof( bool );
      break;
      case aDouble:
      i = sizeof( double ) + sizeof( unsigned char );
      break;
      case aComplex:
      i = 2 * sizeof( double ) + sizeof( unsigned char );
      break;
      case aVector:
      i = sizeof( int ) + ((CVector*)_value)->GetSize() * 2 * sizeof( double );
      break;      
      case aMatrix:
      i = 2 * sizeof( int ) + ((CMatrix*)_value)->RowCount() * ((CMatrix*)_value)->ColCount() * 2 * sizeof( double );
      break;      
      case aMeasure:
      i = ((CMeasure*)(_value))->GetSize() + sizeof( unsigned char );
      break;
      case aString:
      case aExpression:
      case aSymbol:
      i = strlen( (char*)( _value ) ) + 1;
      break;
   }
   return i;
}
//-----------------------------------------------------------------------------
const char* atom::typestr( void )
{
   return __AtomTypes[ _type ];
}
//-----------------------------------------------------------------------------
void atom::makestrbuf( char* buf )
{
   //if( strbuf )
   //{
   //   delete [] strbuf;
   //   strbuf = NULL;
   //}
   //strbuf = new char[ strlen( buf ) + 1 ];
   //strcpy( strbuf, buf );
   astrbuf = AnsiString( buf );
}
//-----------------------------------------------------------------------------
const char* atom::getstring( int prec )
{  
   char tmpbuf[ 200 ];
   //char t1[ 20 ];
   //char t2[ 20 ];
   //int j;
   //bool lExit;
   
   if( prec < 0 )
   {
      prec = 6;
   }
   
   switch( _type )
   {
      case aInteger:
      switch( _format )
      {
         case aifBin:
         sprintf( tmpbuf, "#b%s", bin2str( *(int*)(_value) ).c_str() );
         break;
         
         case aifOct:
         sprintf( tmpbuf, "#o%s", oct2str( *(int*)(_value) ).c_str() );
         break;
         
         case aifHex:
         sprintf( tmpbuf, "#h%X", *(int*)(_value) );
         break;
         
         default:
         sprintf( tmpbuf, "%d", *(int*)(_value) );
         break;
      }
      break;
      
      case aBoolean:
      sprintf( tmpbuf, "%s", *(bool*)(_value) ? "True" : "False" );
      break;
      
      case aDouble:
      switch( _format )
      {
         case aifExp:
         sprintf( tmpbuf, "%s", DoubleToStrE( *(double*)(_value), prec ).c_str() );
         break;
         
         default:
         sprintf( tmpbuf, "%s", DoubleToStr( *(double*)(_value), prec ).c_str() );
         break;
      }
      break;
      
      case aComplex:
      if( ((CComplex*)_value)->imag() )
      {         
         //double2str( dround( ((CComplex*)_value)->real(), prec ), t1 );
         //double2str( dround( ((CComplex*)_value)->imag(), prec ), t2 );
         //sprintf( tmpbuf, "(%s,%s%c)", t1, t2, 'i' );
         switch( _format )
         {
            case aifExp:
            sprintf( tmpbuf, "%s", ComplexToStrE( *((CComplex*)_value), prec ).c_str() );
            break;
            
            default:
            sprintf( tmpbuf, "%s", ComplexToStrF( *((CComplex*)_value), prec ).c_str() );
            break;
         }
      }
      else
      {
         //double2str( dround( ((CComplex*)_value)->real(), prec ), tmpbuf );
         switch( _format )
         {
            case aifExp:
            sprintf( tmpbuf, "%s", DoubleToStrE( ((CComplex*)_value)->real(), prec ).c_str() );
            break;
            
            default:
            sprintf( tmpbuf, "%s", DoubleToStr( ((CComplex*)_value)->real(), prec ).c_str() );
            break;
         }
      }
      break;
      
      case aVector:
      sprintf( tmpbuf, "%s", VectorToStrF( *((CVector*)_value), prec ).c_str() );
      break;
      
      case aMatrix:
      sprintf( tmpbuf, "%s", MatrixToStrF( *((CMatrix*)_value), prec ).c_str() );
      break;
      
      case aMeasure:
      switch( _format )
      {
         case aifExp:
         sprintf( tmpbuf, "%s", MeasureToStrE( *((CMeasure*)_value), prec ).c_str() );
         break;
         
         default:
         sprintf( tmpbuf, "%s", MeasureToStrF( *((CMeasure*)_value), prec ).c_str() );
         break;
      }
      break;
      
      case aString:
      sprintf( tmpbuf, "%c%s%c", '\"', (char*)( _value ), '\"' );
      break;
      
      case aExpression:
      sprintf( tmpbuf, "%s", (char*)( _value ) ); //"%c%s%c", '\'', (char*)( _value ), '\'' );
      break;
      
      case aSymbol:
      sprintf( tmpbuf, "%s", (char*)( _value ) ); //"%c%s%c", '\'', (char*)( _value ), '\'' );
      break;
      
      default:
      sprintf( tmpbuf, "NULL" );
      break;
   }
   //if( _name )
   //{
   //   AnsiString p = "";
   //   if( strlen( _name ) > 0 )
   //   {
   //      p = AnsiString( _name ) + ": ";
   //   }
   //   p += AnsiString( tmpbuf );
   //   makestrbuf( p.c_str() );
   //}
   //else
   //{
      makestrbuf( tmpbuf );
   //}
   //strbuf = (char*)( new char[ strlen( tmpbuf ) + 1 ] );
   //strcpy( (char*)strbuf, tmpbuf );

   return astrbuf.c_str();
}
//-----------------------------------------------------------------------------
const char* atom::getnamed( int prec ) // return string buffer strbuf
{
   AnsiString p = "";
   
   if( _name )
   {
      p = AnsiString( _name ) + ": ";
   }
   p += AnsiString( getstring( prec ) );
   makestrbuf( p.c_str() );
   
   return astrbuf.c_str();
}
//-----------------------------------------------------------------------------
atom::operator int( void )
{
   int i;
   switch( _type )
   {
      case aInteger:
      i = *(int*)(_value);
      break;

      case aBoolean:
      i = ( *(bool*)(_value) ? 1 : 0 );
      break;

      case aDouble:
      i = (int)( *(double*)(_value) );
      break;

      case aComplex:
      i = (int)( ((CComplex*)_value)->real() );
      break;
      
      default:
      i = 0;
      break;
   }
   return i;
}
//-----------------------------------------------------------------------------
atom::operator bool( void )
{
   bool b;
   CComplex c;
   
   switch( _type )
   {
      case aInteger:
      b = ( *(int*)(_value) != 0 );
      break;
      
      case aBoolean:
      b = *(bool*)(_value);
      break;
      
      case aDouble:
      b = ( *(double*)(_value) != 0.0 );
      break;
      
      case aComplex:
      //c = CComplex( 0.0, 0.0 );
      b = ( *((CComplex*)_value) != CComplex( 0.0, 0.0 ) );
      break;

      default:
      b = false;
      break;
   }
   return b;
}
//-----------------------------------------------------------------------------
atom::operator double( void )
{
   double d;
   switch( _type )
   {
      case aInteger:
      d = (double)( *(int*)(_value) );
      break;
      
      case aBoolean:
      d = ( *(bool*)(_value) ? 1.0 : 0.0 );
      break;
      
      case aDouble:
      d = *(double*)(_value);
      break;
      
      case aComplex:
      d = ((CComplex*)_value)->real();
      break;

      default:
      d = 0.0;
      break;
   }
   return d;
}
//-----------------------------------------------------------------------------
atom::operator CComplex( void )
{
   CComplex c;
   
   switch( _type )
   {
      case aInteger:
      c = CComplex( (double)( *(int*)(_value) ), 0.0 );
      break;
      
      case aBoolean:
      c = CComplex( ( *(bool*)(_value) ? 1.0 : 0.0 ), 0.0 );
      break;
      
      case aDouble:
      c = CComplex( *(double*)(_value), 0.0 );
      break;
      
      case aComplex:
      c = CComplex( ((CComplex*)_value)->real(), ((CComplex*)_value)->imag() );
      break;
      
      default:
      c = CComplex( 0.0, 0.0 );
      break;
   }
   return c;
}
//-----------------------------------------------------------------------------
/*
atom::operator adouble( void )
{
   adouble d;
   
   switch( _type )
   {
      case aInteger:
      d = adouble( (double)( *(int*)(_value) ) );
      break;
      
      case aBoolean:
      d = adouble( ( *(bool*)(_value) ? 1.0 : 0.0 ) );
      break;
      
      case aDouble:
      d = adouble( *(double*)(_value) );
      break;
      
      case aComplex:
      d = adouble( ((CComplex*)_value)->real() );
      break;
      
      default:
      d = adouble( 0.0 );
      break;
   }
   return d;
}
//-----------------------------------------------------------------------------
atom::operator acomplex( void )
{
   acomplex c;
   
   switch( _type )
   {
      case aInteger:
      c = acomplex( (double)( *(int*)(_value) ), 0.0 );
      break;
      
      case aBoolean:
      c = acomplex( ( *(bool*)(_value) ? 1.0 : 0.0 ), 0.0 );
      break;
      
      case aDouble:
      c = acomplex( *(double*)(_value), 0.0 );
      break;
      
      case aComplex:
      c = acomplex( ((CComplex*)_value)->real(), ((CComplex*)_value)->imag() );
      break;
      
      default:
      c = acomplex( 0.0, 0.0 );
      break;
   }
   return c;
}
*/
//-----------------------------------------------------------------------------
atom operator+( atom& a, atom& b )
{
   atom x;
   if( a.type() == aInteger && b.type() == aInteger )
   {
      int i = (int)( a ) + (int)( b );
      x = atom( i );
   }
   else if( ( a.type() == aDouble && b.type() == aDouble ) || 
            ( a.type() == aDouble && b.type() == aInteger ) ||
            ( a.type() == aInteger && b.type() == aDouble )
          )  
   {
      double d = (double)( a ) + (double)( b );
      x = atom( d );
   }
   else if( a.type() == aComplex || b.type() == aComplex )
   {    
      CComplex c = (CComplex)( a ) + (CComplex)( b );
      x = atom( c );
   }
   else if( a.type() == aVector || b.type() == aVector )
   {
      CVector v = *((CVector*)(a._value)) + *((CVector*)(b._value));
      x = atom( v );
   }
   else if( a.type() == aMeasure && b.type() == aMeasure )
   {
      CMeasure am = *((CMeasure*)(a._value));
      CMeasure bm = *((CMeasure*)(b._value));
      if( ConsistentUnits( am, bm ) )
      { 
         CMeasure m = am + bm;
         x = atom( m );
      }
   }
   else if( a.type() == aExpression || a.type() == aSymbol ||
            b.type() == aExpression || b.type() == aSymbol )
   {
      x = mix_2_atoms_with_opr( a, b, "+" );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom operator-( atom& a, atom& b )
{
   atom x;
   if( a.type() == aInteger && b.type() == aInteger )
   {
      int i = (int)( a ) - (int)( b );
      x = atom( i );
   }
   else if( ( a.type() == aDouble && b.type() == aDouble ) || 
            ( a.type() == aDouble && b.type() == aInteger ) ||
            ( a.type() == aInteger && b.type() == aDouble )
          )  
   {
      double d = (double)( a ) - (double)( b );
      x = atom( d );
   }
   else if( a.type() == aComplex || b.type() == aComplex )
   {    
      CComplex c = (CComplex)( a ) - (CComplex)( b );
      x = atom( c );
   }
   else if( a.type() == aVector || b.type() == aVector )
   {
      CVector v = *((CVector*)(a._value)) - *((CVector*)(b._value));
      x = atom( v );
   }
   else if( a.type() == aMeasure && b.type() == aMeasure )
   {
      CMeasure am = *((CMeasure*)(a._value));
      CMeasure bm = *((CMeasure*)(b._value));
      if( ConsistentUnits( am, bm ) )
      { 
         CMeasure m = am - bm;
         x = atom( m );
      }
   }
   else if( a.type() == aExpression || a.type() == aSymbol ||
            b.type() == aExpression || b.type() == aSymbol )
   {
      x = mix_2_atoms_with_opr( a, b, "-" );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom operator*( atom& a, atom& b )
{
   atom x;
   if( a.type() == aInteger && b.type() == aInteger )
   {
      int i = (int)( a ) * (int)( b );
      x = atom( i );
   }
   else if( ( a.type() == aDouble && b.type() == aDouble ) || 
            ( a.type() == aDouble && b.type() == aInteger ) ||
            ( a.type() == aInteger && b.type() == aDouble )
          )  
   {
      double d = (double)( a ) * (double)( b );
      x = atom( d );
   }
   else if( ( a.type() == aComplex && b.isnumeric() ) || 
            ( b.type() == aComplex && a.isnumeric() ) )
   {    
      CComplex c = (CComplex)( a ) * (CComplex)( b );
      x = atom( c );
   }
   else if( a.type() == aVector && b.isnumeric() )
   {
      CVector v = *((CVector*)(a._value)) * (CComplex)( b );
      x = atom( v );
   }
   else if( b.type() == aVector && a.isnumeric() )
   {
      CVector v = *((CVector*)(b._value)) * (CComplex)( a );
      x = atom( v );
   }
   else if( a.type() == aMeasure && b.type() == aMeasure )
   {
      CMeasure am = *((CMeasure*)(a._value));
      CMeasure bm = *((CMeasure*)(b._value));
      CMeasure m = am * bm;
      x = atom( m );
   }
   else if( a.type() == aMeasure && ( b.type() == aInteger || b.type() == aDouble ) )
   {
      CMeasure m = *((CMeasure*)(a._value)) * (double)( b );
      x = atom( m );
   }
   else if( b.type() == aMeasure && ( a.type() == aInteger || a.type() == aDouble ) )
   {
      CMeasure m = *((CMeasure*)(b._value)) * (double)( a );
      x = atom( m );
   }
   else if( a.type() == aExpression || a.type() == aSymbol ||
            b.type() == aExpression || b.type() == aSymbol )
   {
      x = mix_2_atoms_with_opr( a, b, "*" );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom operator/( atom& a, atom& b )
{
   atom x = nullatom;
   if( a.type() == aInteger && b.type() == aInteger )
   {
      if( (int)( b ) != 0 )
      {
         int i = (int)( a ) / (int)( b );
         x = atom( i );
      }
   }
   else if( ( a.type() == aDouble && b.type() == aDouble ) || 
            ( a.type() == aDouble && b.type() == aInteger ) ||
            ( a.type() == aInteger && b.type() == aDouble )
          )  
   {
      if( (double)( b ) != 0.0 )
      {
         double d = (double)( a ) / (double)( b );
         x = atom( d );
      }
   }
   else if( ( a.type() == aComplex && b.isnumeric() ) || 
            ( b.type() == aComplex && a.isnumeric() ) )
   {    
      if( (CComplex)( b ) != CComplex( 0.0, 0.0 ) )
      {
         CComplex c = (CComplex)( a ) / (CComplex)( b );
         x = atom( c );
      }
   }
   else if( a.type() == aVector && b.isnumeric() )
   {
      if( (CComplex)( b ) != CComplex( 0.0, 0.0 ) )
      {
         CVector v = *((CVector*)(a._value)) / (CComplex)( b );
         x = atom( v );
      }
   }
   else if( b.type() == aVector && a.isnumeric() )
   {
      //CVector v = (CComplex)( a ) / *((CVector*)(b._value));
      //x = atom( v );
      x = invalidatom;
   }
   else if( a.type() == aMeasure && b.type() == aMeasure )
   {
      CMeasure am = *((CMeasure*)(a._value));
      CMeasure bm = *((CMeasure*)(b._value));
      if( bm.GetValue() != 0.0 )
      {
         CMeasure m = am / bm;
         x = atom( m );
      }
   }
   else if( a.type() == aMeasure && ( b.type() == aInteger || b.type() == aDouble ) )
   {
      if( (double)( b ) != 0.0 )
      {
         CMeasure m = *((CMeasure*)(a._value)) / (double)( b );
         x = atom( m );
      }
   }
   else if( b.type() == aMeasure && ( a.type() == aInteger || a.type() == aDouble ) )
   {
      if( ((CMeasure*)b._value)->GetValue() != 0.0 )
      {
         CMeasure m = (double)( a ) / *((CMeasure*)(b._value));
         x = atom( m );
      }
   }
   else if( a.type() == aExpression || a.type() == aSymbol ||
            b.type() == aExpression || b.type() == aSymbol )
   {
      x = mix_2_atoms_with_opr( a, b, "/" );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom operator==( atom& a, atom& b )
{  
   atom x = equal( a, b );  
   return x;
}
//-----------------------------------------------------------------------------
atom operator!=( atom& a, atom& b )
{
   return !( a == b );
}
//-----------------------------------------------------------------------------
atom operator<( atom& a, atom& b )
{  
   atom x = less( a, b );  
   return x;
}
//-----------------------------------------------------------------------------
atom operator<=( atom& a, atom& b )
{  
   atom x = atom( (bool)( equal( a, b ) ) || (bool)( less( a, b ) ) );  
   return x;
}
//-----------------------------------------------------------------------------
atom operator>( atom& a, atom& b )
{  
   atom x = great( a, b );  
   return x;
}
//-----------------------------------------------------------------------------
atom operator>=( atom& a, atom& b )
{  
   atom x = atom( (bool)( equal( a, b ) ) || (bool)( great( a, b ) ) );  
   return x;
}
//-----------------------------------------------------------------------------
atom sin( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble )
   {
      x = atom( sin( (double)a ) );   
   }
   else if( a.type() == aComplex )
   {
      x = atom( sin( *((CComplex*)(a._value)) ) );
   }
   else if( ( a.type() == aSymbol || a.type() == aExpression ) && a.value() )
   {
      x = mix_1_atom_with_func( a, "Sin" );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom sinh( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble )
   {
      x = atom( sinh( (double)a ) );   
   }
   else if( a.type() == aComplex )
   {
      x = atom( sinh( *((CComplex*)(a._value)) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom asin( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( asin( (CComplex)( a ) ) );
   }
   else if( ( a.type() == aSymbol || a.type() == aExpression ) && a.value() )
   {
      x = mix_1_atom_with_func( a, "ASin" );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom asinh( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( asinh( (CComplex)( a ) ) );   
   }
   return x;
}
//-----------------------------------------------------------------------------
atom cos( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble )
   {
      x = atom( cos( (double)a ) );   
   }
   else if( a.type() == aComplex )
   {
      x = atom( cos( *((CComplex*)(a._value)) ) );
   }
   else if( ( a.type() == aSymbol || a.type() == aExpression ) && a.value() )
   {
      x = mix_1_atom_with_func( a, "Cos" );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom cosh( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble )
   {
      x = atom( cosh( (double)a ) );   
   }
   else if( a.type() == aComplex )
   {
      x = atom( cosh( *((CComplex*)(a._value)) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom acos( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( acos( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom acosh( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( acosh( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom tan( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( tan( (CComplex)( a ) ) );
   }
   else if( ( a.type() == aSymbol || a.type() == aExpression ) && a.value() )
   {
      x = mix_1_atom_with_func( a, "Tan" );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom tanh( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( tanh( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom atan( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( atan( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom atanh( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( atanh( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom cot( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( cot( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom coth( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( coth( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom acot( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( acot( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom acoth( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( acoth( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom sec( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( sec( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom sech( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( sech( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom asec( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( asec( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom asech( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( asech( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom csc( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( csc( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom csch( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( csch( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom acsc( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( acsc( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom acsch( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( acsch( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom deg2rad( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble )
   {
      double d = *(double*)( a._value );
      x = atom( ( d * M_PI ) / 180.0 );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom rad2deg( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      CComplex c = *(CComplex*)( a._value );
      double re = ( c.real() * 180.0 ) / M_PI;
      double im = ( c.imag() * 180.0 ) / M_PI;
      x = atom( re, im );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom exp( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( exp( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom abs( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( abs( (CComplex)( a ) ) );
   }
   else if( a.type() == aVector )
   {
      x = length( a );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom sign( atom& a )
{
   atom x;

   if( a.type() == aInteger || a.type() == aDouble )
   {
      double d = (double)( a );
      if( d >= 0.0 )
      {
         x = atom( 1 );
      }
      else
      {
         x = atom( -1 );
      }
   }
   return x;
}
//-----------------------------------------------------------------------------
atom log( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( log( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom log10( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( log10( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom sqrt( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble || a.type() == aComplex )
   {
      x = atom( sqrt( (CComplex)( a ) ) );
   }
   else if( ( a.type() == aSymbol || a.type() == aExpression ) && a.value() )
   {
      x = mix_1_atom_with_func( a, "Sqrt" );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom expm( atom& a )
{
   atom x;

   if( a.type() == aInteger || a.type() == aDouble )
   {
      double d = (double)( a );
      x = atom( exp( d ) - 1.0 );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom lnp1( atom& a )
{
   atom x;

   if( a.type() == aInteger || a.type() == aDouble )
   {
      double d = (double)( a );
      x = atom( log( d + 1.0 ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom factorial( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble )
   {
      int i = (int)( a );
      double d = (double)( a );
      
      if( (double)( i ) == d && i >= 0 )
      {
         int j, res = 0;
      
         if( i == 0 || i == 1 )
         {
            res = 1;
         }
         else if( i > 1 )
         {
            for( j = 2, res = 1; j <= i; j++ )
            {
               res *= j;
            }
         }
         x = atom( res );
      }
      else
      {
         x = atom( gamma( d + 1.0 ) );
      }
   }
   return x;
}
//-----------------------------------------------------------------------------
atom pow( atom& a, atom& b )
{
   atom x;
   
   if( a.type() == aInteger && b.type() == aInteger )
   {
      int i = (int)pow( (double)( a ), (double)( b ) );
      x = atom( i );
   }
   else if( ( a.type() == aDouble && b.type() == aDouble ) || 
            ( a.type() == aDouble && b.type() == aInteger ) ||
            ( a.type() == aInteger && b.type() == aDouble )
          )  
   {
      double d = pow( (double)( a ), (double)( b ) );
      x = atom( d );
   }
   else if( a.type() == aComplex || b.type() == aComplex )
   {
      CComplex c = pow( (CComplex)( a ), (CComplex)( b ) );
      x = atom( c );
   }
   else if( a.type() == aExpression || a.type() == aSymbol ||
            b.type() == aExpression || b.type() == aSymbol )
   {
      x = mix_2_atoms_with_opr( a, b, "^" );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom rootn( atom& a, atom& b )
{
   atom x;
   
   if( a.type() == aInteger && b.type() == aInteger )
   {
      if( (double)( b ) != 0.0 )
      {
         int i = (int)pow( (double)( a ), 1.0 / (double)( b ) );
         x = atom( i );
      }
   }
   else if( ( a.type() == aDouble && b.type() == aDouble ) || 
            ( a.type() == aDouble && b.type() == aInteger ) ||
            ( a.type() == aInteger && b.type() == aDouble )
          )  
   {
      if( (double)( b ) != 0.0 )
      {
         double d = pow( (double)( a ), 1.0 / (double)( b ) );
         x = atom( d );
      }
   }
   else if( a.type() == aComplex || b.type() == aComplex )
   {
      if( !( ((CComplex)( b )).iszero() ) )
      {
         CComplex c = pow( (CComplex)( a ), CComplex( 1.0, 0.0 ) / (CComplex)( b ) );
         x = atom( c );
      }
   }
   return x;
}
//-----------------------------------------------------------------------------
atom real2exponent( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble )
   {
      double d = (double)( a );
      x = atom( d, aifExp );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom exponent2real( atom& a )
{
   atom x;

   if( a.type() == aInteger || a.type() == aDouble )
   {
      double d = (double)( a );
      x = atom( d, aifDec );
   }
   return x;
}
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
atom real2complex( atom& a, atom& b )
{
   atom x;

   if( ( a.isint() || a.isdouble() ) && ( b.isint() || b.isdouble() ) )
   {
      double n1 = (double)( a );
      double n2 = (double)( b );

      x = atom( n1, n2 );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom real( atom& a )
{
   atom x;
   if( a.isnumeric() )
   {
      x = atom( ((CComplex)( a )).real() );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom imag( atom& a )
{
   atom x;
   if( a.type() == aComplex )
   {
      x = atom( ((CComplex)( a )).imag() );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom arg( atom& a )
{
   atom x;
   if( a.type() == aComplex )
   {
      x = atom( arg( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom conj( atom& a )
{
   atom x;
   if( a.isnumeric() )
   {
      x = atom( conj( (CComplex)( a ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom polar( atom& a, atom& b )
{
   atom x;

   if( ( a.type() == aInteger || a.type() == aDouble ) && ( b.type() == aInteger || b.type() == aDouble ) )
   {
		//double da = (double)( a );
		//double db = (double)( b );
      //x = atom( polar( const_cast<double&>( da ), const_cast<double&>( db ) ) );
      x = atom( polar( (double)( a ), (double)( b ) ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom norm( atom& a )
{
   atom x;
   if( a.isnumeric() )
   {
      CComplex c = (CComplex)( a );
      x = atom( c.real() * c.real() + c.imag() * c.imag() );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom uomconvert( atom& a, atom& b )
{
   atom x = nullatom;
   
   if( a.type() == aMeasure && b.type() == aMeasure )
   {
      if( ConsistentUnits( *((CMeasure*)(a._value)), *((CMeasure*)(b._value)) ) )
      {
         CMeasure m;
         
         m = ((CMeasure*)(a._value))->ConvertTo( ((CMeasure*)(b._value))->GetUnit() );
         x = atom( m );
      }
   }
   return x;
}
//-----------------------------------------------------------------------------
//
//
//
//-----------------------------------------------------------------------------
atom round( atom& a, int prec )
{
   atom x;
   if( a.type() == aInteger )
   {
      x = a;
   }
   else if( a.type() == aDouble ) 
   {
      x = dround( (double)( a ), prec );
   }
   else if( a.type() == aComplex )
   {
      double re, im;
      
      re = dround( ((CComplex)( a )).real(), prec );
      im = dround( ((CComplex)( a )).imag(), prec );
      
      x = atom( CComplex( re, im ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom atom_ip( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble )
   {
      int i = (int)( a );
      x = atom( i );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom atom_fp( atom& a )
{
   atom x;

   if( a.type() == aInteger || a.type() == aDouble )
   {
      double d = (double)( a ) - (double)((int)( a ) );
      x = atom( d );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom equal( atom& a, atom& b )
{
   atom x;
   bool lResult = false;
   
   if( a.type() == b.type() )
   {
      switch( a.type() )
      {
         case aInteger:
         lResult = ( *(int*)(a._value) == *(int*)(b._value) );
         break;
         case aBoolean:
         lResult = ( *(bool*)(a._value) == *(bool*)(b._value) );
         break;
         case aDouble:
         lResult = ( *(double*)(a._value) == *(double*)(b._value) );
         break;
         case aComplex:
         lResult = ( *(CComplex*)(a._value) == *(CComplex*)(b._value) );
         break;
         case aVector:
         lResult = ( *(CVector*)(a._value) == *(CVector*)(b._value) );
         break;
         case aMatrix:
         lResult = ( *(CMatrix*)(a._value) == *(CMatrix*)(b._value) );
         break;
         case aString:
         case aExpression:
         lResult = ( strcmp( (const char*)(a._value), (const char*)(b._value) ) == 0 );
         break;
      }
   }
   else if( a.isnumeric() && b.isnumeric() )
   {
      lResult = ( (CComplex)( a ) == (CComplex)( b ) );
   }
   
   x = atom( lResult );
   return x;
}
//-----------------------------------------------------------------------------
atom great( atom& a, atom& b )
{
   atom x;
   bool lResult = false;
   
   if( a.type() == b.type() )
   {
      switch( a.type() )
      {
         case aInteger:
         lResult = ( *(int*)(a._value) > *(int*)(b._value) );
         break;
         case aBoolean:
         lResult = ( *(bool*)(a._value) > *(bool*)(b._value) );
         break;
         case aDouble:
         lResult = ( *(double*)(a._value) > *(double*)(b._value) );
         break;
         case aComplex:
         lResult = ( *(CComplex*)(a._value) > *(CComplex*)(b._value) );
         break;
         case aVector:
         lResult = ( *(CVector*)(a._value) > *(CVector*)(b._value) );
         break;
         case aString:
         case aExpression:
         lResult = ( strcmp( (const char*)(a._value), (const char*)(b._value) ) > 0 );
         break;
      }
   }
   else if( a.isnumeric() && b.isnumeric() )
   {
      lResult = ( (CComplex)( a ) > (CComplex)( b ) );
   }
   
   x = atom( lResult );
   return x;
}
//-----------------------------------------------------------------------------
atom less( atom& a, atom& b )
{
   atom x;
   bool lResult = false;
   
   if( a.type() == b.type() )
   {
      switch( a.type() )
      {
         case aInteger:
         lResult = ( *(int*)(a._value) < *(int*)(b._value) );
         break;
         case aBoolean:
         lResult = ( *(bool*)(a._value) < *(bool*)(b._value) );
         break;
         case aDouble:
         lResult = ( *(double*)(a._value) < *(double*)(b._value) );
         break;
         case aComplex:
         lResult = ( *(CComplex*)(a._value) < *(CComplex*)(b._value) );
         break;
         case aVector:
         lResult = ( *(CVector*)(a._value) < *(CVector*)(b._value) );
         break;
         case aString:
         case aExpression:
         lResult = ( strcmp( (const char*)(a._value), (const char*)(b._value) ) < 0 );
         break;
      }
   }
   else if( a.isnumeric() && b.isnumeric() )
   {
      lResult = ( (CComplex)( a ) < (CComplex)( b ) );
   }
   
   x = atom( lResult );
   return x;
}
//-----------------------------------------------------------------------------
atom cross( atom& a, atom& b )
{
   atom x;
   
   if( a.type() == aVector && b.type() == aVector )
   {
      CVector v;
      
      v = cross( *( CVector*)( a._value ), *( CVector*)( b._value ) );
      x = atom( v );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom dot( atom& a, atom& b )
{
   atom x;
   
   if( a.type() == aVector && b.type() == aVector )
   {
      CComplex c;
      
      c = dot( *( CVector*)( a._value ), *( CVector*)( b._value ) );
      x = atom( c );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom length( atom& a )
{
   bool lResult = true;
   atom x;
   double z;
   
   if( a.type() == aVector )
   {
      z = ((CVector*)(a._value))->Length();
   }
   else if( a.type() == aString )
   {
      z = strlen( (char*)( a._value ) );
   }
   else
   {
      lResult = false;
   }
   
   if( lResult )
   {
      x = atom( z );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom real2vector2( atom& a, atom& b )
{
   atom x;
   
   if( ( a.type() == aInteger || a.type() == aDouble ) &&
       ( b.type() == aInteger || a.type() == aDouble )
     )
   {
      double n[ 2 ];

      n[ 0 ] = (double)( a );
      n[ 1 ] = (double)( b );
      CVector v = CVector( n, 2 );
      x = atom( v );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom real2vector3( atom& a, atom& b, atom& c )
{
   atom x;

   if( ( a.type() == aInteger || a.type() == aDouble ) &&
       ( b.type() == aInteger || a.type() == aDouble ) &&
       ( c.type() == aInteger || c.type() == aDouble )
     )
   {
      double n[ 3 ];

      n[ 0 ] = (double)( a );
      n[ 1 ] = (double)( b );
      n[ 2 ] = (double)( c );
      CVector v = CVector( n, 3 );
      x = atom( v );
   }
   return x;
}
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
atom int2bin( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble )
   {
      x = atom( (int)( a ), aifBin );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom int2oct( atom& a )
{
   atom x;

   if( a.type() == aInteger || a.type() == aDouble )
   {
      x = atom( (int)( a ), aifOct );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom int2dec( atom& a )
{
   atom x;

   if( a.type() == aInteger || a.type() == aDouble )
   {
      x = atom( (int)( a ), aifDec );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom int2hex( atom& a )
{
   atom x;

   if( a.type() == aInteger || a.type() == aDouble )
   {
      x = atom( (int)( a ), aifHex );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom num2exp( atom& a )
{
   atom x;

   if( a.type() == aInteger || a.type() == aDouble )
   {
      x = atom( (double)( a ), aifExp );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom exp2num( atom& a )
{
   atom x;

   if( a.type() == aInteger || a.type() == aDouble )
   {
      x = atom( (double)( a ), aifDec );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom aand( atom& a, atom& b )
{
   atom x;
   int v;
   bool l;
   
   if( ( a.isint() || a.isdouble() ) && ( b.isint() || b.isdouble() ) )
   {
      v = (int)( a ) & (int)( b );
      x = atom( v, b.format() );
   }
   else if( a.isbool() || b.isbool() )
   {
      l = (bool)( a ) && (bool)( b );
      x = atom( l );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom aor( atom& a, atom& b )
{
   atom x;
   int v;
   bool l;

   if( ( a.isint() || a.isdouble() ) && ( b.isint() || b.isdouble() ) )
   {
      v = (int)( a ) | (int)( b );
      x = atom( v, b.format() );
   }
   else if( a.isbool() || b.isbool() )
   {
      l = (bool)( a ) || (bool)( b );
      x = atom( l );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom axor( atom& a, atom& b )
{
   atom x;
   int v;
   bool l;

   if( ( a.isint() || a.isdouble() ) && ( b.isint() || b.isdouble() ) )
   {
      v = (int)( a ) ^ (int)( b );
      x = atom( v, b.format() );
   }
   else if( a.isbool() || b.isbool() )
   {
      l = ( (bool)( a ) && !(bool)( b ) ) || ( !(bool)( a ) && (bool)( b ) );
      x = atom( l );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom anot( atom& a )
{
   atom x;

   if( a.isint() || a.isdouble() )
   {
      int v = (int)( a );
      x = atom( v ? 0 : 1 );
   }
   else if( a.isbool() )
   {
      bool l = !(bool)( a );
      x = atom( l );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom adiv( atom& a, atom& b )
{
   atom x;
   int v;

   if( ( a.isint() || a.isdouble() ) && ( b.isint() || b.isdouble() ) )
   {
      if( (int)( b ) != 0 )
      {
         v = (int)( a ) / (int)( b );
         x = atom( v, b.format() );
      }
   }
   return x;
}
//-----------------------------------------------------------------------------
atom amod( atom& a, atom& b )
{
   atom x;
   int v;

   if( ( a.isint() || a.isdouble() ) && ( b.isint() || b.isdouble() ) )
   {
      if( (int)( b ) != 0 )
      {
         v = (int)( a ) % (int)( b );
         x = atom( v, b.format() );
      }
   }
   return x;
}
//-----------------------------------------------------------------------------
atom aslb( atom& a )
{
   atom x;

   if( a.isint() || a.isdouble() )
   {
      int v = (int)( a );
      x = atom( v << 1, a.format() );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom asrb( atom& a )
{
   atom x;

   if( a.isint() || a.isdouble() )
   {
      int v = (int)( a );
      x = atom( v >> 1, a.format() );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom arlb( atom& a )
{
   atom x;

   if( a.isint() || a.isdouble() )
   {
      int v = (int)( a );
      int b = ( v & 0x40000000 ? 0x00000001 : 0x00000000 );
      x = atom( ( ( v << 1 ) & 0x4FFFFFFF ) | b, a.format() );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom arrb( atom& a )
{
   atom x;

   if( a.isint() || a.isdouble() )
   {
      int v = (int)( a );
      int b = ( v & 0x00000001 ? 0x40000000 : 0x00000000 );
      x = atom( ( v >> 1 ) | b, a.format() );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom percent( atom& a, atom& b )
{
   atom x;
   
   if( ( a.isint() || a.isdouble() ) && ( b.isint() || b.isdouble() ) )
   {
      double n1 = (double)( a );
      double n2 = (double)( b );
      
      x = atom( ( n1 * n2 ) / 100.0 );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom percentt( atom& a, atom& b )
{
   atom x;

   if( ( a.isint() || a.isdouble() ) && ( b.isint() || b.isdouble() ) )
   {
      double n1 = (double)( a );
      double n2 = (double)( b );

      if( n1 != 0.0 )
      {
         x = atom( ( 100.0 * n2 ) / n1 );
      }
   }
   return x;
}
//-----------------------------------------------------------------------------
atom percentch( atom& a, atom& b )
{
   atom x;

   if( ( a.isint() || a.isdouble() ) && ( b.isint() || b.isdouble() ) )
   {
      double n1 = (double)( a );
      double n2 = (double)( b );

      if( n1 != 0.0 )
      {
         x = atom( ( 100.0 * ( n2 - n1 ) ) / n1 );
      }
   }
   return x;
}
//-----------------------------------------------------------------------------
atom isprime( atom& a )
{
   atom x;
   int i;
   
   if( a.type() == aInteger )
   {
      i = *(int*)( a._value );
      x = atom( isprimenumber( i ) );
   }
   else if( a.type() == aDouble )
   {
      double d = *(double*)( a._value );
      i = (int)( a );
      
      if( (double)( i ) == d )
      {
         x = atom( isprimenumber( i ) );
      }
   }
   return x;
}
//-----------------------------------------------------------------------------
atom gamma( atom& a )
{
   atom x;
   
   if( a.type() == aInteger || a.type() == aDouble )
   {
      double d = (double)( a );
      x = atom( gamma( d ) );
   }
   else if( a.type() == aComplex )
   {
      CComplex c = *(CComplex*)( a._value );
      x = atom( gamma( c ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom psi( atom& a )
{
   atom x;

   if( a.type() == aInteger || a.type() == aDouble )
   {
      double d = (double)( a );
      x = atom( psi( d ) );
   }
   else if( a.type() == aComplex )
   {
      CComplex c = *(CComplex*)( a._value );
      x = atom( psi( c ) );
   }
   return x;
}
//-----------------------------------------------------------------------------
//
// Financial functions 
//
//-----------------------------------------------------------------------------
atom fv( atom& N, atom& I, atom& PV, atom& PMT )
{
   atom x;
   
   if( N.isnumeric() && I.isnumeric() && PV.isnumeric() && 
       ( PMT.isnumeric() || PMT == NULLA ) )
   {
      double n = (double)( N );
      double i = (double)( I );
      double pv = (double)( PV );
      double pmt = (double)( PMT );
      double fv = fin_fv( n, i, pv, pmt );
      
      x = atom( fv );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom pmt( atom& N, atom& I, atom& PV, atom& FV )
{
   atom x;
   
   if( N.isnumeric() && I.isnumeric() && PV.isnumeric() && 
       ( FV.isnumeric() || FV == NULLA ) )
   {
      double n = (double)( N );
      double i = (double)( I );
      double pv = (double)( PV );
      double fv = (double)( FV );
      double pmt = fin_pmt( n, i, pv, fv );
      
      x = atom( pmt );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom pv( atom& N, atom& I, atom& FV, atom& PMT )
{
   atom x;
   
   if( N.isnumeric() && I.isnumeric() && FV.isnumeric() && 
       ( PMT.isnumeric() || PMT == NULLA ) )
   {
      double n = (double)( N );
      double i = (double)( I );
      double fv = (double)( FV );
      double pmt = (double)( PMT );
      double pv = fin_pv( n, i, fv, pmt );
      
      x = atom( pv );
   }
   return x;
}
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
atom str2atom( const char* text )
{
   atom a;
   
   //if( isintstr( text ) )
   //{
   //   a = atom( (int)( StrToDouble( text ) ) ); 
   //}
   //else 
   if( isdoublestr( text ) )
   {
      a = atom( StrToDouble( text ) );
   }
   else if( isexponentstr( text ) )
   {
      a = atom( StrEToDouble( text ), aifExp );
   }
   else if( ishexstr( text ) )
   {
      a = atom( HexStrToInt( text ), aifHex );
   }
   else if( isoctstr( text ) )
   {
      a = atom( OctStrToInt( text ), aifOct );
   }
   else if( isbinstr( text ) )
   {
      a = atom( BinStrToInt( text ), aifBin );
   }
   else if( iscomplexstr( text ) )
   {
      a = atom( StrToComplex( text ) );
   }
   else if( isstringstr( text ) || isexprstr( text ) || issymbolstr( text ) )
   {
      a = atom( text );
   }
   else if( isvectorstr( text ) )
   {
      a = atom( StrToVector( text ) );
   }
   else if( ismeasurestr( text ) )
   {
      a = atom( StrToMeasure( text ) );
   }
   else
   {
      if( text )
      {
         int len = strlen( text ) + 2;
         char* buf = new char[ len + 1 ];
         
         buf[ 0 ] = '\'';
         strcpy( buf + 1, text );
         buf[ len - 1 ] = '\'';
         buf[ len ] = '\0';
         
         a = atom( buf );
         delete [] buf;
      }
      else
      {
         a = nullatom;
      }
   }
   return a;
}
//-----------------------------------------------------------------------------
atom mix_2_atoms_with_opr( atom& a, atom& b, const char* opr )
{
   char buf[ 1000 ];
   atom x;

   buf[ 0 ] = '\0';
   if( opr )
   {
      if( ( a.type() == aExpression ) && a.value() )
      {
         if( b.type() == aInteger || b.type() == aDouble )
         {
            sprintf( buf, "(%s)%s%s",
                     (char*)( a.value() ),
                     opr,
                     DoubleToStr( (double)( b ) ).c_str() );
         }
         else if( ( b.type() == aExpression ) && b.value() )
         {
            sprintf( buf, "(%s)%s(%s)",
                     (char*)( a.value() ),
                     opr,
                     (char*)( b.value() ) );
         }
         else if( ( b.type() == aSymbol ) && b.value() )
         {
            sprintf( buf, "(%s)%s%s",
                     (char*)( a.value() ),
                     opr,
                     (char*)( b.value() ) );
         }
      }
      else if( ( a.type() == aSymbol ) && a.value() )
      {
         if( b.type() == aInteger || b.type() == aDouble )
         {
            sprintf( buf, "%s%s%s",
                     (char*)( a.value() ),
                     opr,
                     DoubleToStr( (double)( b ) ).c_str() );
         }
         else if( ( b.type() == aSymbol ) && b.value() )
         {
            sprintf( buf, "%s%s%s",
                     (char*)( a.value() ),
                     opr,
                     (char*)( b.value() ) );
         }
         else if( ( b.type() == aExpression ) && b.value() )
         {
            sprintf( buf, "%s%s(%s)",
                     (char*)( a.value() ),
                     opr,
                     (char*)( b.value() ) );
         }
      }
      else if( ( b.type() == aSymbol ) && b.value() )
      {
         if( a.type() == aInteger || a.type() == aDouble )
         {
            sprintf( buf, "%s%s%s",
                     DoubleToStr( (double)( a ) ).c_str(),
                     opr,
                     (char*)( b.value() ) );
         }
      }
      else if( ( b.type() == aExpression ) && b.value() )
      {
         if( a.type() == aInteger || a.type() == aDouble )
         {
            sprintf( buf, "%s%s(%s)",
                     DoubleToStr( (double)( a ) ).c_str(),
                     opr,
                     (char*)( b.value() ) );
         }
      }
   }

   if( buf[ 0 ] )
   {
      x = str2atom( buf );
   }
   return x;
}
//-----------------------------------------------------------------------------
atom mix_1_atom_with_func( atom& a, const char* func )
{
   atom x;

   char* buf = new char[ 500 ];
   buf[ 0 ] = '\0';

   if( func && ( a.type() == aExpression || a.type() == aSymbol ) && a.value() )
   {
      sprintf( buf, "%s(%s)",
               func,
               (char*)( a.value() ) );
      //p = AnsiString( func ) + "(" +
      //    AnsiString( (char*)( a.value() ) + ")";

   }

   if( buf[ 0 ] )
   {
      x = str2atom( buf );
   }
   delete [] buf;

   return x;
}
//-----------------------------------------------------------------------------
/*
//-----------------------------------------------------------------------------
//
//
//
//-----------------------------------------------------------------------------
aint::aint( void )
   : atom( 0 )
{
   value = 0;
}
//-----------------------------------------------------------------------------
aint::aint( int avalue )
   : atom( avalue )
{
   value = avalue;
}
//-----------------------------------------------------------------------------
aint::aint( const aint& a )
   : atom( *(int*)(a._value) )
{
   //lprintf( "Making integer in aint( %s )\n\r", *(int*)(a._value) );
}
//-----------------------------------------------------------------------------
void aint::release( void )
{
   atom::release();
   if( _value )
   {
      delete (int*)(_value);
   }
}
//-----------------------------------------------------------------------------
//
//
//
//-----------------------------------------------------------------------------
abool::abool( void )
   : atom( false )
{
   value = false;
}
//-----------------------------------------------------------------------------
abool::abool( bool avalue )
   : atom( avalue )
{
   value = avalue;
}
//-----------------------------------------------------------------------------
abool::abool( const abool& a )
   : atom( *(bool*)(a._value) )
{
   //lprintf( "Making bool in abool(%s)\n", *(bool*)(a._value) ? "True" : "False" );
}
//-----------------------------------------------------------------------------
void abool::release( void )
{
   atom::release();
   if( _value )
   {
      delete (bool*)(_value);
   }
}
//-----------------------------------------------------------------------------
//
//
//
//-----------------------------------------------------------------------------
adouble::adouble( void )
   : atom( 0.0 )
{
   value = 0.0;
}
//-----------------------------------------------------------------------------
adouble::adouble( double avalue )
   : atom( avalue )
{
   value = avalue;
}
//-----------------------------------------------------------------------------
adouble::adouble( const adouble& a )
   : atom( *(double*)(a._value) )
{
   //lprintf( "Making double in adouble(%f)\n", *(double*)(a._value) );
}
//-----------------------------------------------------------------------------
void adouble::release( void )
{
   atom::release();
   if( _value )
   {
      delete (double*)(_value);
   }
}
//-----------------------------------------------------------------------------
//
//
//
//-----------------------------------------------------------------------------
acomplex::acomplex( void )
   : atom( 0.0, 0.0 )
{
   re = im = 0.0;
}
//-----------------------------------------------------------------------------
acomplex::acomplex( double are, double aim )
   : atom( are, aim )
{
   //lprintf( "Making 2 double in acomplex(%f,%f)\n", are, aim );
   re = are;
   im = aim;
}
//-----------------------------------------------------------------------------
acomplex::acomplex( const acomplex& a )
   : atom( *(CComplex*)(a._value) )
{
   //lprintf( "Making CComplex in acomplex(%f,%f)\n", ((CComplex*)(a._value))->real(),((CComplex*)(a._value))->imag() );
}
//-----------------------------------------------------------------------------
void acomplex::release( void )
{
   atom::release();
   if( _value )
   {
      delete (CComplex*)(_value);
   }
}
//-----------------------------------------------------------------------------
double acomplex::getre( void )
{   
   return ((CComplex*)_value)->real();
}
//-----------------------------------------------------------------------------
double acomplex::getim( void )
{
   return ((CComplex*)_value)->imag();
}
*/
//-----------------------------------------------------------------------------

