/************************** * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
*************************** * * * * * * * * * * * * **************************/


#include "qhaccutils.h"
#include "qhacc.h"

#include <qregexp.h>
#include <qstringlist.h>

#include <math.h>

int Utils::QD=2;
	/* debug levels are:
	 *
	 * 0: no debugging nor errors
	 * 1: fatal errors only
	 * 2: errors that affect operation
	 * 3: errors requiring default values to be used
	 * 4: print major debugging information
	 * 5: print minor debugging information
	 * 6: for the curiosity cat
	 *
	 */
const int Utils::PRINT     =0;
const int Utils::ERRFATAL  =1;
const int Utils::ERROPER   =2;
const int Utils::ERRDEFAULT=3;
const int Utils::DBGMAJOR  =4;
const int Utils::DBGMINOR  =5;
const int Utils::CURIOSITY =6;

const int Utils::COLS[]={ QC::PCOLS, QC::LCOLS,
													QC::ACOLS, QC::TCOLS, QC::SCOLS, 
													QC::NCOLS, QC::JCOLS, QC::XCOLS };
const char ** Utils::NAMES[]={ QC::PCOLNAMES, QC::LCOLNAMES,
															 QC::ACOLNAMES, QC::TCOLNAMES, QC::SCOLNAMES,
															 QC::NCOLNAMES, QC::JCOLNAMES, QC::XCOLNAMES };
const ColType * Utils::TYPES[]={ QC::PCOLTYPES, QC::LCOLTYPES,
																 QC::ACOLTYPES, QC::TCOLTYPES, QC::SCOLTYPES,
																 QC::NCOLTYPES, QC::JCOLTYPES, QC::XCOLTYPES };
const int Utils::PKS[]={ QC::PPREF, QC::LID,
												 QC::AID, QC::TID, QC::SID,
												 QC::NID, QC::JID, QC::XSID };

Utils * Utils::obj=0;

Utils * Utils::get(){
	if( obj==0 ) obj=new Utils();
	return obj;
}

Utils::~Utils(){ if( obj ) delete obj; }

bool Utils::debug( int lvl, std::ostream *& str ){
	str=&cout;
	return ( lvl<=QD );
}

bool Utils::error( int lvl, std::ostream *& str ){
	str=&cerr;
	return ( lvl<=QD );
}

void Utils::setDebug( int i ){ QD=i; }
int Utils::getDebug(){ return QD; }

void Utils::parser( const QString& str, const QString& sep, int start,
										QString * tokens, int numT ){
	// parse a string, starting at start, into numT tokens
	// the return array must be big enough to hold the
	// tokens already
	/*
	int finder=0;
	for( int j=0; j<numT; j++ ){
		finder=str.find( sep, start );
		tokens[j]=str.mid( start, finder-start );
		start=finder+sep.length();
	}
	*/

	int finder=0, idx=0;
	while( finder>-1 && idx<numT ){
		finder=str.find( sep, start );
		tokens[idx++]=str.mid( start, finder-start );
		start=finder+sep.length();		
	}

	// in case we are expecting more strings than we
	// actually have in str
	for( int i=idx; i<numT; i++ )	tokens[i]=QString();

}

QDate Utils::dateFromString( const QString& dstring, const QString& sep,
														 int form ){
  int start=0, end=0;
  end=dstring.find( sep );
  int m=dstring.left( end ).toInt();
  start=end+1;
  end=dstring.find( sep, start );  
  int d=dstring.mid( start, end-start ).toInt();
  end=dstring.find( sep, start );
  start=end+1;
  int y=dstring.mid( start, end-start ).toInt();

  if ( form==QC::EUROPEAN ){
    //switch day and month for european formats
    int t=m;
    m=d;
    d=t;
  }
  else if ( form==QC::YEARFIRST ){
    //switch everything for yearfirst
    int t=y;
    y=m;
    m=d;
    d=t;
  }

	return ( QDate::isValid( y, m, d ) ? QDate( y, m, d ) : QDate() );
}

QString Utils::stringFromDate( const QDate& date, const QString& sep,
															 int form ){
  QString format;
  if ( form==QC::AMERICAN ) format="MM"+sep+"dd"+sep+"yyyy";
	else if ( form==QC::EUROPEAN ) format="dd"+sep+"MM"+sep+"yyyy";
	else format="yyyy"+sep+"MM"+sep+"dd";
	return date.toString( format );
}

QString Utils::shortStringFromDate( const QDate& date, const QString& sep,
																		int form ){
	QString s=stringFromDate( date, sep, form );
  return ( form==QC::YEARFIRST ? s.right( 5 ) : s.left( 5 ) );
}

const char * Utils::tcname( Table t, int field ){ 
	return NAMES[( int )t][field];
}

int Utils::tcnum( Table t, const QString& name ){
	// return the field number

	int tdx=( int )t;
	for( int i=0; i<COLS[tdx]; i++ ){
		//cout<<"field "<<i<<" is: "<<COLN[tdx][i]<<endl;
		if( NAMES[tdx][i]==name.upper() ) return i;
	}
	return -1;
}

int Utils::tcols( Table t ) { return COLS[( int )t ]; }
const char ** Utils::tnames( Table t ){ return NAMES[( int )t]; }
const ColType * Utils::ttypes( Table t ){ return TYPES[( int )t]; }
ColType Utils::tctype( Table t, int field ){ return TYPES[( int )t][field]; }
int Utils::tpk( Table t ){ return PKS[( int )t]; }

bool Utils::isLoan( const Account& a, float * interest, 
										QString * payment, int * numpays ){
	QRegExp regexp( QC::LOANREGEXP );
	bool ret=regexp.search( a[QC::AMETA].gets() )>=0;
	if( ret ){
		if( interest ) *interest=regexp.cap( QC::LOANINT ).toFloat();
		if( numpays ) *numpays=regexp.cap( QC::LOANNPR ).toInt();
		if( payment ) *payment=regexp.cap( QC::LOANPMT );
	}
	return ret;
}

bool Utils::isMarket( const Split& split, QString * shares,	QString * price ){
	QRegExp regexp( QC::MARKETREGEXP );
	bool ret=regexp.search( split[QC::SMETA].gets() )>=0;
	if( ret ){
		if( shares ) *shares=regexp.cap( QC::MARKETSHARES );
		if( price ) *price=regexp.cap( QC::MARKETPRICE );
	}	
	return ret;
}

bool Utils::isMarket( const QString& val, QString& shrs, QString& price ){
	QRegExp regexp( QC::MARKETREGEXP );
	bool ret=regexp.search( val )>=0;
	if( ret ){	
		shrs=regexp.cap( QC::MARKETSHARES );
		price=regexp.cap( QC::MARKETPRICE );
	}	
	return ret;
}


/**** the money converter ****/

const QString MonCon::ecurs=QHacc::CURRENCYSEPARATOR;

MonCon::MonCon( QHacc * e ){
	engine=e;

	factor=engine->getFP( "ALTCURRENCYFACTOR" );
	if( factor==0 ) factor=1;

	asymb=engine->getSP( "ALTCSYMBOL" );
	csymb=engine->getSP( "CSYMBOL" );

	curs=engine->getSP( "CURRENCYSEPARATOR" );

	inAlt=engine->getBP( "USEALTCURRENCY" );
	
	figurePrefs();
}

MonCon::~MonCon(){}

void MonCon::decoder( ConvertFlags from, ConvertFlags to, 
											bool& fromenginesep, bool& fromengineval,
											bool& toenginesep, bool& toengineval ) const{
	// a couple notes...
	// this function seems a bit more complicated than necessary
	// because the from and to flags might not contain a value
	// that has both a Sep and Val value. We need to take this
	// into account when figuring out what conversions are
	// necessary. If one of the flags doesn't have a value, use
	// the value from the other flag (i.e., no conversion will be
	// needed). If neither flags has the proper value, we still
	// don't need a conversion, so set the booleans appropriately

	// set the default return results
	fromenginesep=fromengineval=toenginesep=toengineval=true;

	// first, figure out value conversions
	if( from & HasVal ){
		if( from & PrefVal ) fromengineval=prefUseEVal;
		else fromengineval=( from & EngVal );
	}

	if( to & HasVal ){
		if( to & PrefVal ) toengineval=prefUseEVal;
		else toengineval=( to * EngVal );
	}
	else toengineval=fromengineval;
	
	if( !from & HasVal ) fromengineval=toengineval;

	// next, figure out separator conversions
	if( from & HasSep ){
		if( from & PrefSep ) fromenginesep=prefUseESep;
		else fromenginesep=( from & EngSep );
	}

	if( to & HasSep ){
		if( to & PrefSep ) toenginesep=prefUseESep;
		else toenginesep=( to * EngSep );
	}
	else toenginesep=fromenginesep;
	
	if( !from & HasSep ) fromenginesep=toenginesep;
}

QString MonCon::replace( QString str, const QString& find,
												 const QString& repl ){
	// need a separate function for this so we can handle d/f versions of Qt
#if QT_VERSION<320
	int pos=str.find( find );
	if( pos==-1 ) return str;
	
	int ol=find.length();
	QString s( str );
	return s.replace( pos, ol, repl );
#else
	return str.replace( find, repl );
#endif
}

QString MonCon::convert( const QString& currentval, 
												 ConvertFlags from, ConvertFlags to ) const {
	// simple case: no changes
	if( from==to ) return currentval;
	//cout<<"convert s: "<<dec<<currentval<<endl;

	// simplify the params a bit
	bool fromeval, fromesep, toeval, toesep;
	decoder( from, to, fromesep, fromeval, toesep, toeval );

	QString cv( currentval );
	
	// another simple case: not converting sums, just separators
	if( fromeval==toeval ){
		if( fromesep ){	// convert from engine's separator to alternate separator
			return replace( cv, ecurs, curs );
		}
		else{ // convert from alternate separator to engine separator
			return replace( cv, curs, ecurs );
		}
	}

	// we're out of simple cases now...we need to convert a sum and possibly
	// a currency separator. To simplify things, we'll only handle strings
	// formatted for the engine, and make some other function do everything
	if( !fromesep ) cv=replace( cv, curs, ecurs );
	return converte( basic( cv ), to );
}

QString MonCon::convert( int currentval, ConvertFlags from,
												 ConvertFlags to ) const {
	//cout<<"convert i: "<<dec<<currentval<<endl;
	// currency separator obviously doesn't matter here, so we only need to
	// figure out if we're converting the value or not
	bool fromeval, fromesep, toeval, toesep;
	decoder( from, to, fromesep, fromeval, toesep, toeval );

	// we're not converting the value anyway, so just return it
	if( fromeval==toeval ) return basic( currentval );
	else return converte( currentval, to );
}

int MonCon::converti( int currentval, ConvertFlags from,
											ConvertFlags to ) const {
	bool fromeval, fromesep, toeval, toesep;
	decoder( from, to, fromesep, fromeval, toesep, toeval );
	
	// we're not converting the value anyway, so just return it
	if( fromeval==toeval ) return currentval;
	else return ci( currentval, toeval );
}

int MonCon::converti( const QString& currentval, ConvertFlags from,
											ConvertFlags to ) const {
	if( currentval.isEmpty() ) return 0;

	// unfortunately, we always need to convert currentval to the engine
	// format before we can use it for anything.

	//cout<<"converti: "<<currentval<<"; from="<<QString::number( from, 2 )
	//	<<"; engine="<<QString::number( Engine, 2 )<<endl;
	QString cv=convert( currentval, from, Engine );
	//cout<<"  now, cv="<<cv<<"; to="<<QString::number( to, 2 )<<endl;
	//cout<<basic( cv )<<" "<<converti( basic( cv ), Engine, to )<<endl;
	return converti( basic( cv ), Engine, to );
}


QString MonCon::converte( int currentval, ConvertFlags to ) const {
	// this function expects that the to param is not any of the Pref*
	// ConvertFlags. It must be either Eng* or Alt* ones. Also, it will
	// convert val in all cases
	//cout<<"converte: "<<dec<<currentval<<endl;
	bool toesep, toeval;
	if( to & HasSep ) toesep=( to & EngSep );
	else toesep=false;
	
	if( to & HasVal ) toeval=( to & EngVal );
	else toeval=false;

	QString ret=basic( ci( currentval, toeval ) );
	return( toesep ? ret : replace( ret, ecurs, curs ) );
}

int MonCon::ci( int currentval, bool toeval ) const {
	// this function just multiplies or divides by factor. So, if toeval
	// is true, we assume the currentval is the alternate sum and divide
	// else it must be the real sum, so we multiply
	if( toeval ) return ( int ) ( ( float )currentval/factor );
	else return ( int )( ( float )currentval*factor );
}

int MonCon::basic( const QString& st ){
	// convert the string to dollars and cents

	// if we hit something that can't be converted using toInt(), we want to 
	// return 0 to the calling function
	bool ok=true;

	// we only want to deal with positive values here; we'll add the minus later
	bool neg=false;
	QString s;
	if( st.startsWith( "-" ) ){
		neg=true;
		s=st.mid( 1 );
	}
	else if( st.startsWith( QHacc::CURRENCYSEPARATOR ) ) s="0"+st;
	else s=st;	

	int i=s.find( QHacc::CURRENCYSEPARATOR );
	int ret=s.left( i ).toInt( &ok )*QHacc::ROLLOVER;
	
	if( i!=-1 ){ // we have a decimal point to worry about
		// only look at the most significant DECIMALS places
		QString cents=s.mid( ++i, QC::DECIMALS ); 
		int l=cents.length();
		
		bool tok=ok;
		for( i=1; i<=l; i++ ){
			ret+=cents.mid( i-1, 1 ).toInt( &tok )*( int )pow( ( double )10, 
																												 QC::DECIMALS-i );
			ok=( ok && tok );
		}
	}
	
	if( !ok ) ret=0;

	if( neg ) ret=0-ret;
	
	//cout<<"converting "<<s<<" to "<<ret<<" cents"<<endl;
	return ret;
}

QString MonCon::basic( int cents ){
	// convert the cents to a string of dollars and cents
	
	// we want only positive values for our comparisons; we'll add the neg later
	bool neg=false;
	if( cents<0 ){
		neg=true;
		cents=0-cents;
	}
	
	// ROLLOVER will not be a double, GUARANTEED!
	int dollars=cents/QHacc::ROLLOVER;
	int ncents=cents%QHacc::ROLLOVER;
	
	QString ret=QString::number( dollars )+QHacc::CURRENCYSEPARATOR;
	if( ncents<10 ) ret.append( "0" );
	ret.append( QString::number( ncents ) );
	
	// now that we have the string, put a minus sign if necessary
	if( neg ) ret.prepend( "-" );
	
	//cout<<"converting "<<cents<<" to string: "<<ret<<endl;
	return ret;
}

const QString& MonCon::symbol() const { return ( inAlt ? asymb : csymb ); }
const QString& MonCon::cursep() const {	return ( inAlt ? curs : ecurs ); }

void MonCon::figurePrefs(){
	if( inAlt ){
		// realistically, if we want the alternate currency but we haven't
		// overridden the currency separator, it's more efficient to mark
		// the separator as EngSep rather than AltSep. Same with the Vals
		prefUseESep=( curs==ecurs );
		prefUseEVal=( factor==1 );
	}
	else prefUseESep=prefUseEVal=true;
}

void MonCon::changedP( const QString& p, float v ){
	if( p=="ALTCURRENCYFACTOR" ){
		if( v==0 ) factor=1;
		else factor=v;
		figurePrefs();
	}
}

void MonCon::changedP( const QString& p, bool b ){
	if( p=="USEALTCURRENCY" ){
		inAlt=b;
		figurePrefs();
	}
}

void MonCon::changedP( const QString& p, QString s ){
	if( p=="ALTCSYMBOL" ) asymb=s;
	else if( p=="CSYMBOL" ) csymb=s;
	else if ( p=="CURRENCYSEPARATOR" ){
		curs=s;
		figurePrefs();
	}
}

MonCon::MonCon( const MonCon& ){} // no copies are allowed
MonCon& MonCon::operator=( const MonCon& ){
	// no assignments are allowed, either
	return *this;
}
