/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "resultset.h"
#include "qhaccutils.h"
#include "qhacctablerows.h"

#include <iostream>

/*****************************************************************/
/* QHACCRESULTSET                                                */
/*   This is the basis of all data movement in QHacc. It is an   */
/*   add-only, dynamically-growing structure. It is how          */
/*   databases return data to the calling functions              */
/*                                                               */
/*****************************************************************/

// these are error codes
const int QHaccResultSet::VALID=0;
const int QHaccResultSet::COLCOUNT=-1;
const int QHaccResultSet::UNKNOWN=-2;
const int QHaccResultSet::LOWERROR=UNKNOWN;

const char * QHaccResultSet::ERRORS[]={ "no error",
																				"column count does not match",
																				"unknown error" };

const char * QHaccResultSet::error( int code ) const {
	if( code<LOWERROR ) return newerror( code-LOWERROR );
	else return ERRORS[-code];
}

const char * QHaccResultSet::newerror( int ) const { return error( UNKNOWN ); }

QHaccResultSet::QHaccResultSet(	int sz, const ColType * cs, uint, uint ) {
	data.clear();
	cols=sz;

	types=new ColType[cols];
	if( cs ) for( int i=0; i<cols; i++ )	types[i]=cs[i];
	else for( int i=0; i<cols; i++ )	types[i]=CTSTRING;
}

QHaccResultSet::QHaccResultSet( const QHaccResultSet& model ){
	data.clear();
	for( uint i=0; i<model.rows(); i++ )
		data.push_back( new TableRow( model[i] ) );

	cols=model.cols;

	types=new ColType[cols];
	if( model.types ) for( int i=0; i<cols; i++ )	types[i]=model.types[i];
	else for( int i=0; i<cols; i++ )	types[i]=CTSTRING;
}

QHaccResultSet& QHaccResultSet::operator+=( const TableRow& row ){
	add( row );
	return *this;
}

QHaccResultSet& QHaccResultSet::operator+=( const QHaccResultSet& donor ){
	load( &donor );
	return *this;
}

QHaccResultSet& QHaccResultSet::operator=( const QHaccResultSet& model ){
	if( &model!=this ){
		data.clear();
		for( uint i=0; i<model.rows(); i++ )
			data.push_back( new TableRow( model[i] ) );

		cols=model.cols;
		
		delete [] types;
		types=new ColType[cols];
		
		if( model.types ) for( int i=0; i<cols; i++ )	types[i]=model.types[i];
		else for( int i=0; i<cols; i++ ) types[i]=CTSTRING;
	}
	return *this;
}

QHaccResultSet::~QHaccResultSet(){
	delete [] types;
	data.clear();
}

const TableRow& QHaccResultSet::at( uint i ) const { return *data[i]; }
const TableRow& QHaccResultSet::operator[]( uint i ) const { return *data[i]; }

bool QHaccResultSet::load(const QHaccResultSet * donor ){
	// load copies of all the rows in donor table
	/*
	cout<<"loading from donor:"<<endl;
	for( uint i=0; i<donor->rows(); i++ ) cout<<"  "<<i<<": "<<donor->at( i ).toString()<<endl;
	cout<<"to "<<endl;
	for( uint i=0; i<capacity; i++ ) cout<<"  "<<i<<": "<<data[i].toString()<<endl;
	cout<<"--"<<endl;
	*/
	uint drows=donor->rows();
	startLoad( drows );
	for( uint i=0; i<drows; i++ )	add( donor->at( i ) );
	stopLoad();
	return true; // always works?
}

void QHaccResultSet::stopLoad(){ istopLoad(); }
void QHaccResultSet::startLoad( uint adds ){ istartLoad( adds ); }
void QHaccResultSet::istopLoad(){}
void QHaccResultSet::istartLoad( uint ){}

void QHaccResultSet::loadRow( const QString& str ) {
	add( TableRow( str, cols ) );
}

bool QHaccResultSet::isEmpty() const { return data.empty(); }
uint QHaccResultSet::rows() const { return data.size(); }
uint QHaccResultSet::trows() const { return data.capacity(); }
uint QHaccResultSet::columns() const { return cols; }
ColType QHaccResultSet::coltype( int i ) const { 
	if( i>-1 && i<cols ) return types[i]; 
	return CTNULL;
}

int QHaccResultSet::verifyRow( const TableRow& row ) const {
	// verify that the given row has the same number/type fo columns
	int reason=VALID;
	if( row.cols()!=cols ){
		reason=COLCOUNT;
	}

	std::ostream * str=0;
	if( reason<VALID && Utils::error( Utils::ERROPER, str ) ){
		*str<<"INVALID: "<<error( reason )<<endl
				<<"\t"<<row.toString()<<endl;
	}

	return reason;
}

int QHaccResultSet::add( const TableRow& row ){
	// add a new row...first verify that the row is valid
	int valid=verifyRow( row );
	if( valid<VALID )	return valid;

	//if( size+1>=capacity ) resizeTo( size+growBy );
	//cout<<"adding "<<row.toString()<<endl
	//		<<"  size is "<<size<<"; cap is "<<capacity<<"; data["<<size<<"] is ->"
	//  	<<data[size].toString()<<"<-"<<endl;
	//if( data[size] ) delete data[size];
	
	uint ssz=data.size();
	data.push_back( new TableRow( row ) );
	iadd( ssz );
	return VALID;
}

// FIXME: future releases should remove these two functions
void QHaccResultSet::resizeTo( uint ){ iresize(); }
void QHaccResultSet::iresize(){}

void QHaccResultSet::iadd( uint ){}
