/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gpgplugin.h"
#include "qhacctable.h"
#include "qhaccutils.h"

#include <qtextstream.h>
#include <qdatastream.h>
#include <qfile.h>

#include <stdlib.h>
#include <unistd.h>

#include <iostream>
#include <memory>
using namespace std;


// plugin factory calls
extern "C" {
	QHaccPlugin * create(){	return new GPGDBPlugin; }
	void destroy( GPGDBPlugin * p ){ delete p; }
}

const GPGInfo GPGDBPlugin::pinfo;

GPGDBPlugin::GPGDBPlugin() : LocalFileDBPlugin(){
	madectx=false;
}

GPGDBPlugin::~GPGDBPlugin(){ 
	if( madectx )	gpgme_release( gpgctx ); 
}

const PluginInfo& GPGDBPlugin::info() const { return pinfo; }

bool GPGDBPlugin::iconnect( QHacc *, const QString&, QString& error ){
	madectx=( GPG_ERR_NO_ERROR==gpgme_new( &gpgctx ) );
	if( madectx ){
		gpgme_set_passphrase_cb( gpgctx, GPGDBPlugin::passphrase_cb, 
														 ( void * )this );
		gpgme_set_armor( gpgctx, 1 );
		return true;
	}
	else{
		error="error creating GPG context.";
		return false;
	}
}

bool GPGDBPlugin::iload( QString& err ){
	// need to do the load ourselves, because a wrong passphrase should
	// cause QHacc to exit, but the localfiledbplugin will continue anyway

	bool ret=true;
  for( int i=0; i<QC::NUMTABLES; i++ ) {
    dbs[i]->clear();
    ret=( ret && gpgloadt( *dbs[i], home+"/"+QC::TABLENAMES[i], err ) );
  }
  return ret;
}

bool GPGDBPlugin::gpgloadt( QHaccTable& tbl, const QString& fn, QString& err ){
	gpgme_data_t cipher, plain;
	gpgme_error_t rslt;

	rslt=gpgme_data_new( &plain ); // is there really anything to check here?
	rslt=gpgme_data_new_from_file( &cipher, fn, 1 ); // last arg must be non-zero
	
	if( GPG_ERR_INV_VALUE==rslt ){
		err="error opening file";
		return false;
	}

	// if we get to here, at least we know the file should open
	rslt=gpgme_op_decrypt( gpgctx, cipher, plain );

	if( GPG_ERR_NO_ERROR!=rslt ){
		gpgme_data_release( cipher );
		gpgme_data_release( plain );

		if( GPG_ERR_DECRYPT_FAILED==rslt ){
			err="decryption failed";
			return false;
		}
		else if( GPG_ERR_NO_DATA==rslt ){
			err="no encryption found";
			return false;
		}
		else{
			err="wrong passphrase";
			return false;
		}
	}

	// FIXME: something is wrong when this buffer has to be re-filled
	QString buffer;
	QTextStream wdata( buffer, IO_WriteOnly|IO_Append );
	const uint BSZ=1000000;
	char * tbuff=new char[BSZ];
	for( uint i=0; i<BSZ; i++ ) tbuff[i]=0;
	
	int nread=0;
	gpgme_data_seek( plain, 0, SEEK_SET );
	while( ( nread=gpgme_data_read( plain, (void *)tbuff, BSZ ) )>0 ){
		if( ( uint )nread<BSZ ) tbuff[nread]='\0';
		wdata<<tbuff;
	}
	delete [] tbuff;

	QTextStream rdata( buffer, IO_ReadOnly );
	tbl.startLoad();
	while( !rdata.atEnd() )	tbl.loadRow( rdata.readLine() );
	tbl.stopLoad();	 

	std::ostream * str=0;
	if( Utils::debug( Utils::DBGMINOR, str ) ){
		uint rsz=tbl.rows();
		*str<<"loaded "<<rsz<<" row"<<( rsz>1 ? "s" : "" )<<" from "
				<<fn<<" into "<<tbl.getName()<<endl;
	}
	
	gpgme_data_release( cipher );
	gpgme_data_release( plain );

	return true;
}

bool GPGDBPlugin::isavet( QHaccTable& tbl, const QString& fn, QString& error ){
	
	gpgme_data_t cipher, plain;
	gpgme_error_t rslt;

	QString buff;
	const uint RWS=tbl.rows();
	for( uint i=0; i<RWS; i++ )	buff+=tbl[i].toString()+"\n";

	gpgme_data_new_from_mem( &plain, buff, buff.length(), 1 );
	rslt=gpgme_data_new( &cipher ); // is there really anything to check here?
	rslt=gpgme_op_encrypt( gpgctx, 0, (gpgme_encrypt_flags_t)0, plain, cipher );

	if( GPG_ERR_NO_ERROR!=rslt ){
		gpgme_data_release( cipher );
		gpgme_data_release( plain );

		error="could not encrypt data";
		return false;
	}

	const uint BSZ=1000000;
	char * tbuff=new char[BSZ];
	for( uint i=0; i<BSZ; i++ ) tbuff[i]=0;

	QFile file( fn );
	if( file.open( IO_WriteOnly ) ){
		QDataStream out( &file );

		int nread=0;		
		gpgme_data_seek( cipher, 0, SEEK_SET );
		while( ( nread=gpgme_data_read( cipher, (void *)tbuff, BSZ ) )>0 ){
			out.writeRawBytes( tbuff, nread );
		}
		delete [] tbuff;
		file.close();
	}
	else{
		error="could not write "+tbl.getName()+" to "+fn;
		gpgme_data_release( cipher );
		gpgme_data_release( plain );
		return false;
	}

	gpgme_data_release( cipher );
	gpgme_data_release( plain );

	return true;
}

GPGInfo::GPGInfo(){
	description="Gnu Privacy Guard";
	stubby="GPG";
	targ=DIRECTORY;
	atom=true;
	raw=false;

	auto_ptr<QHaccResultSet> tp=LocalFileDBPlugin::pinfo.prefs();
	piprefs=tp;
}

gpgme_error_t GPGDBPlugin::passphrase_cb( void * hook, const char * uid_hint,
																					const char * passphrase_info, 
																					int last_was_bad, int fd ){
	const QString& str=( ( GPGDBPlugin *)hook )->gpgpass( uid_hint,
																												passphrase_info, 
																												last_was_bad );
	write( fd, str, str.length() );
	write( fd, "\n", 1 );
	return 0;
}

const QString& GPGDBPlugin::gpgpass( const char * uid_hint,
																		 const char *, int last_was_bad ){
	if( last_was_bad || passphrase.isEmpty() ){
		if( uid_hint!=0 ) cout<<uid_hint<<endl;
		const char * passer=getpass( "GPG passphrase: " );
		passphrase=QString( passer );
	}
	return passphrase;
}
